/**
 * plugin.js
 *
 * Copyright, Moxiecode Systems AB
 * Released under LGPL License.
 *
 * License: http://www.tinymce.com/license
 * Contributing: http://www.tinymce.com/contributing
 */

/* global tinymce:true, tinyMCE:true */

tinymce.PluginManager.add('oximage', function (editor) {

    var self = this,
        cache = {};

    this.getPendingDeferreds = function (ids) {
        return _(ids)
            .chain()
            .map(function (id) {
                return cache[id];
            })
            .compact()
            .value();
    };

    /*
     * Uploads a blob image file and inserts it at the cursor position.
     */
    this.uploadBlob = function (blob, opt) {
        opt = opt || {};
        var id = _.uniqueId('upload-image-'),
            windowUrl = window.URL || window.webkitURL,
            objectURL = windowUrl.createObjectURL(blob);

        self.addImage({ src: objectURL, id: id, from_clipboard: opt.from_clipboard });

        require(['io.ox/mail/write/inline-images', 'io.ox/core/notifications'], function (inline, notifications) {
            cache[id] = inline.api.inlineImage({ file: blob }).then(
                function success(response) {
                    var url = inline.api.getInsertedImageUrl(response),
                        ed = tinyMCE.activeEditor;

                    if (!ed) return;
                    // use getBody so it works in old and new mail editor
                    var el = $(ed.getBody()),
                        currentImage;

                    if (!el) return;
                    el.trigger('beforeInlineImage', response.data[0]);
                    // tinymce paste plugin seems to remove ids somehow, we added them as classes too
                    currentImage = $('#' + id + ', .' + id, el);

                    currentImage.attr('src', url).removeAttr('data-pending id from_clipboard').css('maxWidth', '100%').addClass('aspect-ratio');
                    el.trigger('addInlineImage', response.data[0]);

                    // Replace old image in the history of the undomanager
                    _(tinymce.activeEditor.undoManager.data).each(function (stage) {
                        var $content = $(stage.content);

                        $content.find('#' + id + ', .' + id).replaceWith(currentImage.clone());
                        stage.content = $content.prop('outerHTML');
                    });

                    windowUrl.revokeObjectURL(objectURL);
                    delete cache[id];
                },
                function (error) {
                    var ed = tinyMCE.activeEditor;
                    // clean up placeholder
                    if (ed) {
                        $('#' + id + ', .' + id, $(ed.getElement())).remove();
                        // Remove old image in the history of the undomanager
                        _(ed.undoManager.data).each(function (stage) {
                            var $content = $(stage.content);

                            $content.find('#' + id + ', .' + id).remove();
                            stage.content = $content.prop('outerHTML');
                        });
                    }

                    notifications.yell(error);
                }
            );
        });
    };

    this.addImage = function (opt) {
        var ed = tinyMCE.activeEditor;
        if (!ed) return;
        ed.undoManager.transact(function () {
            ed.focus();
            // tinymce pasteplugin seems to strip ids so provide as class for now
            var attr = { src: opt.src, style: 'max-width: 100%;', class: 'aspect-ratio' + opt.id ? opt.id : '', id: opt.id, alt: '', 'data-pending': 'true' };
            if (opt.from_clipboard) {
                attr.from_clipboard = true;
            }
            ed.selection.setContent(ed.dom.createHTML('img', attr));
        });
    };

    editor.addButton('image', {
        icon: 'image',
        title: 'Insert image',
        onPostRender: (function () {

            function change() {

                var file = $(this);

                require(['io.ox/core/notifications', 'gettext!io.ox/mail'], function (notifications, gt) {

                    if (!(/\.(gif|bmp|tiff|jpe?g|gmp|png)$/i).test(file.val())) {
                        notifications.yell('error', gt('Please select a valid image File to insert'));
                    } else if (file[0].files && tinyMCE.activeEditor) {
                        self.uploadBlob(file[0].files[0]);
                    }
                });
            }

            return function onPostRender() {
                var self = this;
                var fileInput = $('<input type="file" name="file" capture="camera">').css('display', 'none')
                    .on('change', { editor: editor }, change);

                self.off('click');

                $('button', self.$el)
                    .on('click touchstart', function () {
                        fileInput[0].value = '';
                        fileInput.trigger('click');
                    })
                    .before(fileInput);
            };
        }())
    });

    /*
    Adds dropdown for upload and drive picker.
    This still needs some work and is deactivated for now.

    editor.addButton('image', {
        type: 'menubutton',
        icon: 'image',
        tooltip: 'Insert image',
        menu: [
            {
                text: 'Upload',
                onclick: function () {
                    require(['io.ox/mail/compose/inline-images'], function (inlineimages) {
                        inlineimages.show().done(function (url) {
                            showDialog(url);
                        });
                    });
                }
            },
            {
                text: 'Drive',
                onclick: function() {
                    // open ox filepicker
                    require(['io.ox/files/filepicker', 'io.ox/files/api', 'gettext!io.ox/files']).done(function (Picker, fileAPI, gt) {
                        var picker = new Picker({
                            primaryButtonText: gt('Add'),
                            multiselect: false,
                            filter: function (file) {
                                return (/\.(gif|bmp|tiff|jpe?g|gmp|png)$/i).test(file.filename);
                            },
                        });
                        picker.done(function (files) {
                            var url = '';
                            if (files[0]) {
                                url = fileAPI.getUrl(files[0], 'view');
                            }
                            showDialog(url);
                        });
                    });
                }
            }
        ]
    });
    */
});
