/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.tools.exec;

import java.io.File;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.Mutable;
import org.apache.commons.lang3.mutable.MutableObject;

import com.openxchange.office_communication.tools.exec.impl.ExecutableStreamReader;
import com.openxchange.office_communication.tools.exec.impl.ExecutableWatch;
import com.openxchange.office_communication.tools.exec.impl.GlobalPidProcessHandler;

//==============================================================================
/**
 */
public class Executable implements Runnable
{
    //--------------------------------------------------------------------------
    public Executable ()
    {}
    
    //--------------------------------------------------------------------------
    /** define new working path for following process execution.
     *
     *  @param  sWorkingPath
     *          new working path.
     */
    public void setWorkingPath (final String sWorkingPath)
        throws Exception
    {
        // don't check if it exists here ...
        // its done before execution !
        m_sWorkingPath = sWorkingPath;
    }
    
    //--------------------------------------------------------------------------
    /** define new working path for following process execution.
     *
     *  @param  sExecutable
     *          new working path.
     */
    public void setExecutable (final String sExecutable)
        throws Exception
    {
        // don't check if it exists here ...
        // its done before execution !
        m_sExe = sExecutable;
    }
    
    //--------------------------------------------------------------------------
    public void setForward4StdOut (final OutputStream aForward)
        throws Exception
    {
    	m_aStdOutForward = new PrintStream(aForward);
    }

    //--------------------------------------------------------------------------
    public void setForward4StdErr (final OutputStream aForward)
        throws Exception
    {
    	m_aStdErrForward = new PrintStream(aForward);
    }

    //--------------------------------------------------------------------------
    /** clear list of all currently set command line arguments.
     */
    public void clearArguments ()
        throws Exception
    {
        mem_Arguments ().clear();
    }

    //--------------------------------------------------------------------------
    /** add new command line argument.
     * 
     *  @param  sArgument
     *          name of argument (may including "-" or "--" or ...)
     *          
     *  @param  sValue
     *          value for those argument
     *          In case argument does not have any value passing
     *          null or empty value will be allowed.
     */
    public void addArgument(final String sArgument,
                            final String sValue   )
        throws Exception
    {
        if ( ! StringUtils.isEmpty(sArgument))
            mem_Arguments ().add(sArgument);
        if ( ! StringUtils.isEmpty(sValue))
            mem_Arguments ().add(sValue);
    }
    
    //--------------------------------------------------------------------------
    public void addArgument(final String sArgument)
        throws Exception
    {
        if ( ! StringUtils.isEmpty(sArgument))
            mem_Arguments ().add(sArgument);
    }
    
    //--------------------------------------------------------------------------
    public void registerWatcher (final IExecutableWatcher iWatcher)
        throws Exception
    {
    	ExecutableWatch aWatch = mem_Watch ();
    	aWatch.addWatcher(iWatcher);
    }

    //--------------------------------------------------------------------------
    public void runAsync(final CountDownLatch aSync)
    	throws Exception
    {
    	synchronized(this)
    	{
            if (m_aProcess != null)
                throw new Exception ("Process still running.");

        	// accept null too !
        	// will be handled within run() method right ...

    		m_aSync = aSync;
    	}

    	new Thread(this).start ();
    }

    //--------------------------------------------------------------------------
    @Override
    public void run()
    {
        try
        {
	        if (isAlive ())
	        	return;
	        	
            final List< String > aCmdLine = impl_getCmdLine ();
            //System.out.println ("CMD LINE : "+aCmdLine);
            ProcessBuilder aBuilder = new ProcessBuilder (aCmdLine);
            
            if ( ! StringUtils.isEmpty(m_sWorkingPath))
                aBuilder.directory(new File (m_sWorkingPath));
            
            ExecutableStreamReader aStdOut = mem_StdOut ();
            ExecutableStreamReader aStdErr = mem_StdErr ();

        	Mutable< Process > aProcess = new MutableObject< Process > ();
        	Mutable< Integer > nPid     = new MutableObject< Integer > ();

        	GlobalPidProcessHandler aPidHandler = GlobalPidProcessHandler.get();
        	aPidHandler.startChildProcess(aBuilder, aProcess, nPid);
        	
        	m_aProcess    = aProcess.getValue();
        	m_nProcessPid = nPid    .getValue();
        	
            aStdOut.bind2Input (m_aProcess.getInputStream());
            aStdErr.bind2Input (m_aProcess.getErrorStream());

        	aStdOut.bind2Output(mem_StdOutForward ());
            aStdErr.bind2Output(mem_StdErrForward ());

            aStdOut.bind2Watch (mem_Watch());
            aStdErr.bind2Watch (mem_Watch());
            
            new Thread (aStdOut).start();
            new Thread (aStdErr).start();

            int nResult = 0;
            synchronized (m_aProcess)
            {
                nResult = m_aProcess.waitFor();
            }
            
            m_nLastResult = nResult;
        }
        catch (Throwable ex)
        {
            Logger.getLogger(Logger.GLOBAL_LOGGER_NAME).log(Level.SEVERE, ex.getMessage(), ex);
            m_nLastResult = 1;
        }
        
        if (m_aSync != null)
        	m_aSync.countDown();
    }
    
    //--------------------------------------------------------------------------
    public int getPid ()
    	throws Exception
    {
    	return m_nProcessPid;
    }
    
    //--------------------------------------------------------------------------
    public boolean isAlive ()
    	throws Exception
    {
    	if (m_aProcess == null)
    		return false;
    	
    	if (m_nProcessPid == GlobalPidProcessHandler.INVALID_PID)
    		return false;
    	
    	boolean bIsAlive = GlobalPidProcessHandler.get ().isChildProcessAlive(m_nProcessPid);
    	return  bIsAlive;
    }

    //--------------------------------------------------------------------------
    public void kill ()
        throws Exception
    {
    	if (m_nProcessPid != GlobalPidProcessHandler.INVALID_PID)
        	GlobalPidProcessHandler.get ().killChildProcess(m_nProcessPid);
    	else
    	if (m_aProcess != null)
    		m_aProcess.destroy();
    	
    	m_nProcessPid = GlobalPidProcessHandler.INVALID_PID;
    	m_aProcess    = null;
    	m_aStdOut     = null;
    	m_aStdErr     = null;
    }
    
    //--------------------------------------------------------------------------
    public int getLastResult ()
    	throws Exception
    {
    	return m_nLastResult;
    }
    
    //--------------------------------------------------------------------------
    @Override
    public String toString ()
    {
    	final StringBuffer sString = new StringBuffer (256);
    	
    	sString.append (super.toString ());
    	sString.append (" ["             );
    	sString.append (m_sWorkingPath   );
    	sString.append ("] "             );
    	sString.append (m_sExe           );
    	
    	try
    	{
	    	final List< String > lArgs = mem_Arguments ();
	    	for (final String sArg : lArgs)
	    	{
	    		sString.append (" " );
	    		sString.append (sArg);
	    	}
    	}
    	catch (final Throwable ex)
    	{}
    	
    	return sString.toString ();
    }
    
    //--------------------------------------------------------------------------
    private List< String > impl_getCmdLine ()
        throws Exception
    {
        List< String > lArgs    = mem_Arguments ();
        int            c        = lArgs.size();
        List< String > lCmdLine = new ArrayList< String >(c+1);
        
        lCmdLine.add   (m_sExe);
        lCmdLine.addAll(lArgs );
        
        return lCmdLine;
    }

    //--------------------------------------------------------------------------
    private List< String > mem_Arguments ()
        throws Exception
    {
        if (m_lArguments == null)
            m_lArguments = new ArrayList< String >(10);
        return m_lArguments;
    }
    
    //--------------------------------------------------------------------------
    private ExecutableStreamReader mem_StdOut ()
        throws Exception
    {
        if (m_aStdOut == null)
            m_aStdOut = new ExecutableStreamReader ();
        return m_aStdOut;
    }
    
    //--------------------------------------------------------------------------
    private ExecutableStreamReader mem_StdErr ()
        throws Exception
    {
        if (m_aStdErr == null)
            m_aStdErr = new ExecutableStreamReader ();
        return m_aStdErr;
    }

    //--------------------------------------------------------------------------
    private PrintStream mem_StdOutForward ()
        throws Exception
    {
        if (m_aStdOutForward == null)
        	m_aStdOutForward = System.out;
        return m_aStdOutForward;
    }

    //--------------------------------------------------------------------------
    private PrintStream mem_StdErrForward ()
        throws Exception
    {
        if (m_aStdErrForward == null)
        	m_aStdErrForward = System.err;
        return m_aStdErrForward;
    }

    //--------------------------------------------------------------------------
    private ExecutableWatch mem_Watch ()
        throws Exception
    {
    	if (m_aWatch == null)
    		m_aWatch = new ExecutableWatch ();
    	return m_aWatch;
    }
    
    //--------------------------------------------------------------------------
    private String m_sWorkingPath = null;
    
    //--------------------------------------------------------------------------
    private String m_sExe = null;
    
    //--------------------------------------------------------------------------
    private List< String > m_lArguments = null;
    
    //--------------------------------------------------------------------------
    private Process m_aProcess = null;
    
    //--------------------------------------------------------------------------
    private int m_nProcessPid = GlobalPidProcessHandler.INVALID_PID;
    
    //--------------------------------------------------------------------------
    private ExecutableStreamReader m_aStdOut = null;
    
    //--------------------------------------------------------------------------
    private ExecutableStreamReader m_aStdErr = null;
    
    //--------------------------------------------------------------------------
    private int m_nLastResult = 0;

    //--------------------------------------------------------------------------
    private CountDownLatch m_aSync = null;

    //--------------------------------------------------------------------------
    private ExecutableWatch m_aWatch = null;

    //--------------------------------------------------------------------------
    private PrintStream m_aStdOutForward = null;

    //--------------------------------------------------------------------------
    private PrintStream m_aStdErrForward = null;
}
