/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.cluster_management.core.impl;

import java.awt.Dimension;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Iterator;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.SwingConstants;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openxchange.office_communication.cluster_management.core.ClusterState;
import com.openxchange.office_communication.cluster_management.core.EClusterState;
import com.openxchange.office_communication.cluster_management.core.IClusterManager;
import com.openxchange.office_communication.cluster_management.core.IClusterStateNotification;

//=============================================================================
public class ClusterManagerUI
{
	//-------------------------------------------------------------------------
	private static final Logger LOG = LoggerFactory.getLogger(ClusterManagerUI.class);

	//-------------------------------------------------------------------------
	public ClusterManagerUI ()
		throws Exception
	{
	}

	//-------------------------------------------------------------------------
	public void setManager (final IClusterManager iManager)
	    throws Exception
	{
		m_iManager = iManager;
	}

	//-------------------------------------------------------------------------
	public void show ()
		throws Exception
	{
		impl_initUI ();
		
		m_aWINMain.setVisible(true);
	}
	
	//-------------------------------------------------------------------------
	private void impl_initUI ()
	    throws Exception
	{
		if (m_aWINMain != null)
			return;
		
		impl_createUI          ();
		impl_layoutUI          ();
		impl_registerUIHandler ();
	}
	
	//-------------------------------------------------------------------------
	private void impl_createUI ()
		throws Exception
	{
		final JFrame aWINMain = new JFrame ();
		aWINMain.setLayout               (null                );
//		aWINMain.setAlwaysOnTop          (true                );
		aWINMain.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		
		final JButton aBTStop = new JButton ();
		aBTStop.setText          ("Stop");
		aBTStop.setToolTipText   ("stop the node manager and all sub processes ...");
		aBTStop.setEnabled       (false );
		
		final JLabel aLBClusterState = new JLabel ();
		aLBClusterState.setText        ("Cluster State : ...");
		aLBClusterState.setToolTipText ("show the actual state of the cluster ...");

		final JLabel aLBAppStates = new JLabel ();
		aLBAppStates.setText               ("App States :\n ..." );
		aLBAppStates.setToolTipText        ("show the actual state of all apps within the cluster ...");
		aLBAppStates.setHorizontalAlignment(SwingConstants.LEFT);
		aLBAppStates.setVerticalAlignment  (SwingConstants.TOP );

		aWINMain.add (aBTStop        );
		aWINMain.add (aLBClusterState);
		aWINMain.add (aLBAppStates   );
		
		m_aWINMain        = aWINMain       ;
		m_aBTStop         = aBTStop        ;
		m_aLBClusterState = aLBClusterState;
		m_aLBAppStates    = aLBAppStates   ;
	}
	
	//-------------------------------------------------------------------------
	private void impl_layoutUI ()
		throws Exception
	{
		int nWidth  = 800;
		int nHeight = 600;
		
		m_aWINMain.setSize (nWidth, nHeight);
		
		impl_layoutControl (m_aBTStop        , 25,   5,       200,  50);
		impl_layoutControl (m_aLBClusterState, 25,  60, nWidth-50,  50);
		impl_layoutControl (m_aLBAppStates   , 25, 115, nWidth-50, 400);
	}
	
	//-------------------------------------------------------------------------
	private void impl_layoutControl (final JComponent aControl,
									 final int        nX      ,
									 final int        nY      )
	    throws Exception
	{
		impl_layoutControl (aControl, nX, nY, 0, 0);
	}
	
	//-------------------------------------------------------------------------
	private void impl_layoutControl (final JComponent aControl,
									 final int        nX      ,
									 final int        nY      ,
									 final int        nWidth  ,
									 final int        nHeight )
	    throws Exception
	{
		final Insets    aMainInsets  = m_aWINMain.getInsets();
		      Dimension aControlSize = null;
		      
		if (nWidth > 0 && nHeight > 0)
			aControlSize = new Dimension (nWidth, nHeight);
		else
			aControlSize = aControl.getPreferredSize();
		
		aControl.setBounds(aMainInsets .left + nX,
						   aMainInsets .top  + nY,
						   aControlSize.width    ,
						   aControlSize.height   );
	}

	//-------------------------------------------------------------------------
	private void impl_registerUIHandler ()
		throws Exception
	{
		m_iManager.addStateListener(new IClusterStateNotification ()
		{
			@Override
			public void notifyState(final ClusterState aState)
				throws Exception
			{
				impl_updateClusterState (aState);
			}
		});
		
		m_aBTStop.addActionListener(new ActionListener ()
		{
			@Override
			public void actionPerformed(ActionEvent e)
			{
				impl_stop ();
			}
		});
	}
	
	//-------------------------------------------------------------------------
	private void impl_updateClusterState (final ClusterState aState)
		throws Exception
	{
		final EClusterState eState = aState.getClusterState();
		
		if (eState != null)
			m_aLBClusterState.setText ("Cluster State : "+eState);
		
		if (eState == EClusterState.E_RUNNING)
			m_aBTStop.setEnabled(true );
		else
			m_aBTStop.setEnabled(false);
		
		impl_showAppStates (aState);
	}
	
	//-------------------------------------------------------------------------
	private void impl_showAppStates (final ClusterState aState)
	    throws Exception
	{
		final StringBuffer       sAppStates          = new StringBuffer (256);
		final Iterator< String > lApps               = aState.listApps().iterator();
		      boolean            bAppStatesAvailable = false;
		
		sAppStates.append ("<html>App States<br/>");
		sAppStates.append ("<table border='0'>"   );
		while (lApps.hasNext())
		{
			final String  sApp      = lApps.next();
			final String  sState    = aState.getAppState(sApp);
			final Integer nRestarts = aState.getAppStartCount(sApp);
			
			sAppStates.append ("<tr><td>"+sApp+"</td><td>"+sState+"</td><td>"+nRestarts+"</td></tr>");
			
			bAppStatesAvailable = true;
		}
		sAppStates.append ("</table>");
		sAppStates.append ("</html>" );
		
		if (bAppStatesAvailable)
			m_aLBAppStates.setText (sAppStates.toString ());
	}
	
	//-------------------------------------------------------------------------
	private void impl_stop ()
	{
		try
		{
			if (m_iManager != null)
				m_iManager.stop();
		}
		catch (Throwable ex)
		{
			throw new RuntimeException (ex);
		}
	}
	
	//-------------------------------------------------------------------------
	private IClusterManager m_iManager = null;

	//-------------------------------------------------------------------------
	private JFrame m_aWINMain = null;

	//-------------------------------------------------------------------------
	private JButton m_aBTStop = null;

	//-------------------------------------------------------------------------
	private JLabel m_aLBClusterState = null;

	//-------------------------------------------------------------------------
	private JLabel m_aLBAppStates = null;
}
