/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools.message;

import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.TimeZone;
import java.util.UUID;
import org.apache.commons.io.IOUtils;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.office.tools.json.JSONDebugWriter;
import com.openexchange.office.tools.json.JSONHelper;
import com.openexchange.tools.encoding.Base64;

/**
 * {@link RealtimeHelper}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class OperationHelper {

    final static private String m_actionsKey = "actions";
    final static private String m_operationsKey = "operations";

    /**
     * Comparing, if two positions in the form of JSONArrays have the same
     * parents, grand parents, ... parentLevel = 1 means, both positions
     * have the same parent. parentLevel = 2 means, the grand parents are
     * identical and so on.
     *
     * @param position1
     *  The first position to be compared
     *
     * @param position2
     *  The second position to be compared
     *
     * @param parentLevel
     *  The level, until which the ancestors will be compared. parentLevel = 1
     *  means, that the direct parents must be identical.
     *
     * @return
     *  If the two positions have the same ancestor until the specified
     *  level, true is returned. Otherwise false
     */
    static private boolean hasSameParentComponent(JSONArray position1, JSONArray position2, int parentLevel) {

        boolean sameParent = true;

        if ((position1 == null) || (position2 == null)) {
            return false;
        }

        int length = position1.length();

        if ((length < parentLevel) || (length != position2.length())) {
            return false;
        }

        for (int index = length - parentLevel - 1; index >= 0; index -= 1) {

            try {
                if (position1.getInt(index) != position2.getInt(index)) {
                    return false;
                }
            } catch (JSONException e) {
                return false;
            }
        }

        return sameParent;
    }

    /**
     * Trying to merge two directly following insertText operations. If the
     * merge is possible the operation 'next' that follows the operation
     * 'last' must not be added to the operation array. Therefore the return
     * value 'insertNextOp' will be set to false in the case of successful
     * merging. In this case the content of the operation 'last' will be
     * modified.
     *
     * @param last
     *  The preceding insertText operation. This was already added to the
     *  operations array. Therefore it will be modified within this operation.
     *  And therefore it is necessary, that the insertText operations are
     *  copied before they are added to the operations array, so that the
     *  original operations are not modified.
     *
     * @param position2
     *  The following insertText operation. If merging this two operations
     *  is possible, then this operation will not be added to the operations
     *  array. Therefore the return value of this function is set to
     *  'insertNextOp' false.
     *
     * @return
     *  If the two insertText operations can be merged, it is not necessary
     *  to add the second operation to the operations array. Therefore the
     *  return value will be false. Otherwise it is true.
     *
     * @throws JSONException
     */
    static private boolean mergeInsertOperations(JSONObject last, JSONObject next) throws JSONException {

        boolean insertNextOp = true;

        int parentLevel = 1;
        String nextText = next.getString("text");
        JSONArray nextStart = next.getJSONArray("start");
        JSONArray lastStart = last.getJSONArray("start");

        if ((lastStart != null) && (nextStart != null) && (hasSameParentComponent(lastStart, nextStart, parentLevel))) {

            int lastTextPosStart = lastStart.getInt(lastStart.length() - 1);
            int nextTextPosStart = nextStart.getInt(nextStart.length() - 1);

            JSONObject nextAttrs = null;
            JSONObject lastAttrs = null;

            try {
                nextAttrs = next.getJSONObject("attrs");
            } catch (Exception e) {
                nextAttrs = null;
            }

            try {
                lastAttrs = last.getJSONObject("attrs");
            } catch (Exception e) {
                lastAttrs = null;
            }

            if ((nextAttrs == null) && (!((lastAttrs != null) && (lastAttrs.has("changes"))))) {

            	// TODO: Furter optimization: Both have 'changes' attribute and next has no other attributes

                String lastText = last.getString("text");

                if ((lastTextPosStart + lastText.length() == nextTextPosStart)) {
                    // the new text is inserted directly behind an already existing text
                    last.put("text", lastText + nextText);
                    insertNextOp = false;
                } else if ((lastTextPosStart <= nextTextPosStart) && (nextTextPosStart <= lastTextPosStart + lastText.length())) {
                    // the new text is inserted into an already existing text
                    int relativeStart = nextTextPosStart - lastTextPosStart;
                    last.put("text", lastText.substring(0, relativeStart) + nextText + lastText.substring(relativeStart));
                    insertNextOp = false;
                }
            }
        }

        return insertNextOp;
    }

    /**
     * Trying to merge an insertText operation with a directly following delete
     * operation. If the merge is possible the 'delete' operation 'next' that
     * follows the insertText operation 'last' must not be added to the operation
     * array. Therefore the return value 'insertNextOp' will be set to false in
     * the case of successful merging. In this case the content of the operation
     * 'last' will be modified.
     *
     * @param last
     *  The preceding insertText operation. This was already added to the operations
     *  array. Therefore it will be modified within this operation. And therefore it
     *  is necessary, that the insertText operations are copied before they are added
     *  to the operations array, so that the original operations are not modified.
     *
     * @param position2
     *  The following delete operation. If merging this two operations is possible,
     *  then this operation will not be added to the operations array. Therefore
     *  the return value of this function is set to 'insertNextOp' false.
     *
     * @return
     *  If the 'insertText' and the 'delete' operations can be merged, it
     *  is not necessary to add the 'delete' operation to the operations array.
     *  Therefore the return value will be false. Otherwise it is true.
     *
     * @throws JSONException
     */
    static private boolean mergeInsertAndDeleteOperations(JSONObject last, JSONObject next) throws JSONException {

        boolean insertNextOp = true;
        int parentLevel = 1;
        JSONArray nextStart = next.getJSONArray("start");
        JSONArray lastStart = last.getJSONArray("start");
        // delete has an optional end parameter
        JSONArray nextEnd = null;

        try {
            nextEnd = next.getJSONArray("end");
        } catch (Exception e) {
            nextEnd = null;
        }

        if (nextEnd == null) {
            nextEnd = nextStart;
        }

        if ((lastStart != null) && (nextStart != null) && (hasSameParentComponent(lastStart, nextStart, parentLevel)) && (hasSameParentComponent(
            nextStart,
            nextEnd,
            parentLevel))) {

            int lastTextPosStart = lastStart.getInt(lastStart.length() - 1);
            int nextTextPosStart = nextStart.getInt(nextStart.length() - 1);
            int nextTextPosEnd = nextEnd.getInt(nextEnd.length() - 1);
            String lastText = last.getString("text");

            if ((lastTextPosStart <= nextTextPosStart) && (nextTextPosEnd < lastTextPosStart + lastText.length())) {
                // using 'nextTextPosEnd < lastTextPosStart + lastText.length()', not '<=' !
                // insertText at [0,3], Text 'abc' -> length is 3.
                // deleteText 'abc' is delete from [0,3] to [0,5], not till [0,6]!

                try {
                    // the deleted text is part of an already existing text
                    int relativeStart = nextTextPosStart - lastTextPosStart;
                    int relativeEnd = nextTextPosEnd - lastTextPosStart + 1;
                    String newText = "";
                    if ((relativeStart > 0) && (relativeStart <= lastText.length())) {
                        newText = lastText.substring(0, relativeStart);
                    }
                    if ((relativeEnd >= 0) && (relativeEnd < lastText.length())) {
                        newText += lastText.substring(relativeEnd);
                    }

                    last.put("text", newText);
                    insertNextOp = false;

                } catch (IndexOutOfBoundsException e) {
                    LOG.error("RT connection: Wrong index used within mergeInsertAndDeleteOperations", e);
                }
            }
        }

        return insertNextOp;
    }

    /**
     * Optimizing the operations by trying to merge an insertText operation
     * with a directly following insertText or delete operation.
     *
     * @param jsonResult
     *  The JSONObject that contains the operations below the key "operations"
     *  as a JSONArray. The content of this JSONArray is modified. No return
     *  value is required.
     */
    static private void optimizeOperationsArray(JSONObject jsonResult) {

        if ((null != jsonResult) && (jsonResult.has(m_operationsKey))) {
            try {
                JSONArray operations = jsonResult.getJSONArray(m_operationsKey);
                JSONArray operationArr = new JSONArray();

                if (!operations.isEmpty()) {

                    int nOps = operations.length();

                    for (int i = 0; i < nOps; i++) {

                        JSONObject nextOp = operations.getJSONObject(i);
                        JSONObject next = null;

                        boolean insertNextOp = true;

                        // Optimizing the content of the JSONArray 'operationArr'.
                        // For example several insertText operations can be merged.
                        // INFO: It might be necessary to make this optimization optional,
                        // so that operations from several clients will not be merged.

                        try {

                            String nextName = nextOp.getString("name");

                            if (nextName.equals("insertText")) {
                                // Always make a copy of insertText operations, because they might be modified in
                                // optimization process. Taking care, that the original operations are not modified.
                                next = new JSONObject(nextOp, nextOp.keySet().toArray(new String[0]));
                                // next = new JSONObject(nextOrig.toString()); -> alternative way
                            } else {
                                next = nextOp; // no need to copy operation, because it will not be modified
                            }

                            final JSONObject last = (operationArr.length() > 0) ? operationArr.getJSONObject(operationArr.length() - 1) : null;

                            if (last != null) {

                                String lastName = last.getString("name");

                                if ((lastName.equals("insertText")) && (nextName.equals("insertText"))) {
                                    // First optimization: Merging two following insertText operations
                                    insertNextOp = mergeInsertOperations(last, next);
                                } else if ((lastName.equals("insertText")) && (nextName.equals("delete"))) {
                                    // Second optimization: Merging delete operation following insertText operation
                                    insertNextOp = mergeInsertAndDeleteOperations(last, next);

                                    if ((!insertNextOp) && (last.getString("text").length() == 0)) {
                                        // taking care of empty new texts -> removing the last operation completely
                                        operationArr = JSONHelper.removeItemFromJSONArray(operationArr, operationArr.length() - 1);
                                    }
                                }

                                if (! insertNextOp) {
                                    if (last.has("opl") && (next.has("opl"))) {
                                        last.put("opl", last.getInt("opl") + next.getInt("opl"));
                                    }
                                }

                            }
                        } catch (JSONException e) {
                            LOG.error("RT connection: Exception caught while setting up JSON result within optimizeOperationsArray", e);
                        }

                        if (insertNextOp) {
                            operationArr.put(next);
                        }
                    }

                    jsonResult.put(m_operationsKey, operationArr);
                }
            } catch (JSONException e) {
                LOG.error("RT connection: Exception caught while setting up JSON result within optimizeOperationsArray", e);
            }
        }
    }

    /**
     * Concatenating two JSON objects containing actions
     *
     * @param jsonResult
     *
     * @param jsonAppender
     *
     * @return
     */
    public static JSONObject appendJSON(JSONObject jsonResult, JSONObject jsonAppender) {
        return appendJSON(jsonResult, jsonAppender, false);
    }

    /**
     * Concatenating two JSON objects containing actions
     *
     * @param jsonResult
     *
     * @param jsonAppender
     *
     * @param keepActions
     *
     * @return
     */
    public static JSONObject appendJSON(JSONObject jsonResult, JSONObject jsonAppender, boolean keepActions) {
        if ((null != jsonResult) && (null != jsonAppender)) {
            try {
                final Iterator<String> keys = jsonAppender.keys();

                while (keys.hasNext()) {
                    final String curKey = keys.next();
                    final Object curObject = jsonAppender.get(curKey);

                    if (curKey.equals(m_actionsKey)) {
                        if (!keepActions) {
                            if (!jsonResult.has(m_operationsKey)) {
                                jsonResult.put(m_operationsKey, new JSONArray());
                            }

                            if (curObject instanceof JSONArray) {
                                final JSONArray operationsGroups = (JSONArray) curObject;

                                for (int i = 0, size = operationsGroups.length(); i < size; ++i) {
                                    // [{"operations":[{"text":"a","start":[0,1],"name":"insertText"}]},{"operations":[{"text":"l","start":[0,2],"name":"insertText"}]}]
                                    JSONObject operationsObject = operationsGroups.getJSONObject(i);

                                    if (operationsObject.has(m_operationsKey)) {
                                        final JSONArray newOperations = operationsObject.getJSONArray(m_operationsKey);

                                        for (int j = 0; j < newOperations.length(); j++) {
                                            jsonResult.append(m_operationsKey, newOperations.get(j));
                                        }
                                    } else {
                                        jsonResult.append(m_operationsKey, operationsGroups.get(i));
                                    }
                                }
                            } else {
                                jsonResult.append(m_operationsKey, curObject);
                            }
                        } else {
                            appendJSONElement(jsonResult, curKey, curObject);
                        }
                    } else {
                        appendJSONElement(jsonResult, curKey, curObject);
                    }
                }
            } catch (JSONException e) {
                LOG.error("RT connection: Exception caught while setting up JSON result within appendJSON", e);
            }
        }

        return jsonResult;
    }

    /**
     * @param jsonResult
     * @param key
     * @param element
     */
    static private void appendJSONElement(JSONObject jsonResult, String key, Object element) {
        try {
            if (key.equals(m_operationsKey)) {
                if (!jsonResult.has(m_operationsKey)) {
                    jsonResult.put(m_operationsKey, new JSONArray());
                }

                if (element instanceof JSONArray) {
                    final JSONArray jsonArray = (JSONArray) element;

                    for (int i = 0; i < jsonArray.length(); ++i) {
                        jsonResult.append(m_operationsKey, jsonArray.get(i));
                    }
                } else {
                    jsonResult.append(m_operationsKey, element);
                }
            } else {
                jsonResult.put(key, element);
            }
        } catch (JSONException e) {
            LOG.error("RT connection: Exception caught while setting up JSON result within appendJSONElement", e);
        }
    }

    /**
     * @param jsonResult
     * @param operationChunks
     * @return
     */
    public static JSONObject appendOperationChunks(JSONObject jsonResult, ArrayList<MessageChunk> operationChunks, boolean allowOptimization) {
        if ((null != jsonResult) && (null != operationChunks)) {
            final ListIterator<MessageChunk> listIter = operationChunks.listIterator();

            while (listIter.hasNext()) {
                appendJSON(jsonResult, listIter.next().toJSON());
            }

            // Check if it's safe to optimize the operations array. There are
            // situations where this is NOT safe:
            // If we have just loaded a document and set a OSN for the last operation,
            // this operation could be exchanged by a different operation missing
            // the OSN.
            if (allowOptimization) {
                // optimize operations
                optimizeOperationsArray(jsonResult);
            }
        }

        return jsonResult;
    }

    /**
     * @param jsonResult
     * @param jsonAppender
     * @return
     */
    public static JSONObject appendData(ResourceManager resourceManager, JSONObject jsonResult, InputStream contentStm, String returnType, String mimeType) {
        final JSONObject jsonObject = (null != jsonResult) ? jsonResult : new JSONObject();

        if (null != contentStm) {
            byte[] data = null;

            try {
                data = IOUtils.toByteArray(contentStm);
            } catch (IOException e) {
                LOG.error("RT connection: Exception caught while reading content stream for appendData", e);
            }

            if ((null != data) && (data.length > 0)) {
                if (returnType.equals("resourceid")) {
                    final long uid = resourceManager.addResource(data);

                    if (0 != uid) {
                        try {
                            jsonObject.put("resourceid", Resource.getManagedIdFromUID(uid));
                        } catch (JSONException e) {
                            LOG.error("RT connection: Exception caught while creating resourceid JSON object for appendData", e);
                        }
                    }
                } else if (returnType.equals("dataurl")) {
                    final StringBuilder dataUrlBuilder = new StringBuilder("data:");

                    dataUrlBuilder.append(mimeType).append(";base64,").append(Base64.encode(data));

                    try {
                        jsonObject.put("dataurl", dataUrlBuilder.toString());
                    } catch (JSONException e) {
                        LOG.error("RT connection: Exception caught while creating dataurl JSON object for appendData", e);
                    }
                }
            }
        }

        return jsonObject;
    }

    /**
     * Transforms an operation JSONObject to a string using a special writer to ensure
     * that the output won't be bigger that a maximum number of characters.
     *
     * @param operation The json object containing the data of an operation.
     *
     * @return A string representing the operation.
     */
    public static String operationToString(JSONObject operation) {
        String result = "";

        if (null != operation) {
            JSONDebugWriter myWriter = null;

            try {
                myWriter = new JSONDebugWriter();
                operation.write(myWriter);
                result = myWriter.getData();
            } catch (final Exception e) {
                LOG.warn("RT connection: operationToString catches exception writing operation to string. ", e);
            } finally {
                IOUtils.closeQuietly(myWriter);
            }
        }

        return result;
    }

    /**
     * Provides the json data of the first operation of the first chunk within
     * the provided MessageChunk list.
     *
     * @param chunkList The message chunk list to retrieve the first operation of the first chunk.
     *
     * @return The json data of the first operation of the first chunk or null if no chunk/operation is
     * available.
     */
    public static JSONObject debugGetFirstOperationFromMessageChunkList(ArrayList<MessageChunk> chunkList) {
        JSONObject result = null;

        if (chunkList.size() > 0) {
            MessageChunk chunk = chunkList.get(0);
            JSONObject opsObject = chunk.getOperations();
            try {
                JSONArray actions = opsObject.getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                if (actions.length() > 0) {
                    opsObject = actions.getJSONObject(0);
                    int operationCount = opsObject.getJSONArray("operations").length();
                    if (operationCount > 0) {
                        result = opsObject.getJSONArray("operations").getJSONObject(0);
                    }
                }
            } catch (final JSONException e) {
                LOG.warn("RT connection: debugGetFirstOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            } catch (final Throwable e) {
                LOG.warn("RT connection: debugGetFirstOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            }
        }

        return result;
    }

    /**
     * Provides the json data of the last operation of the last chunk within
     * the provided MessageChunk list.
     *
     * @param chunkList The message chunk list to retrieve the last operation of the last chunk.
     *
     * @return The json data of the last operation of the last chunk or null if no chunk/operation is
     * available.
     */
    public static JSONObject debugGetLastOperationFromMessageChunkList(ArrayList<MessageChunk> chunkList) {
        JSONObject result = null;

        if (chunkList.size() > 0) {
            MessageChunk chunk = chunkList.get(chunkList.size() - 1);
            JSONObject opsObject = chunk.getOperations();
            try {
                JSONArray actions = opsObject.getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                if (actions.length() > 0) {
                    opsObject = actions.getJSONObject(actions.length() - 1);
                    int operationCount = opsObject.getJSONArray("operations").length();
                    if (operationCount > 0) {
                        result = opsObject.getJSONArray("operations").getJSONObject(operationCount - 1);
                    }
                }
            } catch (final JSONException e) {
                LOG.warn("RT connection: debugGetLastOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            } catch (final Throwable e) {
                LOG.warn("RT connection: debugGetLastOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            }
        }

        return result;
    }

    /**
     * Checks that the operations of a message chunk comply to
     * our minimal requirements:
     * 1. always ascending
     * 2. without any gap in osn/opl by consecutive operations
     *
     * @param chunk
     *  The MessageChunk which should be checked.
     *
     * @return
     *  TRUE if the operations inside the message chunk comply
     *  to the requirements or FALSE if not.
     */
    public static boolean hasValidOperations(MessageChunk chunk, int serverOSN) {
        boolean result = false;
        JSONObject chunkOpsObject = chunk.getOperations();

        try {
            JSONArray actionsArray = chunkOpsObject.optJSONArray(MessagePropertyKey.KEY_ACTIONS);
            if (null != actionsArray && actionsArray.length() > 0) {
                int actions = actionsArray.length();

                for (int i=0; i < actions; i++) {
                    JSONObject actionsObject = actionsArray.getJSONObject(i);
                    JSONArray operationsArray = actionsObject.optJSONArray("operations");

                    if ((null != operationsArray) && (operationsArray.length() > 0)) {
                        int operations = operationsArray.length();
                        for (int j=0; j < operations; j++) {
                            JSONObject opsObject = operationsArray.getJSONObject(j);
                            int opsOPL = opsObject.optInt("opl");
                            int opsOSN = opsObject.optInt("osn");

                            if (opsOSN != serverOSN) {
                                // found inconsistent osn in chunk
                                return false;
                            }
                            serverOSN = opsOSN + opsOPL;
                        }
                    }
                }
            }

            result = true;
        } catch (final Exception e) {
            LOG.warn("RT connection: hasValidOperations catches exception trying to retrieve operations from chunkList. ", e);
        }

        return result;
    }

    /**
     * Creates a synchronization no-operation to be added to a list of
     * operations.
     *
     * @param osn
     *  The OSN to be used by the no-op.
     *
     * @return
     *  A JSONObject filled with the necessary properties.
     */
    public static JSONObject createSyncNoOperation(int osn) throws JSONException {
        final JSONObject noOpSync = new JSONObject();
        noOpSync.put("name", "noOp");
        noOpSync.put("osn", (osn - 1));
        noOpSync.put("opl", 1);
        return noOpSync;
    }

    /**
     * Saving performance data into log file on server.
     *  with a directly following insertText or delete operation.
     *
     * @param performanceData
     *  The JSONObject containing all performance information sent
     *  from the client.
     *
     * @param recentFile
     *  The JSONObject containing the information about the file.
     */
    public static void logPerformanceData(JSONObject performanceData, JSONObject recentFile, String loggingFile) {

    	String uid = UUID.randomUUID().toString().substring(0, 8);  // only using 8 digits

    	// Saving performance data in log file
    	try {
    		Writer writer = new java.io.FileWriter(new java.io.File(loggingFile), true);

        	writer.append(uid + ":UTC:" + getISODateFormat(true));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":LAUNCHSTART:" + performanceData.optLong("launchStartAbsolute", 0) + ":" + performanceData.optInt("launchStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":APPCONSTRUCTED:" + performanceData.optLong("appConstructedAbsolute", 0) + ":" + performanceData.optInt("appConstructedDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":GUIPREPARED:" + performanceData.optLong("guiPreparedAbsolute", 0) + ":" + performanceData.optInt("guiPreparedDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":IMPORTHANDLERSTART:" + performanceData.optLong("importHandlerStartAbsolute", 0) + ":" + performanceData.optInt("importHandlerStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":CREATENEWDOCSTART:" + performanceData.optLong("createNewDocStartAbsolute", 0) + ":" + performanceData.optInt("createNewDocStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":CREATENEWDOCAFTERREQUEST:" + performanceData.optLong("createNewDocAfterRequestAbsolute", 0) + ":" + performanceData.optInt("createNewDocAfterRequestDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":PREVIEWSTART:" + performanceData.optLong("previewStartAbsolute", 0) + ":" + performanceData.optInt("previewStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":AFTERLEAVEBUSYEARLY:" + performanceData.optLong("afterLeaveBusyEarlyAbsolute", 0) + ":" + performanceData.optInt("afterLeaveBusyEarlyDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":IMPORTDOCSTART:" + performanceData.optLong("importDocStartAbsolute", 0) + ":" + performanceData.optInt("importDocStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":CONNECTSTART:" + performanceData.optLong("connectStartAbsolute", 0) + ":" + performanceData.optInt("connectStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":CONNECTEND:" + performanceData.optLong("connectEndAbsolute", 0) + ":" + performanceData.optInt("connectEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":PREPROCESSINGSTART:" + performanceData.optLong("preProcessingStartAbsolute", 0) + ":" + performanceData.optInt("preProcessingStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":PREPROCESSINGEND:" + performanceData.optLong("preProcessingEndAbsolute", 0) + ":" + performanceData.optInt("preProcessingEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":APPLYACTIONSSTART:" + performanceData.optLong("applyActionsStartAbsolute", 0) + ":" + performanceData.optInt("applyActionsStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":APPLYACTIONSEND:" + performanceData.optLong("applyActionsEndAbsolute", 0) + ":" + performanceData.optInt("applyActionsEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":POSTPROCESSINGSTART:" + performanceData.optLong("postProcessingStartAbsolute", 0) + ":" + performanceData.optInt("postProcessingStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":POSTPROCESSINGEND:" + performanceData.optLong("postProcessingEndAbsolute", 0) + ":" + performanceData.optInt("postProcessingEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":EDITMODE:" + performanceData.optLong("editModeAbsolute", 0) + ":" + performanceData.optInt("editModeDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":DOCUMENTIMPORTED:" + performanceData.optLong("documentImportedAbsolute", 0) + ":" + performanceData.optInt("documentImportedDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":LEAVEBUSY:" + performanceData.optLong("leaveBusyAbsolute", 0) + ":" + performanceData.optInt("leaveBusyDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":GUIINITIALIZED:" + performanceData.optLong("guiInitializedAbsolute", 0) + ":" + performanceData.optInt("guiInitializedDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":TRIGGERIMPORTSUCCESS:" + performanceData.optLong("triggerImportSuccessAbsolute", 0) + ":" + performanceData.optInt("triggerImportSuccessDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":LAUNCHEND:" + performanceData.optLong("launchEndAbsolute", 0) + ":" + performanceData.optInt("launchEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":LOCALSTORAGE:" + performanceData.optBoolean("LocalStorage", false));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":FASTEMPTYLOAD:" + performanceData.optBoolean("FastEmptyLoad", false));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":USERAGENT:" + performanceData.optString("user-agent", ""));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":PLATFORM:" + performanceData.optString("platform", ""));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":USER:" + performanceData.optString("user", ""));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":OXVERSION:" + performanceData.optString("version", ""));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":SERVERHOSTNAME:" + performanceData.optString("server", ""));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":FILENAME:" + recentFile.optString("filename", ""));

            writer.append(System.getProperty("line.separator"));
            writer.append(System.getProperty("line.separator"));
            writer.close();
		} catch (IOException e) {
			// Do nothing, simply not logging performance data
	    }

    }


    /**
     * The current date in ISO 8601 format.
     *
     * @param useSeconds
     *  Wheter seconds shall be included into the returnes string or not.
     *
     * @return
     *  The date as string in ISO 8601 format.
     */
    public static String getISODateFormat(Boolean useSeconds) {

    	TimeZone tz = TimeZone.getTimeZone("UTC");
    	DateFormat df = null;

    	if (useSeconds) {
    	    df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
    	} else {
    	    df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm':00Z'");
    	}
    	df.setTimeZone(tz);
    	return df.format(new Date());
    }

    // - Members ---------------------------------------------------------------
    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(OperationHelper.class);
}
