/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.jobpool.impl;

import java.lang.ref.WeakReference;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;

import com.openexchange.office.tools.jobpool.IJob;
import com.openexchange.office.tools.jobpool.IJobResultSink;
import com.openexchange.office.tools.jobpool.JobArguments;
import com.openexchange.office.tools.jobpool.JobBase;
import com.openexchange.office.tools.jobpool.JobPoolConfig;
import com.openexchange.office.tools.jobpool.JobResults;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
public class Job implements IJob
{
	//-------------------------------------------------------------------------
	private static final Log LOG = LogFactory.getJclLog(Job.class);
	
	//-------------------------------------------------------------------------
	public static final int DEFAULT_ASYNC_TIMEOUT_IN_MS = 30000;
	
	//-------------------------------------------------------------------------
	public Job ()
		throws Exception
	{}

	//-------------------------------------------------------------------------
	public synchronized void bindToPool (final JobPool aPool)
	    throws Exception
	{
		m_aPool = new WeakReference< JobPool >(aPool);
	}

	//-------------------------------------------------------------------------
	public synchronized void bindToJob (final JobBase aJob)
	    throws Exception
	{
		m_aJob = aJob;
	}
	
	//-------------------------------------------------------------------------
	public synchronized JobBase getJob ()
	    throws Exception
	{
		return m_aJob;
	}

	//-------------------------------------------------------------------------
	public synchronized void setId (final int nId)
	    throws Exception
	{
		m_nJobId = nId;
	}
	
	//-------------------------------------------------------------------------
	@Override
	public synchronized int getId ()
	    throws Exception
	{
		return m_nJobId;
	}

	//-------------------------------------------------------------------------
	@Override
	public /* NO SYNCHRONIZE ! */ JobResults executeSynchronous(final JobArguments iArguments)
		throws Exception
	{
		final ContextAwareLogHelp aLog = mem_Log ().forLevel(ELogLevel.E_TRACE);
		LOG.trace(aLog.toLog ("exec synchronous ..."));
		
		// reuse code ... and implement synchronous execution on top of asynchronous execution .-)
		// the other way around wont be possible.
		//
		// note some details ...
		// - sink is a local instance only - not known by others
		//   so we can be sure there is a 1:1 relation between
		//   our request and it's response
		//
		// - using the asynchronous way in combination with "sink.wait()"
		//   provide the possibility to implement a timeout behaviour ...
		//   which is not (real) possible with a synchronous call

		int nTimeoutInMs = JobPoolConfig.DEFAULT_JOB_EXECUTION_TIMEOUT_IN_MS;
		synchronized(this)
		{
			nTimeoutInMs = impl_getConfiguredTimeout ();
		}
		
		JobResultSink  aSink = new JobResultSink  ( );
		CountDownLatch aSync = new CountDownLatch (1);
		aSink.bind2Latch(aSync);
		
		try
		{
			executeAsynchronoues (iArguments, aSink);
		}
		catch (Throwable exT)
		{
			LOG.error (exT.getMessage(), exT);
		}
		
		LOG.trace (aLog.toLog("... wait for job", "timeout : "+nTimeoutInMs+" ms"));
		boolean bFinished = aSync.await(nTimeoutInMs, TimeUnit.MILLISECONDS);
		boolean bTimedOut = ! bFinished; // :-)
		
		JobResults aResult = null;
		if ( bTimedOut )
		{
			LOG.trace (aLog.toLog("... job timed out"));
			aResult = JobResults.TIMEOUT_RESULT();
		}
		else
		{
			LOG.trace (aLog.toLog("... job finished in time"));
			aResult = aSink.getLastResults();
		}
		
		LOG.trace (aLog.toLog("result = "+aResult));
		return aResult;
	}

	//-------------------------------------------------------------------------
	@Override
	public synchronized void executeAsynchronoues(final JobArguments   iArguments,
									 			  final IJobResultSink iSink     )
		throws Exception
	{
		final ContextAwareLogHelp aLog = mem_Log ();
		LOG.trace(aLog.forLevel(ELogLevel.E_TRACE       )
					  .toLog   ("exec asynchronous ..."));
		
		impl_checkAndRejectReusingSameJob ();

		// job died under the hood ? (disabled by anywhere else ... !)
		// -> return 'nothing' .. because we have nothing

		final JobBase aJob = getJob ();
		if (aJob == null)
		{
			LOG.warn(aLog.forLevel(ELogLevel.E_WARNING                                 )
					     .toLog   ("NOT BOUND TO A JOB ?! -> return empty results ..."));
			if (iSink != null)
				iSink.setResults(JobResults.EMPTY_RESULT());

			return;
		}

		// execute job asynchronous
		
		new Thread ()
		{
			@Override
			public void run ()
			{
				Throwable aError = null;
				try
				{
					LOG.trace(aLog.forLevel(ELogLevel.E_TRACE        )
							      .toLog   ("... trigger native job"));
					
					aJob.setArguments(iArguments);
					aJob.execute();

					LOG.trace(aLog.forLevel(ELogLevel.E_TRACE         )
						          .toLog   ("... native job finished"));
					
					if (iSink != null)
					{
						final JobResults iResults = aJob.getResults();
						iResults.setSuccessfully();
						
						LOG.trace(aLog.forLevel(ELogLevel.E_TRACE              )
							          .toLog   ("... pass results to listener"));
						iSink.setResults(iResults);
					}
					else
						LOG.trace(aLog.forLevel(ELogLevel.E_TRACE                           )
						              .toLog   ("... no listener for results - ignore that"));
					
					return;
				}
				catch (Throwable ex)
				{
					aError = ex;
				}
				
				try
				{
					impl_handleJobError (aError);

					if (iSink != null)
					{
						LOG.trace(aLog.forLevel(ELogLevel.E_TRACE            )
						              .toLog   ("... pass error to listener"));
						iSink.setResults(JobResults.ERROR_RESULT(aError));
					}
				}
				catch (Throwable ex)
				{
					LOG.error (ex.getMessage(), ex);
					// ignore that "inner" error !
				}
			}
			
		}.start ();
	}
	
	//-------------------------------------------------------------------------
	private void impl_handleJobError (final Throwable aError)
	{
		try
		{
			final ContextAwareLogHelp aLog = mem_Log ();
			LOG.error (aLog.forLevel (ELogLevel.E_ERROR   )
					       .toLog    ("... job has errors"),
					   aError                             );
			
			JobPool aPool = null;
			synchronized(this)
			{
				aPool = m_aPool.get();
			}
			
			if (aPool != null)
				aPool.deactivateJob (this);
		}
		catch (Throwable ex)
		{
			LOG.error (ex.getMessage(), ex);
			// ignore errors inside errors !
		}
	}
	
	//-------------------------------------------------------------------------
	private void impl_checkAndRejectReusingSameJob ()
	    throws Exception
	{
		if (m_bWasUsed)
			throw new Exception ("Job has not to be reused for multiple requests ! Please lease new job always ...");
		m_bWasUsed = true;
	}
	
	//-------------------------------------------------------------------------
	private int impl_getConfiguredTimeout ()
	    throws Exception
	{
		JobPool aPool = m_aPool.get ();
		if (aPool == null)
			return DEFAULT_ASYNC_TIMEOUT_IN_MS;
		
		final JobPoolConfig aConfig  = aPool.getConfig();
		final int           nTimeout = aConfig.get(JobPoolConfig.PROP_JOB_EXECUTION_TIMEOUT, JobPoolConfig.DEFAULT_JOB_EXECUTION_TIMEOUT_IN_MS);
		
		return nTimeout;
	}

	//-------------------------------------------------------------------------
	private ContextAwareLogHelp mem_Log ()
	    throws Exception
	{
		if (m_aLog == null)
		{
			m_aLog = new ContextAwareLogHelp (LOG);
			m_aLog.enterContext("job-id", Integer.toString(m_nJobId));
		}
		return m_aLog;
	}

	//-------------------------------------------------------------------------
	private WeakReference< JobPool > m_aPool = null;

	//-------------------------------------------------------------------------
	private JobBase m_aJob = null;

	//-------------------------------------------------------------------------
	private boolean m_bWasUsed = false;

	//-------------------------------------------------------------------------
	private int m_nJobId = IJob.INVALID_JOB_ID;

	//-------------------------------------------------------------------------
	private ContextAwareLogHelp m_aLog = null;
}
