package com.openexchange.office.tools;

import org.apache.commons.logging.Log;

import com.openexchange.exception.OXException;
import com.openexchange.file.storage.FileStorageFolder;
import com.openexchange.file.storage.FileStorageFolderType;
import com.openexchange.file.storage.FileStoragePermission;
import com.openexchange.file.storage.TypeAware;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class FolderHelper
{
    private static final Log LOG = com.openexchange.log.Log.loggerFor(FolderHelper.class);

    public static final int FOLDER_PERMISSIONS_READ = 0;

    public static final int FOLDER_PERMISSIONS_WRITE = 1;

    public static final int FOLDER_PERMISSIONS_FOLDER = 2;

    public static final int FOLDER_PERMISSIONS_SIZE = 3;

    /**
     * Determines if a folder is a trash folder or not.
     *
     * @param folder
     *  The folder to be checked.
     *
     * @return
     *  TRUE if the folder is a trash folder, otherwise FALSE.
     */
    public static boolean isTrashFolder(FileStorageFolder folder) {
        return ((null != folder) && (TypeAware.class.isInstance(folder) && FileStorageFolderType.TRASH_FOLDER.equals(((TypeAware)folder).getType())));
    }

    /**
     * Determines if a folder is a root folder or not.
     *
     *  The folder to be checked.
     *
     * @return
     *  TRUE if the folder is a trash folder, otherwise FALSE.
     */
    public static boolean isRootFolder(FileStorageFolder folder) {
        // Uses a workaround in case isRootFolder is false but the id = 0
        // Treat this folder as root folder regarding the result of
        // isRootFolder.
        return (folder.isRootFolder() || folder.getId().equals("0"));
    }

    /**
     * Retrieve user read permission for a certain folder.
     *
     * @param fileAccess
     *  The file access to be used to retrieve the write
     *  permissions of the user.
     *
     * @param folderId
     *  The folder id where want to know the write permissions
     *  of the user.
     *
     * @return
     *  TRUE if the user has read permission in the provided folder,
     *  other FALSE.
     */
    public static boolean folderHasReadAccess(IDBasedFolderAccess folderAccess, String folderId) {
        boolean ret = false;

        int[] result = getFolderPermissions(folderAccess, folderId);
        if (null != result && result.length >= 3) {
            int readPermission = result[FOLDER_PERMISSIONS_READ];
            ret = (readPermission > FileStoragePermission.NO_PERMISSIONS);
        }
        return ret;
    }

    /**
     * Retrieve user write permission for a certain folder.
     *
     * @param fileAccess
     *  The file access to be used to retrieve the write
     *  permissions of the user.
     *
     * @param folderId
     *  The folder id where want to know the write permissions
     *  of the user.
     *
     * @return
     *  TRUE if the user has write permission in the provided folder,
     *  other FALSE.
     */
    public static boolean folderHasWriteAccess(IDBasedFolderAccess folderAccess, String folderId) {
        boolean ret = false;

        int[] result = getFolderPermissions(folderAccess, folderId);
        if (null != result && result.length >= 3) {
            int writePermission = result[FOLDER_PERMISSIONS_WRITE];
            ret = (writePermission > FileStoragePermission.NO_PERMISSIONS);
        }
        return ret;
    }

    /**
     * Retrieve user create permission for a certain folder.
     *
     * @param fileAccess
     *  The file access to be used to retrieve the create
     *  permissions of the user.
     *
     * @param folderId
     *  The folder id where want to know the create permissions
     *  of the user.
     *
     * @return
     *  TRUE if the user has the create permission in the provided folder,
     *  other FALSE.
     */
    public static boolean folderHasCreateAccess(IDBasedFolderAccess folderAccess, String folderId) {
        boolean ret = false;

        int[] result = getFolderPermissions(folderAccess, folderId);
        if (null != result && result.length >= 3) {
            int createPermission = result[FOLDER_PERMISSIONS_FOLDER];
            ret = (createPermission >= FileStoragePermission.CREATE_OBJECTS_IN_FOLDER);
        }
        return ret;
    }

    /**
     * Retrieve the user permissions for the provided folder id.
     * @param fileAccess
     *  The file access used to retrieve the user permissions.
     * @param folderId
     *  The folder id to retrieve the user permissions.
     *
     * @return
     *  An array with three entries for read, write and folder permissions or
     *  null if the method is unable to retrieve the permissions.
     */
    public static int[] getFolderPermissions(IDBasedFolderAccess folderAccess, String folderId) {
        int[] result = null;


        try {
            // retrieve the folder instance from the folder id
            final FileStorageFolder folder = folderAccess.getFolder(folderId);

            if (folder != null) {
                final FileStoragePermission permission = folder.getOwnPermission();

                int readPermission = permission.getReadPermission();
                int writePermission = permission.getWritePermission();
                int folderPermission = permission.getFolderPermission();

                // create array with user permissions for the folder
                result = new int[FOLDER_PERMISSIONS_SIZE];
                result[FOLDER_PERMISSIONS_READ] = readPermission;
                result[FOLDER_PERMISSIONS_WRITE] = writePermission;
                result[FOLDER_PERMISSIONS_FOLDER] = folderPermission;
            }
        } catch (final Exception e) {
            LOG.warn("GetFolderPermissions - retrieving user permissions for folder failed with exception", e);
        }

        return result;
    }

    /**
     * Retrieve a folder access.
     *
     * @param session
     *  The session associated with the folder access.
     *
     * @return
     *  The folder access or null if no folder access is available.
     *
     * @throws OXException
     */
    public static IDBasedFolderAccess getFolderAccess(ServiceLookup services, Session session) throws OXException {
        IDBasedFolderAccessFactory factory = services.getService(IDBasedFolderAccessFactory.class);
        return factory.createAccess(session);
    }

}
