/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.realtime.impl.calc;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import com.openexchange.config.ConfigurationService;
import com.openexchange.server.ServiceLookup;
import com.openexchange.threadpool.ThreadPoolService;

public class RequestExecutor {
	
    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(RequestExecutor.class);
    private static final int DEFAULT_LOAD_WORKER = 3;
    private ServiceLookup m_serviceLookup = null;
	private ExecutorService m_executor = null;

    /**
     * Initialize a new RequestExecutor instance.
     *
     * @param services
     *  The ServiceLookup instance to be used for this instance.
     */
	public RequestExecutor(ServiceLookup services) {
    	m_serviceLookup = services;
    }

    /**
     * Starts the RequestExecutor which is a requirement to use the execute
     * method. The method uses the thread pool service to create executors
     * which will be used to asynchronously process requests.
     *
     * @return
     *  TRUE if starting the executor instances was successful, otherwise
     *  FALSE will be returned.
     */
	public boolean start() {
    	boolean result = false;

    	if (null == m_executor) {
            int numOfWorker = getNumberOfCalcEngineWorkers();    		
    		ThreadPoolService threadPoolService = m_serviceLookup.getService(ThreadPoolService.class);
    		if (null != threadPoolService) {
    			m_executor = threadPoolService.getFixedExecutor(numOfWorker);
    		}
    	} else {
    		result = true;
    	}

    	return result;
    }
	
    /**
     * Prepares to shutdown the RequestExecutor instance, trying to close the
     * executor instance created with start().
     *
     * @return
     *  TRUE if the shutdown was successful, otherwise FALSE.
     */
	public boolean shutdown() {
        boolean result = true;

        if (null != m_executor) {
            m_executor = null;
        }

    	return result;
    }

	/**
	 * Executes a runnable using the internal executor.
	 *
	 * @param runnable
	 *  A runnable to be executed by an executor.
	 */
	public void execute(Runnable runnable) throws Exception {
	    final ExecutorService executor = m_executor;

		if (null != executor) {
			executor.execute(runnable);
		}
	}

	/**
	 * Provides the number of CalcEngine workers that are currently configured
	 * on the system.
	 *
	 * @return
	 *  The number of CalcEngine workers configured.
	 */
	private int getNumberOfCalcEngineWorkers() {
		int numOfWorker = DEFAULT_LOAD_WORKER;
	    ConfigurationService cfgService = m_serviceLookup.getService(com.openexchange.config.ConfigurationService.class);
		
	    if (null != cfgService) {
			int minRange = cfgService.getIntProperty("calcengine.worker.port_range.min", 5001);
		    int maxRange = cfgService.getIntProperty("calcengine.worker.port_range.max", 5003);
		    numOfWorker = (maxRange-minRange+1);
	    }
	    return numOfWorker;
	}
}
