/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.groupware.ldap.User;
import com.openexchange.java.Strings;
import com.openexchange.log.Log;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.IExporter;
import com.openexchange.office.IImporter;
import com.openexchange.office.IPreviewImporter;
import com.openexchange.office.tools.DocFileHelper;
import com.openexchange.office.tools.DocFileHelper.WriteInfo;
import com.openexchange.office.tools.DocumentFormat;
import com.openexchange.office.tools.DocumentMetaData;
import com.openexchange.office.tools.ErrorCode;
import com.openexchange.office.tools.ErrorCode2BackupErrorCode;
import com.openexchange.office.tools.ExceptionToBackupErrorCode;
import com.openexchange.office.tools.FileHelper;
import com.openexchange.office.tools.Resource;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.office.tools.SessionUtils;
import com.openexchange.office.tools.StorageHelper;
import com.openexchange.office.tools.StreamInfo;
import com.openexchange.office.tools.message.MessageChunk;
import com.openexchange.office.tools.message.MessageHelper;
import com.openexchange.office.tools.message.OperationHelper;
import com.openexchange.realtime.packet.ID;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.iterator.SearchIterator;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link OXDocument}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class OXDocument {

    private static final Field updateMetaDataFields[] = { Field.VERSION };

    public static class ResolvedStreamInfo {

        public InputStream resolvedStream;

        public ErrorCode errorCode;

        public boolean debugStream;

        public ResolvedStreamInfo(InputStream inputStream, final ErrorCode errorCode, boolean debugStream) {
            this.resolvedStream = inputStream;
            this.errorCode = errorCode;
            this.debugStream = debugStream;
        }
    }

    /**
     * Contains the result data of a save operation.
     *
     * @author Carsten Driesner
     */
    public static class SaveResult {

        public String version;

        public ErrorCode errorCode;

        public SaveResult(final String version, final ErrorCode errorCode) {
            this.version = version;
            this.errorCode = errorCode;
        }
    }

    /**
     * Initializes a new {@link OXDocument}.
     *
     * @param session
     *   The session of the client that requested a service which needs documents access.
     * @param services
     *   The service lookup instance to be used by this instance.
     * @param fileHelper
     *   The FileHelper instance that is responsible for the file access regarding the document.
     * @param newDoc
     *   Specifies, if this instance references a new document or not.
     * @param resourceManager
     *   The ResourceManager instance which stores resources addressed by the document to be available via REST-API and
     *   other documents.
     * @param storageHelper
     *   A helper instance which provides data to the referenced document storage. The instance must be
     *   initialized with the same folderId, where the document itself is stored.
     */
    OXDocument(final Session session, final ServiceLookup services, final DocFileHelper fileHelper, boolean newDoc, final ResourceManager resourceManager, final StorageHelper storageHelper) {
        super();

        m_resourceManager = resourceManager;
        m_session = session;
        m_services = services;
        m_newDoc = newDoc;
        m_storageHelper = storageHelper;

        if (null != fileHelper) {
            final StreamInfo streamInfo = fileHelper.getDocumentStream(session, storageHelper);
            final InputStream documentStm = streamInfo.getDocumentStream();

            if (null != documentStm) {
                try {
                    m_metaData = streamInfo.getMetaData();
                    m_documentBuffer = IOUtils.toByteArray(documentStm);
                    m_documentFormat = fileHelper.getDocumentFormat();
                    m_documentMetaData = new DocumentMetaData(streamInfo);
                } catch (IOException e) {
                    LOG.error("RT connection: unable to retrieve document stream, exception catched", e);
                } finally {
                    IOUtils.closeQuietly(documentStm);
                }
            }

            if (null != m_documentBuffer) {
                final ZipInputStream zipInputStm = new ZipInputStream(new ByteArrayInputStream(m_documentBuffer));
                ZipEntry entry = null;
                byte[] documentBuffer = null;
                byte[] operationsBuffer = null;
                byte[] extendedDataBuffer = null;

                try {
                    // create a dummy resource manager, if no one is given
                    if (null == m_resourceManager) {
                        m_resourceManager = new ResourceManager(services);
                    }

                    while ((entry = zipInputStm.getNextEntry()) != null) {
                        final String entryName = entry.getName();

                        if (entryName.equals(DocFileHelper.OX_DOCUMENT_EXTENDEDDATA)) {
                            try {
                                // read document identifier
                                extendedDataBuffer = IOUtils.toByteArray(zipInputStm);
                            } catch (IOException e) {
                                LOG.error("RT connection: unable to retrieve extended data stream, exception catched", e);
                            }
                        } else if (entryName.equals(DocFileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME)) {
                            try {
                                // read original document
                                documentBuffer = IOUtils.toByteArray(zipInputStm);
                            } catch (IOException e) {
                                LOG.error("RT connection: unable to retrieve original document stream, exception catched", e);
                            }
                        } else if (entryName.equals(DocFileHelper.OX_RESCUEDOCUMENT_OPERATIONSSTREAM_NAME)) {
                            try {
                                // read original document
                                final String op = IOUtils.toString(zipInputStm);
                                // non continuous osn numbers lead to a load error in the frontend, so we have to remove them as some
                                // operations are filtered by the calcengine
                                operationsBuffer = op.replaceAll("\\\"osn\\\":[0-9]+,?", "").getBytes();
                            } catch (IOException e) {
                                LOG.error("RT connection: unable to retrieve operations stream, exception catched", e);
                            }
                        } else if (entryName.startsWith(DocFileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX)) {
                            // read resource and put into the resource manager, if not already contained
                            final String resourceName = entryName.substring(DocFileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX.length());

                            if (!m_resourceManager.containsResource(Resource.getUIDFromResourceName(resourceName))) {
                                byte[] resourceBuffer = null;

                                try {
                                    resourceBuffer = IOUtils.toByteArray(zipInputStm);
                                } catch (IOException e) {
                                    LOG.error("RT connection: unable to retrieve resource document stream, exception catched", e);
                                }

                                m_resourceManager.addResource(resourceBuffer);
                            }
                        }

                        zipInputStm.closeEntry();
                    }
                } catch (IOException e) {
                    LOG.error("RT connection: Exception catched while reading the document stream", e);
                } finally {
                    IOUtils.closeQuietly(zipInputStm);
                    try {
                        streamInfo.close();
                    } catch (IOException e) {
                        LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                    }

                    // set content of original document
                    if (null != documentBuffer) {
                        m_documentBuffer = documentBuffer;
                    }
                    if (null != extendedDataBuffer) {
                        try {
                            m_uniqueDocumentId = new JSONObject(new String(extendedDataBuffer));
                        } catch (JSONException e) {
                            LOG.error("RT connection: unable to setup json object for unique document identifier", e);
                        }
                    }

                    // set content of already available operations
                    if (null != operationsBuffer) {
                        try {
                            m_documentOperations = new JSONObject(new String(operationsBuffer));
                        } catch (JSONException e) {
                            LOG.error("RT connection: unable to setup json object for document operations", e);
                        }
                    }
                }
            } else {
                try {
                    streamInfo.close();
                } catch (IOException e) {
                    LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                }
            }

            if (null == m_documentBuffer) {
                // set last error code using the stream info from the file helper getDocumentStream()
                m_lastErrorCode = streamInfo.getErrorCode();
                if (m_lastErrorCode.isNoError()) {
                    m_lastErrorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
                }
                m_lastErrorCode = ErrorCode.setErrorClass(m_lastErrorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                LOG.warn("OXDocument Ctor: could not retrieve a valid document buffer from given session.");
            }
        }
    }

    /**
     * Prepares the preview operations access which is mandatory if preview functions should be used.
     *
     * @param importer The importer instance to be used for the preview. The instance must support the IPreviewImporter interface otherwise
     *            preview is not possible.
     * @return TRUE if preview operations are possible otherwise FALSE.
     */
    boolean preparePreviewOperations(IImporter importer, JSONObject importerProps) throws FilterException {
        boolean result = false;

        final Session session = m_session;

        if ((importer instanceof IPreviewImporter) && (null != session)) {
            final DocumentProperties documentProperties = new DocumentProperties();
            m_previewImporter = (IPreviewImporter) importer;
            m_docProperties = documentProperties;

            // provide properties for the importer
            documentProperties.put(DocumentProperties.PROP_USER_LANGUAGE, getUserLangCode(session));
            documentProperties.put(DocumentProperties.PROP_MAX_MEMORY_USAGE, importerProps.opt("maxMemoryUsage"));
            documentProperties.put(DocumentProperties.PROP_MEMORYOBSERVER, importerProps.opt(DocumentProperties.PROP_MEMORYOBSERVER));

            if (null != m_documentBuffer) {
                final InputStream documentStm = new ByteArrayInputStream(m_documentBuffer);
                m_previewImporter.createDocumentModel(session, documentStm, documentProperties);
                result = true;
            }
        }

        return result;
    }

    /**
     * @return
     */
    JSONObject getPreviewOperations() throws FilterException {
        JSONObject result = null;

        final IPreviewImporter importer = m_previewImporter;
        final DocumentProperties documentProperties = m_docProperties;
        final Session session = m_session;

        if ((null != session) && (null != importer) && (null != documentProperties)) {

            result = importer.createPreviewOperations(documentProperties);

            // retrieve properties from the document properties and set member
            m_activeSheetIndex = documentProperties.optInteger(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX, 0);
            m_sheetCount = documentProperties.optInteger(DocumentProperties.PROP_SPREADSHEET_SHEET_COUNT, 0);
        }

        return result;
    }

    /**
     * @return
     */
    JSONObject getRemainingOperations(JSONObject appendOperations) throws FilterException {
        JSONObject result = null;

        final IPreviewImporter importer = m_previewImporter;
        final DocumentProperties documentProperties = m_docProperties;
        final Session session = m_session;

        if ((null != session) && (null != importer) && (null != documentProperties)) {

            result = importer.createOutstandingOperations(documentProperties);

            // retrieve properties from the document properties and set member
            m_activeSheetIndex = documentProperties.optInteger(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX, 0);
            m_sheetCount = documentProperties.optInteger(DocumentProperties.PROP_SPREADSHEET_SHEET_COUNT, 0);
        }

        return result;
    }

    /**
     * Retrieves the operations from the document and optional additional operations.
     *
     * @param session The session of the client requesting the document operations.
     * @param importer The importer instance to be used to read the document stream and provide the operations.
     * @param appendOperations Optional operations which should be appended to the document operations. Normally used if there are queued
     *            operations which are not part of the last document stream.
     * @param previewData Optional preview data object which contains data to request a special preview data from the filter including a
     *            possible callback.
     * @return The JSONObject containing all operations from the document, the saved document operations and the given operations.
     * @throws FilterException in case the filter implementation is not able to provide the operations.
     */
    JSONObject getOperations(IImporter importer, final JSONObject importerProps, final JSONObject appendOperations) throws FilterException {
        JSONObject jsonResult = null;
        Session session = m_session;

        if ((null != session) && (null != m_documentBuffer) && (null != importer)) {
            final InputStream documentStm = new ByteArrayInputStream(m_documentBuffer);
            try {
                String userLangCode = getUserLangCode(session);
                long startTime = System.currentTimeMillis();

                DocumentProperties docProps = new DocumentProperties();
                docProps.put(DocumentProperties.PROP_USER_LANGUAGE, userLangCode);
                docProps.put(DocumentProperties.PROP_MEMORYOBSERVER, importerProps.opt(DocumentProperties.PROP_MEMORYOBSERVER));

                // request the document operations, preview data will be provided via callback
                jsonResult = importer.createOperations(session, documentStm, docProps);

                String uniqueDocumentId = docProps.optString(DocumentProperties.PROP_UNIQUE_DOCUMENT_IDENTIFIER, null);
                if (null != uniqueDocumentId) {
                    try {
                        this.setUniqueDocumentId(Integer.parseInt(uniqueDocumentId));
                    } catch (NumberFormatException e) {
                        // nothing to do as the unique document identifier
                        // is only an optional value
                    }
                }
                // set the optional active sheet index (in case this is not a
                // spreadsheet document).
                m_activeSheetIndex = docProps.optInteger(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX, 0);
                m_sheetCount = docProps.optInteger(DocumentProperties.PROP_SPREADSHEET_SHEET_COUNT, 0);

                LOG.debug("TIME getOperations: " + (System.currentTimeMillis() - startTime));
            } finally {
                IOUtils.closeQuietly(documentStm);
            }
        }

        // getting operations from the original document is mandatory
        if (null != jsonResult) {
            // append possible operations from the rescue document
            if (null != m_documentOperations) {
                OperationHelper.appendJSON(jsonResult, m_documentOperations);
            }

            // append given operations
            if (null != appendOperations) {
                OperationHelper.appendJSON(jsonResult, appendOperations);
            }
        }

        return jsonResult;
    }

    /**
     *
     * @return
     */
    public final DocumentMetaData getDocumentMetaData() {
        return m_documentMetaData;
    }

    /**
     * Determines if an importer supports preview operations or not. This method can only provide this information if
     * preparePreviewOperations() has been called, otherwise it will always return FALSE.
     *
     * @return TRUE if the importer supports preview operations and FALSE if not.
     */
    boolean supportsPreview() {
        return (m_previewImporter instanceof IPreviewImporter);
    }

    /**
     * Determines if an importer supports preview operations or not.
     *
     * @param importer An import instance to check if preview operations are supported or not.
     * @return TRUE if the importer supports preview operations and FALSE if not.
     */
    static boolean supportsPreview(IImporter importer) {
        return (importer instanceof IPreviewImporter);
    }

    /**
     * Retrieves the last error code set by the latest method.
     *
     * @return The last error code set by the last method called.
     */
    ErrorCode getLastError() {
        return m_lastErrorCode;
    }

    /**
     * Retrieves the resource manager used by this document.
     *
     * @return the ResourceManager used to read the document
     */
    ResourceManager getResourceManager() {
        return m_resourceManager;
    }

    /**
     * Provides the unique document id stored in the document.
     *
     * @return Returns the unique document id stored in the document or -1 if no id is available.
     */
    public int getUniqueDocumentId() {
        int id = -1;

        if (null != m_uniqueDocumentId) {
            id = m_uniqueDocumentId.optInt("uniqueDocumentIdentifier", -1);
        }

        return id;
    }

    /**
     * Provides the active sheet index.
     *
     * @return The active sheet index or 0 if the index is unknown (in case of a non- spreadsheet document) or cannot be retrieved.
     */
    public int getActiveSheetIndex() {
        return m_activeSheetIndex;
    }

    /**
     * Provides the sheet count.
     *
     * @return The sheet count or 0 if the count is unknown (in case of a non- spreadsheet document) or cannot be retrieved.
     */
    public int getSheetCount() {
        return m_sheetCount;
    }

    /**
     * Provides the folder id of the document file used by this document instance.
     *
     * @return The folder id or null if no document file is associated with this document instance.
     */
    public String getFolderId() {
        return (m_metaData != null) ? m_metaData.getFolderId() : null;
    }

    /**
     * Provides the file id of the document file used by this document instance.
     *
     * @return The file id or null if no document file is associated with this document instance.
     */
    public String getFileId() {
        return (m_metaData != null) ? m_metaData.getId() : null;
    }

    /**
     * Provides the version of the document at the time this instance was
     * created.
     *
     * @return
     *  The version of the document file, can be null.
     */
    public String getVersion() {
        return (m_metaData != null) ? m_metaData.getVersion() : null;
    }

    /**
     * Provides the meta data of the document file at the time this
     * instance was created.
     *
     * @return
     *  The meta data of the document file.
     */
    public File getMetaData() {
        return m_metaData;
    }

    /**
     * Provides the storage helper used by this instance.
     *
     * @return
     *  The storage helper instance of the document file.
     */
    public StorageHelper getStorageHelper() {
        return m_storageHelper;
    }

    /**
     * Sets a new unique document id to be stored in the document on the next successful flushDocument.
     *
     * @param uniqueId The new unique document id to be stored.
     */
    public void setUniqueDocumentId(int uniqueId) {
        if (null == m_uniqueDocumentId) {
            m_uniqueDocumentId = new JSONObject();
        }

        try {
            m_uniqueDocumentId.put("uniqueDocumentIdentifier", uniqueId);
        } catch (JSONException e) {
            // this is not a fatal error as we can work without this unique
            // id. just the speed up of the local cache won't work
            LOG.error("RT connection: Couldn't set unique document id to JSONObject", e);
        }
    }

    /**
     * Provides the document format of the document.
     *
     * @return The document format or null if the document instance was not initialized with a valid & existing file & folder id.
     */
    public DocumentFormat getDocumentFormat() {
        return m_documentFormat;
    }

    /**
     * Provides the new document stream that can be used to store the document to the info store.
     *
     * @param session The session of the client requesting the document operations.
     * @param exporter The exporter instance used to create the new document stream.
     * @param resourceManager The resource manager used by this document.
     * @param messageChunkList The message chunk list containing the pending operations to be stored in the new document.
     * @param version The version of the document used for document properties.
     * @param errorCode The array containing a possible error code.
     * @param finalSave Specifies if the new document stream should be handled as the final save. This can be used by the exporter to
     *            optimize the document, e.g. finally remove non-referenced resources.
     * @return The stream of the new document containing the message chunk operations. Can be null if the creation of the stream failed or
     *         contains a debug document stream. The result depends on the errorCode.
     */
    ResolvedStreamInfo getResolvedDocumentStream(final IExporter exporter, ResourceManager resourceManager, ArrayList<MessageChunk> messageChunkList, final String version, boolean finalSave, boolean saveAsTemplate) {

        boolean debugStreamCreated = false;
        InputStream resolvedDocumentStream = (null != m_documentBuffer) ? new ByteArrayInputStream(m_documentBuffer) : null;
        final JSONObject combinedOperations = new JSONObject();

        // set error code to no error
        ErrorCode errorCode = ErrorCode.NO_ERROR;
        m_lastErrorCode = errorCode;

        // get combined operations
        if ((null != m_documentOperations) || !messageChunkList.isEmpty()) {
            // append possible operations from the rescue document
            if (null != m_documentOperations) {
                OperationHelper.appendJSON(combinedOperations, m_documentOperations);
            }

            // apply collected operations
            if (null != messageChunkList) {
                OperationHelper.appendOperationChunks(combinedOperations, messageChunkList, true);
            }
        }

        // if there are any operations, try to export them with the document exporter first
        if ((null != resolvedDocumentStream) && (combinedOperations.has("operations") || saveAsTemplate) && (null != exporter) && (null != resourceManager)) {
            InputStream combinedStm = null;

            // set error code to worst case first
            errorCode = ErrorCode.SAVEDOCUMENT_FAILED_NOBACKUP_ERROR;

            try {
                String userLangCode = null;
                String userName = "";
                Session session = m_session;

                final User user = SessionUtils.getUserInfo(session, m_services);
                if (null != user) {
                    userLangCode = DocFileHelper.mapUserLanguageToLangCode(user.getPreferredLanguage());
                    userName = user.getDisplayName();
                }

                long startTime = System.currentTimeMillis();

                DocumentProperties docProps = new DocumentProperties();
                docProps.put(DocumentProperties.PROP_USER_LANGUAGE, userLangCode);
                docProps.put(DocumentProperties.PROP_LAST_MODIFIED_BY, userName);
                docProps.put(DocumentProperties.PROP_SAVE_TEMPLATE_DOCUMENT, saveAsTemplate);
                int documentOSN = this.getUniqueDocumentId();
                if (documentOSN != -1) {
                    docProps.put(DocumentProperties.PROP_UNIQUE_DOCUMENT_IDENTIFIER, String.valueOf(documentOSN));
                }
                if (null != version) {
                    // only use a valid version string
                    docProps.put(DocumentProperties.PROP_VERSION, version);
                }

                final String mergeOperations = combinedOperations.has("operations") ? combinedOperations.getJSONArray("operations").toString() : null;

                combinedStm = exporter.createDocument(
                    session,
                    resolvedDocumentStream,
                    mergeOperations,
                    resourceManager,
                    docProps,
                    finalSave);

                LOG.debug("TIME createDocument: " + (System.currentTimeMillis() - startTime));
            } catch (FilterException e) {
                if (e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED) {
                    LOG.error("Document export filter run out of memory while exporting the document");
                    errorCode = ErrorCode.GENERAL_MEMORY_TOO_LOW_ERROR;
                    // setErrorClass provides a clone with changed error class!
                    errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                } else {
                    LOG.error("Document export filter threw a filter exception while exporting the document", e);
                    errorCode = ErrorCode.SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR;
                    // setErrorClass provides a clone with changed error class!
                    errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                }
            } catch (Exception e) {
                LOG.error("Document export filter threw an exception while exporting the document", e);
                errorCode = ErrorCode.SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR;
                // setErrorClass provides a clone with changed error class!
                errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
            } finally {
                IOUtils.closeQuietly(resolvedDocumentStream);

                if (null != (resolvedDocumentStream = combinedStm)) {
                    errorCode = ErrorCode.NO_ERROR;
                }
            }
        } else if (combinedOperations.has("operations")) {
            errorCode = ErrorCode.SAVEDOCUMENT_FAILED_NOBACKUP_ERROR;

            if (null == exporter) {
                errorCode = ErrorCode.SAVEDOCUMENT_FAILED_NO_FILTER_ERROR;
                // setErrorClass provides a clone with changed error class!
                errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                LOG.error("Document export filter not available - not possible to provide resolved stream");
            }
            if (null == resourceManager) {
                errorCode = ErrorCode.SAVEDOCUMENT_FAILED_NO_RESOURCEMANAGER_ERROR;
                // setErrorClass provides a clone with changed error class!
                errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                LOG.error("Resource manager not available - not possible to provide resolved stream");
            }
            if (null == resolvedDocumentStream) {
                errorCode = ErrorCode.SAVEDOCUMENT_FAILED_NO_DOCUMENTSTREAM_ERROR;
                // setErrorClass provides a clone with changed error class!
                errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                LOG.error("Document stream not available - not possible to provide resolved stream");
            }
        }
        /*
         * { // !!! DEBUG code to create OX rescue documents by setting the document name to the !!! // !!! name of the env var
         * 'OX_RESCUEDOCUMENT_NAME'; may be commented for final release!!! final String oxRescueDocumentName_debug =
         * System.getenv("OX_RESCUEDOCUMENT_NAME"); if ((null != oxRescueDocumentName_debug) && (null != m_fileHelper)) { try { final String
         * filename = m_fileHelper.getFilenameFromFile(session); if ((null != filename) && filename.startsWith(oxRescueDocumentName_debug))
         * { errorCode = ErrorCode.SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR; } } catch (OXException e) {
         * LOG.error("Could not retrieve file name from file for saving OX rescue document"); } } }
         */
        m_lastErrorCode = errorCode;

        // if the there was an error before, we write a zipped OXDocument with
        // the original document and the operations content in separate streams
        if (errorCode != ErrorCode.NO_ERROR) {
            InputStream inputStm = null;
            final ByteArrayOutputStream byteArrayStm = new ByteArrayOutputStream(8192);
            final ZipOutputStream zipOutputStm = new ZipOutputStream(byteArrayStm);

            // set error code to worst case
            errorCode = ErrorCode.SAVEDOCUMENT_FAILED_NOBACKUP_ERROR;

            // write original document as well as collected operations
            // and collected resources into different zip entry streams
            try {
                // write original document stream
                if (null != m_documentBuffer) {
                    inputStm = new ByteArrayInputStream(m_documentBuffer);

                    try {
                        zipOutputStm.putNextEntry(new ZipEntry(DocFileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME));
                        IOUtils.copy(inputStm, zipOutputStm);
                        zipOutputStm.flush();
                    } catch (IOException e) {
                        LOG.error("Exception while writing original document stream to backup document", e);
                    } finally {
                        zipOutputStm.closeEntry();
                        IOUtils.closeQuietly(inputStm);
                        inputStm = null;
                    }

                }

                // write collected operations, that were added
                // by editing users, into separate zip entry stream
                if (!combinedOperations.isEmpty()) {
                    inputStm = new ByteArrayInputStream(combinedOperations.toString().getBytes("UTF-8"));

                    try {
                        zipOutputStm.putNextEntry(new ZipEntry(DocFileHelper.OX_RESCUEDOCUMENT_OPERATIONSSTREAM_NAME));
                        IOUtils.copy(inputStm, zipOutputStm);
                        zipOutputStm.flush();
                    } catch (IOException e) {
                        LOG.error("Exception while writing operations stream to backup document", e);
                    } finally {
                        zipOutputStm.closeEntry();
                        IOUtils.closeQuietly(inputStm);
                        inputStm = null;
                    }

                }

                // write collected resources, that were added by
                // editing users, into separate zip entry streams
                if (null != resourceManager) {
                    final Hashtable<Long, byte[]> resourceBuffers = resourceManager.getByteArrayTable();

                    for (final Long curKey : resourceBuffers.keySet()) {
                        final byte[] curResourceBuffer = resourceBuffers.get(curKey);

                        if (null != curResourceBuffer) {
                            final String curResourceName = DocFileHelper.OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX + Resource.getResourceNameFromUID(curKey.longValue());

                            inputStm = new ByteArrayInputStream(curResourceBuffer);

                            try {
                                zipOutputStm.putNextEntry(new ZipEntry(curResourceName));
                                IOUtils.copy(inputStm, zipOutputStm);
                                zipOutputStm.flush();
                            } catch (IOException e) {
                                LOG.error("Exception while writing collected resources to backup document", e);
                            } finally {
                                zipOutputStm.closeEntry();
                                IOUtils.closeQuietly(inputStm);
                                inputStm = null;
                            }
                        }
                    }
                }

                zipOutputStm.flush();
            } catch (IOException e) {
                LOG.error("Exception while writing backup document", e);
            } finally {
                IOUtils.closeQuietly(zipOutputStm);
                final byte[] byteArray = byteArrayStm.toByteArray();

                if (byteArray.length > 0) {
                    resolvedDocumentStream = new ByteArrayInputStream(byteArray);
                    errorCode = ErrorCode.SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR;
                    // setErrorClass provides a clone with changed error class!
                    errorCode = ErrorCode.setErrorClass(errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                    debugStreamCreated = true;
                }
            }
        }

        // don't overwrite the memory error code
        if (m_lastErrorCode.getCode() != ErrorCode.GENERAL_MEMORY_TOO_LOW_ERROR.getCode()) {
            // set the final error code and create the result object
            m_lastErrorCode = errorCode;
        }

        return new ResolvedStreamInfo(resolvedDocumentStream, m_lastErrorCode, debugStreamCreated);
    }

    /**
     * Saves the new document to the stroage.
     *
     * @param exporter
     *   The exporter instance used to create the new document stream.
     * @param resourceManager
     *   The resource manager used by this document.
     * @param messageChunkList
     *   The message chunk list containing the pending operations to be stored in the new document.
     * @param creator
     *   An optional creator to be set at the written file. Can be null.
     * @param revisionless
     *   Specifies if the new document should be saved as a new version or using the latest
     * @param finalSave
     *   Specifies if the new document stream should be handled as the final save. This can be used by the exporter to
     *   optimize the document, e.g. finally remove non-referenced resources.
     * @return
     *   The result of the save operation stored in the SaveResult instance. The different properties (error code, version)
     *   can be retrieved from the instance.
     */
    SaveResult save(IExporter exporter, ResourceManager resourceManager, ArrayList<MessageChunk> messageChunkList, ID creator, boolean revisionless, boolean finalSave) throws OXDocumentException {
        final SaveResult saveResult = new SaveResult(null, ErrorCode.NO_ERROR);
        String lastVersion = null;
        String fileVersion = null;
        Session session = m_session;
        ErrorCode backupError = ErrorCode.NO_ERROR;
        IDBasedFileAccess fileAccess = null;
        ResolvedStreamInfo docStreamInfo = null;

        try {
            final boolean storageSupportsVersions = m_storageHelper.supportsFileVersions();
            final Session creatorSession = MessageHelper.getServerSession(null, creator);
            final Integer creatorID = (null != creatorSession) ?  creatorSession.getUserId() : null;
            fileAccess = m_services.getService(IDBasedFileAccessFactory.class).createAccess(session);

            // No versions support && create a new version && not a new document => we
            // have to create a backup file with the current doc content
            if (!storageSupportsVersions && !revisionless && !m_newDoc) {
                final ArrayList<MessageChunk> emptyList = new ArrayList<MessageChunk>();
                docStreamInfo = getResolvedDocumentStream(exporter, resourceManager, emptyList, FileStorageFileAccess.CURRENT_VERSION, finalSave, false);
                if ((null != docStreamInfo) && (null != docStreamInfo.resolvedStream)) {
                    try {
                        final String folder_id = getFolderId();
                        final File currentFile = m_metaData;
                        final String fileName = currentFile.getFileName();

                        SearchIterator<File> iter = null;
                        if (!Strings.isEmpty(fileName)) {
                            final String backupFileName = FileHelper.createFilenameWithPostfix(fileName, BACKUP_FILE_POSTFIX);
                            // check file existence using the current file name
                            iter = FileHelper.getMetaDataFromFileName(fileAccess, folder_id, backupFileName);

                            if (!StringUtils.isEmpty(backupFileName)) {
                                if ((iter == null) || (!iter.hasNext())) {
                                    // create a new BAK and use the meta data from the original file
                                    backupError = FileHelper.createFileCopyWithStream(
                                        fileAccess,
                                        currentFile.getFolderId(),
                                        backupFileName,
                                        currentFile,
                                        docStreamInfo.resolvedStream,
                                        true);
                                } else {
                                    // overwrite existing backup file
                                    final File oldBackFile = iter.next();
                                    backupError = FileHelper.writeStreamToFile(fileAccess, oldBackFile, docStreamInfo.resolvedStream, currentFile);
                                }
                                // map all general error codes to the specific backup error codes
                                backupError = ErrorCode2BackupErrorCode.mapToBackupErrorCode(backupError, ErrorCode.SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR);
                            } else {
                                backupError = ErrorCode.SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR;
                            }
                        } else {
                            backupError = ErrorCode.SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR;
                        }
                    } catch (OXException e) {
                        LOG.warn("Exception catched while trying to store backup file", e);
                        backupError = ExceptionToBackupErrorCode.map(e, ErrorCode.SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR, false);
                    } catch (Throwable e) {
                        LOG.warn("Exception catched while trying to store backup file", e);
                        backupError = ErrorCode.SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR;
                    } finally {
                        // close stream for the backup file
                        IOUtils.closeQuietly((docStreamInfo != null) ? docStreamInfo.resolvedStream : null);
                    }
                } else {
                    backupError = ErrorCode.SAVEDOCUMENT_BACKUPFILE_CREATE_FAILED_ERROR;
                }

                // In case we detect anything wrong, including warnings, set the
                // the error class correctly to warning.
                if (backupError.getCode() != ErrorCode.CODE_NO_ERROR) {
                    // setErrorClass provides a clone with changed error class!
                    backupError = ErrorCode.setErrorClass(backupError, ErrorCode.ERRORCLASS_WARNING);
                }
            }

            // Normal save, retrieve stream with latest actions apply. This includes
            // creating a rescue file, if saving failed due to filter problems.
            docStreamInfo = getResolvedDocumentStream(exporter, resourceManager, messageChunkList, FileStorageFileAccess.CURRENT_VERSION, finalSave, false);
            if ((null != docStreamInfo) && (null != docStreamInfo.resolvedStream)) {
                final ErrorCode errorOnDocumentStream = docStreamInfo.errorCode;
                ErrorCode writeDocErrorCode = ErrorCode.NO_ERROR;

                boolean setActiveVersionToPrevious = false;
                boolean errorOnCreatingResolvedStream = (errorOnDocumentStream != ErrorCode.NO_ERROR);
                if (errorOnCreatingResolvedStream) {
                    // ATTENTION: There is an error creating the document stream from
                    // the operations. We have to assume that the document cannot be
                    // opened afterwards. Therefore we have to protect the last known
                    // good version to not be overwritten by "revisionless" save!
                    // Therefore we set "revisionless" to false here, if the current
                    // storage supports versions.
                    revisionless = false;
                    setActiveVersionToPrevious = storageSupportsVersions;
                }

                if (storageSupportsVersions || !errorOnCreatingResolvedStream) {
                    // just write the document even in case of a broken document
                    final WriteInfo writeInfo = DocFileHelper.writeDocumentStream(
                        m_services,
                        session,
                        docStreamInfo.resolvedStream,
                        getFileId(),
                        getFolderId(),
                        m_metaData,
                        m_storageHelper,
                        Properties.OX_RESCUEDOCUMENT_EXTENSION_APPENDIX,
                        docStreamInfo.debugStream,
                        revisionless,
                        creatorID);
                    writeDocErrorCode = writeInfo.errorCode;
                    fileVersion = writeInfo.version;
                    if (writeDocErrorCode.isError()) {
                        // in case of an error set fatal error class
                        // setErrorClass provides a clone with changed error class!
                        writeDocErrorCode = ErrorCode.setErrorClass(writeDocErrorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                    }
                } else {
                    // Versions are not supported and we encountered an error
                    // creating the new document stream. Make sure that we
                    // store the content to a file with the same name and
                    // the _ox extension. If it exists, overwrite it.
                    final String folder_id = getFolderId();
                    final File currentFile = m_metaData;

                    SearchIterator<File> iter = null;
                    String rescueFileName = currentFile.getFileName();
                    if (!Strings.isEmpty(rescueFileName)) {
                        rescueFileName += Properties.OX_RESCUEDOCUMENT_EXTENSION_APPENDIX;
                        // check file existence using file name
                        iter = FileHelper.getMetaDataFromFileName(fileAccess, folder_id, rescueFileName);
                    }

                    ErrorCode rescueError = ErrorCode.NO_ERROR;
                    if ((iter == null) || (!iter.hasNext())) {
                        // write new rescue file
                        rescueError = FileHelper.createFileWithStream(
                            fileAccess,
                            currentFile.getFolderId(),
                            rescueFileName,
                            currentFile.getFileMIMEType(),
                            docStreamInfo.resolvedStream);
                    } else {
                        // overwrite existing rescue file
                        final File oldRescueFile = iter.next();
                        rescueError = FileHelper.writeStreamToFile(fileAccess, oldRescueFile, docStreamInfo.resolvedStream, null);
                    }

                    // set error code dependent on the fact that we wrote/didn't write the rescue file
                    writeDocErrorCode = (rescueError.getCode() == ErrorCode.CODE_NO_ERROR) ? ErrorCode.SAVEDOCUMENT_FAILED_FILTER_OPERATION_ERROR : ErrorCode.SAVEDOCUMENT_FAILED_NOBACKUP_ERROR;
                    // setErrorClass provides a clone with changed error class!
                    writeDocErrorCode = ErrorCode.setErrorClass(writeDocErrorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                }

                if (writeDocErrorCode != ErrorCode.NO_ERROR) {
                    // error evaluation
                    saveResult.errorCode = writeDocErrorCode;
                    if (setActiveVersionToPrevious) {
                        // setErrorClass provides a clone with changed error class!
                        saveResult.errorCode = ErrorCode.setErrorClass(saveResult.errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
                    }
                    m_lastErrorCode = saveResult.errorCode;
                } else if (setActiveVersionToPrevious) {
                    // Set the active version to last known good version as
                    // we detected a filter problem storing the latest version
                    // correctly.

                    // First set the error information correctly
                    saveResult.errorCode = docStreamInfo.errorCode;
                    // setErrorClass provides a clone with changed error class!
                    saveResult.errorCode = ErrorCode.setErrorClass(saveResult.errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);

                    final String folderId = getFolderId();
                    final String fileId = getFileId();

                    if ((null != fileAccess) && (null != folderId) && (null != fileId)) {
                        final DefaultFile metadata = new DefaultFile();

                        // set the last known good version
                        metadata.setId(fileId);
                        metadata.setFolderId(folderId);
                        metadata.setVersion(lastVersion);

                        // and save the meta data
                        fileAccess.saveFileMetadata(metadata, FileStorageFileAccess.DISTANT_FUTURE, Arrays.asList(updateMetaDataFields));
                        fileAccess.commit();
                        fileVersion = lastVersion;
                    }
                }
            } else {
                // unknown information about creating the document stream - create
                // a generic error and set the fatal error class
                saveResult.errorCode = (null != docStreamInfo) ? docStreamInfo.errorCode : ErrorCode.SAVEDOCUMENT_FAILED_ERROR;
                // setErrorClass provides a clone with changed error class!
                saveResult.errorCode = ErrorCode.setErrorClass(saveResult.errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
            }
        } catch (Exception e) {
            LOG.error("Failed to write document stream on OXDocument.save().", e);
            saveResult.errorCode = ErrorCode.SAVEDOCUMENT_FAILED_ERROR;
            // setErrorClass provides a clone with changed error class!
            saveResult.errorCode = ErrorCode.setErrorClass(saveResult.errorCode, ErrorCode.ERRORCLASS_FATAL_ERROR);
        } finally {
            IOUtils.closeQuietly((docStreamInfo != null) ? docStreamInfo.resolvedStream : null);
            finishFileAccessSafely(fileAccess);
        }

        // in case of no save error, we check the backup error state for errors/warnings
        if (!saveResult.errorCode.isError() && (backupError.isError() || backupError.isWarning())) {
            saveResult.errorCode = backupError;
        }

        saveResult.version = fileVersion;
        return saveResult;
    }

    private void finishFileAccessSafely(IDBasedFileAccess fileAccess) {
        if (null != fileAccess) {
            try {
                fileAccess.finish();
            } catch (OXException e) {
                LOG.error("Failed to finish file access.", e);
            }
        }
    }

    /**
     * Retrieve the user language code.
     *
     * @param session
     *  The user session.
     *
     * @return
     *  The language code of the user.
     */
    private String getUserLangCode(Session session) {
        String result = null;

        if (session instanceof ServerSession) {
            final User user = ((ServerSession) session).getUser();

            if (null != user) {
                result = DocFileHelper.mapUserLanguageToLangCode(user.getPreferredLanguage());
            }
        }

        return result;
    }

    // - Members ---------------------------------------------------------------

    @SuppressWarnings("deprecation")
    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(OXDocument.class);

    static private final String BACKUP_FILE_POSTFIX = "-bak";

    private boolean m_newDoc = false;

    private File m_metaData = null;

    private ErrorCode m_lastErrorCode = ErrorCode.NO_ERROR;

    private byte[] m_documentBuffer = null;

    private StorageHelper m_storageHelper = null;

    private DocumentMetaData m_documentMetaData = null;

    private JSONObject m_documentOperations = null;

    private ResourceManager m_resourceManager = null;

    private JSONObject m_uniqueDocumentId = null;

    private Session m_session = null;

    private DocumentFormat m_documentFormat = null;

    private final ServiceLookup m_services;

    private IPreviewImporter m_previewImporter = null;

    private DocumentProperties m_docProperties = null;

    private int m_activeSheetIndex = 0;

    private int m_sheetCount = 0;
}
