/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.drawingml;

import java.util.List;
import java.util.logging.Logger;

import org.docx4j.dml.BaseStyles;
import org.docx4j.dml.CTBackgroundFillStyleList;
import org.docx4j.dml.CTBlip;
import org.docx4j.dml.CTBlipFillProperties;
import org.docx4j.dml.CTColor;
import org.docx4j.dml.CTFillStyleList;
import org.docx4j.dml.CTGradientFillProperties;
import org.docx4j.dml.CTGradientStop;
import org.docx4j.dml.CTGradientStopList;
import org.docx4j.dml.CTHslColor;
import org.docx4j.dml.CTLineProperties;
import org.docx4j.dml.CTLineStyleList;
import org.docx4j.dml.CTNoFillProperties;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPatternFillProperties;
import org.docx4j.dml.CTPoint2D;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTPresetColor;
import org.docx4j.dml.CTPresetGeometry2D;
import org.docx4j.dml.CTPresetLineDashProperties;
import org.docx4j.dml.CTRegularTextRun;
import org.docx4j.dml.CTRelativeRect;
import org.docx4j.dml.CTSRgbColor;
import org.docx4j.dml.CTScRgbColor;
import org.docx4j.dml.CTSchemeColor;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.CTShapeStyle;
import org.docx4j.dml.CTSolidColorFillProperties;
import org.docx4j.dml.CTStyleMatrix;
import org.docx4j.dml.CTStyleMatrixReference;
import org.docx4j.dml.CTSystemColor;
import org.docx4j.dml.CTTextBody;
import org.docx4j.dml.CTTextBodyProperties;
import org.docx4j.dml.CTTextCharacterProperties;
import org.docx4j.dml.CTTextField;
import org.docx4j.dml.CTTextLineBreak;
import org.docx4j.dml.CTTextListStyle;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.dml.CTTextParagraphProperties;
import org.docx4j.dml.CTTransform2D;
import org.docx4j.dml.IColorChoice;
import org.docx4j.dml.IFillProperties;
import org.docx4j.dml.ITransform2D;
import org.docx4j.dml.STPresetColorVal;
import org.docx4j.dml.STPresetLineDashVal;
import org.docx4j.dml.STSchemeColorVal;
import org.docx4j.dml.STShapeType;
import org.docx4j.dml.STTextAnchoringType;
import org.docx4j.dml.STTextStrikeType;
import org.docx4j.dml.STTextUnderlineType;
import org.docx4j.dml.STTextVertOverflowType;
import org.docx4j.dml.STTextVerticalType;
import org.docx4j.dml.STTextWrappingType;
import org.docx4j.dml.TextFont;
import org.docx4j.dml.Theme;
import org.docx4j.dml.chart.CTBoolean;
import org.docx4j.dml.chart.CTCatAx;
import org.docx4j.dml.chart.CTChartLines;
import org.docx4j.dml.chart.CTLayout;
import org.docx4j.dml.chart.CTNumFmt;
import org.docx4j.dml.chart.CTPlotArea;
import org.docx4j.dml.chart.CTScaling;
import org.docx4j.dml.chart.CTTickLblPos;
import org.docx4j.dml.chart.CTTickMark;
import org.docx4j.dml.chart.CTTitle;
import org.docx4j.dml.chart.CTTx;
import org.docx4j.dml.chart.CTValAx;
import org.docx4j.dml.chart.IAxisDescription;
import org.docx4j.dml.chart.IListSer;
import org.docx4j.dml.chart.STTickLblPos;
import org.docx4j.dml.chart.STTickMark;
import org.docx4j.dml.chart.chartStyle.CTChartColor;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.tools.json.JSONHelper;

final public class DMLHelper {

	private static final CTTextCharacterProperties GERMAN = new CTTextCharacterProperties();
	static {
		GERMAN.setLang("de-DE");
	}

    public static void createJsonFromPositiveSize2D(JSONObject drawingProperties, CTPositiveSize2D positiveSize2D)
        throws JSONException {

        if(positiveSize2D!=null) {
            drawingProperties.put("width", Commons.coordinateTo100TH_MM(positiveSize2D.getCx()));
            drawingProperties.put("height", Commons.coordinateTo100TH_MM(positiveSize2D.getCy()));
        }
    }

    public static JSONObject createJsonFromBlipFillProperties(Part contextPart, JSONObject attrs, CTBlipFillProperties blipFillProperties)
        throws JSONException {

        if(blipFillProperties!=null) {

			final JSONObject initialImageAttrs = attrs.optJSONObject("image");
			final JSONObject imageAttrs = initialImageAttrs!=null ? initialImageAttrs : new JSONObject();

            imageAttrs.put("imageUrl", DMLHelper.getBlipUrl(contextPart, blipFillProperties.getBlip()));

            final CTRelativeRect srcRect = blipFillProperties.getSrcRect();
            if (srcRect!=null) {
                final int l = srcRect.getL() / 1000;
                final int t = srcRect.getT() / 1000;
                final int r = srcRect.getR() / 1000;
                final int b = srcRect.getB() / 1000;

                // add possible crop properties
                if (l != 0)
                    imageAttrs.put("cropLeft", l);
                if (t != 0)
                    imageAttrs.put("cropTop", t);
                if (r != 0)
                    imageAttrs.put("cropRight", r);
                if (b != 0)
                    imageAttrs.put("cropBottom", b);
            }
	        if(initialImageAttrs==null&&!imageAttrs.isEmpty()) {
				attrs.put("image", imageAttrs);
			}
        }
        return attrs;
    }

    public static void createJsonFromTransform2D(JSONObject attrs, ITransform2D xFrm)
    	throws JSONException {

    	if(xFrm!=null) {
			final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
			final JSONObject drawingAttrs = initialDrawingAttrs!=null ? initialDrawingAttrs : new JSONObject();

            final CTPoint2D point2D = xFrm.getOff();
            if(point2D!=null) {
                if(point2D.getX()!=0) {
                    if(!drawingAttrs.hasAndNotNull("left")) {
                    	drawingAttrs.put("left", Commons.coordinateTo100TH_MM(point2D.getX()));
                    }
                }
                if(point2D.getY()!=0) {
                    if(!drawingAttrs.hasAndNotNull("top")) {
                    	drawingAttrs.put("top", Commons.coordinateTo100TH_MM(point2D.getY()));
                    }
                }
            }
            final CTPositiveSize2D positiveSize2D = xFrm.getExt();
            if(positiveSize2D!=null) {
                if(!drawingAttrs.hasAndNotNull("width")) {
                	drawingAttrs.put("width", Commons.coordinateTo100TH_MM(positiveSize2D.getCx()));
                }
                if(!drawingAttrs.hasAndNotNull("height")) {
                	drawingAttrs.put("height", Commons.coordinateTo100TH_MM(positiveSize2D.getCy()));
                }
            }
            if(xFrm.isFlipH()) {
                if(!drawingAttrs.hasAndNotNull("flipH")) {
                	drawingAttrs.put("flipH", true );
                }
            }
            if(xFrm.isFlipV()) {
                if(!drawingAttrs.hasAndNotNull("flipV")) {
                	drawingAttrs.put("flipV", true );
                }
            }
	        if(initialDrawingAttrs==null&&!drawingAttrs.isEmpty()) {
				attrs.put("drawing", drawingAttrs);
			}
        }
    }

    public static void applyTransform2DFromJson(ITransform2D transform2D, JSONObject attrs) {

    	final JSONObject drawingAttrs = attrs.optJSONObject("drawing");
    	if(drawingAttrs!=null) {
			final Object width = drawingAttrs.opt("width");
			if(width!=null) {
	            if(width instanceof Integer) {
	                CTPositiveSize2D extend = transform2D.getExt();
	                if(extend==null) {
	                	extend = Context.getDmlObjectFactory().createCTPositiveSize2D();
	                	transform2D.setExt(extend);
	                }
	                extend.setCx(((Integer)width).intValue()*360);
	            }
			}
			final Object height = drawingAttrs.opt("height");
			if(height!=null) {
				if(height instanceof Integer) {
	                CTPositiveSize2D extend = transform2D.getExt();
	                if(extend==null) {
	                	extend = Context.getDmlObjectFactory().createCTPositiveSize2D();
	                	transform2D.setExt(extend);
	                }
	                extend.setCy(((Integer)height).intValue()*360);
				}
			}
			final Object left = drawingAttrs.opt("left");
			if(left!=null) {
			    if(left instanceof Integer) {
			        CTPoint2D pos = transform2D.getOff();
			        if(pos==null) {
			            pos = Context.getDmlObjectFactory().createCTPoint2D();
			            transform2D.setOff(pos);
			        }
			        pos.setX(((Integer)left).longValue()*360);
			    }
			}
            final Object top = drawingAttrs.opt("top");
            if(top!=null) {
                if(top instanceof Integer) {
                    CTPoint2D pos = transform2D.getOff();
                    if(pos==null) {
                        pos = Context.getDmlObjectFactory().createCTPoint2D();
                        transform2D.setOff(pos);
                    }
                    pos.setY(((Integer)top).longValue()*360);
                }
            }
			final Object flipH = drawingAttrs.opt("flipH");
			if(flipH!=null) {
				if(flipH instanceof Boolean) {
	                transform2D.setFlipH((Boolean)flipH);
				}
				else {
					transform2D.setFlipH(null);
				}
			}
			final Object flipV = drawingAttrs.opt("flipV");
			if(flipV!=null) {
				if(flipV instanceof Boolean) {
	                transform2D.setFlipV((Boolean)flipV);
				}
				else {
					transform2D.setFlipV(null);
				}
			}
    	}
    }

    public static void applyFillPropertiesFromJson(IFillProperties fillProperties, JSONObject attrs)
    	throws JSONException {

    	final JSONObject fillAttrs = attrs.optJSONObject("fill");
    	if(fillAttrs!=null) {
	    	fillProperties.setNoFill(null);
			fillProperties.setSolidFill(null);
			fillProperties.setGradFill(null);
			fillProperties.setPattFill(null);
			fillProperties.setBlipFill(null);
			fillProperties.setGrpFill(null);

			final Object fillProp = createFillPropsFromJson(fillAttrs);
			if (fillProp instanceof CTNoFillProperties) {
				fillProperties.setNoFill((CTNoFillProperties)fillProp);
			} else if (fillProp instanceof CTSolidColorFillProperties) {
				fillProperties.setSolidFill((CTSolidColorFillProperties)fillProp);
			}
    	}
    }

    public static void createJsonFromShapeProperties(JSONObject attrs, CTShapeProperties shapeProperties, boolean createTransform)
        throws JSONException {

        createJsonFromShapeProperties(attrs, shapeProperties, null, null, createTransform);
    }

    private static enum ShapeStyleType {
        FILL,
        LINE,
        FONT,
        EFFECT
    }

    // styleMatrixReference can be zero, if available then the color from the styleMatrix is used
    private static void createJsonFromFillStyle(Object source, CTStyleMatrixReference styleMatrixReference, JSONObject dest)
        throws JSONException {

        if(source instanceof CTSolidColorFillProperties) {
            dest.put("type", "solid");
            createJsonColorFromSolidColorFillProperties(dest, (CTSolidColorFillProperties)source);
        }
        else if(source instanceof CTNoFillProperties) {
            dest.put("type", "none");
        }
        else if(source instanceof CTGradientFillProperties) {
            // mapping gradient to solid fill...
            dest.put("type", "solid");
            final CTGradientStopList gradientStopList = ((CTGradientFillProperties)source).getGsLst();
            if(gradientStopList!=null) {
                final List<CTGradientStop> list = gradientStopList.getGs();
                if(!list.isEmpty()) {
                    final CTGradientStop gradientStop = list.get(0);
                    createJsonColorFromSolidColorFillProperties(dest, gradientStop);
                }
            }
        }
        else if(source instanceof CTPatternFillProperties) {
            // mapping pattern fill to solid fill
            dest.put("type", "solid");
            final CTColor fgFill = ((CTPatternFillProperties)source).getFgClr();
            if(fgFill!=null) {
                createJsonColorFromSolidColorFillProperties(dest, fgFill);
            }
        }
        else if(source instanceof CTBlipFillProperties) {
            // mapping blipFill to solid fill
            dest.put("type", "solid");
        }
        if(styleMatrixReference!=null) {
            createJsonColorFromSolidColorFillProperties(dest, styleMatrixReference);
        }
    }

    // styleMatrixReference can be zero, if available then the color from the styleMatrix is used
    private static void createJsonFromLineStyle(CTLineProperties lineProperties, CTStyleMatrixReference styleMatrixReference, JSONObject dest)
        throws JSONException {

        if(lineProperties.getNoFill()!=null) {
            createJsonFromFillStyle(lineProperties.getNoFill(), styleMatrixReference, dest);
        }
        else if(lineProperties.getSolidFill()!=null) {
            createJsonFromFillStyle(lineProperties.getSolidFill(), styleMatrixReference, dest);
        }
        else if(lineProperties.getGradFill()!=null) {
            createJsonFromFillStyle(lineProperties.getGradFill(), styleMatrixReference, dest);
        }
        else if(lineProperties.getPattFill()!=null) {
            createJsonFromFillStyle(lineProperties.getPattFill(), styleMatrixReference, dest);
        }
        if(lineProperties.getW()!=null) {
            dest.put("width", lineWidthToThmm(lineProperties.getW()));
        }
        if(lineProperties.getPrstDash()!=null) {
            switch(lineProperties.getPrstDash().getVal()) {
            case SYS_DOT:
            case DOT:
                dest.put("style", "dotted");
                break;
            case LG_DASH:
            case SYS_DASH:
            case DASH:
                dest.put("style", "dashed");
                break;
            case LG_DASH_DOT:
            case SYS_DASH_DOT:
            case DASH_DOT:
                dest.put("style", "dashDot");
                break;
            case LG_DASH_DOT_DOT:
            case SYS_DASH_DOT_DOT:
                dest.put("style", "dashDotDot");
                break;
            case SOLID:
            default:
                break;
            }
        }
    }
    private static void createJsonFromStyleMatrix(Theme theme, CTShapeStyle shapeStyle, ShapeStyleType type, JSONObject dest)
        throws JSONException {

        if(shapeStyle==null&&theme==null) {
            return;
        }
        final BaseStyles themeElements = theme.getThemeElements();
        if(themeElements==null) {
            return;
        }
        final CTStyleMatrix fmtScheme = themeElements.getFmtScheme();
        if(fmtScheme==null) {
            return;
        }
        if(type==ShapeStyleType.FILL) {
            final CTStyleMatrixReference fillStyleReference = shapeStyle.getFillRef();
            if(fillStyleReference!=null) {
                final long idx = fillStyleReference.getIdx();
                if(idx==0||idx==1000) {
                    dest.put("type", "none");
                    return;
                }
                else {
                    List<Object> fillMatrix = null;
                    if(idx>=1&&idx<1000) {
                        final CTFillStyleList fillStyleList = fmtScheme.getFillStyleLst();
                        if(fillStyleList!=null) {
                            fillMatrix = fillStyleList.getContent();
                        }
                    }
                    else if(idx>1000) {
                        final CTBackgroundFillStyleList bgFillStyleList = fmtScheme.getBgFillStyleLst();
                        if(bgFillStyleList!=null) {
                            fillMatrix = bgFillStyleList.getContent();
                        }
                    }
                    if(fillMatrix!=null) {
                        final int index = (int)(idx>1000?idx-1001:idx-1);
                        if(index<fillMatrix.size()) {
                            createJsonFromFillStyle(fillMatrix.get(index), fillStyleReference, dest);
                        }
                    }
                }
            }
        }
        else if(type==ShapeStyleType.LINE) {
           final CTStyleMatrixReference lineStyleReference = shapeStyle.getLnRef();
           if(lineStyleReference!=null) {
               final int idx = (int)lineStyleReference.getIdx();
               final CTLineStyleList lineStyleList = fmtScheme.getLnStyleLst();
               if(lineStyleList!=null) {
                   final List<CTLineProperties> lineMatrix = lineStyleList.getLn();
                   if(idx>=0&&idx<lineMatrix.size()) {
                       createJsonFromLineStyle(lineMatrix.get(idx), lineStyleReference, dest);
                   }
               }
           }
        }
        return;
    }

    public static void createJsonFromShapeProperties(JSONObject attrs, CTShapeProperties shapeProperties, Theme theme, CTShapeStyle shapeStyle, boolean createTransform)
        throws JSONException {

        final JSONObject initialFillAttrs = attrs.optJSONObject("fill");
        final JSONObject fillAttrs = initialFillAttrs!=null ? initialFillAttrs : new JSONObject(1);

        if(shapeProperties!=null) {
        	if(createTransform) {
	            createJsonFromTransform2D(attrs, shapeProperties.getXfrm());
        	}
        	createJsonFromLineProperties(attrs, theme, shapeStyle, shapeProperties.getLn());
        	if(shapeStyle!=null) {
        	    // try to apply the style first...
        	    createJsonFromStyleMatrix(theme, shapeStyle, ShapeStyleType.FILL, fillAttrs);
        	}
        	// then each hard applied fill attribute
        	if(shapeProperties.getNoFill()!=null) {
        	    createJsonFromFillStyle(shapeProperties.getNoFill(), null, fillAttrs);
	    	}
	    	else if(shapeProperties.getSolidFill()!=null) {
	    	    createJsonFromFillStyle(shapeProperties.getSolidFill(), null, fillAttrs);
            }
	    	else if(shapeProperties.getBlipFill()!=null) {
	    	    createJsonFromFillStyle(shapeProperties.getBlipFill(), null, fillAttrs);
	    	}
	    	else if(shapeProperties.getGradFill()!=null) {
	    	    createJsonFromFillStyle(shapeProperties.getGradFill(), null, fillAttrs);
	    	}
	    	else if(shapeProperties.getPattFill()!=null) {
	    	    createJsonFromFillStyle(shapeProperties.getPattFill(), null, fillAttrs);
	    	}
        	/*
        	if(shapeProperties.getGrpFill()!=null) {

        	}
        	*/
        }
        // TODO: special chart handling... should be removed if frontend is no longer depenending to a type solid default
        if(shapeStyle==null) {
            if(!fillAttrs.has("type")) {
                fillAttrs.put("type", "solid");
            }
            JSONObject lineAttrs = attrs.optJSONObject("line");
            if (null == lineAttrs) {
                lineAttrs = new JSONObject();
                attrs.put("line", lineAttrs);
            }
            if (!lineAttrs.has("type")) {
                lineAttrs.put("type", "auto");
            }
        }
        if(initialFillAttrs==null&&!fillAttrs.isEmpty()) {
            attrs.put("fill", fillAttrs);
        }
    }

	public static CTShapeProperties applyShapePropertiesFromJson(CTShapeProperties shapeProperties, JSONObject attrs)
		throws JSONException {

		if(shapeProperties==null) {
			shapeProperties = Context.getDmlObjectFactory().createCTShapeProperties();
		}
		CTTransform2D transform2d = shapeProperties.getXfrm();
		if(transform2d==null) {
		    transform2d = Context.getDmlObjectFactory().createCTTransform2D();
		    shapeProperties.setXfrm(transform2d);
		}
		applyTransform2DFromJson(transform2d, attrs);
		shapeProperties.setLn(applyLinePropertiesFromJson(shapeProperties.getLn(), attrs));
		applyFillPropertiesFromJson(shapeProperties, attrs);
		return shapeProperties;
	}

    public static void createJsonFromLineProperties(JSONObject attrs, Theme theme, CTShapeStyle shapeStyle, CTLineProperties lineProperties)
        throws JSONException {

        final JSONObject initialLineAttrs = attrs.optJSONObject("line");
        final JSONObject lineAttrs = initialLineAttrs!=null ? initialLineAttrs : new JSONObject(2);

        if(shapeStyle!=null) {
            // try to apply the style first...
            createJsonFromStyleMatrix(theme, shapeStyle, ShapeStyleType.LINE, lineAttrs);
        }
    	if(lineProperties!=null) {
    	    createJsonFromLineStyle(lineProperties, null, lineAttrs);
    	}

    	// TODO: the following code is superfluous if line type "auto" is removed and the default is "none"
    	if(!lineAttrs.has("type")) {
    	    lineAttrs.put("type", "none");
    	}

    	if(initialLineAttrs==null&&!lineAttrs.isEmpty()) {
            attrs.put("line", lineAttrs);
        }

    }

	public static CTLineProperties applyLinePropertiesFromJson(CTLineProperties lineProperties, JSONObject attrs) throws JSONException {

		final JSONObject lineAttrs = attrs.optJSONObject("line");
		if(lineAttrs!=null) {
			if(lineProperties==null) {
				lineProperties = Context.getDmlObjectFactory().createCTLineProperties();
			}
			final Object type = lineAttrs.opt("type");
			if(type!=null) {
                final CTSolidColorFillProperties oldSolidFill = lineProperties.getSolidFill();
			    lineProperties.setNoFill(null);
                lineProperties.setSolidFill(null);
                lineProperties.setGradFill(null);
                lineProperties.setPattFill(null);
			    if(type instanceof String) {
                    if(((String)type).equals("none")||((String)type).equals("auto")) {
                        lineProperties.setNoFill(Context.getDmlObjectFactory().createCTNoFillProperties());
                    }
                    else if(((String)type).equals("solid")) {
                        if(oldSolidFill!=null) {
                            lineProperties.setSolidFill(oldSolidFill);
                        }
                        else {
                            final CTSolidColorFillProperties solidColorFillProperties = Context.getDmlObjectFactory().createCTSolidColorFillProperties();
                            final CTPresetColor presetColor = Context.getDmlObjectFactory().createCTPresetColor();
                            presetColor.setVal(STPresetColorVal.BLACK);
                            solidColorFillProperties.setPrstClr(presetColor);
                            lineProperties.setSolidFill(solidColorFillProperties);
                        }
                    }
			    }
			    else {
			        lineProperties.setNoFill(Context.getDmlObjectFactory().createCTNoFillProperties());
			    }
			}
			final Object color = lineAttrs.opt("color");
			if(color!=null) {
			    // applying color only possible if fill type is solid
			    final CTSolidColorFillProperties solidFill = lineProperties.getSolidFill();
			    if(solidFill!=null) {
			        if(color instanceof JSONObject) {
			            createSolidColorFillPropertiesFromJson(solidFill, (JSONObject)color);
	                }
			        else {
                        final CTPresetColor presetColor = Context.getDmlObjectFactory().createCTPresetColor();
                        presetColor.setVal(STPresetColorVal.BLACK);
                        solidFill.setPrstClr(presetColor);
			        }
			    }
			}
			final Object width = lineAttrs.opt("width");
			if(width!=null) {
			    if(width instanceof Number) {
                    lineProperties.setW(thmmToLineWidth(lineAttrs.getLong("width")));
			    }
			    else {
			        lineProperties.setW(null);
			    }
			}
			final Object style = lineAttrs.opt("style");
			if(style!=null) {
			    CTPresetLineDashProperties presetDash = null;
			    if(style instanceof String) {
			        if(((String)style).equals("dotted")) {
			            presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
			            presetDash.setVal(STPresetLineDashVal.DOT);
			        }
			        else if (((String)style).equals("dashed")) {
                        presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
                        presetDash.setVal(STPresetLineDashVal.DASH);
			        }
			        else if (((String)style).equals("dashDot")) {
                        presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
                        presetDash.setVal(STPresetLineDashVal.DASH_DOT);
			        }
			        else if (((String)style).equals("dashDotDot")) {
                        presetDash = Context.getDmlObjectFactory().createCTPresetLineDashProperties();
                        presetDash.setVal(STPresetLineDashVal.LG_DASH_DOT_DOT);
			        }
			    }
		        lineProperties.setPrstDash(presetDash);
			}
		}
		return lineProperties;
	}

    public static void createJsonColorFromSolidColorFillProperties(JSONObject dest, IColorChoice fillProperties)
        throws JSONException {

        if(fillProperties!=null) {
            if (fillProperties.getScrgbClr()!= null) {
                createJsonFromScrgbClr(dest, fillProperties.getScrgbClr());
            }
            else if(fillProperties.getSrgbClr()!= null) {
                createJsonFromSrgbClr(dest, fillProperties.getSrgbClr());
            }
            else if (fillProperties.getHslClr()!= null) {
                createJsonFromHslClr(dest, fillProperties.getHslClr());
            }
            else if (fillProperties.getSchemeClr()!= null) {
                createJsonFromSchemeClr(dest, fillProperties.getSchemeClr());
            }
            else if (fillProperties.getSysClr()!= null) {
                createJsonFromSysClr(dest, fillProperties.getSysClr());
            }
            else if (fillProperties.getPrstClr()!= null) {
                createJsonFromPrstClr(dest, fillProperties.getPrstClr());
            }
        }
    }

	public static boolean createSolidColorFillPropertiesFromJson(CTSolidColorFillProperties docx, JSONObject json) throws JSONException {
		docx.setSysClr(null);
		docx.setScrgbClr(null);
		docx.setSrgbClr(null);
		docx.setHslClr(null);
		docx.setPrstClr(null);
		docx.setSchemeClr(null);


		final String type = json.getString("type");

		if (type.equals("auto")) {
			return false;
		}else if (type.equals("scheme")) {
			final CTSchemeColor schemeClr = new CTSchemeColor();
			createSchemeColorFromJson(schemeClr, json);
			docx.setSchemeClr(schemeClr);
		} else if (type.equals("rgb")) {
			final CTSRgbColor srgbClr = new CTSRgbColor();
			createSRgbColorFromJson(srgbClr, json);
			docx.setSrgbClr(srgbClr);
		} else {
			Logger.getAnonymousLogger().warning("colortype is not implemented " + type);
			// TODO: other colors?
		}
		return true;
	}

	public static Object createFillPropsFromJson(JSONObject prop) throws JSONException {
		if (prop == null) {
			return null;
		}
		final String type = prop.optString("type", null);
		if (type != null && type.equals("none")) {
			return new CTNoFillProperties();
		} else if (type == null || type.equals("solid")) {
			final CTSolidColorFillProperties res = new CTSolidColorFillProperties();
			final JSONObject color = prop.optJSONObject("color");

			if(color!=null){
				if(!createSolidColorFillPropertiesFromJson(res, color)){
					return null;
				}
			}
			return res;
		}
		return null;
	}

    public static void createJsonFromScrgbClr(JSONObject dest, CTScRgbColor color)
        throws JSONException {

        final byte[] value = new byte[3];
        value[0] = JSONHelper.gammaChannel(color.getR());
        value[1] = JSONHelper.gammaChannel(color.getG());
        value[2] = JSONHelper.gammaChannel(color.getB());
        dest.put("color", JSONHelper.makeColor("rgb", Commons.bytesToHexString(value, 0, 3), color.getEGColorTransform()));
    }

    public static void createJsonFromSrgbClr(JSONObject dest, CTSRgbColor color)
        throws JSONException {

        final byte[] value = color.getVal();
        dest.put("color", JSONHelper.makeColor("rgb", Commons.bytesToHexString(value, 0, 3), color.getEGColorTransform()));
    }

    public static void createJsonFromHslClr(JSONObject dest, CTHslColor color)
        throws JSONException {

        dest.put("color", JSONHelper.makeColor("hsl", JSONHelper.makeJSON("h", color.getHue(), "s", color.getSat(), "l", color.getLum()), color.getEGColorTransform()));
    }

    public static void createJsonFromSchemeClr(JSONObject dest, CTSchemeColor color)
        throws JSONException {

        String value = null;
        final STSchemeColorVal sourveVal = color.getVal();
        if (sourveVal != null) {
            switch (sourveVal) {
			case ACCENT_1:
			case ACCENT_2:
			case ACCENT_3:
			case ACCENT_4:
			case ACCENT_5:
			case ACCENT_6:
				value = sourveVal.value();
				break;
			case BG_1:
				value = "background1";
				break;
			case BG_2:
				value = "background2";
				break;
			case DK_1:
				value = "dark1";
				break;
			case DK_2:
				value = "dark2";
				break;
			case TX_1:
				value = "text1";
				break;
			case TX_2:
				value = "text2";
				break;
			case LT_1:
				value = "light1";
				break;
			case LT_2:
				value = "light2";
				break;
			case HLINK:
				value = "hyperlink";
				break;
			case FOL_HLINK:
				value = "followedHyperlink";
				break;
			case PH_CLR:
				value = "style";
				break;
			}
        }
        dest.put("color", JSONHelper.makeColor("scheme", value, color.getEGColorTransform()));
    }

	public static void createSchemeColorFromJson(CTSchemeColor docx, JSONObject json) throws JSONException {
		final String value = json.getString("value");
		final STSchemeColorVal colorVal;
		if (value.equals("background1")) {
			colorVal = STSchemeColorVal.BG_1;
		} else if (value.equals("background2")) {
			colorVal = STSchemeColorVal.BG_2;
		} else if (value.equals("dark1")) {
			colorVal = STSchemeColorVal.DK_1;
		} else if (value.equals("dark2")) {
			colorVal = STSchemeColorVal.DK_2;
		} else if (value.equals("text1")) {
			colorVal = STSchemeColorVal.TX_1;
		} else if (value.equals("text2")) {
			colorVal = STSchemeColorVal.TX_2;
		} else if (value.equals("light1")) {
			colorVal = STSchemeColorVal.LT_1;
		} else if (value.equals("light2")) {
			colorVal = STSchemeColorVal.LT_2;
		} else if (value.equals("hyperlink")) {
			colorVal = STSchemeColorVal.HLINK;
		} else if (value.equals("followedHyperlink")) {
			colorVal = STSchemeColorVal.FOL_HLINK;
		} else if (value.equals("style")) {
			colorVal = STSchemeColorVal.PH_CLR;
		} else {
			STSchemeColorVal tmpVal;
			try {
				tmpVal = STSchemeColorVal.fromValue(value);
			} catch (final IllegalArgumentException e) {
				Logger.getAnonymousLogger().warning("could not find STSchemeColorVal for " + value);
				tmpVal = STSchemeColorVal.TX_1;
			}
			colorVal = tmpVal;
		}
		docx.setVal(colorVal);
		JSONHelper.saveTransformations(docx.getEGColorTransform(), json);
	}

	public static void createSRgbColorFromJson(CTSRgbColor docx, JSONObject json) throws JSONException {
		docx.setVal(Commons.hexStringToBytes(json.getString("value")));
		JSONHelper.saveTransformations(docx.getEGColorTransform(), json);
	}

    public static void createJsonFromSysClr(JSONObject dest, CTSystemColor color)
        throws JSONException {

        dest.put("color", JSONHelper.makeColor("system", color.getVal(), color.getEGColorTransform()));
    }

    public static void createJsonFromPrstClr(JSONObject dest, CTPresetColor color)
    	throws JSONException {

        dest.put("color", JSONHelper.makeColor("preset", color.getVal().value(), color.getEGColorTransform()));
    }

    public static JSONObject createJsonFromChartColor(CTChartColor color)
        throws JSONException {

        JSONObject jsonChartColor = null;
        if(color!=null) {
            jsonChartColor = new JSONObject();
            jsonChartColor.put("id", color.getId());
            jsonChartColor.put("meth", color.getMeth());
            final List<CTSchemeColor> schemeClrList = color.getSchemeClr();
            if(schemeClrList!=null) {
                final JSONArray jsonArray = new JSONArray();
                for(final CTSchemeColor schemeClr:schemeClrList) {
                    final JSONObject jsonSchemeColor = new JSONObject();
                    createJsonFromSchemeClr(jsonSchemeColor, schemeClr);
                    jsonArray.put(jsonSchemeColor);
                }
                jsonChartColor.put("schemeClr", jsonArray);
            }
            final List<CTSchemeColor> variationList = color.getVariation();
            if(variationList!=null ) {
                final JSONArray jsonArray = new JSONArray();
                for(final CTSchemeColor schemeClr:variationList) {
                    final JSONObject resColor = new JSONObject();
                	createJsonFromSchemeClr(resColor, schemeClr);
					if (resColor != null){
						jsonArray.put(resColor);
					}
                }
				if (!jsonArray.isEmpty()){
					jsonChartColor.put("variation", jsonArray);
				}
            }
        }
        return jsonChartColor!=null&&!jsonChartColor.isEmpty()?jsonChartColor:null;
    }

    public static void createJsonFromNonVisualDrawingProperties(JSONObject attrs, CTNonVisualDrawingProps nonVisualDrawingProps)
        throws JSONException {

        if(nonVisualDrawingProps==null) {
            return;
        }
		final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
		final JSONObject drawingAttrs = initialDrawingAttrs!=null ? initialDrawingAttrs : new JSONObject();

        final String name = nonVisualDrawingProps.getName();
        if(name!=null&&!name.isEmpty()) {
            drawingAttrs.put("name", name);
        }
        final String description = nonVisualDrawingProps.getDescr();
        if(description!=null&&!description.isEmpty()) {
            drawingAttrs.put("description", description);
        }
        if(initialDrawingAttrs==null&&!drawingAttrs.isEmpty()) {
			attrs.put("drawing", drawingAttrs);
		}
    }

    public static void applyNonVisualDrawingProperties(JSONObject drawingProperties, CTNonVisualDrawingProps nonVisualDrawingProps) {

        if(nonVisualDrawingProps==null) {
            return;
        }

        if(drawingProperties.has("name")) {
            final Object optName = drawingProperties.opt("name");
            if(optName instanceof String) {
                nonVisualDrawingProps.setName((String)optName);
            }
            else {
                // name is a required xml attribute... so we are setting a dummy
                nonVisualDrawingProps.setName("...");
            }
        }
        if(drawingProperties.has("description")) {
            final Object optDescription = drawingProperties.opt("description");
            if(optDescription instanceof String) {
                nonVisualDrawingProps.setDescr((String)optDescription);
            }
            else {
                nonVisualDrawingProps.setDescr(null);
            }
        }
    }

    public static CTTransform2D createTransform2D(int x100thmm, int y100thmm, int width100thmm, int height100thmm) {
        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTTransform2D transform2D = dmlObjectFactory.createCTTransform2D();
        final CTPoint2D point2D = dmlObjectFactory.createCTPoint2D();
        point2D.setX(Commons.coordinateFrom100TH_MM(x100thmm));
        point2D.setY(Commons.coordinateFrom100TH_MM(y100thmm));
        transform2D.setOff(point2D);
        final CTPositiveSize2D positiveSize2D = dmlObjectFactory.createCTPositiveSize2D();
        positiveSize2D.setCx(Commons.coordinateFrom100TH_MM(width100thmm));
        positiveSize2D.setCy(Commons.coordinateFrom100TH_MM(height100thmm));
        return transform2D;
    }

    public static CTPresetGeometry2D createPresetGeometry2D(STShapeType presetType) {
        final CTPresetGeometry2D presetGeometry2D = Context.getDmlObjectFactory().createCTPresetGeometry2D();
        presetGeometry2D.setPrst(presetType);
        return presetGeometry2D;
    }

    public static CTBlipFillProperties createBlipFillProperties(OperationDocument operationDocument, Part part, String imageUrl)
        throws Exception {

        final org.docx4j.dml.ObjectFactory dmlObjectFactory = Context.getDmlObjectFactory();
        final CTBlipFillProperties blipFillProperties = dmlObjectFactory.createCTBlipFillProperties();
        final CTBlip blip = dmlObjectFactory.createCTBlip();
        blipFillProperties.setBlip(blip);
        blipFillProperties.setStretch(dmlObjectFactory.createCTStretchInfoProperties());
        final Relationship relationship = Commons.createGraphicRelation(operationDocument, part, imageUrl);
        if(relationship!=null) {
            final boolean linked = relationship.getTargetMode() != null && relationship.getTargetMode().equals("External");
            if(linked) {
                blip.setLink(relationship.getId());
            }
            else {
                blip.setEmbed(relationship.getId());
            }
        }
        return blipFillProperties;
    }

    public static CTNonVisualDrawingProps createNonVisualDrawingProps(String name) {
        final CTNonVisualDrawingProps nonVisualDrawingProps = Context.getDmlObjectFactory().createCTNonVisualDrawingProps();
        nonVisualDrawingProps.setName(name);
        return nonVisualDrawingProps;
    }

    public static void addTo(CTRegularTextRun textRun, StringBuffer stringBuffer) {
    	stringBuffer.append(textRun.getT());
    }

    public static void addTo(CTTextField textRun, StringBuffer stringBuffer) {
    	stringBuffer.append(textRun.getT());
    }

    public static void addTo(CTTextLineBreak textRun, StringBuffer stringBuffer) {
    	stringBuffer.append("\n");
    }

	public static String getText(CTTextBody rich) {
		final StringBuffer buf = new StringBuffer();

		final List<CTTextParagraph> paras = rich.getP();
		for (int i = 0; i < paras.size(); i++) {
			final CTTextParagraph p = paras.get(i);
			final List<Object> textes = p.getEGTextRun();
			for (final Object text : textes) {
				if(text instanceof CTRegularTextRun) {
					addTo((CTRegularTextRun)text, buf);
				}
				else if (text instanceof CTTextField) {
					addTo((CTTextField)text, buf);
				}
				else if (text instanceof CTTextLineBreak) {
					addTo((CTTextLineBreak)text, buf);
				}
			}
			if (i < paras.size() - 1) {
				buf.append("\n");
			}
		}
		return buf.toString();
	}

	public static JSONObject createJsonFromProperties(CTTitle title) throws JSONException {
		if (title == null) {
			return null;
		}
		final CTTx tx = title.getTx();
		if (tx == null) {
			return null;
		}
		final CTTextBody rich = tx.getRich();
		if (rich == null) {
			return null;
		}
		final String first = getText(rich);

		final JSONObject attrs = new JSONObject();
		final JSONArray link = new JSONArray();
		link.put(first);
		final JSONObject text = new JSONObject();
		text.put("link", link);
		attrs.put("text", text);

		final JSONObject character = DMLChartSpace.createJsonFromProperties(rich);
		if (character != null) {
			attrs.put("character", character);
		}
		return attrs;
	}

	public static void setTitleFromAttrs(CTTitle title, JSONObject attrs) throws JSONException {
		handleStandardTypes(title);
		final CTTx tx = title.getTx();
		final CTTextBody rich = tx.getRich();
		final CTTextBodyProperties body = rich.getBodyPr();
		final CTTextBody txPr = title.getTxPr();
		final CTTextBodyProperties body2 = txPr.getBodyPr();
		body.setRot(0);
		body2.setRot(0);

		final JSONObject text = attrs.optJSONObject("text");
		if(text==null){
			return;
		}


		final Object olink = text.opt("link");

		if (olink == null || !(olink instanceof JSONArray))
		{

			if (olink == null || olink.equals(JSONObject.NULL))
			{
				return;
			}
			else
			{
				Logger.getAnonymousLogger().warning("title text has no correct content " + text.opt("link"));
				return;
			}
		}

		final JSONArray links = (JSONArray) olink;
		if (links.length() > 0)
		{


			final List<CTTextParagraph> paras = rich.getP();
			paras.clear();
			for (int i = 0; i < links.length(); i++)
			{
				final CTTextParagraph p = new CTTextParagraph();
				setTextFromAttrs(p, attrs);
				paras.add(p);
				final List<Object> textHolder = p.getEGTextRun();
				textHolder.clear();

				final String link = links.getString(i);
				final String[] textes = link.split("\n");
				for (int j = 0; j < textes.length; j++)
				{
					final CTRegularTextRun reg = new CTRegularTextRun();
					reg.setRPr(GERMAN);
					reg.setT(textes[j]);
					textHolder.add(reg);
					if (j < textes.length - 1)
					{
						final CTTextLineBreak lb = new CTTextLineBreak();
						lb.setRPr(GERMAN);
						textHolder.add(lb);
					}
				}
			}
		}
	}

	public static void refreshTitleRotations(
		CTPlotArea area)
	{
		final List<IAxisDescription> axes = area.getValAxOrCatAxOrDateAx();
		final IListSer chart = area.getAreaChartOrArea3DChartOrLineChart().get(0);
		final String dir = chart.getBarDirection();
		for (int i = 0; i < axes.size(); i++)
		{
			boolean rotate = false;
			if (i == 1)
			{
				rotate = true;
			}
			if (dir == null || dir.equals("col"))
			{
				// normal
			}
			else
			{
				rotate = !rotate;
			}
			int rot = 0;
			if (rotate)
			{
				rot = -5400000;
			}
			final IAxisDescription axis = axes.get(i);
			final CTTitle title = axis.getTitle();

			if (title != null && title.getTx() != null)
			{
				final CTTx tx = title.getTx();
				final CTTextBody rich = tx.getRich();
				final CTTextBodyProperties body = rich.getBodyPr();
				body.setRot(rot);

				final CTTextBody txPr = title.getTxPr();
				if (null != txPr) {
				    final CTTextBodyProperties body2 = txPr.getBodyPr();
				    body2.setRot(rot);
				}
			}

			final CTTextBody lbl = axis.getTxPr();
			if (lbl != null && lbl.getBodyPr() != null)
			{
				final CTTextBodyProperties body = lbl.getBodyPr();
				body.setRot(0);
			}
		}
	}

	public static void handleStandardTypes(CTTextBody txPr) {
		CTTextBodyProperties body = txPr.getBodyPr();
		if(body==null){
			body = new CTTextBodyProperties();
			txPr.setBodyPr(body);
		}
		body.setSpcFirstLastPara(true);
		body.setVertOverflow(STTextVertOverflowType.ELLIPSIS);
		body.setVert(STTextVerticalType.HORZ);
		body.setWrap(STTextWrappingType.SQUARE);
		body.setAnchor(STTextAnchoringType.CTR);
		body.setAnchorCtr(true);

		CTTextListStyle lst2 = txPr.getLstStyle();
		if (lst2 == null) {
			lst2 = new CTTextListStyle();
			txPr.setLstStyle(lst2);
		}
	}

	private static void handleStandardTypes(CTTitle title) {
		CTTx tx = title.getTx();
		if (tx == null) {
			tx = new CTTx();
			title.setTx(tx);
		}
		CTTextBody rich = tx.getRich();
		if (rich == null) {
			rich = new CTTextBody();
			tx.setRich(rich);
		}
		handleStandardTypes(rich);

		CTLayout layout = title.getLayout();
		if (layout == null) {
			layout = new CTLayout();
			title.setLayout(layout);
		}
		CTBoolean overlay = title.getOverlay();
		if (overlay == null) {
			overlay = new CTBoolean();
			title.setOverlay(overlay);
		}
		overlay.setVal(false);
		CTShapeProperties spr = title.getSpPr();
		if (spr == null) {
			spr = getHiddenShapeProperties();
		}
		CTTextBody txpr = title.getTxPr();
		if (txpr == null) {
			txpr = new CTTextBody();
			title.setTxPr(txpr);
		}
		handleStandardTypes(txpr);

		final List<CTTextParagraph> ps = rich.getP();
		ps.clear();

		final List<CTTextParagraph> par2 = txpr.getP();
		par2.clear();

		final CTTextParagraph p = new CTTextParagraph();
		p.setPPr(getTextProp());
		par2.add(p);
		ps.add(p);

	}

	public static CTTextParagraphProperties getTextProp() {
		final CTTextParagraphProperties props = new CTTextParagraphProperties();
		final CTTextCharacterProperties def = new CTTextCharacterProperties();
		props.setDefRPr(def);

		def.setSz(1100);
		def.setB(false);
		def.setI(false);
		def.setU(STTextUnderlineType.NONE);
		def.setStrike(STTextStrikeType.NO_STRIKE);
		def.setKern(1200);
		def.setBaseline(0);

		final TextFont latin = new TextFont();
		latin.setTypeface("+mn-lt");
		def.setLatin(latin);
		final TextFont ea = new TextFont();
		ea.setTypeface("+mn-ea");
		def.setEa(ea);
		final TextFont cs = new TextFont();
		cs.setTypeface("+mn-cs");
		def.setCs(cs);

		def.setSolidFill(null);
		def.setNoFill(null);

		return props;
	}

	public static void addToOperations(CTPlotArea plotArea, JSONArray operations, JSONArray position) throws JSONException {
		final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();
		if (axes != null && axes.size() > 0) {
			for (int i = 0; i < axes.size(); i++) {
				final IAxisDescription axe = axes.get(i);

				final String axisParam;
				if (i == 0) {
					axisParam = "x";
				} else if (i == 1) {
					axisParam = "y";
				} else if (i == 2) {
					axisParam = "z";
				} else {
					return;
				}

				getAxeOperation(axe, operations, axisParam, position);
				getGridOperation(axe, operations, axisParam, position);
			}
		}
	}

	private static final void getAxeOperation(IAxisDescription axe, JSONArray operations, String axisParam, JSONArray position) throws JSONException {

		final JSONObject axis = new JSONObject();
        final CTScaling scaling = axe.getScaling();
        if(scaling!=null) {
            if (scaling.getMin()!=null) {
                axis.put("min", scaling.getMin().getVal());
            }
            if(scaling.getMax()!=null) {
                axis.put("max", scaling.getMax().getVal());
            }
        }
        JSONObject character = null;
		final JSONObject axisOp = new JSONObject();
		final CTBoolean del = axe.getDelete();
		if (del != null && del.isVal()) {
			final JSONObject line = new JSONObject();
			line.put("type", "none");
			axisOp.put("line", line);
		} else {
			final CTShapeProperties shape = axe.getSpPr();
			if (shape != null) {
                DMLHelper.createJsonFromShapeProperties(axisOp, shape, false);
			}
			if (!axisOp.has("line")) {
				axisOp.put("line", makeStandardShapeType());
			}

			final CTTickLblPos label = axe.getTickLblPos();
			if (label != null && label.getVal() != STTickLblPos.NONE) {
				axis.put("label", true);
				character = DMLChartSpace.createJsonFromProperties(axe.getTxPr());
			}

			getAxeTitleOperation(axe, operations, axisParam, position);
		}

		if (axis.length() > 0) {
			axisOp.put("axis", axis);
		}
		if (character != null) {
			axisOp.put("character", character);
		}


		if (axisOp.length() > 0) {
			final JSONObject op = new JSONObject();
			op.put("name", "setChartAxisAttributes");
			op.put("start", position);
			op.put("axis", axisParam);
			op.put("attrs", axisOp);
			operations.put(op);
		}
	}

	public static void setAxisAttributes(CTPlotArea plotArea, String axis, JSONObject attrs) throws JSONException {
		final IAxisDescription ax = getAxis(plotArea, axis);
		if (ax == null) {
			return;
		}
		final CTShapeProperties shape = applyShapePropertiesFromJson(ax.getSpPr(), attrs);
		ax.setSpPr(shape);
		CTTickMark tickmark = ax.getMajorTickMark();
		if (tickmark == null)
		{
			tickmark = new CTTickMark();
			ax.setMajorTickMark(tickmark);
		}
		if (shape.getLn() != null && shape.getLn().getNoFill() != null)
		{
			tickmark.setVal(STTickMark.NONE);
		}
		else
		{
			tickmark.setVal(STTickMark.OUT);
		}

		final JSONObject a = attrs.optJSONObject("axis");
		if (a != null) {
			CTTickLblPos lblPos = ax.getTickLblPos();
			if (lblPos == null) {
				lblPos = new CTTickLblPos();
				ax.setTickLblPos(lblPos);
			}
			final boolean label = a.optBoolean("label", false);

			if (label) {
				lblPos.setVal(STTickLblPos.LOW);


				CTTextBody txPr = ax.getTxPr();

				if (txPr == null) {
					txPr = new CTTextBody();
					ax.setTxPr(txPr);
					DMLHelper.handleStandardTypes(txPr);
				}
				setCharacterFromAttrs(txPr, attrs);

			} else {
				lblPos.setVal(STTickLblPos.NONE);
			}
		}
	}

	public static void setCharacterFromAttrs(CTTextBody txPr, JSONObject attrs) throws JSONException {
		final List<CTTextParagraph> ps = txPr.getP();
		ps.clear();
		final CTTextParagraph p = new CTTextParagraph();
		ps.add(p);
		setTextFromAttrs(p, attrs);
	}

	public static void setTextFromAttrs(CTTextParagraph p, JSONObject attrs) throws JSONException {
		final JSONObject character = attrs.optJSONObject("character");


		CTTextParagraphProperties prop = p.getPPr();
		if(prop == null){
			prop = getTextProp();
			p.setPPr(prop);
		}

		if (character != null)
		{

			final CTTextCharacterProperties def = prop.getDefRPr();

			def.setSz(ptTofontSize(character.optDouble("fontSize", 10)));

			CTSolidColorFillProperties solid = def.getSolidFill();
			if (solid == null)
			{
				solid = new CTSolidColorFillProperties();
				def.setSolidFill(solid);
			}

			final JSONObject color = character.optJSONObject("color");
			if (color != null)
			{
				if (!createSolidColorFillPropertiesFromJson(def.getSolidFill(), color))
				{
					def.setSolidFill(null);
				}
			}
		}
	}

	private static IAxisDescription getAxis(CTPlotArea plotArea, String axis) {
		final int index;
		if (axis.equals("x")) {
			index = 0;
		} else if (axis.equals("y")) {
			index = 1;
		} else if (axis.equals("z")) {
			index = 2;
		} else {
			return null;
		}
		final List<IAxisDescription> axes = plotArea.getValAxOrCatAxOrDateAx();

		if (index >= axes.size()) {
			return null;
		}

		IAxisDescription res = axes.get(index);

		if (res == null) {
			if (axis.equals("x")) {
				final CTCatAx cat = new CTCatAx();
				cat.setMinorGridlines(getHiddenChartLines());
				cat.setMinorTickMark(getHiddenTickMark());
				cat.setNumFmt(createNumFmt());
				res = cat;
			} else {
				final CTValAx val = new CTValAx();
				val.setMinorGridlines(getHiddenChartLines());
				val.setMinorTickMark(getHiddenTickMark());
				val.setNumFmt(createNumFmt());
				res = val;
			}

			res.getAxId().setVal(index);
			axes.set(index, res);
		}
		if (res.getDelete() == null || res.getDelete().isVal()) {
			hideAxis(res);
		}

		return res;
	}

	private static void hideAxis(
		IAxisDescription axis)
	{
		final CTBoolean delete = new CTBoolean();
		delete.setVal(false);
		axis.setDelete(delete);

		axis.setSpPr(getHiddenShapeProperties());

		axis.setMajorGridlines(getHiddenChartLines());
		axis.setTickLblPos(getHiddenLblPos());

		axis.setTitle(null);
		axis.setTxPr(null);

	}

	public static CTNumFmt createNumFmt()
	{
		final CTNumFmt res = new CTNumFmt();
		res.setFormatCode("0");
		res.setSourceLinked(true);
		return res;
	}


	private static CTChartLines getHiddenChartLines()
	{
		final CTChartLines line = new CTChartLines();
		line.setSpPr(getHiddenShapeProperties());
		return line;
	}

	private static CTTickLblPos getHiddenLblPos()
	{
		final CTTickLblPos lbl = new CTTickLblPos();
		lbl.setVal(STTickLblPos.NONE);
		return lbl;
	}

	private static CTTickMark getHiddenTickMark()
	{
		final CTTickMark mark = new CTTickMark();
		mark.setVal(STTickMark.NONE);
		return mark;
	}

	public static CTShapeProperties getHiddenShapeProperties()
	{
		final CTShapeProperties shape = new CTShapeProperties();
		shape.setNoFill(new CTNoFillProperties());
		final CTLineProperties line = new CTLineProperties();
		line.setNoFill(new CTNoFillProperties());
		shape.setLn(line);
		return shape;
	}

	private static final void getAxeTitleOperation(IAxisDescription axe, JSONArray operations, String axisParam, JSONArray position) throws JSONException {
		final CTTitle title = axe.getTitle();

		final JSONObject attrs = DMLHelper.createJsonFromProperties(title);
		if (attrs != null) {

			final JSONObject op = new JSONObject();
			op.put("name", "setChartTitleAttributes");
			op.put("start", position);
			op.put("axis", axisParam);
			op.put("attrs", attrs);
			operations.put(op);
		}

	}

	public static void setChartTitleAttributes(CTPlotArea plotArea, String axis, JSONObject attrs) throws JSONException {
		final IAxisDescription ax = getAxis(plotArea, axis);
		if (ax == null) {
			return;
		}
		CTTitle title = ax.getTitle();
		if (title == null) {
			title = new CTTitle();
			ax.setTitle(title);
		}
		setTitleFromAttrs(title, attrs);
		refreshTitleRotations(plotArea);
	}

	private static final void getGridOperation(IAxisDescription axe, JSONArray operations, String axisParam, JSONArray position) throws JSONException {

		final CTChartLines grid = axe.getMajorGridlines();
		if (grid != null) {
			final JSONObject gridOp = new JSONObject();
			final CTShapeProperties shape = grid.getSpPr();
			if (shape != null) {
				createJsonFromShapeProperties(gridOp, shape, false);
			}
			if (!gridOp.has("line")) {
				gridOp.put("line", makeStandardShapeType());
			}
			final JSONObject op = new JSONObject();
			op.put("name", "setChartGridlineAttributes");
			op.put("start", position);
			op.put("axis", axisParam);
			op.put("attrs", gridOp);
			operations.put(op);
		}
	}

	public static void setChartGridlineAttributes(CTPlotArea plotArea, String axis, JSONObject attrs) throws JSONException {
		final IAxisDescription ax = getAxis(plotArea, axis);
		if (ax == null) {
			return;
		}

		CTChartLines grids = ax.getMajorGridlines();
		if (grids == null) {
			grids = new CTChartLines();
			ax.setMajorGridlines(grids);
		}
		grids.setSpPr(applyShapePropertiesFromJson(grids.getSpPr(), attrs));
	}

    /**
     * @param part (partName Uri should be /word/document.xml)
     * @return the imageUrl in following form: word/media/xxxx.png
     */
    public static String getBlipUrl(Part part, CTBlip blip) {
        String imageUrl = "";
        if(blip!=null) {
            imageUrl = Commons.getUrl(part, blip.getEmbed().length() > 0 ? blip.getEmbed() : blip.getLink() );
        }
        return imageUrl;
    }

    public static float fontSizeToPt(Integer sz) {
		if (sz == null) {
			return 10;
		}
		return ((float) sz) / 100f;
	}

    public static Integer ptTofontSize(double pt) {
		return (int) (pt * 100);
	}

    public static int lineWidthToThmm(Integer w) {
        if (w == null) {
            return 1;
        }
        return (int)((w/360f)+0.5);
    }

    private static Integer thmmToLineWidth(Long w) {
    	return (int)(w*360);
	}

	//FIXME: nessecary?
	private static JSONObject makeStandardShapeType() throws JSONException {
		final JSONObject shape = new JSONObject();
		shape.put("type", "solid");
		final JSONObject color = new JSONObject();
		color.put("type", "auto");
		shape.put("color", color);
		return shape;
	}
}
