/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.List;

import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.CTAutoFilter;
import org.xlsx4j.sml.CTFilter;
import org.xlsx4j.sml.CTFilterColumn;
import org.xlsx4j.sml.CTFilters;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SmlUtils;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;

import com.openexchange.office.ooxml.xlsx.operations.ApplyOperationHelper;

public class AutoFilterHelper {

    enum FilterType {
        NONE,
        DISCRETE
    };

    public static void createAutoFilterOperations(JSONArray operationsArray, int sheetIndex, CTAutoFilter autoFilter)
        throws JSONException {

        if(autoFilter==null) {
            return;
        }
        final SmlUtils.CellRefRange cellRefRange = SmlUtils.createCellRefRange(autoFilter.getRef());
        if(cellRefRange==null) {
            return;
        }
        JSONObject attrs = new JSONObject(1);
        final JSONObject tableAttrs = new JSONObject(2);
        tableAttrs.put("headerRow", true);
        tableAttrs.put("filtered", true);
        attrs.put("table", tableAttrs);
        addInsertTableOperation(operationsArray, sheetIndex, cellRefRange, "", attrs);

        final List<CTFilterColumn> filterColumns = autoFilter.getFilterColumn();
        for(CTFilterColumn filterColumn:filterColumns) {
            attrs = new JSONObject();
            final JSONObject filterAttrs = new JSONObject(3);
            final CTFilters filters = filterColumn.getFilters();
            if(filterColumn.isShowButton()==false) {
                filterAttrs.put("hideMergedButton", true);
            };
            if(filters!=null) {
                filterAttrs.put("type", "discrete");
                final List<CTFilter> filterList = filters.getFilter();
                final JSONArray filterArray = new JSONArray(filterList.size()+1);
                for(CTFilter filter:filterList) {
                    filterArray.put(filter.getVal());
                }
                if(filters.isBlank()) {
                    filterArray.put("");
                }
                filterAttrs.put("entries", filterArray);
            }
            attrs.put("filter", filterAttrs);
            addChangeTableColumnOperation(operationsArray, sheetIndex, "", filterColumn.getColId(), attrs);
        }
    }

    public static void addInsertTableOperation(JSONArray operationsArray, int sheetIndex, SmlUtils.CellRefRange range, String tableName, JSONObject attrs)
            throws JSONException {

        final JSONObject addInsertTableObject = new JSONObject(6);
        addInsertTableObject.put("name", "insertTable");
        addInsertTableObject.put("sheet", sheetIndex);
        addInsertTableObject.put("start", range.getStart().getJSONArray());
        addInsertTableObject.put("end", range.getEnd().getJSONArray());
        if(!attrs.isEmpty()) {
            addInsertTableObject.put("attrs", attrs);
        }
        if(tableName!=null) {
            addInsertTableObject.put("table", tableName);
        }
        operationsArray.put(addInsertTableObject);
    }

    public static void addChangeTableColumnOperation(JSONArray operationsArray, int sheetIndex, String tableName, long columnIndex, JSONObject attrs)
        throws JSONException {
        if(!attrs.isEmpty()) {
            final JSONObject addInsertChangeTableColumnObject = new JSONObject();
            addInsertChangeTableColumnObject.put("name", "changeTableColumn");
            addInsertChangeTableColumnObject.put("sheet", sheetIndex);
            if(tableName!=null) {
                addInsertChangeTableColumnObject.put("table", tableName);
            }
            addInsertChangeTableColumnObject.put("col", columnIndex);
            addInsertChangeTableColumnObject.put("attrs", attrs);
            operationsArray.put(addInsertChangeTableColumnObject);
        }
    }

    public static void insertTable(ApplyOperationHelper applyOperationHelper, int sheetIndex, String tableName, JSONArray start, JSONArray end, JSONObject attrs)
        throws JSONException {

        final Worksheet worksheet = applyOperationHelper.getWorksheet(sheetIndex);
        final String cellRefRange = SmlUtils.getCellRefRange(start.getInt(0), start.getInt(1), end.getInt(0), end.getInt(1));
        if(tableName==null||tableName.isEmpty()) {
            CTAutoFilter autoFilter = worksheet.getAutoFilter();
            if(autoFilter==null) {
                autoFilter = Context.getsmlObjectFactory().createCTAutoFilter();
                autoFilter.setParent(worksheet);
                worksheet.setAutoFilter(autoFilter);
            }
            autoFilter.setRef(cellRefRange);
            insertDefinedTableName(applyOperationHelper, Long.valueOf(sheetIndex), null, SmlUtils.createCellRefRange(cellRefRange));
            if(attrs!=null) {
                applyTableAttributes(autoFilter, attrs);
            }
        }
    }

    public static void changeTable(ApplyOperationHelper applyOperationHelper, int sheetIndex, String tableName, JSONArray start, JSONArray end, JSONObject attrs)
        throws JSONException {

        if(tableName==null||tableName.isEmpty()) {
            insertTable(applyOperationHelper, sheetIndex, null, start, end, attrs);
        }
    }

    public static void deleteTable(ApplyOperationHelper applyOperationHelper, int sheetIndex, String tableName) {

        final Worksheet worksheet = applyOperationHelper.getWorksheet(sheetIndex);
        worksheet.setAutoFilter(null);
        deleteDefinedTableName(applyOperationHelper, tableName, Long.valueOf(sheetIndex));
    }

    public static void deleteDefinedTableName(ApplyOperationHelper applyOperationHelper, String tableName, Long sheet) {
        applyOperationHelper.deleteName(sheet, tableName==null||tableName.isEmpty()?"_xlnm._FilterDatabase":tableName);
    }

    public static void insertDefinedTableName(ApplyOperationHelper applyOperationHelper, Long sheetIndex, String tableName, SmlUtils.CellRefRange cellRefRange) {

        String formula = SmlUtils.getAbsoluteCellRefRange(cellRefRange);
        if(sheetIndex!=null) {
            final WorkbookPart workbookPart = applyOperationHelper.getOperationDocument().getPackage().getWorkbookPart();
            final Workbook workbook = workbookPart.getJaxbElement();
            final Sheet sheet = workbook.getSheets().getSheet().get(sheetIndex.intValue());
            formula
                = ApplyOperationHelper.encodeSheetName(sheet.getName())
                + "!"
                + formula;
        }
        applyOperationHelper.insertName(sheetIndex, tableName==null||tableName.isEmpty()?"_xlnm._FilterDatabase":tableName, formula, Boolean.TRUE);
    }

    public static void changeTableColumn(ApplyOperationHelper applyOperationHelper, int sheetIndex, String tableName, long col, JSONObject attrs)
        throws JSONException {

        final JSONObject filterAttrs = attrs.optJSONObject("filter");
        if(filterAttrs==null) {
            return;
        }
        FilterType filterType = FilterType.NONE;
        final String v = filterAttrs.optString("type", null);
        if(v!=null) {
            if(v.equals("none")) {
                filterType = FilterType.NONE;
            }
            else if(v.equals("discrete")) {
                filterType = FilterType.DISCRETE;
            }
        }
        final JSONArray entries = filterAttrs.optJSONArray("entries");
        final Worksheet worksheet = applyOperationHelper.getWorksheet(sheetIndex);
        if(tableName==null||tableName.isEmpty()) {
            final CTAutoFilter autoFilter = worksheet.getAutoFilter();
            final List<CTFilterColumn> filterColumns = autoFilter.getFilterColumn();
            int filterColumnIndex = 0;
            CTFilterColumn filterColumn = null;
            for(;filterColumnIndex<filterColumns.size();filterColumnIndex++) {
                if(filterColumns.get(filterColumnIndex).getColId()==col) {
                    filterColumn = filterColumns.get(filterColumnIndex);
                    break;
                }
            }
            if(filterColumn==null) {
                filterColumn = Context.getsmlObjectFactory().createCTFilterColumn();
                filterColumn.setParent(autoFilter);
                filterColumn.setColId(col);
                filterColumns.add(filterColumn);
            }
            if(filterType==FilterType.NONE) {
                final Object hideMergedButton = filterAttrs.opt("hideMergedButton");
                if(hideMergedButton!=null) {
                    filterColumn.setShowButton(hideMergedButton instanceof Boolean?!((Boolean)hideMergedButton).booleanValue():null);
                }
                filterColumn.setFilters(null);
            }
            else if(filterType==FilterType.DISCRETE) {
                CTFilters filters = filterColumn.getFilters();
                if(filters==null) {
                    filters = Context.getsmlObjectFactory().createCTFilters();
                    filters.setParent(filterColumn);
                    filterColumn.setFilters(filters);
                }
                if(entries!=null) {
                    final List<CTFilter> filterList = filters.getFilter();
                    filterList.clear();
                    for(int i=0; i<entries.length(); i++) {
                        final String filterValue = entries.getString(i);
                        if(filterValue.isEmpty()) {
                            filters.setBlank(Boolean.TRUE);
                        }
                        else {
                            final CTFilter filter = Context.getsmlObjectFactory().createCTFilter();
                            filter.setParent(filters);
                            filter.setVal(filterValue);
                            filterList.add(filter);
                        }
                    }
                }
            }
        }
    }

    private static void applyTableAttributes(CTAutoFilter autoFilter, JSONObject attrs) {

    }
}
