/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.lang.annotation.Annotation;
import java.math.BigInteger;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.datatype.XMLGregorianCalendar;

import org.apache.commons.logging.Log;
import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.docx4j.mce.AlternateContent;
import org.docx4j.mce.AlternateContent.Choice;
import org.docx4j.mce.AlternateContent.Fallback;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.wml.Br;
import org.docx4j.wml.CTMarkup;
import org.docx4j.wml.CTPPrChange;
import org.docx4j.wml.CTParaRPrOriginal;
import org.docx4j.wml.CTSimpleField;
import org.docx4j.wml.CTTblGridChange;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblPrBase.TblStyle;
import org.docx4j.wml.CTTblPrChange;
import org.docx4j.wml.CTTcPrChange;
import org.docx4j.wml.CTTrPrBase;
import org.docx4j.wml.CTTrPrChange;
import org.docx4j.wml.CTTrackChange;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Drawing;
import org.docx4j.wml.IText;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase;
import org.docx4j.wml.PPrBase.PStyle;
import org.docx4j.wml.ParaRPr;
import org.docx4j.wml.ParaRPrChange;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.RStyle;
import org.docx4j.wml.RunDel;
import org.docx4j.wml.RunIns;
import org.docx4j.wml.STBrType;
import org.docx4j.wml.SdtBlock;
import org.docx4j.wml.SdtRun;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.TblGrid;
import org.docx4j.wml.TblGridBase;
import org.docx4j.wml.TblPr;
import org.docx4j.wml.TblWidth;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.TcPrInner;
import org.docx4j.wml.Text;
import org.docx4j.wml.Tr;
import org.docx4j.wml.TrPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;

import com.openexchange.log.LogFactory;
import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.docx.tools.DrawingML.DrawingML_root;
import com.openexchange.office.ooxml.docx.tools.PictVML.VMLShape;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public abstract class Component extends ComponentContext {

    protected static Log log = LogFactory.getLog(Component.class);

    private static HashSet<Class<?>> rootElements = new HashSet<Class<?>>();

    //--------------------------------------------------------------------------------------------------------------------------------------------------------
    //                                                                                          Paragraph                             Paragraph
    //                                   Paragraph                                             ?(hyperlink)                          ?(hyperlink)
    //          TextRun*                SimpleField*       Hyperlink*                             Ins*                                   Del*
    //  Text*|Drawing*|Pict*|Tab*|Br*    TextRun(1)         TextRun*                             TextRun*                              TextRun*
    //                                    Text(1)    Text*|Drawing*|Pict*|Tab*|Br*      Text*|Drawing*|Pict*|Tab*|Br*       DelText*|Drawing*|Pict*|Tab*|Br*
    //--------------------------------------------------------------------------------------------------------------------------------------------------------

    // change track flags, the  in the highword specifies the value that should be used
    final public static int CT_NONE  			= 0;
    final public static int CT_INSERTED  		= 1;	// corresponding HIWORD bit == VALUE
    final public static int CT_DELETED   		= 2;	// corresponding HIWORD bit == VALUE
    final public static int CT_MODIFIED 		= 4;	// corresponding HIWORD bit == VALUE

    public enum Type {
    	PARAGRAPH,
    	TABLE,
    	TR,
    	TC,
    	TAB,
    	HARDBREAK_DEFAULT,
    	HARDBREAK_PAGE,
    	HARDBREAK_COLUMN,
    	AC_SHAPE,
    	AC_GROUP,
    	AC_IMAGE
    }

    public enum SplitMode {
        ATTRIBUTE,
        DELETE
    }

    /*
     * returns the corresponding component for the given oxo position. if positionCount is zero a
     * RootComponent is returned. The contextPart is usually the MainDocumentPart and changed only
     * for Header & Footer.
     *
     */
    public static Component getComponent(Part contextPart, JSONArray position, int positionCount) {
        try {
            if(positionCount==0) {
                return new RootComponent((ContentAccessor)contextPart);
            }
            // a part has to implement the ContentAccessor interface, otherwise no component can be accessed
            final RootComponent rootComponent = new RootComponent((ContentAccessor)contextPart);
            Component c = rootComponent.getNextChildComponent(null, null);
            for(int i=0; c!=null&&i<positionCount;) {
                c = c.getComponent(position.getInt(i++));
                if(i!=positionCount) {
                    c = c.getNextChildComponent(null, null);
                }
            }
            return c;
        }
        catch(JSONException e) {
            // ups
        }
        return null;
    }

    /*
     * simply returns o, but in case o is a JAXBElement and a RootElement annotation is available
     * then the parent content list is updated and the contentModel is returned.
     * */
    public static Object getContentModel(IndexedNode<Object> node, Object parent) {

        if (node.getData() instanceof javax.xml.bind.JAXBElement) {
            boolean hasRootElement = rootElements.contains(((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType());
            if(!hasRootElement) {
                final Annotation[] annotations = ((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType().getAnnotations();
                for(Annotation annotation : annotations) {
                    if(annotation instanceof XmlRootElement) {
                        rootElements.add(((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType());
                        hasRootElement = true;
                        break;
                    }
                }
            }
            if(hasRootElement) {
                node.setData(((JAXBElement<?>)node.getData()).getValue());
            }
        }
        if(node.getData() instanceof Child) {
        	((Child)node.getData()).setParent(parent);
        }
        return node.getData();
    }

    private IndexedNode<Object> contextNode;
    protected int componentNumber;

    public Component(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber) {
        super(parentContext, _node);
        contextNode = _contextNode;
        componentNumber = _componentNumber;
    }
    public Object getObject() {
        return getNode().getData();
    }
    public IndexedNode<Object> getContextNode() {
    	return contextNode;
    }
    public Component getNextComponent() {
        ComponentContext parentContext = this;
        Component nextComponent = null;
        do {
            ComponentContext previousContext = parentContext;
            parentContext = parentContext.getParentContext();
            nextComponent = parentContext.getNextChildComponent(previousContext, this);
        }
        while(nextComponent==null&&!(parentContext instanceof Component));
        return nextComponent;
    }
    // the ComponentContext c is inserted behind one of the classes in the
    // parentContextList, it is inserted at least behind the parent Component,
    // so the parentContextList is allowed to be null
    //
    // the parentContext of c is inserted, so c does not need to have a parentContext
    public void insertContext(ComponentContext c, HashSet<Class<?>> parentContextList) {
        ComponentContext childContext = this;
        ComponentContext parentContext = getParentContext();
        while(!(parentContext instanceof Component)&&((parentContextList==null)||(!parentContextList.contains(parentContext.getClass())))) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
        c.setParentContext(parentContext);
        childContext.setParentContext(c);
    }
    public void removeContext(Class<?> contextClass) {
        ComponentContext childContext = this;
        ComponentContext parentContext = getParentContext();
        while(!(parentContext instanceof Component)) {
            if(parentContext.getClass()==contextClass) {
                childContext.setParentContext(parentContext.getParentContext());
                break;
            }
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
    }
    public ComponentContext getContextChild(HashSet<Class<?>> parentContextList) {
        ComponentContext childContext = this;
        ComponentContext parentContext = getParentContext();
        while(!(parentContext instanceof Component)&&((parentContextList==null)||(!parentContextList.contains(parentContext.getClass())))) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
        return childContext;
    }

    public abstract Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent);

    public Component getChildComponent(int childComponentNumber) {
    	Component c = getNextChildComponent(null, null);
    	if(c!=null) {
    		c = c.getComponent(childComponentNumber);
    	}
    	return c;
    }
    public static Component insertChildComponent(ComponentContext parentContext, IndexedNode<Object> contextNode, int number, Component child, Type type) {

        int index;
        IndexedNodeList<Object> indexedNodeList;
        if(child!=null&&child.getComponentNumber()==number) {
            final HashSet<Class<?>> parentContextList = new HashSet<Class<?>>(1);
            parentContextList.add(SdtRootContext.class);
            final ComponentContext contextChild = child.getContextChild(parentContextList);
            indexedNodeList = (IndexedNodeList<Object>)((ContentAccessor)contextChild.getParentContext().getNode().getData()).getContent();
            index = contextChild.getNode().getIndex();
        }
        else {
            indexedNodeList = (IndexedNodeList<Object>)((ContentAccessor)contextNode.getData()).getContent();
            index = indexedNodeList.size();
        }
        switch(type) {
            case PARAGRAPH : {
                final Child newChild = Context.getWmlObjectFactory().createP();
                newChild.setParent(contextNode.getData());
                indexedNodeList.add(index, newChild);
                return new ParagraphComponent(parentContext, indexedNodeList.getNode(index), contextNode, number);
            }
            case TABLE : {
                final Tbl newTbl = Context.getWmlObjectFactory().createTbl();
                final TblPr tblPr = Table.getTableProperties(newTbl);

                // turning on each conditional Style, each of our conditional styles has to be used by default,
                // (they can be deactivated by using the exclude property)

                // TODO: tblLook changed in docx4j3.0.1...
                Table.initLookAndRowBandSize(tblPr);
                newTbl.setParent(contextNode.getData());
                indexedNodeList.add(index, newTbl);
                return new TableComponent(parentContext, indexedNodeList.getNode(index), contextNode, number);
            }
            default : {
                throw new UnsupportedOperationException();
            }
        }
    }
    public Component insertChildComponent(OperationDocument operationDocument, int number, JSONObject attrs, Type type)
    	throws UnsupportedOperationException, JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        final Component c = insertChildComponent(this, getNode(), number, getChildComponent(number), type);
        if(attrs!=null) {
            c.applyAttrsFromJSON(operationDocument, attrs);
        }
        return c;
    }
    public Object getContextObject() {
        return contextNode.getData();
    }
    public IndexedNode<Object> getContextChildNode() {
        if(getParentContext() instanceof SdtRootContext) {
            return getParentContext().getNode();
        }
        return getNode();
    }
    public Component getComponent(int number) {
    	Component c = this;
    	while(c!=null&&c.getNextComponentNumber()<=number) {
    		c = c.getNextComponent();
    	}
    	return c;
    }
    public int getComponentNumber() {
        return componentNumber;
    }
    public int getNextComponentNumber() {
        return componentNumber + 1;
    }
    public void delete(int count) {
        final int endComponent = (getComponentNumber()+count)-1;
        Component component = this;
        final int startIndex = component.getContextChildNode().getIndex();
        while(component.getNextComponentNumber()<=endComponent) {
            component = component.getNextComponent();
        }
        component.splitEnd(endComponent, SplitMode.DELETE);
        final IndexedNodeList<Object> content = (IndexedNodeList<Object>)((ContentAccessor)component.getContextObject()).getContent();
        content.removeNodes(startIndex, (component.getContextChildNode().getIndex()-startIndex)+1);
    }
    public void splitStart(int n, SplitMode splitMode) {
        final int index = getNode().getIndex();
        if(index!=0) {
            final ComponentContext pContext = getParentContext();
            if(pContext instanceof SdtRootContext) {
                final IndexedNode<Object> sdtSourceNode = pContext.getNode();
                final SdtBlock sdtSource = (SdtBlock)(sdtSourceNode.getData());
                final SdtBlock sdtDest = Context.getWmlObjectFactory().createSdtBlock();
                sdtDest.setParent(sdtSource.getParent());
                final IndexedNodeList<Object> sdtSourceContent = sdtSource.getContent();
                final IndexedNodeList<Object> sdtDestContent = sdtDest.getContent();
                IndexedNodeList.moveNodes(sdtSourceContent, sdtDestContent, index, 0, sdtSourceContent.size()-index, sdtDest);
                final ContentAccessor sdtParent = (ContentAccessor)(pContext.getParentContext().getNode().getData());
                final IndexedNode<Object> sdtDestNode = new IndexedNode<Object>(sdtDest);
                ((IndexedNodeList<Object>)(sdtParent.getContent())).addNode(sdtSourceNode.getIndex()+1, sdtDestNode);
                setParentContext(new SdtRootContext(pContext.getParentContext(), sdtDestNode, getContextNode()));
            }
        }
    }
    public void splitEnd(int n, SplitMode splitMode) {
        final ComponentContext pContext = getParentContext();
        if(pContext instanceof SdtRootContext) {
            final IndexedNode<Object> sdtSourceNode = pContext.getNode();
            final SdtBlock sdtSource = (SdtBlock)(sdtSourceNode.getData());
            final IndexedNodeList<Object> sdtSourceContent = sdtSource.getContent();
            final int index = getNode().getIndex();
            if(index+1!=sdtSourceContent.size()) {
                final SdtBlock sdtDest = Context.getWmlObjectFactory().createSdtBlock();
                sdtDest.setParent(sdtSource.getParent());
                final IndexedNodeList<Object> sdtDestContent = sdtDest.getContent();
                IndexedNodeList.moveNodes(sdtSourceContent, sdtDestContent, 0, 0, index+1, sdtDest);
                final ContentAccessor sdtParent = (ContentAccessor)(pContext.getParentContext().getNode().getData());
                final IndexedNode<Object> sdtDestNode = new IndexedNode<Object>(sdtDest);
                ((IndexedNodeList<Object>)(sdtParent.getContent())).addNode(sdtSourceNode.getIndex(), sdtDestNode);
                setParentContext(new SdtRootContext(pContext.getParentContext(), sdtDestNode, getContextNode()));
            }
        }
    }
    public abstract void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException;

    public abstract JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
    	throws JSONException, ParseException, FilterException;

    public void postFixNonConformance(AtomicInteger id) {
    	final Object o = getObject();
    	if (o instanceof CTMarkup) {
			((CTMarkup)o).setId(BigInteger.valueOf(id.incrementAndGet()));
		}
    }

    public interface IParagraph {

    	public void insertText(OperationDocument operationDocument, int textPosition, String text, JSONObject attrs)
   			throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException;

        public void splitParagraph(int textPosition);

        public void mergeParagraph();

        public void insertField(OperationDocument operationDocument, int textPosition, String type, String representation, JSONObject attrs)
            throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException;
    }

    public interface ITable {

        public void insertRows(OperationDocument operationDocument, int rowPosition, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
            throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException;

        public void splitTable(int rowPosition);

        public void mergeTable();

        public void insertColumn(OperationDocument operationDocument, JSONArray tableGrid, int gridPosition, String insertMode)
        	throws JSONException;

        public void deleteColumns(OperationDocument operationDocument, int gridStart, int gridEnd)
        	throws JSONException;
    }

    public interface IRow {

    	public void insertCells(OperationDocument operationDocument, int cellPosition, int count, JSONObject attrs)
    	    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException;
    }

    public static class RootComponent extends Component {

        public RootComponent(ContentAccessor contentAccessor) {
            super(null, new IndexedNode<Object>(contentAccessor), null, 0);
        }
        @Override
        public Component getNextComponent() {
            return null;
        }
        @Override
        public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
            final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

            Component nextComponent = null;
            final IndexedNode<Object> rootNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)rootNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = getContentModel(childNode, rootNode.getData());
                if(o instanceof P) {
                    nextComponent = new ParagraphComponent(this, childNode, rootNode, nextComponentNumber);
                }
                else if(o instanceof Tbl) {
                    nextComponent = new TableComponent(this, childNode, rootNode, nextComponentNumber);
                }
                else if(o instanceof SdtBlock) {
                    final SdtRootContext sdtRootContext = new SdtRootContext(this, childNode, rootNode);
                    nextComponent = sdtRootContext.getNextChildComponent(null, previousChildComponent);
                }
            }
            return nextComponent;
        }
        @Override
        public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs) {

        }
        @Override
        public JSONObject createJSONAttrs(CreateOperationHelper operationDocument, JSONObject attrs) {
            return attrs;
        }
    }

    public static class MultiComponent extends Component implements IParagraph, ITable, IRow {

    	final private List<Component> componentList;

    	public MultiComponent(List<Component> componentList) {
    		super(null, new IndexedNode<Object>(componentList), null, 0);

    		this.componentList = componentList;
    	}
    	public List<Component> getComponentList() {
    		return componentList;
    	}
    	@Override
    	public Component getComponent(int number) {
    		final List<Component> nextComponents = new ArrayList<Component>(componentList.size());
			for(Component component:componentList) {
				final Component nextComponent = component.getComponent(number);
				if(nextComponent!=null) {
					nextComponents.add(nextComponent);
				}
			}
			return nextComponents.isEmpty() ? null : new MultiComponent(nextComponents);
        }
    	@Override
        public int getComponentNumber() {
            return componentList.get(0).getComponentNumber();
        }
        @Override
        public int getNextComponentNumber() {
			return componentList.get(0).getNextComponentNumber();
        }
		@Override
		public Component getNextComponent() {
			final List<Component> nextComponents = new ArrayList<Component>(componentList.size());
			for(Component component:componentList) {
				final Component nextComponent = component.getNextComponent();
				if(nextComponent!=null) {
					nextComponents.add(nextComponent);
				}
			}
			return nextComponents.isEmpty() ? null : new MultiComponent(nextComponents);
		}
		@Override
		public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
			final List<Component> childComponents = new ArrayList<Component>(componentList.size());
			for(Component component:componentList) {
				final Component childComponent = component.getNextChildComponent(previousChildContext, previousChildComponent);
				if(childComponent!=null) {
					childComponents.add(childComponent);
				}
			}
			return childComponents.isEmpty() ? null : new MultiComponent(childComponents);
		}
		@Override
        public Component insertChildComponent(OperationDocument operationDocument, int number, JSONObject attrs, Type type)
            throws UnsupportedOperationException, JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

		    final List<Component> childComponents = new ArrayList<Component>(componentList.size());
			for(Component component:componentList) {
				childComponents.add(component.insertChildComponent(operationDocument, number, attrs, type));
			}
			return new MultiComponent(childComponents);
		}
		@Override
		public void applyAttrsFromJSON(OperationDocument operationDocument,
				JSONObject attrs) throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

			for(Component component:componentList) {
				component.applyAttrsFromJSON(operationDocument, attrs);
			}
		}
		@Override
		public JSONObject createJSONAttrs(
				CreateOperationHelper createOperationHelper, JSONObject attrs)
				throws JSONException, ParseException, FilterException {

			return componentList.get(0).createJSONAttrs(createOperationHelper, attrs);
		}
		@Override
		public void delete(int count) {
		    for(Component component:componentList) {
		        component.delete(count);
		    }
		}
		@Override
	    public void splitStart(int n, SplitMode splitMode) {
			for(Component component:componentList) {
				component.splitStart(n, splitMode);
			}
	    }
		@Override
	    public void splitEnd(int n, SplitMode splitMode) {
			for(Component component:componentList) {
				component.splitEnd(n, splitMode);
			}
	    }
		@Override
		public void postFixNonConformance(AtomicInteger id) {

			for(Component component:componentList) {
				component.postFixNonConformance(id);
			}
	    }
		// IParagraph interfaces
		@Override
    	public void insertText(OperationDocument operationDocument, int textPosition, String text, JSONObject attrs)
       		throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

			for(Component component:componentList) {
				((IParagraph)component).insertText(operationDocument, textPosition, text, attrs);
			}
		}
		@Override
        public void splitParagraph(int textPosition) {
			for(Component component:componentList) {
				((IParagraph)component).splitParagraph(textPosition);
			}
		}
		public void mergeParagraph() {
			for(Component component:componentList) {
				((IParagraph)component).mergeParagraph();
			}
		}
        public void insertField(OperationDocument operationDocument, int textPosition, String type, String representation, JSONObject attrs)
            throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

            for(Component component:componentList) {
        		((IParagraph)component).insertField(operationDocument, textPosition, type, representation, attrs);
        	}
        }
		// ITable interface
		@Override
		public void insertRows(OperationDocument operationDocument, int rowPosition, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
		    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

		    for(Component component:componentList) {
				((ITable)component).insertRows(operationDocument, rowPosition, count, insertDefaultCells, referenceRow, attrs);
			}
		}
		@Override
		public void splitTable(int rowPosition) {
			for(Component component:componentList) {
				((ITable)component).splitTable(rowPosition);
			}
		}
		@Override
		public void mergeTable() {
			for(Component component:componentList) {
				((ITable)component).mergeTable();
			}
		}
		@Override
		public void insertColumn(OperationDocument operationDocument, JSONArray tableGrid, int gridPosition, String insertMode)
			throws JSONException {

			for(Component component:componentList) {
				((ITable)component).insertColumn(operationDocument, tableGrid, gridPosition, insertMode);
			}
		}
		@Override
        public void deleteColumns(OperationDocument operationDocument, int gridStart, int gridEnd)
        	throws JSONException {

			for(Component component:componentList) {
				((ITable)component).deleteColumns(operationDocument, gridStart, gridEnd);
			}
		}
		// IRow interface
		@Override
		public void insertCells(OperationDocument operationDocument, int cellPosition, int count, JSONObject attrs)
		    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

			for(Component component:componentList) {
				((IRow)component).insertCells(operationDocument, cellPosition, count, attrs);
			}
		}
    }

    /**
     * {@link ParagraphComponent}
     *
     * @author <a href="mailto:Sven.Jacobi@open-xchange.com">Sven Jacobi</a>
     */
    public static class ParagraphComponent extends Component implements IParagraph {

        public ParagraphComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber) {
            super(parentContext, _node, _contextNode, _componentNumber);
        }
        @Override
		public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
            final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

            Component nextComponent = null;
            final IndexedNode<Object> paragraphNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)paragraphNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = getContentModel(childNode, paragraphNode.getData());
                if(o instanceof R) {
                    final ComponentContext.TextRunContext textRunContext = new ComponentContext.TextRunContext(this, childNode, paragraphNode);
                    nextComponent = textRunContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof P.Hyperlink) {
                    final ComponentContext.HyperlinkContext hyperlinkContext = new ComponentContext.HyperlinkContext(this, childNode, paragraphNode);
                    nextComponent = hyperlinkContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof RunIns) {
                    final ComponentContext.RunInsContext runInsContext = new ComponentContext.RunInsContext(this, childNode, paragraphNode);
                    nextComponent = runInsContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof RunDel) {
                    final ComponentContext.RunDelContext runDelContext = new ComponentContext.RunDelContext(this, childNode, paragraphNode);
                    nextComponent = runDelContext.getNextChildComponent(null, previousChildComponent);
                }
                else if(o instanceof CTSimpleField) {
                    nextComponent = new FldSimpleComponent(this, childNode, paragraphNode, nextComponentNumber);
                }
                else if(o instanceof SdtRun) {
                    final SdtParagraphContext sdtParagraphContext = new SdtParagraphContext(this, childNode, paragraphNode);
                    nextComponent = sdtParagraphContext.getNextChildComponent(null, previousChildComponent);
                }
            }
            return nextComponent;
        }
        @Override
        public Component insertChildComponent(OperationDocument operationDocument, int textPosition, JSONObject attrs, Type childType)
            throws JSONException, JAXBException, InvalidFormatException, PartUnrecognisedException {

            Child newTextRunChild = null;
            switch(childType) {
                case TAB : {
                    newTextRunChild = Context.getWmlObjectFactory().createRTab();
                    break;
                }
                case HARDBREAK_DEFAULT :
                case HARDBREAK_PAGE :
                case HARDBREAK_COLUMN : {
                    newTextRunChild = Context.getWmlObjectFactory().createBr();
                    if(childType==Type.HARDBREAK_PAGE) {
                        ((Br)newTextRunChild).setType(STBrType.PAGE);
                    }
                    else if(childType==Type.HARDBREAK_COLUMN) {
                        ((Br)newTextRunChild).setType(STBrType.COLUMN);
                    }
                    break;
                }
                case AC_IMAGE : {
                    final Drawing drawing = Context.getWmlObjectFactory().createDrawing();
                    drawing.getAnchorOrInline().add(DrawingML_root.createInline(false, "", 500000, 500000));
                    newTextRunChild = drawing;
                    break;
                }
                case AC_SHAPE : {
                    newTextRunChild = DrawingML.createAlternateContentShape();
                    break;
                }
                case AC_GROUP : {
                    newTextRunChild = DrawingML.createAlternateContentGroup();
                    break;
                }
                default : {
                    throw new UnsupportedOperationException();
                }
            }
            final P paragraph = (P)getObject();
            final IndexedNodeList<Object> paragraphContent = paragraph.getContent();

            Component childComponent = getNextChildComponent(null, null);
            if(childComponent!=null) {
                if(textPosition>0) {
                    childComponent = childComponent.getComponent(textPosition-1);
                    childComponent.splitEnd(textPosition-1, SplitMode.ATTRIBUTE);
                }
                if(childComponent instanceof TextRun_Base) {
                    final R run = ((TextRun_Base)childComponent).getTextRun();
                    final int index = ((TextRun_Base)childComponent).getNode().getIndex();
                    newTextRunChild.setParent(run);
                    run.getContent().add(textPosition>0?index+1:index, newTextRunChild);
                }
                else if(childComponent instanceof FldSimpleComponent) {
                    final R newRun = Context.getWmlObjectFactory().createR();
                    newRun.setParent(paragraph);
                    paragraphContent.add(newRun);
                    final int index = childComponent.getContextChildNode().getIndex();
                    paragraphContent.add(textPosition>0?index+1:index, newRun);
                    final RPr rPr = ((FldSimpleComponent)childComponent).getTextRun().getRPr();
                    if(rPr!=null) {
                        final RPr newRPr = XmlUtils.deepCopy(rPr);
                        newRPr.setParent(newRun);
                        newRun.setRPr(newRPr);
                    }
                }
            }
            else {
                final R newRun = Context.getWmlObjectFactory().createR();
                newRun.setParent(paragraph);
                paragraphContent.add(newRun);
                newTextRunChild.setParent(newRun);
                newRun.getContent().add(newTextRunChild);
                if(paragraph.getPPr()!=null) {
                    final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                    if(rPr!=null) {
                        rPr.setParent(newRun);
                        newRun.setRPr(rPr);
                    }
                }
            }
            if(textPosition>0) {
                childComponent = childComponent.getNextComponent();
            }
            else {
                childComponent = getNextChildComponent(null, null);
            }
            if(childComponent instanceof TextRun_Base) {
                if(((TextRun_Base)childComponent).getCT()!=Component.CT_NONE) {
                    attrs = TextUtils.forceRemoveChangeTrack(attrs);
                }
            }
            if(attrs!=null) {
                childComponent.splitStart(textPosition, SplitMode.ATTRIBUTE);
                childComponent.splitEnd(textPosition, SplitMode.ATTRIBUTE);
                childComponent.applyAttrsFromJSON(operationDocument, attrs);
            }
            return childComponent;
        }

		@Override
		public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs) {
			if(attrs==null) {
				return;
			}
			try {
		        final P paragraph = (P)getObject();
		        PPr pPr = paragraph.getPPr();
		        if (pPr==null) {
		            pPr = Context.getWmlObjectFactory().createPPr();
		            pPr.setParent(paragraph);
		            paragraph.setPPr(pPr);
		        }
		        if(attrs.has("styleId")) {
		            final Object styleId = attrs.get("styleId");
		            if(styleId instanceof String) {
		                PPrBase.PStyle pStyle = pPr.getPStyle();
		                if(pStyle==null) {
		                    pStyle = Context.getWmlObjectFactory().createPPrBasePStyle();
		                    pStyle.setParent(pPr);
		                }
		                pStyle.setVal((String)styleId);
		                pPr.setPStyle(pStyle);
		            }
		            else {
		                pPr.setPStyle(null);
		            }
		        }
		        Paragraph.applyParagraphProperties(operationDocument, attrs.optJSONObject("paragraph"), pPr);
		        final Object characterAttrs = attrs.opt("character");
		        if(characterAttrs instanceof JSONObject) {
		            ParaRPr paraRPr = pPr.getRPr();
		            if(paraRPr==null) {
		                paraRPr = Context.getWmlObjectFactory().createParaRPr();
		                paraRPr.setParent(pPr);
		                pPr.setRPr(paraRPr);
		            }
		            com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties(operationDocument, (JSONObject)characterAttrs, paraRPr);
		        }
	            final Object changes = attrs.opt("changes");
	            if(changes!=null) {
		            if(changes instanceof JSONObject) {
	            		ParaRPr paraRPr = pPr.getRPr();
	            		if(paraRPr==null) {
	            			paraRPr = Context.getWmlObjectFactory().createParaRPr();
	            			paraRPr.setParent(pPr);
	            			pPr.setRPr(paraRPr);
	            		}
		            	final Object attrsInserted = ((JSONObject)changes).opt("inserted");
		            	if(attrsInserted!=null) {
		            		if(attrsInserted instanceof JSONObject) {
		            			CTTrackChange ins = paraRPr.getIns();
		            			if(ins==null) {
		            				ins = Context.getWmlObjectFactory().createCTTrackChange();
		            				ins.setParent(paraRPr);
		            				paraRPr.setIns(ins);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsInserted, ins);
		            		}
		            		else {
		            			paraRPr.setIns(null);
		            		}
		            	}
		            	final Object attrsRemoved = ((JSONObject)changes).opt("removed");
		            	if(attrsRemoved!=null) {
		            		if(attrsRemoved instanceof JSONObject) {
		            			CTTrackChange del = paraRPr.getDel();
		            			if(del==null) {
		            				del = Context.getWmlObjectFactory().createCTTrackChange();
		            				del.setParent(paraRPr);
		            				paraRPr.setDel(del);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsRemoved, del);
		            		}
		            		else {
		            			paraRPr.setDel(null);
		            		}
		            	}
		            	final Object modified = ((JSONObject)changes).opt("modified");
		            	if(modified!=null) {
		            		if(modified instanceof JSONObject) {
	      				        ParaRPrChange rPrChange = paraRPr.getRPrChange();
			            		if(rPrChange==null) {
			            			rPrChange = Context.getWmlObjectFactory().createParaRPrChange();
			            			rPrChange.setParent(paraRPr);
			            			paraRPr.setRPrChange(rPrChange);
			            		}
	        					CTPPrChange pPrChange = pPr.getPPrChange();
	        					if(pPrChange==null) {
	        						pPrChange = Context.getWmlObjectFactory().createCTPPrChange();
	        						pPrChange.setParent(pPr);
	        						pPr.setPPrChange(pPrChange);
	        					}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)modified, rPrChange);
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)modified, pPrChange);
	                			final Object attrsModified = ((JSONObject)modified).opt("attrs");
	                			if(attrsModified!=null) {
	                				if(attrsModified instanceof JSONObject) {
			                			// Style
	                					if(((JSONObject) attrsModified).has("styleId")) {
            					            final Object styleId = attrs.get("styleId");
            					            PPrBase pPrBase = pPrChange.getPPr();
            					            if(pPrBase==null) {
            					            	pPrBase = Context.getWmlObjectFactory().createPPrBase();
            					            	pPrBase.setParent(pPrChange);
            					            	pPrChange.setPPr(pPrBase);
            					            }
             					            if(styleId instanceof String) {
            					                PPrBase.PStyle pStyle = pPrBase.getPStyle();
            					                if(pStyle==null) {
            					                    pStyle = Context.getWmlObjectFactory().createPPrBasePStyle();
            					                    pStyle.setParent(pPrBase);
            					                }
            					                pStyle.setVal((String)styleId);
            					                pPrBase.setPStyle(pStyle);
            					            }
            					            else {
            					                pPrBase.setPStyle(null);
            					            }
	                					}
	                					final Object characterChanges = ((JSONObject)attrsModified).opt("character");
				            			if(characterChanges!=null) {
				            				if(characterChanges instanceof JSONObject) {
				            					CTParaRPrOriginal paraRPrOriginal = rPrChange.getRPr();
				            					if(paraRPrOriginal==null) {
				            						paraRPrOriginal = Context.getWmlObjectFactory().createCTParaRPrOriginal();
				            						paraRPrOriginal.setParent(rPrChange);
				            						rPrChange.setRPr(paraRPrOriginal);
				            					}
				            					Character.applyCharacterProperties(operationDocument, (JSONObject)characterChanges, paraRPrOriginal);
				            				}
				            				else {
				            					rPrChange.setRPr(null);
				            				}
				            			}
				            			final Object paragraphChanges = ((JSONObject)attrsModified).opt("paragraph");
				            			if(paragraphChanges!=null) {
				            				if(paragraphChanges instanceof JSONObject) {
		            							PPrBase pPrBase = pPrChange.getPPr();
		            							if(pPrBase==null) {
		            								pPrBase = Context.getWmlObjectFactory().createPPrBase();
		            								pPrBase.setParent(pPrChange);
		            								pPrChange.setPPr(pPrBase);
		            							}
		            							Paragraph.applyParagraphProperties(operationDocument, (JSONObject)paragraphChanges, pPrBase);
				            				}
				            				else {
				            					pPr.setPPrChange(null);
				            				}
				            			}
	                				}
	                				else {
	                					paraRPr.setRPrChange(null);
	                					pPr.setPPrChange(null);
	                				}
	                			}
		            		}
		            		else {
		            			paraRPr.setRPrChange(null);
		            			pPr.setPPrChange(null);
		            		}
		            	}
		            }
		            else {
		            	final ParaRPr paraRPr = pPr.getRPr();
		            	if(paraRPr!=null) {
		            		paraRPr.setIns(null);
		            		paraRPr.setDel(null);
		            		paraRPr.setRPrChange(null);
		            	}
		            }
	            }
			}
	        catch (Exception e) {
				log.error("docx, ParagraphComponent::setAttributes", e);
			}
		}

		@Override
		public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
			throws JSONException {

			final PPr pPr = ((P)getObject()).getPPr();
	        if(pPr!=null) {
	            final PPrBase.PStyle pStyle = pPr.getPStyle();
	            if(pStyle!=null) {
	                attrs.put("styleId", pStyle.getVal());
	            }
	            final JSONObject paragraphProperties = Paragraph.createParagraphProperties(pPr);
	            if(paragraphProperties!=null) {
	                attrs.put("paragraph", paragraphProperties);
	            }
	            final ParaRPr paraRPr = pPr.getRPr();
	            if(paraRPr!=null) {
	            	// only creating paragraph character attributes if there are no child elements #35344#, #35571#
	            	if(getNextChildComponent(null, null)==null) {
		                final JSONObject characterProperties = Character.createCharacterProperties(createOperationHelper.getThemeFonts(), paraRPr);
		                if(characterProperties!=null) {
		                    attrs.put("character", characterProperties);
		                }
	            	}
	            }

	            CTTrackChange rPrIns = null;
            	CTTrackChange rPrDel = null;
            	ParaRPrChange rPrChange = null;
            	if(paraRPr!=null) {
            		rPrIns = paraRPr.getIns();
            		rPrDel = paraRPr.getDel();
            		rPrChange = paraRPr.getRPrChange();
            	}
            	final CTPPrChange pPrChange = pPr.getPPrChange();
                if(rPrIns!=null||rPrDel!=null||rPrChange!=null||pPrChange!=null) {
                	final JSONObject changes = new JSONObject(2);
                	if(rPrIns!=null) {
	            		changes.put("inserted", Utils.createJSONFromTrackInfo(createOperationHelper.getOperationDocument(), rPrIns));
	                }
	                if(rPrDel!=null) {
	                	changes.put("removed", Utils.createJSONFromTrackInfo(createOperationHelper.getOperationDocument(), rPrDel));
	                }
	                if(rPrChange!=null||pPrChange!=null) {
	                	JSONObject modified = null;
	                	JSONObject modifiedAttrs = new JSONObject(3);
	                	if(rPrChange!=null) {
	 	                	if(modified==null) {
	 	                		modified = Utils.createJSONFromTrackInfo(createOperationHelper.getOperationDocument(), rPrChange);
	 	                	}
		                	final CTParaRPrOriginal rPrOriginal  = rPrChange.getRPr();
	                    	if(rPrOriginal!=null) {
		                    	final JSONObject changedJsonCharacterAttributes = Character.createCharacterProperties(createOperationHelper.getThemeFonts(), rPrOriginal);
		                    	if(changedJsonCharacterAttributes!=null) {
		                    		modifiedAttrs.put("character", changedJsonCharacterAttributes);
		                    	}
	                    	}
	                	}
	                	if(pPrChange!=null) {
	 	                	if(modified==null) {
	 	                		modified = Utils.createJSONFromTrackInfo(createOperationHelper.getOperationDocument(), pPrChange);
	 	                	}
	 	                	final PPrBase pPrOriginal = pPrChange.getPPr();
	 	                	if(pPrOriginal!=null) {
		 	                	final PStyle originalStyle = pPrOriginal.getPStyle();
			 	   	            if(originalStyle!=null) {
			 		                modifiedAttrs.put("styleId", originalStyle.getVal());
			 		            }
	 	                	}
	 	                	final JSONObject changedJsonParagraphAttributes = Paragraph.createParagraphProperties(pPrOriginal);
	 	                	if(changedJsonParagraphAttributes!=null) {
	 	                		modifiedAttrs.put("paragraph", changedJsonParagraphAttributes);
	 	                	}
	                	}

	                	if(!modifiedAttrs.isEmpty()) {
	                		modified.put("attrs", modifiedAttrs);
	                	}
                    	if(!modified.isEmpty()) {
                    		changes.put("modified", modified);
                    	}
	                }
	                if(!changes.isEmpty()) {
	                	attrs.put("changes", changes);
	                }
	            }
	        }
	        return attrs;
		}

    	public void insertText(OperationDocument operationDocument, int textPosition, String text, JSONObject attrs)
            throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

        	if(text.length()>0) {
            	final P paragraph = (P)getObject();
            	final IndexedNodeList<Object> paragraphContent = paragraph.getContent();

                IText t = null;
                Component childComponent = getNextChildComponent(null, null);
                TextRun_Base textRun_Base = null;

                if(childComponent!=null) {
                	if(textPosition>0) {
                		childComponent = childComponent.getComponent(textPosition-1);
                	}
                    // check if the character could be inserted into an existing text:
                    if(childComponent instanceof TextComponent) {
                        t = (IText)childComponent.getObject();
                        final StringBuffer s = new StringBuffer(t.getValue());
                        s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                        t.setValue(s.toString());
                        textRun_Base = (TextRun_Base)childComponent;
                    }
                    else {
                    	t = Context.getWmlObjectFactory().createText();
                        t.setValue(text);
                    	if(childComponent instanceof TextRun_Base) {
                        	// we can insert the text into a existing run
                        	final TextRun_Base runBase = (TextRun_Base)childComponent;
                        	final R oldRun = runBase.getTextRun();
                            t.setParent(oldRun);
                            final int i = runBase.getNode().getIndex();
                            oldRun.getContent().add(textPosition>0?i+1:i, t);
                    	}
                    	else if(childComponent instanceof FldSimpleComponent) {
                        	// new run needs to be created ... (simpleField...)
                            final R newRun = Context.getWmlObjectFactory().createR();
                            newRun.setParent(getObject());
                            final int i = childComponent.getContextChildNode().getIndex();
                            paragraphContent.add(textPosition>0?i+1:i, newRun);
                        	final R run = ((FldSimpleComponent)childComponent).getTextRun();
                        	final RPr rPr = run.getRPr();
                        	if(rPr!=null) {
                    			final RPr newRPr = XmlUtils.deepCopy(rPr);
                    			if(newRPr!=null) {
                    				newRun.setRPr(newRPr);
                    				newRPr.setParent(newRun);
                    			}
                        	}
                            t.setParent(newRun);
                            newRun.getContent().add(t);
                    	}
                        if(textPosition>0) {
                            textRun_Base = (TextRun_Base)childComponent.getNextComponent();
                        }
                        else {
                            textRun_Base = (TextRun_Base)getNextChildComponent(null, null);
                        }
                    }
                }
                else {
                    // the paragraph is empty, we have to create R and its text
                    final R newRun = Context.getWmlObjectFactory().createR();
                    newRun.setParent(getObject());
                    paragraphContent.add(newRun);
                    if(paragraph.getPPr()!=null) {
                        final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                        if(rPr!=null) {
                            rPr.setParent(newRun);
                            newRun.setRPr(rPr);
                        }
                    }
                    t = Context.getWmlObjectFactory().createText();
                    t.setParent(newRun);
                    t.setValue(text);
                    newRun.getContent().add(t);
                    textRun_Base = (TextRun_Base)getNextChildComponent(null, null);
                }
    			if(t!=null) {
    				TextComponent.preserveSpace(t);
    			}
    			if(textRun_Base.getCT()!=Component.CT_NONE) {
    				attrs = TextUtils.forceRemoveChangeTrack(attrs);
    			}
                if(attrs!=null) {
                	textRun_Base.splitStart(textPosition, SplitMode.ATTRIBUTE);
                	textRun_Base.splitEnd(textPosition+text.length()-1, SplitMode.ATTRIBUTE);
                	textRun_Base.applyAttrsFromJSON(operationDocument, attrs);
                }
            }
    	}

        public void splitParagraph(int textPosition) {

        	final P paragraph = (P)getObject();

            Component component = getChildComponent(textPosition);
            if(component!=null) {
            	component.splitStart(textPosition, SplitMode.ATTRIBUTE);
            }
            // creating our new paragraph
            final P destParagraph = Context.getWmlObjectFactory().createP();
            destParagraph.setParent(paragraph.getParent());
            final IndexedNode<Object> destParagraphNode = new IndexedNode<Object>(destParagraph);
            Utils.getContent(paragraph.getParent()).addNode(getNode().getIndex()+1, destParagraphNode);
            final PPr sourceParagraphProperties = paragraph.getPPr();
            if(sourceParagraphProperties!=null){
                final PPr destParagraphProperties = XmlUtils.deepCopy(sourceParagraphProperties);
                // no inheritance for pageBreakBefore attribute... this attribute is always false
                destParagraphProperties.setPageBreakBefore(null);
                destParagraphProperties.setParent(destParagraph);
                destParagraph.setPPr(destParagraphProperties);
            }

            // TODO: to be optimized.. the component is already the correct source component,
            // the iteration is only needed to get the previous RPr
            RPr lastRPr = null;
            while(component!=null) {
                if(textPosition>=component.getComponentNumber()&&textPosition<component.getNextComponentNumber()) {
                    boolean skipSdt = false;
                    ComponentContext childContext = component;
                    ComponentContext parentContext = component.getParentContext();
                    while(!(childContext instanceof ParagraphComponent)) {
                        if(parentContext instanceof SdtParagraphContext) {
                            if(childContext.getNode().getIndex()>0) {
                                // moving end-part of the sdt into the next paragraph
                                final IndexedNodeList<Object> sourceContent = (IndexedNodeList<Object>)((ContentAccessor)parentContext.getNode().getData()).getContent();
                                final int index = childContext.getNode().getIndex();
                                final int count = sourceContent.size() - index;
                                IndexedNodeList.moveNodes(sourceContent, destParagraph.getContent(), index, destParagraph.getContent().size(), count, destParagraph);
                                skipSdt = true;
                            }
                        }
                        else if(parentContext instanceof ParagraphComponent) {
                            final IndexedNode<Object> contextChildNode = childContext.getNode();
                            final IndexedNodeList<Object> parentContent = (IndexedNodeList<Object>)((ContentAccessor)parentContext.getNode().getData()).getContent();
                            int index = contextChildNode.getIndex();
                            if(skipSdt) {
                                index++;
                            }
                            final int count = parentContent.size() - index;
                            if(count>0) {
                                IndexedNodeList.moveNodes(parentContent, destParagraph.getContent(), index, destParagraph.getContent().size(), count, destParagraph);
                            }
                        }
                        childContext = parentContext;
                        parentContext = childContext.getParentContext();
                    }
                    break;
                } else if (component instanceof TextRun_Base) {
                    lastRPr = ((TextRun_Base)component).getTextRun().getRPr();
                }
                component = component.getNextComponent();
            }

            // taking care of paragraph attributes
            if(lastRPr!=null) {

                // if available, we have to get the character attributes from the last textrun
                PPr paraProps = destParagraph.getPPr();
                if(paraProps==null) {
                    paraProps = Context.getWmlObjectFactory().createPPr();
                    paraProps.setParent(destParagraph);
                    destParagraph.setPPr(paraProps);
                }

                // 30793 ... only setting character attributes if no listStyle is used at the paragraph
                if(paraProps.getNumPr()==null||paraProps.getNumPr().getNumId()==null) {
                    ParaRPr paraRPr = TextUtils.cloneRPrToParaRPr(lastRPr);
                    if(paraRPr!=null) {
                        paraRPr.setParent(paraProps);
                        paraProps.setRPr(paraRPr);
                    }
                }
            }
        }

        public void mergeParagraph() {
        	final Component nextParagraphComponent = getNextComponent();
        	if(nextParagraphComponent instanceof ParagraphComponent) {
            	final P paragraph = (P)getObject();
            	final IndexedNodeList<Object> sourceContent = ((P)nextParagraphComponent.getObject()).getContent();
                final IndexedNodeList<Object> destContent = paragraph.getContent();
                if(sourceContent.size()>0) {
                	IndexedNodeList.moveNodes(sourceContent, destContent, 0, destContent.size(), sourceContent.size(), paragraph);
                }
                IndexedNode<Object> parentContextNode = nextParagraphComponent.getParentContext().getNode();
            	Utils.getContent(parentContextNode.getData()).remove(nextParagraphComponent.getNode().getIndex());
            	if(parentContextNode.getData() instanceof SdtBlock) {
            	    final ComponentContext sdtRootContext = nextParagraphComponent.getParentContext();
            	    final ComponentContext parentOfSdtRootContext = sdtRootContext.getParentContext();
                    if(((ContentAccessor)sdtRootContext.getNode().getData()).getContent().isEmpty()) {
                        Utils.getContent(parentOfSdtRootContext.getNode().getData()).remove(sdtRootContext.getNode().getIndex());
                    }
            	}
        	}
        }

        public void insertField(OperationDocument operationDocument, int textPosition, String type, String representation, JSONObject attrs)
            throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

        	final P paragraph = (P)getObject();
        	final IndexedNodeList<Object> paragraphContent = paragraph.getContent();

            final CTSimpleField ctSimpleField = Context.getWmlObjectFactory().createCTSimpleField();
            ctSimpleField.setParent(paragraph);
            ctSimpleField.setInstr(type);
            final R simpleFieldRun = Context.getWmlObjectFactory().createR();
            simpleFieldRun.setParent(ctSimpleField);
            ctSimpleField.getContent().add(simpleFieldRun);
            final Text simpleFieldText = Context.getWmlObjectFactory().createText();
            simpleFieldText.setValue(representation);
            simpleFieldText.setParent(simpleFieldRun);
            simpleFieldRun.getContent().add(simpleFieldText);

            RPr destRPr = null;
            int simpleFieldIndex = 0;

            Component childComponent = getNextChildComponent(null, null);
            if(childComponent!=null) {
            	if(textPosition>0) {
            		childComponent = childComponent.getComponent(textPosition-1);
            		childComponent.splitEnd(textPosition, SplitMode.ATTRIBUTE);
            		simpleFieldIndex = childComponent.getContextChildNode().getIndex()+1;
            	}
            	RPr sourceRPr = null;
            	if(childComponent instanceof TextRun_Base) {
            		sourceRPr = ((TextRun_Base)childComponent).getTextRun().getRPr();
            	}
            	else if(childComponent instanceof FldSimpleComponent) {
            		sourceRPr = ((FldSimpleComponent)childComponent).getTextRun().getRPr();
            	}
            	if(sourceRPr!=null) {
            		destRPr = XmlUtils.deepCopy(sourceRPr);
            	}
            	paragraphContent.add(simpleFieldIndex, ctSimpleField);
            }
            else {
            	if(paragraph.getPPr()!=null) {
            		destRPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
            	}
                paragraphContent.add(simpleFieldIndex, ctSimpleField);
            }
    		if(destRPr!=null) {
    			destRPr.setParent(simpleFieldRun);
    			simpleFieldRun.setRPr(destRPr);
    		}
            if(attrs!=null) {
            	if(textPosition>0) {
            		childComponent = childComponent.getNextComponent();
            	}
            	else {
            		childComponent = getNextChildComponent(null, null);
            	}
            	childComponent.applyAttrsFromJSON(operationDocument, attrs);
            }
        }
    }

    public static class TableComponent extends Component implements ITable {

        public TableComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber) {
            super(parentContext, _node, _contextNode, _componentNumber);
        }
        @Override
		public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
            final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

            Component nextComponent = null;
            final IndexedNode<Object> tblNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)tblNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = getContentModel(childNode, tblNode.getData());
                if(o instanceof Tr) {
                    nextComponent = new TrComponent(this, childNode, tblNode, nextComponentNumber);
                }
            }
            return nextComponent;
        }
        @Override
        public IndexedNode<Object> getContextChildNode() {
            if(getParentContext() instanceof SdtRootContext) {
                return getParentContext().getNode();
            }
            return getNode();
        }
		@Override
		public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs) {
			if(attrs==null) {
				return;
			}
			try {
		        final Tbl tbl = (Tbl)getObject();
        		TblPr tblPr = tbl.getTblPr();
	        	if(tblPr==null) {
	        		tblPr = Context.getWmlObjectFactory().createTblPr();
	        		tblPr.setParent(tbl);
	        		tbl.setTblPr(tblPr);
	        	}
        		if(attrs.has("styleId")) {
		            Table.applyTableStyle(attrs.getString("styleId"), tblPr);
		        }
        		TblGrid tblGrid = tbl.getTblGrid();
        		if(tblGrid==null) {
        			tblGrid = Context.getWmlObjectFactory().createTblGrid();
        			tblGrid.setParent(tbl);
        			tbl.setTblGrid(tblGrid);
        		}
		        Table.applyTableProperties(operationDocument, attrs.optJSONObject("table"), this, tblGrid, tblPr);
	        	final Object changes = attrs.opt("changes");
	        	if(changes!=null) {
	        		boolean removeTableChangeTrack = false;
	        		if(changes instanceof JSONObject) {
	        			final Object modified = ((JSONObject)changes).opt("modified");
	        			if(modified!=null) {
	        				if(modified instanceof JSONObject) {
	        					CTTblPrChange tblPrChange = tblPr.getTblPrChange();
	        					if(tblPrChange==null) {
	        						tblPrChange = Context.getWmlObjectFactory().createCTTblPrChange();
	        						tblPrChange.setParent(tblPr);
	        						tblPr.setTblPrChange(tblPrChange);
	        					}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)modified, tblPrChange);
	        					final Object attrsModified = ((JSONObject)modified).opt("attrs");
	        					if(attrsModified!=null) {
	        						if(attrsModified instanceof JSONObject) {
	        							CTTblPrBase tblPrChangePr = tblPrChange.getTblPr();
	        							if(tblPrChangePr==null) {
	        								tblPrChangePr = Context.getWmlObjectFactory().createCTTblPrBase();
	        								tblPrChangePr.setParent(tblPrChange);
	        								tblPrChange.setTblPr(tblPrChangePr);
	        							}
	        							if(((JSONObject)attrsModified).has("styleId")) {
	        								Table.applyTableStyle(((JSONObject) attrsModified).getString("styleId"), tblPrChangePr);
	        							}
	        							final JSONObject tableAttrChanges = ((JSONObject)attrsModified).optJSONObject("table");
	        							if(tableAttrChanges!=null) {
	        								TblGridBase tblGridBase = null;
	        								Object tblGridObj = tableAttrChanges.opt("tableGrid");
	        								if(tblGridObj!=null) {
	        									if(tblGridObj instanceof JSONArray) {
			        								CTTblGridChange tblGridChange = tblGrid.getTblGridChange();
			        								if(tblGridChange==null&&tableAttrChanges.has("tableGrid")) {
			        									tblGridChange=Context.getWmlObjectFactory().createCTTblGridChange();
			        									tblGridChange.setParent(tblGrid);
			        									tblGrid.setTblGridChange(tblGridChange);
			        								}
			        								tblGridBase = tblGridChange.getTblGrid();
			        								if(tblGridBase==null) {
			        									tblGridBase = Context.getWmlObjectFactory().createTblGridBase();
			        									tblGridBase.setParent(tblGridChange);
			        									tblGridChange.setTblGrid(tblGridBase);
			        								}
	        									}
	        									else {
	        										tblGrid.setTblGridChange(null);
	        									}
	        								}
	        								Table.applyTableProperties(operationDocument, tableAttrChanges, null, tblGridBase, tblPrChangePr);
	        							}
	        						}
	        						else {
	        							tblPrChange.setTblPr(null);
	        						}
	        					}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsModified, tblPrChange);
	        				}
	        				else {
	        					removeTableChangeTrack = true;
	        				}
	        			}
	        			// removed / inserted changetracking has to be applied to each row
	        			if(((JSONObject)changes).opt("inserted")!=null||((JSONObject)changes).opt("removed")!=null) {
	        				Component trComponent = getNextChildComponent(null, null);
	        				while(trComponent!=null) {
	        					trComponent.applyAttrsFromJSON(operationDocument, attrs);
	        					trComponent = trComponent.getNextComponent();
	        				}
	        			}
	        		}
	        		else {
	        			removeTableChangeTrack = true;
	        		}
	        		if(removeTableChangeTrack) {
	        			if(tblPr!=null) {
	        				tblPr.setTblPrChange(null);
	        			}
	        			if(tblGrid!=null) {
	        				tblGrid.setTblGridChange(null);
	        			}
	        		}
	        	}
			}
			catch(Exception e) {
				log.error("docx, TableComponent::setAttributes:", e);
			}
		}

		@Override
		public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
			throws JSONException {

			final Tbl tbl = (Tbl)getObject();
			final OperationDocument operationDocument = createOperationHelper.getOperationDocument();
	        final ServiceLookup services = operationDocument.getServiceLookup();
	        final Session session = operationDocument.getSession();
	        final int maxTableColumns = ConfigurationHelper.getIntegerOfficeConfigurationValue(services, session, "//module/maxTableColumns", 15);
	        final int maxTableRows = ConfigurationHelper.getIntegerOfficeConfigurationValue(services, session, "//module/maxTableRows", 1500);
	        final int maxTableCells = ConfigurationHelper.getIntegerOfficeConfigurationValue(services, session, "//module/maxTableCells", 1500);

	        final TblGrid tableGrid = tbl.getTblGrid();
	        final JSONArray tableGridArray = Table.getTableGrid(tableGrid);
	        int gridCount = tableGridArray!=null?tableGridArray.length() : 0;
	        int columns = gridCount;
	        int rows = 0;
	        int rowWidth = 0;
	        boolean useCellWidth = true;

	        Component rowComponent = getNextChildComponent(null, null);
	        while(rowComponent!=null) {
	            rows++;
	            // check, if table cells in the first row have a width defined (in "dxa")
	            if (rows == 1) {
	                Component cellComponent = rowComponent.getNextChildComponent(null, null);
	                while(cellComponent!=null) {
	                    TcPr cellProperties = ((Tc)cellComponent.getObject()).getTcPr();
	                    if (cellProperties != null) {
	                        int cellWidth = 0;
	                        TblWidth cellWidthObject = cellProperties.getTcW();
	                        if (cellWidthObject != null) {
	                            String cellWidthType = cellWidthObject.getType();
	                            BigInteger bigInt = cellWidthObject.getW();
	                            if (bigInt != null) {
	                                cellWidth = bigInt.intValue();
	                                if ((cellWidthType != null) && (cellWidthType.equals("dxa"))) {
	                                    rowWidth = rowWidth + cellWidth * 2540 / 72 / 20;
	                                } else {
	                                    useCellWidth = false;
	                                }
	                            } else {
	                                useCellWidth = false;
	                            }
	                        } else {
	                            useCellWidth = false;
	                        }
	                    } else {
	                        useCellWidth = false;
	                    }
	                    cellComponent = cellComponent.getNextComponent();
	                }
	            }
	            rowComponent = rowComponent.getNextComponent();
	        }

	        if (useCellWidth == false) {
	            rowWidth = 0;  // <- do not use the value
	        }

	        boolean maxTableSizeExceeded = (columns>maxTableColumns)||(rows>maxTableRows)||((columns*rows)>maxTableCells);

	        JSONObject tableChanges = null;
	        TblPr tblPr = tbl.getTblPr();
	        if(tblPr!=null) {
	            // setting the table style that is used...
	            TblStyle tblStyle = tblPr.getTblStyle();
	            if(tblStyle!=null) {
	                String style = tblStyle.getVal();
	                if(style!=null&&style.length()>0) {
	                    attrs.put("styleId", style);
	                }
	            }
	        }
	        final boolean isRootTable = !(tbl.getParent() instanceof Tc);
	        Table.createTableProperties(operationDocument.getPackage(), tbl.getTblPr(), tableGridArray, isRootTable, rowWidth, attrs);
	        final CTTblPrChange tblPrChange = tbl.getTblPr().getTblPrChange();
	        if(tblPrChange!=null) {
	        	tableChanges = new JSONObject(2);
	        	final JSONObject tableModified = Utils.createJSONFromTrackInfo(operationDocument, tblPrChange);
	        	JSONArray tableGridChangeArray = null;
	        	if(tableGrid!=null) {
	        		final CTTblGridChange tblGridChange = tableGrid.getTblGridChange();
	        		if(tblGridChange!=null) {
	        			tableGridChangeArray = Table.getTableGrid(tblGridChange.getTblGrid());
	        		}
	        	}
	        	final CTTblPrBase tblChangePr = tblPrChange.getTblPr();
	        	if(tblChangePr!=null) {
	            	final JSONObject tableChangedAttrs = new JSONObject();
	            	final TblStyle tblStyle = tblChangePr.getTblStyle();
	            	if(tblStyle!=null) {
	            		String style = tblStyle.getVal();
	            		if(style!=null&&!style.isEmpty()) {
	            			tableChangedAttrs.put("styleId", style);
	            		}
	            	}
	            	Table.createTableProperties(operationDocument.getPackage(), tblPrChange.getTblPr(), tableGridChangeArray, isRootTable, rowWidth, tableChangedAttrs);
	        		if(!tableChangedAttrs.isEmpty()) {
	        			tableModified.put("attrs", tableChangedAttrs);
	        		}
	        	}
	        	if(!tableModified.isEmpty()) {
	        		tableChanges.put("modified", tableModified);
	        	}
	        	if(!tableChanges.isEmpty()) {
	        		attrs.put("changes", tableChanges);
	        	}
	        }
	        JSONObject sizeExceeded = null;
	        if(maxTableSizeExceeded) {
	            sizeExceeded = new JSONObject();
	            sizeExceeded.put("rows", rows);
	            sizeExceeded.put("columns", columns);
	            attrs.put("sizeExceeded", sizeExceeded);
	        }
			return attrs;
		}

		// ITable interfaces
		@Override
		public void insertRows(OperationDocument operationDocument, int destinationRowComponent, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
		    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

	    	final TrComponent trReferenceRow = referenceRow!=-1 ? (TrComponent)getChildComponent(referenceRow) : null;
	    	final Tbl tbl = (Tbl)getObject();
	        final IndexedNodeList<Object> tblContent = tbl.getContent();
	        final Component oldTr = getChildComponent(destinationRowComponent);
	        int destinationIndex = oldTr!=null ? oldTr.getNode().getIndex() : tblContent.size();

	        for(int i = 0; i<count; i++) {
	            final Tr tr = Context.getWmlObjectFactory().createTr();
	            tr.setParent(tbl);
	            tblContent.add(destinationIndex++, tr);
	            if(insertDefaultCells) {
	                JSONArray tableGrid = Table.getTableGrid(tbl.getTblGrid());
	                if(tableGrid!=null) {
	                    for(int j = 0; j<tableGrid.length(); j++) {
	                        Tc tc = Context.getWmlObjectFactory().createTc();
	                        tc.setParent(tr);
	                        tr.getContent().add(tc);
	                    }
	                }
	            }
	            else if(trReferenceRow!=null) {       // we try to create the new row from the referenceRow
	                final Tr trReference = (Tr)trReferenceRow.getObject();
	                final TrPr sourceTrPr = trReference.getTrPr();
	                if(sourceTrPr!=null) {
	                    final TrPr newTrPr = XmlUtils.deepCopy(sourceTrPr);
	                    newTrPr.setParent(tr);
	                    tr.setTrPr(newTrPr);
	                }
	                TcComponent tcReferenceComponent = (TcComponent)trReferenceRow.getNextChildComponent(null, null);
	                while(tcReferenceComponent!=null) {
	                    final Tc tcReference = (Tc)tcReferenceComponent.getObject();
	                    final Tc newTc = Context.getWmlObjectFactory().createTc();
	                    newTc.setParent(tr);
	                    tr.getContent().add(newTc);
	                    TcPr tcPrReference = tcReference.getTcPr();
	                    if(tcPrReference!=null) {
	                        TcPr newTcPr = XmlUtils.deepCopy(tcPrReference);
	                        newTcPr.setParent(newTc);
	                        newTc.setTcPr(newTcPr);
	                    }
	                    tcReferenceComponent = (TcComponent)tcReferenceComponent.getNextComponent();
	                }
	            }
	        }
	        if(attrs!=null) {
	        	Component c = getNextChildComponent(null, null).getComponent(destinationRowComponent);
		        for(int i=0; i<count; i++) {
		        	c.applyAttrsFromJSON(operationDocument, attrs);
		        	c = c.getNextComponent();
		        }
	        }
		}

		@Override
		public void splitTable(int rowPosition) {
	    	final Tbl sourceTbl = (Tbl)getObject();
	    	// to clone the table we temporarily remove the content
	    	final IndexedNodeList<Object> sourceTblContent = sourceTbl.getContent();
	    	sourceTbl.setContent(null);
	    	final Tbl destTbl = XmlUtils.deepCopy(sourceTbl);
	    	sourceTbl.setContent(sourceTblContent);
	    	final ContentAccessor sourceTblParent = (ContentAccessor)getContextObject();
	    	destTbl.setParent(sourceTblParent);
	    	sourceTblParent.getContent().add(getNode().getIndex()+1, destTbl);
	    	IndexedNodeList.moveNodes(sourceTblContent, destTbl.getContent(), rowPosition, 0, sourceTblContent.size()-rowPosition, destTbl);
		}

		@Override
		public void mergeTable() {
	    	final TableComponent sourceTable = (TableComponent)getNextComponent();
	    	final IndexedNodeList<Object> sourceContent = ((Tbl)sourceTable.getObject()).getContent();
	    	final IndexedNodeList<Object> destContent = ((Tbl)getObject()).getContent();
	    	IndexedNodeList.moveNodes(sourceContent, destContent, 0, destContent.size(), sourceContent.size(), getObject());
	    	((ContentAccessor)((Child)sourceTable.getObject()).getParent()).getContent().remove(sourceTable.getNode().getIndex());
		}

		@Override
		public void insertColumn(OperationDocument operationDocument, JSONArray tableGrid, int gridPosition, String insertMode)
			throws JSONException {

			boolean before = insertMode.equals("before");
            TrComponent trComponent = (TrComponent)getNextChildComponent(null, null);
            while(trComponent!=null) {
                TcComponent tcReference = (TcComponent)trComponent.getNextChildComponent(null, null);
                TcComponent destination = null;
                while(tcReference!=null) {
                    if(gridPosition>=tcReference.getGridPosition()&&gridPosition<tcReference.getNextGridPosition()) {
                        destination = tcReference;
                        break;
                    }
                    tcReference = (TcComponent)tcReference.getNextComponent();
                }
                List<Object> rowContent = ((Tr)trComponent.getObject()).getContent();
                int destinationIndex = destination==null?rowContent.size():before?destination.getNode().getIndex():destination.getNode().getIndex()+1;
                Tc tc = Context.getWmlObjectFactory().createTc();
                tc.setParent(trComponent.getObject());
                if(tcReference!=null) {
                    TcPr referenceTcPr = ((Tc)tcReference.getObject()).getTcPr();
                    if(referenceTcPr!=null) {
                        TcPr newTcPr = XmlUtils.deepCopy(referenceTcPr);
                        newTcPr.setParent(tc);
                        tc.setTcPr(newTcPr);
                    }
                }
                rowContent.add(destinationIndex, tc);
                trComponent = (TrComponent)trComponent.getNextComponent();
            }
            final Tbl tbl = (Tbl)getObject();
            TblGrid tblGrid = tbl.getTblGrid();
            if(tblGrid==null) {
            	tblGrid = Context.getWmlObjectFactory().createTblGrid();
            	tblGrid.setParent(tbl);
            	tbl.setTblGrid(tblGrid);
            }
            Table.setTableGrid(operationDocument, this, tblGrid, tableGrid);
		}

		@Override
        public void deleteColumns(OperationDocument operationDocument, int gridStart, int gridEnd)
        	throws JSONException {

        	Component trComponent = getNextChildComponent(null, null);
            while(trComponent!=null) {
                int startIndex = -1;
                int endIndex = -1;
                TcComponent tcComponent = (TcComponent)trComponent.getNextChildComponent(null, null);
                while(tcComponent!=null) {
                    if((tcComponent.getGridPosition()>=gridStart&&tcComponent.getGridPosition()<=gridEnd)||
                        (tcComponent.getNextGridPosition()-1>=gridStart&&tcComponent.getNextGridPosition()-1<=gridEnd)){
                        if (startIndex==-1)
                            startIndex = tcComponent.getNode().getIndex();
                        endIndex = tcComponent.getNode().getIndex();
                    }
                    if(tcComponent.getNextGridPosition()>gridEnd)
                        break;
                    tcComponent = (TcComponent)tcComponent.getNextComponent();
                }
                if(startIndex!=-1) {
                    List<Object> rowContent = ((Tr)trComponent.getObject()).getContent();
                    for(int i=endIndex;i>=startIndex;i--) {
                        rowContent.remove(i);
                    }
                }
                trComponent = trComponent.getNextComponent();
            }
            JSONArray tableGrid = Table.getTableGrid(((Tbl)getObject()).getTblGrid());
            JSONArray newTableGrid = new JSONArray();
            for(int i=0;i<tableGrid.length();i++) {
                if(i<gridStart||i>gridEnd) {
                    newTableGrid.put(tableGrid.get(i));
                }
            }
            final Tbl tbl = (Tbl)getObject();
            TblGrid tblGrid = tbl.getTblGrid();
            if(tblGrid==null) {
            	tblGrid = Context.getWmlObjectFactory().createTblGrid();
            	tblGrid.setParent(tbl);
            	tbl.setTblGrid(tblGrid);
            }
            Table.setTableGrid(operationDocument, this, tblGrid, newTableGrid);
		}
    }

    public static class TrComponent extends Component implements IRow {

        public TrComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber) {
            super(parentContext, _node, _contextNode, _componentNumber);
        }
        @Override
		public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
            final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;
            final int nextGridPosition = previousChildComponent instanceof TcComponent?((TcComponent)previousChildComponent).getGridPosition()+1:0;

            Component nextComponent = null;
            final IndexedNode<Object> trNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)trNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = getContentModel(childNode, trNode.getData());
                if(o instanceof Tc) {
                    nextComponent = new TcComponent(this, childNode, trNode, nextComponentNumber, nextGridPosition);
                }
            }
            return nextComponent;
        }
		@Override
		public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs) {
			if(attrs==null) {
				return;
			}
			try {
		        final Tr tr = (Tr)getObject();
		        TrPr trPr = tr.getTrPr();
		        if(trPr==null) {
		            trPr = Context.getWmlObjectFactory().createTrPr();
		            trPr.setParent(tr);
		            tr.setTrPr(trPr);
		        }
		        Table.applyRowProperties(attrs.optJSONObject("row"), trPr);
		        final Object changes = attrs.opt("changes");
		        if(changes!=null) {
			        if(changes instanceof JSONObject) {
			        	final Object attrsInserted = ((JSONObject)changes).opt("inserted");
			        	if(attrsInserted!=null) {
			        		if(attrsInserted instanceof JSONObject) {
		            			CTTrackChange ins = trPr.getIns();
		            			if(ins==null) {
		            				ins = Context.getWmlObjectFactory().createCTTrackChange();
		            				ins.setParent(trPr);
		            				trPr.setIns(ins);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsInserted, ins);
			        		}
			        		else {
			        			trPr.setIns(null);
			        		}
			        	}
			        	final Object attrsRemoved = ((JSONObject)changes).opt("removed");
			        	if(attrsRemoved!=null) {
				        	if(attrsRemoved instanceof JSONObject) {
		            			CTTrackChange del = trPr.getDel();
		            			if(del==null) {
		            				del = Context.getWmlObjectFactory().createCTTrackChange();
		            				del.setParent(trPr);
		            				trPr.setDel(del);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsRemoved, del);
				        	}
				        	else {
				        		trPr.setDel(null);
				        	}
			        	}
			        	final Object attrsModified = ((JSONObject)changes).opt("modified");
			        	if(attrsModified!=null) {
			        		if(attrsModified instanceof JSONObject) {
			        			CTTrPrChange trPrChange = trPr.getTrPrChange();
			        			if(trPrChange==null) {
			        				trPrChange = Context.getWmlObjectFactory().createCTTrPrChange();
			        				trPrChange.setParent(trPr);
			        				trPr.setTrPrChange(trPrChange);
			        			}
			        			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsModified, trPrChange);
	        					final Object attrsModifiedAttrs = ((JSONObject)attrsModified).opt("attrs");
	        					if(attrsModifiedAttrs!=null) {
	        						if(attrsModifiedAttrs instanceof JSONObject) {
	        							CTTrPrBase trPrBase = trPrChange.getTrPr();
	        							if(trPrBase==null) {
	        								trPrBase = Context.getWmlObjectFactory().createCTTrPrBase();
	        								trPrBase.setParent(trPrChange);
	        								trPrChange.setTrPr(trPrBase);
	        							}
	        							Table.applyRowProperties(((JSONObject)attrsModifiedAttrs).optJSONObject("row"), trPrBase);
	        						}
	        						else {
	        							trPrChange.setTrPr(null);
	        						}
	        					}
			        		}
			        		else {
			        			trPr.setTrPrChange(null);
			        		}
			        	}
			        }
			        else {
			        	trPr.setIns(null);
			        	trPr.setDel(null);
			        	trPr.setTrPrChange(null);
			        }
		        }
			}
			catch(Exception e) {
				log.error("docx, TrComponent::setAttributes:", e);
			}
		}

		@Override
		public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
			throws JSONException {

			final Tr tr = (Tr)getObject();
            final TrPr trPr = tr.getTrPr();
            if(trPr!=null) {
                Table.createRowProperties(tr.getTrPr(), attrs);
                if(trPr.getDel()!=null||trPr.getIns()!=null||trPr.getTrPrChange()!=null) {
                    final JSONObject jsonRowChanges = new JSONObject(2);
                    if(trPr.getDel()!=null) {
                    	jsonRowChanges.put("removed", Utils.createJSONFromTrackInfo(createOperationHelper.getOperationDocument(), trPr.getDel()));
                    }
                    if(trPr.getIns()!=null) {
                    	jsonRowChanges.put("inserted", Utils.createJSONFromTrackInfo(createOperationHelper.getOperationDocument(), trPr.getIns()));
                    }
                    if(trPr.getTrPrChange()!=null) {
                    	JSONObject rowModified = Utils.createJSONFromTrackInfo(createOperationHelper.getOperationDocument(), trPr.getTrPrChange());
                    	final CTTrPrBase trPrChangePr = trPr.getTrPrChange().getTrPr();
                    	if(trPrChangePr!=null) {
                    		final JSONObject jsonRowAttrChanges = new JSONObject();
                    		Table.createRowProperties(trPrChangePr, jsonRowAttrChanges);
                    		if(!jsonRowAttrChanges.isEmpty()) {
                    			rowModified.put("attrs", jsonRowAttrChanges);
                    		}
                    	}
                    	if(!rowModified.isEmpty()) {
	                    	jsonRowChanges.put("modified", rowModified);
                    	}
                    }
                    if(!jsonRowChanges.isEmpty()) {
                    	attrs.put("changes", jsonRowChanges);
                    }
                }
            }
            return attrs;
		}

		// IRow interface
		@Override
		public void insertCells(OperationDocument operationDocument, int cellPosition, int count, JSONObject attrs)
		    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

        	Component c = getChildComponent(cellPosition);
        	final Tr tr = (Tr)getObject();
            final IndexedNodeList<Object> trContent = tr.getContent();
            final int destinationIndex = c!=null ? c.getNode().getIndex() : trContent.size();
            for (int i=0; i<count; i++) {
                final Tc tc = Context.getWmlObjectFactory().createTc();
                tc.setParent(tr);
                trContent.add(destinationIndex, tc);
            }
            if(attrs!=null) {
                c = getChildComponent(cellPosition);
                for(int i=0; i<count; i++) {
                	c.applyAttrsFromJSON(operationDocument, attrs);
                	c = c.getNextComponent();
                }
            }
		}
    }

    public static class TcComponent extends Component {

        private final int gridPosition;

        public TcComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber, int _gridPosition) {
            super(parentContext, _node, _contextNode, _componentNumber);
            gridPosition = _gridPosition;
        }

        public int getGridPosition() {
            return gridPosition;
        }

        public int getNextGridPosition() {
            int gridSpan = 1;
            TcPr tcPr = ((Tc)getNode().getData()).getTcPr();
            if(tcPr!=null) {
                TcPrInner.GridSpan span = tcPr.getGridSpan();
                if (span!=null) {
                    gridSpan = span.getVal().intValue();
                    if (gridSpan==0) {
                        System.out.println("Error: empty gridSpan is zero... this results in problems with the component iterator");
                        gridSpan++; // increasing.. preventing an endless loop
                    }
                }
            }
            return gridPosition + gridSpan;
        }
		@Override
		public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
            final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

            Component nextComponent = null;
		    final IndexedNode<Object> cellNode = getNode();
            final IndexedNodeList<Object> nodeList = (IndexedNodeList<Object>)((ContentAccessor)cellNode.getData()).getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = getContentModel(childNode, cellNode.getData());
                if(o instanceof P) {
                    nextComponent = new ParagraphComponent(this, childNode, cellNode, nextComponentNumber);
                }
                else if(o instanceof Tbl) {
                    nextComponent = new TableComponent(this, childNode, cellNode, nextComponentNumber);
                }
            }
            return nextComponent;
		}
		@Override
		public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs) {
			if(attrs==null) {
				return;
			}
			try {
		        final Tc tc = (Tc)getObject();
		        TcPr tcPr = tc.getTcPr();
		        if(tcPr==null) {
		            tcPr = Context.getWmlObjectFactory().createTcPr();
		            tcPr.setParent(tc);
		            tc.setTcPr(tcPr);
		        }
		        Table.applyCellProperties(operationDocument, attrs.optJSONObject("cell"), tcPr);
		        final Object changes = attrs.opt("changes");
		        if(changes!=null) {
			        if(changes instanceof JSONObject) {
			        	final Object attrsInserted = ((JSONObject)changes).opt("inserted");
			        	if(attrsInserted!=null) {
			        		if(attrsInserted instanceof JSONObject) {
		            			CTTrackChange ins = tcPr.getCellIns();
		            			if(ins==null) {
		            				ins = Context.getWmlObjectFactory().createCTTrackChange();
		            				ins.setParent(tcPr);
		            				tcPr.setCellIns(ins);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsInserted, ins);
			        		}
				        	else {
				        		tcPr.setCellIns(null);
				        	}
			        	}
			        	final Object attrsRemoved = ((JSONObject)changes).opt("removed");
			        	if(attrsRemoved!=null) {
		            		if(attrsRemoved instanceof JSONObject) {
		            			CTTrackChange del = tcPr.getCellDel();
		            			if(del==null) {
		            				del = Context.getWmlObjectFactory().createCTTrackChange();
		            				del.setParent(tcPr);
		            				tcPr.setCellDel(del);
		            			}
		            			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsRemoved, del);
		            		}
		            		else {
		            			tcPr.setCellDel(null);
		            		}
			        	}
			        	final Object attrsModified = ((JSONObject)changes).opt("modified");
			        	if(attrsModified!=null) {
			        		if(attrsModified instanceof JSONObject) {
			        			CTTcPrChange tcPrChange = tcPr.getTcPrChange();
			        			if(tcPrChange==null) {
			        				tcPrChange = Context.getWmlObjectFactory().createCTTcPrChange();
			        				tcPrChange.setParent(tcPr);
			        				tcPr.setTcPrChange(tcPrChange);
			        			}
		            			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsModified, tcPrChange);
	        					final Object attrsModifiedAttrs = ((JSONObject)attrsModified).opt("attrs");
	        					if(attrsModifiedAttrs!=null) {
	        						if(attrsModifiedAttrs instanceof JSONObject) {
	        							TcPrInner tcPrInner = tcPrChange.getTcPr();
	        							if(tcPrInner==null) {
	        								tcPrInner = Context.getWmlObjectFactory().createTcPrInner();
	        								tcPrInner.setParent(tcPrChange);
	        								tcPrChange.setTcPr(tcPrInner);
	        							}
	        					        Table.applyCellProperties(operationDocument, ((JSONObject)attrsModifiedAttrs).optJSONObject("cell"), tcPrInner);
	        						}
	        						else {
	        							tcPrChange.setTcPr(null);
	        						}
	        					}
			        		}
			        		else {
			        			tcPr.setTcPrChange(null);
			        		}
			        	}
			        }
			        else {
			        	tcPr.setCellDel(null);
			        	tcPr.setCellIns(null);
			        	tcPr.setCellMerge(null);
			        }
		        }
			}
			catch(Exception e) {
				log.error("docx, TcComponent::setAttributes:", e);
			}
		}

		@Override
		public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
			throws JSONException {

            final Tc cell = (Tc)getObject();
            final Tbl tbl = (Tbl)((Tr)cell.getParent()).getParent();
            final TcPr tcPr = cell.getTcPr();
            final OperationDocument operationDocument = createOperationHelper.getOperationDocument();
            if(tcPr!=null) {
            	Table.createCellProperties(tcPr, tbl, attrs);
                if(tcPr.getCellDel()!=null||tcPr.getCellIns()!=null||tcPr.getTcPrChange()!=null||tcPr.getCellMerge()!=null) {
                    final JSONObject jsonCellChanges = new JSONObject(2);
                    if(tcPr.getCellDel()!=null) {
                    	jsonCellChanges.put("removed", Utils.createJSONFromTrackInfo(operationDocument, tcPr.getCellDel()));
                    }
                    if(tcPr.getCellIns()!=null) {
                    	jsonCellChanges.put("inserted", Utils.createJSONFromTrackInfo(operationDocument, tcPr.getCellIns()));
                    }
                    if(tcPr.getTcPrChange()!=null) {
                    	final JSONObject jsonCellModified = Utils.createJSONFromTrackInfo(operationDocument, tcPr.getTcPrChange());
                    	final JSONObject jsonCellAttrChanges = new JSONObject();
                    	final TcPrInner tcPrInner = tcPr.getTcPrChange().getTcPr();
                    	if(tcPrInner!=null) {
                        	Table.createCellProperties(tcPrInner, tbl, jsonCellAttrChanges);
                    	}
                    	if(!jsonCellAttrChanges.isEmpty()) {
                    		jsonCellModified.put("attrs", jsonCellAttrChanges);
                    	}
                    	if(!jsonCellModified.isEmpty()) {
                    		jsonCellChanges.put("modified", jsonCellModified);
                    	}
                    }
                    if(tcPr.getCellMerge()!=null) {
                    	jsonCellChanges.put("merged", Utils.createJSONFromTrackInfo(operationDocument, tcPr.getCellMerge()));
                    }
                    if(!jsonCellChanges.isEmpty()) {
                    	attrs.put("changes", jsonCellChanges);
                    }
                }
            }
			return attrs;
		}
		@Override
	    public void postFixNonConformance(AtomicInteger id) {
			super.postFixNonConformance(id);

			// check if this cell is having P at the end
			Component component = getNextChildComponent(null, null);
	        boolean hasP = false;
	        while(component!=null) {
	            if(component instanceof ITable) {
	                hasP = false;                                           // :-( there must be a P within Tc... and there must be a P behind Tbl
	            }
	            else if(component instanceof ParagraphComponent) {
	                hasP = true;
	            }
	            component = component.getNextComponent();
	        }
	        if(!hasP) {
	            P p = Context.getWmlObjectFactory().createP();
	            p.setRsidP("47110815");
	            p.setParent(getObject());
	            ((Tc)getObject()).getContent().add(p);
	        }
	    }
    }

    public abstract static class TextRun_Base extends Component {

        // change tracking mode of this component
        private int changeTrackFlags = 0;

        private IndexedNode<Object> textRunNode = null;
        private IndexedNode<Object> hyperlinkNode = null;
        private IndexedNode<Object> runInsNode = null;
        private IndexedNode<Object> runDelNode = null;

        protected TextRun_Base(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(parentContext, _node, _paragraphNode, _componentNumber);
            ComponentContext parent = this;
            do {
                parent = parent.getParentContext();
                if(parent instanceof TextRunContext) {
                    textRunNode = parent.getNode();
                }
                else if(parent instanceof HyperlinkContext) {
                    hyperlinkNode = parent.getNode();
                }
                else if(parent instanceof RunInsContext) {
                    runInsNode = parent.getNode();
                    changeTrackFlags |= (Component.CT_INSERTED<<16)|Component.CT_INSERTED;
                }
                else if(parent instanceof RunDelContext) {
                    runDelNode = parent.getNode();
                    changeTrackFlags |= (Component.CT_DELETED<<16)|Component.CT_DELETED;
                }
            }
            while(!(parent instanceof ParagraphComponent));
        }

        @Override
        public Object getContextObject() {
        	Object o = ((Child)getNode().getData()).getParent();
        	while(!(o instanceof P)&&o instanceof Child) {
        		o = ((Child)o).getParent();
        	}
        	return o;
        }
        @Override
		public IndexedNode<Object> getContextChildNode() {
        	if(hyperlinkNode!=null) {
        		return hyperlinkNode;
        	}
        	else if(runInsNode!=null) {
        		return runInsNode;
        	}
        	else if(runDelNode!=null) {
        		return runDelNode;
        	}
        	else {
        		return textRunNode;
        	}
        }
		@Override
		public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
			return null;
		}
		@Override
		public void splitStart(int n, SplitMode splitMode) {
			splitAccessor(this, true, splitMode);
		}
		@Override
		public void splitEnd(int n, SplitMode splitMode) {
			splitAccessor(this, false, splitMode);
		}
		@Override
		public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
		    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

		    try {
				TextUtils.setTextRunAttributes(operationDocument, attrs, this);
			}
	        catch (Exception e) {
				log.error("docx, TextRun_Base::setAttributes", e);
			}
		}
		@Override
		public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
			throws JSONException, ParseException, FilterException {

			JSONObject changes = null;
            JSONObject jsonCharacterProperties = null;

            final OperationDocument operationDocument = createOperationHelper.getOperationDocument();
            final IndexedNode<Object> textRunNode = getTextRunNode();
            final R textRun = (R)textRunNode.getData();
            final RPr rPr = textRun.getRPr();
            if(rPr!=null) {
                final RStyle rStyle = rPr.getRStyle();
                if(rStyle!=null)
                    attrs.put("styleId", rStyle.getVal());
                jsonCharacterProperties = Character.createCharacterProperties(createOperationHelper.getThemeFonts(), rPr);
                if(rPr.getRPrChange()!=null) {
                	changes = new JSONObject(2);
                	final JSONObject changedAttrs = new JSONObject();
                	final org.docx4j.wml.CTRPrChange.RPr rPrChange = rPr.getRPrChange().getRPr();
                	final JSONObject changedJsonCharacterAttributes = Character.createCharacterProperties(createOperationHelper.getThemeFonts(),  rPrChange);
                	if(changedJsonCharacterAttributes!=null) {
                		changedAttrs.put("character", changedJsonCharacterAttributes);
                	}
                	final JSONObject modifiedAttrs = Utils.createJSONFromTrackInfo(operationDocument, rPr.getRPrChange());
                	modifiedAttrs.put("attrs", changedAttrs);
                	changes.put("modified", modifiedAttrs);
                }
            }
            final int ctFlags = getCT();
            if(ctFlags!=Component.CT_NONE) {
            	if(changes==null) {
            		changes = new JSONObject(2);
            	}
            	final IndexedNode<Object> runInsNode = getRunInsNode();
            	if(runInsNode!=null) {
            		changes.put("inserted", Utils.createJSONFromTrackInfo(operationDocument, (RunIns)runInsNode.getData()));
            	}
            	final IndexedNode<Object> runDelNode = getRunDelNode();
            	if(runDelNode!=null) {
            		changes.put("removed", Utils.createJSONFromTrackInfo(operationDocument, (RunDel)runDelNode.getData()));
            	}
            }
        	if(changes!=null&&!changes.isEmpty()) {
        		attrs.put("changes", changes);
        	}
            final IndexedNode<Object> hyperlinkNode = getHyperlinkNode();
            if(hyperlinkNode!=null) {
                String url = Commons.getUrl(operationDocument.getContextPart(), ((P.Hyperlink)hyperlinkNode.getData()).getId());
                if (url!=null&&url.length()>0) {
                    if(jsonCharacterProperties==null)
                        jsonCharacterProperties = new JSONObject();
                    jsonCharacterProperties.put("url", url);
                }
            }
            if(jsonCharacterProperties!=null&&!jsonCharacterProperties.isEmpty()) {
                attrs.put("character", jsonCharacterProperties);
            }
			return attrs;
		}
        public IndexedNode<Object> getTextRunNode() {
            return textRunNode;
        }
        public R getTextRun() {
            return (R)textRunNode.getData();
        }
        public int getCT() {
        	return changeTrackFlags;
        }
        public void setCT(int _changeTrackFlags) {
        	changeTrackFlags = _changeTrackFlags;
        }
        public IndexedNode<Object> getHyperlinkNode() {
        	return hyperlinkNode;
        }
        public void removeHyperlink() {
            hyperlinkNode = null;
            removeContext(HyperlinkContext.class);
        }
        public void insertHyperlink(IndexedNode<Object> _hyperlinkNode) {
            final HashSet<Class<?>> parentContextList = new HashSet<Class<?>>(1);
            parentContextList.add(SdtParagraphContext.class);
            insertContext(new HyperlinkContext(null, _hyperlinkNode, getContextNode()), parentContextList);
            hyperlinkNode = _hyperlinkNode;
        }
        public IndexedNode<Object> getRunInsNode() {
        	return runInsNode;
        }
        public void removeRunInsNode() {
            runInsNode = null;
            removeContext(RunInsContext.class);
        }
        public void insertRunInsNode(IndexedNode<Object> _runInsNode) {
            final HashSet<Class<?>> parentContextList = new HashSet<Class<?>>(2);
            parentContextList.add(SdtParagraphContext.class);
            parentContextList.add(HyperlinkContext.class);
            insertContext(new RunInsContext(null, _runInsNode, getContextNode()), parentContextList);
            runInsNode = _runInsNode;
        }
        public IndexedNode<Object> getRunDelNode() {
        	return runDelNode;
        }
        public void removeRunDelNode() {
            runDelNode = null;
            removeContext(RunDelContext.class);
        }
        public void insertRunDelNode(IndexedNode<Object> _runDelNode) {
            final HashSet<Class<?>> parentContextList = new HashSet<Class<?>>(3);
            parentContextList.add(SdtParagraphContext.class);
            parentContextList.add(HyperlinkContext.class);
            parentContextList.add(RunInsContext.class);
            insertContext(new RunDelContext(null, _runDelNode, getContextNode()), parentContextList);
            runDelNode = _runDelNode;
        }
        protected static void splitAccessor(TextRun_Base component, boolean splitStart, SplitMode splitMode) {
            IndexedNode<Object> node = component.getNode();
            ComponentContext parentContext = component.getParentContext();
            while(!(parentContext instanceof ParagraphComponent||(splitMode==SplitMode.ATTRIBUTE&&parentContext instanceof SdtParagraphContext))&&parentContext!=null) {
                final IndexedNode<Object> parentNode = parentContext.getNode();
                final IndexedNodeList<Object> parentNodeContent = (IndexedNodeList<Object>)((ContentAccessor)parentNode.getData()).getContent();
                if((splitStart&&node.getIndex()!=0)||(!splitStart&&node.getIndex()!=parentNodeContent.size()-1)) {
                    IndexedNode<Object> newParentNode = null;
                    if(parentContext instanceof TextRunContext) {
                        final R parent = (R)parentNode.getData();
                        final R newParent = Context.getWmlObjectFactory().createR();
                        newParentNode = new IndexedNode<Object>(newParent);
                        final RPr rPr = ((R)parent).getRPr();
                        if(rPr!=null) {
                            final RPr rPrNew = XmlUtils.deepCopy(rPr);
                            rPrNew.setParent(newParent);
                            newParent.setRPr(rPrNew);
                        }
                        if(splitStart) {
                            component.textRunNode = newParentNode;
                        }
                    }
                    else if(parentContext instanceof HyperlinkContext) {
                        final P.Hyperlink parent = (P.Hyperlink)parentNode.getData();
                        final P.Hyperlink newParent = Context.getWmlObjectFactory().createPHyperlink();
                        newParentNode = new IndexedNode<Object>(newParent);
                        newParent.setAnchor(parent.getAnchor());
                        newParent.setDocLocation(parent.getDocLocation());
                        newParent.setHistory(parent.isHistory());
                        newParent.setId(parent.getId());
                        newParent.setTgtFrame(parent.getTgtFrame());
                        newParent.setTooltip(parent.getTooltip());
                        if(splitStart) {
                            component.hyperlinkNode = newParentNode;
                        }
                    }
                    else if(parentContext instanceof RunInsContext) {
                        final RunIns parent = (RunIns)parentNode.getData();
                        final RunIns newParent = Context.getWmlObjectFactory().createRunIns();
                        newParentNode = new IndexedNode<Object>(newParent);
                        newParent.setAuthor(parent.getAuthor());
                        final XMLGregorianCalendar date = parent.getDate();
                        if(date!=null) {
                            newParent.setDate((XMLGregorianCalendar) date.clone());
                        }
                        newParent.setId(((RunIns)parent).getId());
                        if(splitStart) {
                            component.runInsNode = newParentNode;
                        }
                    }
                    else if(parentContext instanceof RunDelContext) {
                        final RunDel parent = (RunDel)parentNode.getData();
                        final RunDel newParent = Context.getWmlObjectFactory().createRunDel();
                        newParentNode = new IndexedNode<Object>(newParent);
                        newParent.setAuthor(parent.getAuthor());
                        final XMLGregorianCalendar date = parent.getDate();
                        if(date!=null) {
                            newParent.setDate((XMLGregorianCalendar)date.clone());
                        }
                        newParent.setId(((RunDel)parent).getId());
                        if(splitStart) {
                            component.runDelNode = newParentNode;
                        }
                    }
                    else if(parentContext instanceof SdtParagraphContext&&splitMode==SplitMode.DELETE) {
                        final SdtRun parent = (SdtRun)parentNode.getData();
                        final SdtRun newParent = Context.getWmlObjectFactory().createSdtRun();
                        newParentNode = new IndexedNode<Object>(newParent);
                        newParent.setSdtEndPr(parent.getSdtEndPr());
                        newParent.setSdtPr(parent.getSdtPr());
                    }
                    if(newParentNode!=null) {
                        ((Child)newParentNode.getData()).setParent(parentNode.getData());
                        final IndexedNodeList<Object> rContent = (IndexedNodeList<Object>)((ContentAccessor)parentNode.getData()).getContent();
                        final IndexedNodeList<Object> rContentNew = (IndexedNodeList<Object>)((ContentAccessor)newParentNode.getData()).getContent();
                        final Object pParent = ((Child)parentNode.getData()).getParent();
                        final Object pNewParent = newParentNode.getData();
                        ((Child)pNewParent).setParent(pParent);
                        final IndexedNodeList<Object> pParentNode = (IndexedNodeList<Object>)((ContentAccessor)pParent).getContent();
                        pParentNode.addNode(parentNode.getIndex() + 1, newParentNode);
                        int index = node.getIndex();
                        if(!splitStart) {
                            index++;
                        }
                        final int count = rContent.size() - index;
                        if(count>0) {
                            IndexedNodeList.moveNodes(rContent, rContentNew, index, rContentNew.size(), count, pNewParent);
                        }
                        if(splitStart) {
                            parentContext.setNode(newParentNode);
                        }
                    }
                }
                node = parentContext.getNode();
                parentContext = parentContext.getParentContext();
            }
        }
        @Override
        public void postFixNonConformance(AtomicInteger id) {
            super.postFixNonConformance(id);

            if(runDelNode!=null) {
                final Object o = runDelNode.getData();
                if (o instanceof CTMarkup) {
                    ((CTMarkup)o).setId(BigInteger.valueOf(id.incrementAndGet()));
                }
            }
            if(runInsNode!=null) {
                final Object o = runInsNode.getData();
                if (o instanceof CTMarkup) {
                    ((CTMarkup)o).setId(BigInteger.valueOf(id.incrementAndGet()));
                }
            }
        }
    }

    public static class TextComponent extends TextRun_Base {

        public TextComponent(ComponentContext parentContext, IndexedNode<Object> _textNode, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(parentContext, _textNode, _paragraphNode, _componentNumber);
        }

        @Override
        public int getNextComponentNumber() {
            int textLength = ((IText)getNode().getData()).getValue().length();
            if (textLength==0) {
                System.out.println("Error: empty text string is no component... this results in problems with the component iterator");
                textLength++;       // increasing.. preventing an endless loop
            }
            return componentNumber + textLength;
        }
        @Override
        public void splitStart(int componentPosition, SplitMode splitMode) {
        	final int splitPosition = componentPosition-componentNumber;
        	if(splitPosition>0) {
        		splitText(splitPosition, true);
        		componentNumber+=splitPosition;
        	}
    		splitAccessor(this, true, splitMode);
        }
        @Override
        public void splitEnd(int componentPosition, SplitMode splitMode) {
        	if(getNextComponentNumber()>++componentPosition) {
        		splitText(componentPosition-componentNumber, false);
        	}
        	splitAccessor(this, false, splitMode);
        }
        public static void preserveSpace(IText t) {
            if(t!=null) {
                String s = t.getValue();
                if(s!=null&&s.length()>0) {
                    if(s.charAt(0)==' '||s.charAt(s.length()-1)==' ') {
                        t.setSpace("preserve");
                        return;
                    }
                }
                t.setSpace(null);
            }
        }
        private void splitText(int textSplitPosition, boolean splitStart) {
        	final IText text = (IText)getNode().getData();
            if(textSplitPosition>0&&textSplitPosition<text.getValue().length()) {
            	final IText newText = text.createText();
                newText.setParent(text.getParent());
                final IndexedNodeList<Object> runContent = getTextRun().getContent();
                final StringBuffer s = new StringBuffer(text.getValue());
                if(splitStart) {
                    runContent.add(getNode().getIndex(), newText);
                    newText.setValue(s.substring(0, textSplitPosition));
                    text.setValue(s.substring(textSplitPosition));
                }
                else {
                	runContent.add(getNode().getIndex()+1, newText);
                    text.setValue(s.substring(0, textSplitPosition));
                    newText.setValue(s.substring(textSplitPosition));
                }
                preserveSpace(newText);
                preserveSpace(text);
            }
        }
    }

    public static class TabComponent extends TextRun_Base {

        public TabComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(parentContext, _node, _paragraphNode, _componentNumber);
        }
    }

    public static class HardBreakComponent extends TextRun_Base {

        public HardBreakComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(parentContext, _node, _paragraphNode, _componentNumber);
        }
    }

    public static class FldSimpleComponent extends Component {

        public FldSimpleComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(parentContext, _node, _paragraphNode, _componentNumber);
        }
		@Override
		public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
			return null;
		}
		@Override
		public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs) {
			try {
		    	final JSONObject characterAttributes = attrs.optJSONObject("character");
		    	com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties
		    		(operationDocument, characterAttributes, getTextRun().getRPr());
			}
			catch(Exception e) {
				log.error("docx, FldSimpleComponent::setAttributes:", e);
			}
		}
		@Override
		public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
			throws JSONException {

            final R textRun = getTextRun();
            final RPr rPr = textRun.getRPr();
            if(rPr!=null) {
                final RStyle rStyle = rPr.getRStyle();
                if(rStyle!=null) {
                    attrs.put("styleId", rStyle.getVal());
                }
                final JSONObject jsonCharacterProperties = Character.createCharacterProperties(createOperationHelper.getThemeFonts(), rPr);
                if (jsonCharacterProperties!=null&&!jsonCharacterProperties.isEmpty()) {
                    attrs.put("character", jsonCharacterProperties);
                }
            }
			return attrs;
		}

		public R getTextRun() {
        	final CTSimpleField simpleField = (CTSimpleField)getNode().getData();
        	final IndexedNodeList<Object> content = simpleField.getContent();
        	for(int i=0; i<content.size();i++) {
        		Object o = getContentModel(content.getNode(i), getNode().getData());
        		if(o instanceof R)
        			return (R)o;
        	}
        	final R r = Context.getWmlObjectFactory().createR();
        	r.setParent(simpleField);
        	content.add(r);
        	return r;
        }

        public String getInstr() {
            return ((CTSimpleField)getNode().getData()).getInstr();
        }

        public String getRepresentation() {
            String representation = "";
            IndexedNodeList<Object> content = ((CTSimpleField)getNode().getData()).getContent();
            if(content!=null) {
                for(int i=0; i<content.size();i++) {
                    Object o = getContentModel(content.getNode(i), getNode().getData());
                    if(o instanceof R) {
                        IndexedNodeList<Object> run = ((R)o).getContent();
                        if(run!=null) {
                            for(int j=0; j<run.size();j++) {
                                Object ro = getContentModel(run.getNode(j), o);
                                if(ro instanceof IText) {
                                    String t = ((IText)ro).getValue();
                                    if(t!=null)
                                        representation += t;
                                }
                            }
                        }
                    }
                }
            }
            return representation;
        }
    }

    public enum ShapeType {
        UNDEFINED,
        GROUP,
        IMAGE,                  // undoable
        SHAPE,
        OLE,
        DIAGRAM,
        CHART,
        HORIZONTAL_LINE
    }

    public static interface IShapeType {

    	public ShapeType getType();
    }

    public static interface IShape extends IShapeType {

    	public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
    	    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException;

    	public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
   			throws JSONException, ParseException, FilterException;

    	// returns the next shape (the parent must be a group shape)
    	public IShape getNext();

    	// returns the first child shape (the parent must be a group shape)
    	public Object getChild();

    	// inserts a new shape (image, shape and or group into the parent group)
    	// returns either IShape or Component
        public Object insertChild(int number, Type type)
            throws JAXBException;

		public void postFixNonConformance(AtomicInteger id);
    }

    public static class ShapeComponent extends Component implements IShapeType {

    	private final IShape shape;

    	public ShapeComponent(IShape shape, IndexedNode<Object> _contextNode, int _componentNumber) {
			super(null, new IndexedNode<Object>(null), _contextNode, _componentNumber);
			this.shape = shape;
		}
    	@Override
    	public ShapeType getType() {
    		return shape.getType();
    	}
		@Override
		public Component getNextComponent() {
		    final IShape next = shape.getNext();
		    return next!=null?new ShapeComponent(next, getContextNode(), getComponentNumber()+1):null;
		}
		@Override
		public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
		    final Object child = shape.getChild();
		    if(child instanceof IShape) {
		        return new ShapeComponent((IShape)child, getContextNode(), 0);
		    }
		    else if(child instanceof Component) {
		        return (Component)child;
		    }
		    return null;
		}
		@Override
		public Component insertChildComponent(OperationDocument operationDocument, int number, JSONObject attrs, Type type)
		    throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

		    final Object child = shape.insertChild(number, type);
            Component childComponent = null;
		    if(child instanceof IShape) {
                childComponent = new ShapeComponent((IShape)child, getContextNode(), number);
            }
            else if(child instanceof Component) {
                childComponent = (Component)child;
            }
		    if(attrs!=null) {
		        childComponent.applyAttrsFromJSON(operationDocument, attrs);
		    }
		    return childComponent;
		}
		@Override
		public void postFixNonConformance(AtomicInteger id) {

			shape.postFixNonConformance(id);
		}
		@Override
		public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
		    throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

			shape.applyAttrsFromJSON(operationDocument, attrs);
		}
		@Override
		public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
			throws JSONException, ParseException, FilterException {

			return shape.createJSONAttrs(createOperationHelper, attrs);
		}
    }

    public abstract static class ShapeComponent_Root extends TextRun_Base implements IShapeType {

        protected ShapeType shapeType;
        protected Component shapeComponent;

        protected ShapeComponent_Root(ComponentContext parentContext, IndexedNode<Object> _rootShapeNode, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(parentContext, _rootShapeNode, _paragraphNode, _componentNumber);
            shapeType = ShapeType.UNDEFINED;
        }
		@Override
		public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {

		    return shapeComponent!=null?shapeComponent.getNextChildComponent(previousChildContext, previousChildComponent):null;
		}
		@Override
        public Component insertChildComponent(OperationDocument operationDocument, int number, JSONObject attrs, Type type)
            throws JAXBException, UnsupportedOperationException, InvalidFormatException, PartUnrecognisedException, JSONException {

		    return shapeComponent!=null?shapeComponent.insertChildComponent(operationDocument, number, attrs, type):null;
		}
		@Override
		public void postFixNonConformance(AtomicInteger id) {

		    if(shapeComponent!=null) {
		        shapeComponent.postFixNonConformance(id);
		    }
	    }
        @Override
        public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
            throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

            if(shapeComponent!=null) {
                shapeComponent.applyAttrsFromJSON(operationDocument, attrs);
            }
            super.applyAttrsFromJSON(operationDocument, attrs);
        }

        @Override
        public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
            throws ParseException, JSONException, FilterException {

            attrs = super.createJSONAttrs(createOperationHelper, attrs);
            if(shapeComponent!=null) {
                attrs = shapeComponent.createJSONAttrs(createOperationHelper, attrs);
            }
            return attrs;
        }
        @Override
        public ShapeType getType() {
        	return shapeType;
        }
    }

    public static class AlternateContentComponent extends ShapeComponent_Root {

    	public AlternateContentComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(parentContext, _node, _paragraphNode, _componentNumber);

            final List<Component> shapes = new ArrayList<Component>(2);
    		final AlternateContent alternateContent = (AlternateContent)getObject();
			final List<Choice> choices = alternateContent.getChoice();
			for (int i=0; i<choices.size(); i++) {
				final Choice choice = choices.get(i);
				final String requires = choice.getRequires();
				if(requires.equals("wps")||requires.equals("wpg")) {
					final List<Object> any = choice.getAny();
					final Object o = any.get(0);
					if(o instanceof Drawing) {
						shapes.add(new ShapeComponent(new DrawingML.DrawingML_root((Drawing)o), getContextNode(), getComponentNumber()));
					}
				}
				else if (requires.equals("wpc")) {
					// canvas ... what to do ? ...
				}
				else {
					// we don't support changes of this type, so we remove it...
					// changes will be applied to the fallback
					choices.remove(i--);
				}
			}
			final Fallback fallback = alternateContent.getFallback();
			if(fallback!=null) {
				final List<Object> any = fallback.getAny();
				if(any.size()==1) {
					final Object o = any.get(0);
					if(o instanceof ContentAccessor) {
		                final VMLShape vmlShape = new VMLShape(((ContentAccessor)o).getContent());
		        		shapes.add(new ShapeComponent(vmlShape.getVMLShapeImpl(), getContextNode(), getComponentNumber()));
					}
				}
			}
			if(!shapes.isEmpty()) {
			    shapeType = ((ShapeComponent)shapes.get(0)).getType();
			    if(shapes.size()==1) {
			        shapeComponent = shapes.get(0);
			    }
			    else {
	                shapeComponent = new MultiComponent(shapes);
			    }
			}
        }
    }

    // parent can either be a Pict or a CTObject

    public static class PictVMLComponent extends ShapeComponent_Root {

        public PictVMLComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(parentContext, _node, _paragraphNode, _componentNumber);
    		final List<Object> pictContent = ((ContentAccessor)getObject()).getContent();
    		if(!pictContent.isEmpty()) {
    			VMLShape vmlShape = new VMLShape(pictContent);
        		shapeComponent = new ShapeComponent(vmlShape.getVMLShapeImpl(), getContextNode(), getComponentNumber());
        		shapeType = vmlShape.getVMLShapeImpl().getType();
    		}
        }
    }

    public static class DrawingComponent extends ShapeComponent_Root {

        public DrawingComponent(ComponentContext parentContext, IndexedNode<Object> _node, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(parentContext, _node, _paragraphNode, _componentNumber);
            shapeComponent = new ShapeComponent(new DrawingML.DrawingML_root((Drawing)_node.getData()), getContextNode(), getComponentNumber());
            shapeType = ((ShapeComponent)shapeComponent).getType();
        }
    }
}
