/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.operations;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import javax.xml.bind.JAXBException;

import org.apache.commons.logging.Log;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.WordprocessingML.NumberingDefinitionsPart;
import org.docx4j.wml.BooleanDefaultTrue;
import org.docx4j.wml.CTLanguage;
import org.docx4j.wml.CTSettings;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.CTTabStop;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblStylePr;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.DocDefaults;
import org.docx4j.wml.DocDefaults.RPrDefault;
import org.docx4j.wml.JcEnumeration;
import org.docx4j.wml.Lvl;
import org.docx4j.wml.NumFmt;
import org.docx4j.wml.NumberFormat;
import org.docx4j.wml.Numbering;
import org.docx4j.wml.Numbering.AbstractNum;
import org.docx4j.wml.Numbering.Num;
import org.docx4j.wml.Numbering.Num.AbstractNumId;
import org.docx4j.wml.ObjectFactory;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase.Ind;
import org.docx4j.wml.RPr;
import org.docx4j.wml.STTblStyleOverrideType;
import org.docx4j.wml.Style;
import org.docx4j.wml.Styles;
import org.docx4j.wml.Tabs;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.TrPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.tools.Component;
import com.openexchange.office.ooxml.docx.tools.Component.ParagraphComponent;
import com.openexchange.office.ooxml.docx.tools.Component.SplitMode;
import com.openexchange.office.ooxml.docx.tools.Component.Type;
import com.openexchange.office.ooxml.docx.tools.Paragraph;
import com.openexchange.office.ooxml.docx.tools.Table;
import com.openexchange.office.ooxml.docx.tools.Utils;

public class ApplyOperationHelper extends com.openexchange.office.ooxml.operations.ApplyOperationHelper {

    protected final static Log log = LogFactory.getLog(ApplyOperationHelper.class);

    private final OperationDocument operationDocument;
    private final WordprocessingMLPackage wordMLPackage;
    private final ObjectFactory objectFactory;
    private final Map<String, String> conditionalTableAttrNames;

    public ApplyOperationHelper(OperationDocument _operationDocument) {
        super();

        operationDocument = _operationDocument;
        wordMLPackage = _operationDocument.getPackage();
        objectFactory = Context.getWmlObjectFactory();

        conditionalTableAttrNames = new HashMap<String, String>();
        conditionalTableAttrNames.put("wholeTable", "wholeTable");
        conditionalTableAttrNames.put("firstRow", "firstRow");
        conditionalTableAttrNames.put("lastRow", "lastRow");
        conditionalTableAttrNames.put("firstCol", "firstCol");
        conditionalTableAttrNames.put("lastCol", "lastCol");
        conditionalTableAttrNames.put("band1Vert", "band1Vert");
        conditionalTableAttrNames.put("band2Vert", "band2Vert");
        conditionalTableAttrNames.put("band1Hor", "band1Horz");
        conditionalTableAttrNames.put("band2Hor", "band2Horz");
        conditionalTableAttrNames.put("northEastCell", "neCell");
        conditionalTableAttrNames.put("northWestCell", "nwCell");
        conditionalTableAttrNames.put("southEastCell", "seCell");
        conditionalTableAttrNames.put("southWestCell", "swCell");
    }

    @Override
    public OperationDocument getOperationDocument() {
        return operationDocument;
    }

    /**
     *
     * After applying operations to the document, this method is called
     * once to fix the output so that it fulfills the OOXML specification.
     * (E.g: A Tc element must have at least one P. In our operation model
     * we create Tc and P elements independently, so there might exist
     * a Tc without P... this is fixed here)
     *
     * additionally we are also creating unique ids for each changeTrack
     *  ... wtf, why is this id required
     *
     * @author Sven Jacobi <sven.jacobi@open-xchange.com>
     */
    public void postFixNonConformance(Component component, AtomicInteger id) {
        while(component!=null) {
        	component.postFixNonConformance(id);
        	final Component child = component.getNextChildComponent(null, null);
        	if(child!=null) {
        		postFixNonConformance(child, id);
        	}
            component = component.getNextComponent();
        }
    }

    public void delete(JSONArray start, JSONArray end)
        throws JSONException, InvalidFormatException, JAXBException, PartUnrecognisedException {

        if(start==null||start.length()==0)
            return;

        int startComponent = start.getInt(start.length()-1);
        int endComponent = startComponent;
        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endComponent = end.getInt(end.length()-1);
        }
        Component component = Component.getComponent(getOperationDocument().getContextPart(), start, start.length());
		component.splitStart(startComponent, SplitMode.DELETE);
        component.delete((endComponent-startComponent)+1);
    }

    /**
     * moving components, only DrawingComponents are supported
     *
     * @param startPosition
     * @param endPosition
     *
     */

    public void move(JSONArray startPosition, JSONArray endPosition, JSONArray toPosition)
        throws JSONException {

        if(startPosition.length()!=endPosition.length())
            throw new JSONException("ooxml export: move operation, size of startPosition != size of endPosition");

        int startIndex = startPosition.getInt(startPosition.length()-1);
        int endIndex = endPosition.getInt(startPosition.length()-1);
        int toIndex = toPosition.getInt(toPosition.length()-1);

        if((endIndex - startIndex) < 0)
            throw new JSONException("ooxml export: move operation, component count < 1");

        Component sourceParent = null;
        Component toParent = null;

        // check if split is necessary (only needed if parent is a paragraph, if length == 1 then the parent is always root)
        sourceParent = Component.getComponent(getOperationDocument().getContextPart(), startPosition, startPosition.length()-1);
        if(sourceParent instanceof ParagraphComponent) {
        	Component c = sourceParent.getChildComponent(startIndex);
        	if(c!=null) {
        		c.splitStart(startIndex, SplitMode.ATTRIBUTE);
        	}
        	c = c.getComponent(endIndex+1);
        	if(c!=null) {
        		c.splitEnd(endIndex, SplitMode.ATTRIBUTE);
        	}
        }
        toParent = Component.getComponent(getOperationDocument().getContextPart(), toPosition, toPosition.length()-1);
        if(toParent instanceof ParagraphComponent) {
        	Component c = toParent.getChildComponent(toIndex);
        	if(c!=null) {
        		c.splitStart(toIndex, SplitMode.ATTRIBUTE);
        	}
        }

        Component startComponent = sourceParent.getChildComponent(startIndex);
        if(startComponent==null)
        	throw new JSONException("ooxml export: move operation, could not get component for start position");

        final int parentStartIndex = startComponent.getContextChildNode().getIndex();
        int parentEndIndex = parentStartIndex;

        while(startComponent!=null&&startComponent.getNextComponentNumber()<=endIndex) {
            parentEndIndex = startComponent.getContextChildNode().getIndex();
            startComponent = startComponent.getNextComponent();
        }

        final IndexedNodeList<Object> sourceContent = (IndexedNodeList<Object>)((ContentAccessor)sourceParent.getObject()).getContent();
        final IndexedNodeList<Object> destContent = (IndexedNodeList<Object>)((ContentAccessor)toParent.getObject()).getContent();

        final Component toComponent = toParent.getChildComponent(toIndex);
        final int destIndex = toComponent!=null?toComponent.getContextChildNode().getIndex():destContent.size();
    	IndexedNodeList.moveNodes(sourceContent, destContent, parentStartIndex, destIndex, (parentEndIndex - parentStartIndex)+1, toParent.getObject());
    }

    public void insertParagraph(JSONArray start, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1).insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, Component.Type.PARAGRAPH);
    }

    public void splitParagraph(JSONArray start)
        throws JSONException {

        ((Component.IParagraph)Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1)).splitParagraph(start.getInt(start.length()-1));
    }

    public void mergeParagraph(JSONArray start) {

    	((Component.IParagraph)Component.getComponent(getOperationDocument().getContextPart(), start, start.length())).mergeParagraph();
    }

    public void insertText(JSONArray start, String text, JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

    	if(text.length()>0) {
    		final Component component = Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1);
    		((Component.IParagraph)component).insertText(operationDocument, start.getInt(start.length()-1), text, attrs);
    	}
    }

    public void insertTab(JSONArray start, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1)
    	    .insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, Type.TAB);
    }

    public void insertHardBreak(JSONArray start, String type, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	Type textRunChild = Type.HARDBREAK_DEFAULT;
		if(type.equals("page")) {
			textRunChild = Type.HARDBREAK_PAGE;
		}
		else if(type.equals("column")) {
			textRunChild = Type.HARDBREAK_COLUMN;
		}
    	Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1)
			.insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, textRunChild);
    }

    public void insertDrawing(JSONArray start, String type, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	Type childComponentType = Type.AC_IMAGE;
    	if(type.equals("shape")) {
    		// we will create an AlternateContent for our shape (drawingML + VML)
    		childComponentType = Type.AC_SHAPE;
    	}
    	else if(type.equals("group")) {
            // we will create an AlternateContent for our group (drawingML + VML)
    	    childComponentType = Type.AC_GROUP;
    	}
    	else if(type.equals("image")) {
            // we will create an AlternateContent for our group (drawingML + VML)
    	    childComponentType = Type.AC_IMAGE;
    	}
    	Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1).insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, childComponentType);
    }

    public void insertTable(JSONArray start, JSONObject attrs)
        throws JSONException, JAXBException, InvalidFormatException, PartUnrecognisedException {

    	Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1).insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, Component.Type.TABLE);
    }

    public void splitTable(JSONArray start)
    	throws JSONException {

    	((Component.ITable)Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1)).splitTable(start.getInt(start.length()-1));
    }

    public void mergeTable(JSONArray start) {

    	((Component.ITable)Component.getComponent(getOperationDocument().getContextPart(), start, start.length())).mergeTable();
    }

    public void insertRows(JSONArray start, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
        throws JSONException, JAXBException, InvalidFormatException, PartUnrecognisedException {

    	((Component.ITable)Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1))
    		.insertRows(operationDocument, start.getInt(start.length()-1), count, insertDefaultCells, referenceRow, attrs);
    }

    public void insertCells(JSONArray start, int count, JSONObject attrs)
        throws JSONException, JAXBException, InvalidFormatException, PartUnrecognisedException {

    	((Component.IRow)Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1)).insertCells(operationDocument, start.getInt(start.length()-1), count, attrs);
    }

    public void insertColumn(JSONArray start, JSONArray tableGrid, int gridPosition, String insertMode)
        throws JSONException {

    	((Component.ITable)Component.getComponent(getOperationDocument().getContextPart(), start, start.length())).insertColumn(operationDocument, tableGrid, gridPosition, insertMode);
    }

    public void deleteColumns(JSONArray position, int gridStart, int gridEnd)
        throws JSONException {

    	((Component.ITable)Component.getComponent(getOperationDocument().getContextPart(), position, position.length())).deleteColumns(operationDocument, gridStart, gridEnd);
    }

    public void setAttributes(JSONObject attrs, JSONArray start, JSONArray end)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	if(attrs==null) {
    		return;
    	}
        int startIndex = start.getInt(start.length()-1);
        int endIndex = startIndex;

        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endIndex = end.getInt(end.length()-1);
        }
    	Component component = Component.getComponent(getOperationDocument().getContextPart(), start, start.length());
		component.splitStart(startIndex, SplitMode.ATTRIBUTE);
        while(component!=null&&component.getComponentNumber()<=endIndex) {
        	if(component.getNextComponentNumber()>=endIndex+1) {
        		component.splitEnd(endIndex, SplitMode.ATTRIBUTE);
        	}
        	component.applyAttrsFromJSON(operationDocument, attrs);
            component = component.getNextComponent();
        }
    }

    public void insertField(JSONArray start, String type, String representation, JSONObject attrs)
            throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	((Component.IParagraph)Component.getComponent(getOperationDocument().getContextPart(), start, start.length()-1))
    		.insertField(operationDocument, start.getInt(start.length()-1), type, representation, attrs);
    }

    public void insertStyleSheet(String type, String styleId, String styleName, JSONObject attrs, String parentId, Boolean hidden, Integer uipriority, Boolean isDefault)
        throws Exception {

        final Styles styles = getOperationDocument().getStyles(true);
        final List<Style> stylesList = styles.getStyle();

        // removing style if it already exists
        Iterator<Style> styleIter = stylesList.iterator();
        while(styleIter.hasNext()){
            Style aStyle = styleIter.next();
            if ((aStyle.getType().equals(type)) && (aStyle.getStyleId().equals(styleId))) {
                styleIter.remove();
                break;
            }
        };

        // always creating a new style...
        Style style = objectFactory.createStyle();
        stylesList.add(style);

        // style properties
        style.setParent(styles);
        style.setType(type);
        style.setStyleId(styleId);
        if (styleName != null) {
            Style.Name name = objectFactory.createStyleName();
            name.setParent(style);
            name.setVal(styleName);
            style.setName(name);
        }
        if (hidden != null) {
            BooleanDefaultTrue hiddenValue = objectFactory.createBooleanDefaultTrue();
            hiddenValue.setParent(style);
            hiddenValue.setVal(hidden.booleanValue());
            style.setHidden(hiddenValue);
        }
        if (isDefault != null) {
            style.setDefault(isDefault.booleanValue());
        }
        if (uipriority != null) {
            Style.UiPriority uiPrio = objectFactory.createStyleUiPriority();
            uiPrio.setParent(style);
            uiPrio.setVal(BigInteger.valueOf(uipriority.intValue()));
            style.setUiPriority(uiPrio);
        }
        if (parentId != null) {
            Style.BasedOn basedOn = objectFactory.createStyleBasedOn();
            basedOn.setParent(style);
            basedOn.setVal(parentId);
            style.setBasedOn(basedOn);
        }

        // paragraph properties
        JSONObject paragraphProps = attrs.optJSONObject("paragraph");
        if (paragraphProps != null) {
            String nextStyle = paragraphProps.optString("nextStyleId");
            if (nextStyle != null) {
               Style.Next next = objectFactory.createStyleNext();
               next.setParent(style);
               next.setVal(nextStyle);
               style.setNext(next);
            }
            PPr ppr = style.getPPr();
            if (ppr == null) {
                ppr = objectFactory.createPPr();
                ppr.setParent(style);
                style.setPPr(ppr);
            }
            Paragraph.applyParagraphProperties(operationDocument, paragraphProps, ppr);
        }

        // character properties
        JSONObject characterProps = attrs.optJSONObject("character");
        if (characterProps != null) {
            RPr rpr = style.getRPr();
            if (rpr == null) {
                rpr = objectFactory.createRPr();
                rpr.setParent(style);
                style.setRPr(rpr);
            }
            com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties(operationDocument, characterProps, rpr);
        }

        if(type.equals("table")) {
            Iterator<String> keys = attrs.keys();
            while(keys.hasNext()) {
                String key = keys.next();
                if(conditionalTableAttrNames.containsKey(key)) {
                    JSONObject conditionalTableProps = attrs.optJSONObject(key);
                    if (conditionalTableProps != null) {
                        String conditionalTableName = conditionalTableAttrNames.get(key);
                        List<CTTblStylePr> conditionalStyles = style.getTblStylePr();
                        CTTblStylePr conditionalStyle = objectFactory.createCTTblStylePr();
                        conditionalStyle.setType(STTblStyleOverrideType.fromValue(conditionalTableName));
                        Iterator<String> tablePropKeys = conditionalTableProps.keys();
                        while(tablePropKeys.hasNext()) {
                            String tablePropKey = tablePropKeys.next();
                            JSONObject tablePropAttrs = conditionalTableProps.optJSONObject(tablePropKey);
                            if(tablePropAttrs!=null) {
                                if(tablePropKey.equals("table")) {
                                    CTTblPrBase tblPrBase = conditionalStyle.getTblPr();
                                    if (tblPrBase==null) {
                                        tblPrBase = objectFactory.createCTTblPrBase();
                                        tblPrBase.setParent(conditionalStyle);
                                        conditionalStyle.setTblPr(tblPrBase);
                                    }
                                    Table.applyTableProperties(operationDocument, tablePropAttrs, null, null, tblPrBase);
                                } else if(tablePropKey.equals("row")) {
                                     TrPr trPr = conditionalStyle.getTrPr();
                                    if (trPr==null) {
                                        trPr = objectFactory.createTrPr();
                                        trPr.setParent(conditionalStyle);
                                        conditionalStyle.setTrPr(trPr);
                                    }
                                    Table.applyRowProperties(tablePropAttrs, trPr);
                                } else if(tablePropKey.equals("cell")) {
                                    TcPr tcPr = conditionalStyle.getTcPr();
                                   if (tcPr==null) {
                                       tcPr = objectFactory.createTcPr();
                                       tcPr.setParent(conditionalStyle);
                                       conditionalStyle.setTcPr(tcPr);
                                   }
                                   Table.applyCellProperties(operationDocument, tablePropAttrs, tcPr);
                                } else if(tablePropKey.equals("paragraph")) {
                                    PPr pPr = conditionalStyle.getPPr();
                                    if (pPr==null) {
                                        pPr = objectFactory.createPPr();
                                        pPr.setParent(conditionalStyle);
                                        conditionalStyle.setPPr(pPr);
                                    }
                                    Paragraph.applyParagraphProperties(operationDocument, tablePropAttrs, pPr);
                                } else if(tablePropKey.equals("character")) {
                                    RPr rPr = conditionalStyle.getRPr();
                                    if (rPr==null) {
                                        rPr = objectFactory.createRPr();
                                        rPr.setParent(conditionalStyle);
                                        conditionalStyle.setRPr(rPr);
                                    }
                                    com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties(operationDocument, tablePropAttrs, rPr);
                                }
                            }
                        }

                        // we will not create a whole_table style, because it does not seem to work,
                        // (even the TblLook does not mention the conditional WholeTable style) instead
                        // we will set the properties direct as tblPr
                        if(conditionalStyle.getType()==STTblStyleOverrideType.WHOLE_TABLE) {
                            style.setTblPr(conditionalStyle.getTblPr());
                            style.setTrPr(conditionalStyle.getTrPr());
                            style.setTcPr(conditionalStyle.getTcPr());
                            style.setPPr(conditionalStyle.getPPr());
                            style.setRPr(conditionalStyle.getRPr());

                            // if no shading is used within the TblPr, then we have to copy the shading from
                            // the cell into TblPr (LO does not take care about cell shading, so we have to do this workaround)
                            CTTblPrBase tblPr = style.getTblPr();
                            if(tblPr==null||tblPr.getShd()==null&&style.getTcPr()!=null) {
                            	final CTShd tableShade = style.getTcPr().getShd();
                            	if(tableShade!=null) {
	                            	if(tblPr==null) {
	                                	tblPr = objectFactory.createCTTblPrBase();
	                                	style.setTblPr(tblPr);
	                                	tblPr.setParent(style);
	                                }
	                            	tblPr.setShd(XmlUtils.deepCopy(tableShade));
	                            	tblPr.getShd().setParent(tblPr);
                            	}
                            }
                        }
                        else {
                            conditionalStyles.add(conditionalStyle);
                        }
                    }
                }
            }
        }
    }

    @SuppressWarnings("unused")
    public void deleteStylesheet(String type, String styleName) {
        //
    }
    public void insertListStyle(String listStyleId, JSONObject listDefinition)
        throws InvalidFormatException, JAXBException, JSONException {

        NumberingDefinitionsPart numDefPart = wordMLPackage.getMainDocumentPart().getNumberingDefinitionsPart();
        if( numDefPart == null ) {
            numDefPart = new NumberingDefinitionsPart();
            numDefPart.unmarshalDefaultNumbering();
            Numbering numbering = numDefPart.getJaxbElement();
            // remove defaults
            numbering.getAbstractNum().clear();
            numbering.getNum().clear();
            wordMLPackage.getMainDocumentPart().addTargetPart(numDefPart);
        }
        Numbering numbering = numDefPart.getJaxbElement();
        List<AbstractNum> abstractNumList = numbering.getAbstractNum();

        //find a 'free' abstractNumId
        Iterator<AbstractNum> abstractIter = abstractNumList.iterator();
        HashSet<Long> existingAbstractIds = new HashSet<Long>();
        while( abstractIter.hasNext() ){
            AbstractNum abstractNum = abstractIter.next();
            long abstractNumId = abstractNum.getAbstractNumId().longValue();
            existingAbstractIds.add(new Long(abstractNumId));
        }
        long newAbstractId = 0;
        while( existingAbstractIds.contains( new Long( newAbstractId ) ) )
            ++newAbstractId;

        AbstractNum newAbstractNum = objectFactory.createNumberingAbstractNum();
        newAbstractNum.setParent(numDefPart);
        BigInteger abstractId = BigInteger.valueOf(newAbstractId);
        newAbstractNum.setAbstractNumId(abstractId);
        List<Lvl> lvlList = newAbstractNum.getLvl();
        for( long lvlIndex = 0; lvlIndex < 9; ++lvlIndex){
            JSONObject levelObject = listDefinition.getJSONObject("listLevel" + lvlIndex);
            Lvl lvl = objectFactory.createLvl();
            lvl.setParent(newAbstractNum);
            lvl.setIlvl(BigInteger.valueOf(lvlIndex));
            if( levelObject.has("numberFormat")) {
                NumFmt numFmt = new NumFmt();
                numFmt.setVal( NumberFormat.fromValue( levelObject.getString("numberFormat") ));
                lvl.setNumFmt(numFmt);
            }
            if( levelObject.has("listStartValue")) {
                Lvl.Start lvlStart = objectFactory.createLvlStart();
                lvlStart.setVal(BigInteger.valueOf(levelObject.getLong("listStartValue")));
                lvl.setStart(lvlStart);
            }

            if( levelObject.has("levelText")) {
                if( lvl.getLvlText() == null )
                    lvl.setLvlText(objectFactory.createLvlLvlText());
                lvl.getLvlText().setVal(levelObject.getString("levelText"));
            }
            if( levelObject.has("textAlign") ){
                lvl.setLvlJc( objectFactory.createJc() );
                lvl.getLvlJc().setVal(JcEnumeration.fromValue(levelObject.getString("textAlign")));
            }
            RPr runProperties = objectFactory.createRPr();
            lvl.setRPr(runProperties);
            if( levelObject.has("fontName") ){
                runProperties.setRFonts(objectFactory.createRFonts());
                String font = levelObject.getString("fontName");
                runProperties.getRFonts().setAscii(font);
                runProperties.getRFonts().setHAnsi(font);
            }
            if( levelObject.has("paraStyle") ){
                Lvl.PStyle pstyle = objectFactory.createLvlPStyle();
                pstyle.setVal(levelObject.getString("paraStyle"));
            }

            PPr paraProperties = objectFactory.createPPr();
            lvl.setPPr(paraProperties);
            Ind ind = objectFactory.createPPrBaseInd();
            if( levelObject.has("indentLeft")) {
                ind.setLeft(Utils.map100THMMToTwip( levelObject.getInt("indentLeft")) );
            }
            if( levelObject.has("indentFirstLine")) {
                int firstline = levelObject.getInt("indentFirstLine");
                if( firstline > 0){
                    ind.setFirstLine( Utils.map100THMMToTwip( firstline ));
                    ind.setHanging( null );
                } else if( firstline < 0) {
                    ind.setFirstLine( null );
                    ind.setHanging( Utils.map100THMMToTwip( -firstline));
                }
            }
            paraProperties.setInd( ind );
            if( levelObject.has("tabStopPosition")){
                Tabs tabs = objectFactory.createTabs();
                CTTabStop tabStop = objectFactory.createCTTabStop();
                tabStop.setPos( Utils.map100THMMToTwip( levelObject.getInt("tabStopPosition") ) );
                tabs.getTab().add(tabStop);
                paraProperties.setTabs(tabs);
            }
            lvlList.add(lvl);
        }
        abstractNumList.add(newAbstractNum);
//        newAbstractNum.setName(value);
//        newAbstractNum.setMultiLevelType(value);

        List<Num> numList = numbering.getNum();
        Num num = objectFactory.createNumberingNum();

        num.setParent(numDefPart);
        AbstractNumId abstractNumId = new AbstractNumId();
        abstractNumId.setVal(abstractId);
        num.setAbstractNumId( abstractNumId );
        BigInteger listId;
        try {
            if(!Character.isDigit(listStyleId.charAt(0))){
                listId = BigInteger.valueOf(Integer.parseInt(listStyleId.substring(1)));
            } else {
                listId = BigInteger.valueOf(Integer.parseInt(listStyleId));
            }
            num.setNumId(listId);

            // removing list style if it already exists
            Iterator<Num> numIter = numList.iterator();
            while(numIter.hasNext()){
                Num aNum = numIter.next();
                if (aNum.getNumId().equals(listId)) {
                    numIter.remove();
                    break;
                }
            };
            numList.add(num);
        }
        catch (NumberFormatException e) {
            log.info("docx export insertListStyle: invalid listStyleId found!");
        }
    }

    public void setDocumentLanguageIfNotSet(String langStr)
        throws Exception {

        final Styles styles = getOperationDocument().getStyles(true);
        DocDefaults docDefaults = styles.getDocDefaults();
        if (docDefaults == null)
            docDefaults = objectFactory.createDocDefaults();
        if(docDefaults!=null) {
            RPrDefault rDefault = docDefaults.getRPrDefault();
            if (rDefault == null)
                rDefault = objectFactory.createDocDefaultsRPrDefault();
            if(rDefault!=null) {
                RPr rPrDefault = rDefault.getRPr();
                if (rPrDefault == null)
                    rPrDefault = objectFactory.createRPr();
                if(rPrDefault!=null) {
                    CTLanguage ctLanguage = rPrDefault.getLang();
                    if (ctLanguage == null)
                        ctLanguage = objectFactory.createCTLanguage();

                    if (ctLanguage != null) {
                        String lang = ctLanguage.getVal();
                        if (lang == null || lang.length() == 0) {
                            if (langStr != null && langStr.length() > 0)
                                ctLanguage.setVal(langStr);
                            else
                                ctLanguage.setVal("en-US"); // default according to MS
                        }
                        lang = ctLanguage.getEastAsia();
                        if (lang == null || lang.length() == 0) {
                            ctLanguage.setEastAsia("en-US"); // default according to MS
                        }
                        lang = ctLanguage.getBidi();
                        if (lang == null || lang.length() == 0) {
                            ctLanguage.setBidi("ar-SA"); // default according to MS
                        }
                    }
                }
            }
        }
    }

    public void setDocumentAttributes(JSONObject attrs)
    	throws Exception {

    	final JSONObject documentAttrs = attrs.optJSONObject("document");
    	if(documentAttrs!=null) {
			final Object changeTracking = documentAttrs.opt("changeTracking");
			if(changeTracking instanceof Boolean) {
    			final CTSettings settings = getOperationDocument().getSettings(true);
    			if((Boolean)changeTracking) {
	    			final BooleanDefaultTrue trackRevision = Context.getWmlObjectFactory().createBooleanDefaultTrue();
	    			trackRevision.setVal(true);
	    			settings.setTrackRevisions(trackRevision);
    			}
    			else {
    				settings.setTrackRevisions(null);
    			}
    		}
    	}
    }
}

