/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.openexchange.office.odf;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.json.JSONObject;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.pkg.OdfPackage;
import org.xml.sax.SAXException;

import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

/**
 * The access point to the ODF Toolkit, obfuscating all OX Office dependencies
 * from the Toolkit.
 *
 */
public class OdfOperationDoc {

    private OdfOperationDocument mDocument;
    private static final Logger LOG = Logger.getLogger(OdfOperationDoc.class.getName());
    private ServiceLookup mServiceLookup;
    private static final String STATUS_ID_FILE = "debug/statusid.txt";

    /**
     * Creates an empty ODF text document.
     */
    public OdfOperationDoc() {
        try {
            mDocument = OdfOperationDocument.newTextOperationDocument();
        } catch (Exception ex) {
            Logger.getLogger(OdfOperationDoc.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    /**
     * Creates an OdtOperationDocument from the OpenDocument provided by a
     * resource Stream.
     * <p>
     * Since an InputStream does not provide the arbitrary (non sequential) read
     * access needed by OdtOperationDocument, the InputStream is cached. This
     * usually takes more time compared to the other createInternalDocument
     * methods. An advantage of caching is that there are no problems
     * overwriting an input file.
     * </p>
     *
     * @param inputStream - the InputStream of the ODF text document.
     * @return the text document created from the given InputStream
     */
    public OdfOperationDoc(InputStream aInputDocumentStm) {
        try {
            mDocument = new OdfOperationDocument(aInputDocumentStm);
        } catch (Exception ex) {
            Logger.getLogger(OdfOperationDoc.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    private static final String MODULE = "//module/";
    private static final String SPREADSHEET = "//spreadsheet/";
    private static final String DEBUG_OPERATIONS = "debugoperations";
    private static final String MAX_TABLE_COLUMNS = "maxTableColumns";
    private static final String MAX_TABLE_ROWS = "maxTableRows";
    private static final String MAX_TABLE_CELLS = "maxTableCells";
    private static final String MAX_SHEETS = "maxSheets";

    public OdfOperationDoc(Session session, ServiceLookup _services, InputStream inputDocumentStream, ResourceManager resourceManager, Map<String, Object> configuration) throws Exception {
        mServiceLookup = _services;
        boolean saveDebugOperations = Boolean.parseBoolean(ConfigurationHelper.getOfficeConfigurationValue(_services, session, MODULE + DEBUG_OPERATIONS, "false"));
        configuration.put(DEBUG_OPERATIONS, saveDebugOperations);

        int maxTableColumns = ConfigurationHelper.getIntegerOfficeConfigurationValue(_services, session, MODULE + MAX_TABLE_COLUMNS, 15);
        configuration.put(MAX_TABLE_COLUMNS, maxTableColumns);
        int maxTableRows = ConfigurationHelper.getIntegerOfficeConfigurationValue(_services, session, MODULE + MAX_TABLE_ROWS, 1500);
        configuration.put(MAX_TABLE_ROWS, maxTableRows);
        int maxTableCells = ConfigurationHelper.getIntegerOfficeConfigurationValue(_services, session, MODULE + MAX_TABLE_CELLS, 1500);
        configuration.put(MAX_TABLE_CELLS, maxTableCells);
        int maxSheets = ConfigurationHelper.getIntegerOfficeConfigurationValue(_services, session, SPREADSHEET + MAX_SHEETS, 200);
        configuration.put(MAX_SHEETS, maxSheets);
        mDocument = new OdfOperationDocument(inputDocumentStream, (null != resourceManager) ? resourceManager.getByteArrayTable() : null, configuration);
    }

    /**
     * Receives the (known) operations of the ODF text document
     *
     * @return the operations as JSON
     */
    // ToDo OX - JSONObject is to be considered..
    public JSONObject getOperations() throws SAXException, FilterException {
        JSONObject ops = mDocument.getOperations();
        if (ops != null) {
            LOG.log(Level.FINER, "\n\n*** ALL OPERATIONS:\n{0}", mDocument.getOperations().toString());
        } else {
            LOG.finer("\n\n*** ALL OPERATIONS:\n" + null);
        }
        return ops;
    }

    /**
     * Applies the (known) operations to upon the latest state of the ODF text
     * document
     *
     * @param operationString ODF operations as String
     * @return the number of operations being accepted
     * @throws FilterException
     */
    public int applyOperations(String operationString) throws FilterException {
        LOG.log(Level.FINER, "\n*** EDIT OPERATIONS:\n{0}", operationString);
        int operationsCount = 0;
        try {
            operationsCount = mDocument.applyOperations(operationString);
        } catch (Exception ex) {
            LOG.fine("Error simluated");
            if (ex.getMessage().equals("ERROR_SIMULATED")) {
                throw new FilterException("Simulated error triggered by operation: createError", ErrorCode.UNSUPPORTED_OPERATION_USED);
            }
        }
        return operationsCount;
    }

    /**
     * Applies the (known) operations to upon the latest state of the ODF text
     * document
     *
     * @param operations ODF operations as JSONArray within an JSONObject with
     * "operations" key.
     * @return the number of operations being accepted
     * @throws FilterException
     */
    public int applyOperations(JSONObject operations) throws FilterException {
        LOG.log(Level.FINER, "\n*** EDIT OPERATIONS:\n{0}", operations.toString());
        int operationsCount = 0;
        try {
            operationsCount = mDocument.applyOperations(operations);
        } catch (Exception ex) {
            LOG.fine("Error simluated");
            if (ex.getMessage().equals("ERROR_SIMULATED")) {
                throw new FilterException("Simulated error triggered by operation: createError", ErrorCode.UNSUPPORTED_OPERATION_USED);
            }
        }
        return operationsCount;
    }

    /**
     * writes the status ID - which reflects the latest changes of the document,
     * used as cashing ID of web representation to enable performance
     */
    void writeStatusID(String statusId) {
        OdfPackage pkg = mDocument.getPackage();
        LOG.finest("Overwriting ODT document status ID " + STATUS_ID_FILE);
        pkg.insert(statusId.getBytes(), STATUS_ID_FILE, "text/plain");
    }

    /**
     * Read the status ID - which reflects the latest changes of the document,
     * used as cashing ID of web representation to enable performance
     */
    String readStatusID() {
        OdfPackage pkg = mDocument.getPackage();
        String statusId = null;
        if (pkg != null && pkg.contains(STATUS_ID_FILE)) {
            // ..from the ODF ZIP
            byte[] revisionByteArray = pkg.getBytes(STATUS_ID_FILE);
            if (revisionByteArray != null && revisionByteArray.length != 0) {
                BufferedReader reader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(revisionByteArray)));
                // read the first line of the file, only containing one number
                try {
                    statusId = reader.readLine();
                } catch (IOException e) {
                    LOG.log(Level.FINE, "No existing status ID file found!");
                    Logger.getLogger(OdfOperationDocument.class.getName()).log(Level.FINEST, null, e);
                }
                LOG.log(Level.FINE, "Found an existing status ID:{0}", statusId);
            }
        }
        return statusId;
    }

    public long getContentSize() {
        return mDocument.getContentSize();
    }

    /**
     * Returns the TextOperationDocument encapsulating the DOM view
     *
     * @return ODF text document
     */
    public OdfDocument getDocument() {
        return mDocument.getDocument();
    }

    /**
     * Close the OdfPackage and release all temporary created data. After
     * execution of this method, this class is no longer usable. Do this as the
     * last action to free resources. Closing an already closed document has no
     * effect.
     */
    public void close() {
        mDocument.close();
    }
}
