/*
 * Copyright 2014 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.util.logging.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import static org.odftoolkit.odfdom.component.Table.MAX_ROW_NUMBER;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableCellPropertiesElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElementBase;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.dom.style.props.OdfStylePropertiesSet;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 *
 * @author svante.schubert AT gmail DOT com
 */
final class TableChanges implements TableChange {

    private static final Logger LOG = Logger.getLogger(TableChanges.class.getName());

    /**
     * Only column or rows can be added, not both as the multiple inheritance is
     * not resolved, yet.
     *
     */
    private TableChanges(TableTraversal tableTraversal, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo, JSONArray dataRange) {
        mColumnStartNo = columnStartNo;
        mColumnEndNo = columnEndNo;
        mRowStartNo = rowStartNo;
        mRowEndNo = rowEndNo;
        mNewRowCount = mRowEndNo - mRowStartNo + 1;
        mNewColumnCount = mColumnEndNo - mColumnStartNo + 1;
        // setCellContent property only
        mRangeData = dataRange;
        mTableTraversal = tableTraversal;

        if (mNewRowCount > 0 || mRowEndNo == -1) {
            hasRowEffect = Boolean.TRUE;
            hasCellEffect = Boolean.TRUE;
        } else {
            hasRowEffect = Boolean.FALSE;
        }
        if (mNewColumnCount > 0 || (mColumnEndNo == -1 && mColumnStartNo == 0 && mRangeData != null)) {
            hasColumnEffect = Boolean.TRUE;
            hasCellEffect = Boolean.TRUE;
        } else {
            hasColumnEffect = Boolean.FALSE;
        }
    }

    /**
     * Only column or rows can be added, not both as the multiple inheritance is
     * not resolved, yet.
     *
     */
    private TableChanges(TableTraversal tableTraversal, JSONArray dataRange, int columnStartNo, int rowStartNo) {
        mTableTraversal = tableTraversal;
        mRangeData = dataRange;
        mColumnStartNo = columnStartNo;
        mRowStartNo = rowStartNo;

        hasRowEffect = Boolean.FALSE;
        hasCellEffect = Boolean.TRUE;
        hasColumnEffect = Boolean.FALSE;

        // in case there is a range of data with possibly multiple rows of different length
        if (mRangeData != null) {
            nextDataRangeRow();
        }
    }

    /**
     * Only column or rows can be added, not both as the multiple inheritance is
     * not resolved, yet.
     *
     */
    private TableChanges(TableTraversal tableTraversal, Object cellValue, JSONObject cellFormat, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo) {
        mTableTraversal = tableTraversal;
        mRangeValue = cellValue;
        mFormatProps = cellFormat;
        mColumnStartNo = columnStartNo;
        mColumnEndNo = columnEndNo;
        mRowStartNo = rowStartNo;
        mRowEndNo = rowEndNo;

        hasRowEffect = Boolean.FALSE;
        hasCellEffect = Boolean.TRUE;
        hasColumnEffect = Boolean.FALSE;
    }

    /**
     * @param formatProps the format/style properties to adjust the range with.
     */
    private TableChanges(TableTraversal tableTraversal, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo) {
        init(tableTraversal, null, columnStartNo, columnEndNo, rowStartNo, rowEndNo, true);
        determineRange(null);
    }

    /**
     * @param formatProps the format/style properties to adjust the range with.
     */
    private TableChanges(TableTraversal tableTraversal, JSONObject formatProps, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo) {
        init(tableTraversal, formatProps, columnStartNo, columnEndNo, rowStartNo, rowEndNo, false);
        determineRange(formatProps);
    }

    private void init(TableTraversal tableTraversal, JSONObject formatProps, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo, boolean isColumnDelete) {
        mColumnStartNo = columnStartNo;
        mColumnEndNo = columnEndNo;
        mRowStartNo = rowStartNo;
        mRowEndNo = rowEndNo;
        mTableTraversal = tableTraversal;
        this.mFormatProps = formatProps;
        mIsColumnDeletion = isColumnDelete;
    }

    private OdfStylableElement mPrecedingColumnElement;
    private OdfStylableElement mPrecedingRowElement;

    // format properties only
    private JSONObject mFormatProps;

    // setCellContent properties only
    private JSONArray mRangeData;
    private int mDataRangeRowPos = -1;
    private int mDataRangeColumnPos = -1;
    private JSONArray mRangeRowData = null;
    private TableTraversal mTableTraversal = null;

    private Object mRangeValue;

    private OdfFileDom mXMLDoc;
    private Boolean hasColumnEffect;
    private Boolean hasRowEffect;
    private Boolean hasCellEffect;

    int mColumnStartNo;
    int mColumnEndNo;
    int mRowStartNo;
    int mRowEndNo;
    private int mNewRowCount;
    private int mNewColumnCount;
    private OdfStylableElement mNewContainerElement;
    private TableTableCellElementBase mPriorCellElement;
    private boolean mIsColumnDeletion;
//    private StyleTableCellPropertiesElement mPrecedingDefaultCellProperties;
//    private StyleTableCellPropertiesElement mNextDefaultCellProperties;

    private int mPrecedingCellPos = -1;
    // the position within a repeating cell. Required to know which position have to be split, when a border has to be adapted (removed)
    private int mCellPositionWithinReptition = 1;

    static TableChanges newFormatation(TableTraversal tableTraversal, JSONObject formatProps, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo) {
        return new TableChanges(tableTraversal, formatProps, columnStartNo, columnEndNo, rowStartNo, rowEndNo);
    }

    static TableChanges newColumnDeletion(TableTraversal tableTraversal, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo) {
        return new TableChanges(tableTraversal, columnStartNo, columnEndNo, rowStartNo, rowEndNo);
    }

    static TableChanges newCellsChange(TableTraversal tableTraversal, Object cellValue, JSONObject cellFormat, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo) {
        return new TableChanges(tableTraversal, cellValue, cellFormat, columnStartNo, columnEndNo, rowStartNo, rowEndNo);
    }

    static TableChanges newCellsChange(TableTraversal tableTraversal, JSONArray dataRange, int columnStartNo, int rowStartNo) {
        return new TableChanges(tableTraversal, dataRange, columnStartNo, rowStartNo);
    }

    static TableChanges newCellChange(TableTableElement tableElement, int columnStartNo, int columnEndNo, int rowStartNo, int rowEndNo, JSONArray dataRange) {
        return new TableChanges(((Table) tableElement.getComponent()).getTableTraversal(), columnStartNo, columnEndNo, rowStartNo, rowEndNo, dataRange);
    }

    /**
     * Starts the change on the last position of the given table traversal
     *
     * @param tableTraversal finds the table range where the change has to be
     * applied
     */
    public void start() {
        // simply start the taversal of the table
        mTableTraversal.findRange(mRowStartNo, mRowEndNo, mColumnStartNo, mColumnEndNo, this);
    }

    protected void determineRange(JSONObject formatProps) {
        if (formatProps != null && (formatProps.hasAndNotNull("column") || formatProps.hasAndNotNull("cell")) || mIsColumnDeletion) {
            hasColumnEffect = Boolean.TRUE;
        } else {
            hasColumnEffect = Boolean.FALSE;
        }
        if (formatProps != null && formatProps.hasAndNotNull("row")) {
            hasRowEffect = Boolean.TRUE;
        } else {
            hasRowEffect = Boolean.FALSE;
        }
        // for column & row format:
        // although column format will use default cell styles to apply "text" and "cell" style to a column, but still need empty cells to have those default styles express
        if (formatProps != null && (formatProps.hasAndNotNull("cell") || formatProps.hasAndNotNull("character") || formatProps.hasAndNotNull("paragraph")) || mIsColumnDeletion) {
            hasCellEffect = Boolean.TRUE;
        } else {
            hasCellEffect = Boolean.FALSE;
        }
    }

    /**
     * The element within the range provided by the TableTraversal is being
     * executed
     */
    public Object execute(OdfStylableElement tableStyleableElement, OdfElement nextElement, int rowStartNo, int rowEndNo, int currentRowNo, int columnStartNo, int columnEndNo, int currentColumnNo, TableChange change, Object... args) {
        // doing an addition
        if (tableStyleableElement != null) {
            if (mRangeData != null || mRangeValue != null || mFormatProps != null) {
                // *** CELL ***
                if (tableStyleableElement instanceof TableTableCellElement && hasCellEffect) {
                    modifyCells((TableTableCellElement) tableStyleableElement);
                } else if (tableStyleableElement instanceof TableTableRowElement && hasRowEffect
                    || tableStyleableElement instanceof TableTableColumnElement && hasColumnEffect) {
                    // if the complete column or row is selected
                    if (tableStyleableElement instanceof TableTableColumnElement && rowEndNo == (Table.MAX_ROW_NUMBER - 1) && rowStartNo == 0
                        || tableStyleableElement instanceof TableTableRowElement && columnEndNo == (Table.MAX_COLUMN_NUMBER_CALC - 1) && columnStartNo == 0
                        || tableStyleableElement instanceof TableTableRowElement && columnEndNo == (Table.MAX_COLUMN_NUMBER_EXCEL - 1) && columnStartNo == 0) {
                        JsonOperationConsumer.setAttributes(null, tableStyleableElement, null, null, mFormatProps, Boolean.TRUE);
                    } else {
                        JsonOperationConsumer.setAttributes(null, tableStyleableElement, null, null, mFormatProps, Boolean.FALSE);
                    }
                }

            } else if (mIsColumnDeletion && !(tableStyleableElement instanceof TableTableRowElement)) {
                OdfElement parent = (OdfElement) tableStyleableElement.getParentNode();
                parent.removeChild(tableStyleableElement);
            } else if (mNewColumnCount > 0 || mNewRowCount > 0) {
                if (tableStyleableElement instanceof TableTableRowElement) {
                    addEmptyRange(tableStyleableElement, currentRowNo);
                } else {
                    addEmptyRange(tableStyleableElement, currentColumnNo);
                }

//                // doing a formatation
//            } else if (tableStyleableElement != null && mFormatProps != null) {
//                if (tableStyleableElement instanceof TableTableCellElement) {
//                    formatCell((TableTableCellElement) tableStyleableElement, mFormatProps);
//                }
//                if (tableStyleableElement instanceof TableTableRowElement) {
//                    formatRow((TableTableRowElement) tableStyleableElement, mFormatProps);
//                }
//                if (tableStyleableElement instanceof TableTableColumnElement) {
//                    formatColumn((TableTableColumnElement) tableStyleableElement, mFormatProps);
//                }
            }
        }
        return Boolean.TRUE;
    }

    private static final String NUMBER_ROWS_REPEATED = "number-rows-repeated";
    private static final String TABLE_NUMBER_ROWS_REPEATED = "table:number-rows-repeated";

    private static final String NUMBER_COLUMNS_REPEATED = "number-columns-repeated";
    private static final String TABLE_NUMBER_COLUMNS_REPEATED = "table:number-columns-repeated";

    /**
     * This method adds (or removes) content and/or format to (or from) a cell.
     * No column/row attributes can be changed when changing the cells.
     */
    private void modifyCells(TableTableCellElement mostLeftCell) {
        if (mRangeRowData != null || mRangeValue != null || mFormatProps != null) {
            // if there are changes for this position
            Cell cellComponent = (Cell) mostLeftCell.getComponent();
            Component rootComponent = cellComponent.getRootComponent();

            // if only a single cell should be modified
            if (mRangeRowData == null) {
                cellComponent.addCellStyleAndContent(rootComponent, mRangeValue, mFormatProps);
            } else {
                // if a range should be modified, starting from the upper left corner of the range
                Integer rowsRepeated = 1;
                TableTableRowElement row = (TableTableRowElement) mostLeftCell.getParentNode();
                TableTableCellElementBase currentCell = mostLeftCell;
                TableTableRowElement newRow = null;
                do {
                    rowsRepeated = row.getTableNumberRowsRepeatedAttribute();

                    // before we add content to a repeated table, we have to split one row off
                    if (rowsRepeated != null && rowsRepeated > 1) {
                        row.removeAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-repeated");

                        // add the new row behind the old
                        newRow = (TableTableRowElement) row.cloneNode(true);
                        Node nextSibling = row.getNextSibling();
                        row.getParentNode().insertBefore(newRow, nextSibling);

                        // keep the cellsRepeated minus the one split off now
                        if (rowsRepeated > 2) {
                            newRow.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:number-rows-repeated", Integer.toString(rowsRepeated - 1));
                        }
                    } else {
                        newRow = null;
                    }
                    int cellsRepeated = 1;
                    TableTableCellElementBase nextRepetitonCellPart = null;
                    JSONObject newCellFormat;
                    Object newCellValue;
                    if (currentCell == null) {
                        currentCell = getCell(row, mColumnStartNo);
                    }
                    do {
                        JSONObject newCellData = mRangeRowData.optJSONObject(mDataRangeColumnPos);
                        cellsRepeated = currentCell.getRepetition();
                        // if there is a cellsRepeated, take only the first cell (
                        if (cellsRepeated > 1) {
                            nextRepetitonCellPart = (TableTableCellElementBase) currentCell.split(1);
                        }else{
                            nextRepetitonCellPart = null;
                        }
                        if (newCellData != null) {
                            newCellFormat = newCellData.optJSONObject("attrs");
                            newCellValue = newCellData.opt("value");
                            cellComponent = (Cell) currentCell.getComponent();
                            cellComponent.addCellStyleAndContent(rootComponent, newCellValue, newCellFormat);
                        }
                        // if there is no repetition part it is null
                        currentCell = (TableTableCellElementBase) nextRepetitonCellPart;
                        mDataRangeColumnPos++;
                    } while (currentCell != null && mDataRangeColumnPos <= mRangeRowData.length());
                    if (mDataRangeColumnPos >= mRangeRowData.length()) {
                        nextDataRangeRow();
                        // if there are no repeated rows being split continue to traverse to get possibly new row..
                        if (newRow == null) {
                            break;
                        }
                    }
                    row = newRow;
                    currentCell = null;
                } while (mRangeRowData != null && row != null);
            }
        }
    }

    /** @return # the cell position at the given position even with repetition */
    private static TableTableCellElementBase getCell(TableTableRowElement row, int columnPos) {
        TableTableCellElementBase firstElementChild = null;
        NodeList nodeList = row.getChildNodes();
        int cellPos = -1;
        for (int i = 0; i < nodeList.getLength(); i++) {
            Node node = nodeList.item(i);
            if (node instanceof OdfElement) {
                if (node instanceof TableTableCellElementBase) {
                    // if the cell element starts at the column position
                    if (cellPos + 1 == columnPos) {
                        firstElementChild = (TableTableCellElementBase) node;
                        break;
                    }else {
                        int cellPosEarlier = cellPos;
                        cellPos += ((TableTableCellElementBase) node).getRepetition();                        
                        if (cellPos < columnPos) {
                            // if the column pos was not reached, check the next cell element
                            continue;
                        } else if (cellPos == columnPos) {
                            firstElementChild = (TableTableCellElementBase) node;
                            break;
                        } else {
                            firstElementChild = (TableTableCellElementBase) ((TableTableCellElementBase) node).split(cellPos - cellPosEarlier - 1);
                            break;
                        }             
                    }
                }
            }
        }
        return firstElementChild;
    }

    /**
     * Initializes the data set for the next data row.
     */
    private void nextDataRangeRow() {
        // increment to new position
        mDataRangeRowPos++;
        // reset to the first column in new row
        mDataRangeColumnPos = 0;

        // the horizontal position is equal to the length
        if (mRangeData.length() > 0) {
            // estimate new first row
            mRowEndNo = mRowStartNo + mRangeData.length() - 1;
        }
        mRangeRowData = mRangeData.optJSONArray(mDataRangeRowPos);
        if (mRangeRowData != null && mRangeRowData.length() > 0) {
            mColumnEndNo = mColumnStartNo + mRangeRowData.length() - 1;
        } else {
            mColumnEndNo = 0;
        }
    }

    /**
     * This method adds the given element (either a column, row or cell) to the
     * table and handles formatting according to MSO behavior: The styles are
     * related from the table cell before and after the insertion. In general
     * the styles are overtaken from the prior cell. Formulas are never copied.
     * Drawings and merged cells, which are overlapping the insertion point
     * (i.e. the following and preceding cell) are being extended. Cell borders
     * are overtaken for a row/column addition only if, the same side (right,
     * left, top or bottom) are identical for the preceding and following cell.
     */
    private void addEmptyRange(OdfStylableElement tableStyleableElement, int currentPos) {
        // *** ROW ***
        if (tableStyleableElement instanceof TableTableRowElement) {
            if (mNewRowCount > 0) {
                // if there is a new row to be inserted in the first position, there is no preceding element
                if (mPrecedingRowElement == null && mRowStartNo != Table.BEFORE_START) {
                    mPrecedingRowElement = receivePrecedingElement(tableStyleableElement, NUMBER_ROWS_REPEATED, TABLE_NUMBER_ROWS_REPEATED, currentPos, mRowEndNo, mNewRowCount);
                } else { // we have the following element
                    if (mNewContainerElement == null) {
                        mNewContainerElement = receiveNewContainerElement(tableStyleableElement, NUMBER_ROWS_REPEATED, TABLE_NUMBER_ROWS_REPEATED, mNewRowCount, TableTableRowElement.class);
                    }
                }
            }
            // *** COLUMN ***
        } else if (tableStyleableElement instanceof TableTableColumnElement) {
            if (mNewColumnCount > 0) {
                // if there is a new column to be inserted in the first position, there is no preceding element
                if (mPrecedingColumnElement == null && mColumnStartNo != Table.BEFORE_START) {
                    mPrecedingColumnElement = receivePrecedingElement(tableStyleableElement, NUMBER_COLUMNS_REPEATED, TABLE_NUMBER_COLUMNS_REPEATED, currentPos, mColumnEndNo, mNewColumnCount);
                } else {
                    // The following would inherit the format from the first column
                    adaptStyleAndInsert(null, tableStyleableElement, false);
                }
            }

            // *** CELL ***
        } else { // it is a table cell
            // column insertion
            if (mNewColumnCount > 0) {
                // prio can be the vertical or horizontal adjacent cell dependent if a row/column has been inserted
                if (mPriorCellElement == null && mColumnStartNo != Table.BEFORE_START) {
                    // if the preceding and next element are the same
                    if (tableStyleableElement.getRepetition() >= 2) {
                        // ARGH: SPLIT ONLY WHEN THERE IS CONTENT
                        TableTableCellElementBase followingCell = (TableTableCellElementBase) tableStyleableElement.split(1);
                        adaptStyleAndInsert((TableTableCellElementBase) tableStyleableElement, followingCell, true);
                        mPriorCellElement = null;
                    } else {
                        mPriorCellElement = (TableTableCellElementBase) tableStyleableElement;
                        if (mColumnStartNo == Table.BEFORE_START) {
                            adaptStyleAndInsert(mPriorCellElement, (TableTableCellElementBase) tableStyleableElement, true);
                            mPriorCellElement = null;
                        }
                    }
                } else {
                    adaptStyleAndInsert(mPriorCellElement, (TableTableCellElementBase) tableStyleableElement, true);
                    mPriorCellElement = null;
                }

                // ROWS: for a row insertion
                // if the preceding row was already passed, the new basic cloned row already exists..
            } else if (mNewContainerElement != null) {
                // for every following cell element with repeated, check all previous elements
                for (int coveringPreviousCells = tableStyleableElement.getRepetition(); coveringPreviousCells >= 1; coveringPreviousCells--) {
                    if (mPriorCellElement == null) {
                        // get the first cell to compare..
                        mPriorCellElement = getNextTableCell((TableTableRowElement) mNewContainerElement, null);
                        if (mPriorCellElement != null) {
                            // position within the cellsRepeated/span (counting starts with 0)
                            mCellPositionWithinReptition = 0;
                            mPrecedingCellPos += mPriorCellElement.getRepetition();
                        } else {
                            // first row with no preceding rows to inherit styles
                            TableTableRowElement firstRow = (TableTableRowElement) tableStyleableElement.getParentNode();
                            OdfElement rowParent = (OdfElement) firstRow.getParentNode();
                            rowParent.insertBefore(mNewContainerElement, firstRow); // ADD ROW REPEITION
                            TableTableCellElement newCell = (TableTableCellElement) ((OdfFileDom) ((OdfElement) tableStyleableElement).getOwnerDocument()).newOdfElement(((OdfElement) tableStyleableElement).getClass());
                            if (mTableTraversal.getColumnMaxNo() > 1) {
                                newCell.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:number-columns-repeated", String.valueOf(mTableTraversal.getColumnMaxNo()));
                            } else {
                                newCell.removeAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated");
                            }
                            // If there was a column with default style, we need to overwrite that default with an empty style, as the new inserted first row has no style
                            // 2DO: We could check for table:default-cell-style-name and we could reuse one automatic style (perhaps with a mnemonic naming)
                            newCell.getOrCreateUnqiueAutomaticStyle();
                            mNewContainerElement.appendChild(newCell);
                            break;
                        }
                    } else { // compare the cells and adapt them..
                        // keep the given current cell in synch with the relative cell from the cloned row
                        if (currentPos > mPrecedingCellPos) {
                            mPriorCellElement = getNextTableCell((TableTableRowElement) mNewContainerElement, (TableTableCellElementBase) mPriorCellElement);
                            if (mPriorCellElement != null) {
                                mPrecedingCellPos += mPriorCellElement.getRepetition();
                            }
                            mCellPositionWithinReptition = 0;
                        }
                    }
                    if (mPriorCellElement != null) {
                        TableTableCellElementBase splittedPrecedingCellElement = adaptFormat(mPriorCellElement, (TableTableCellElementBase) tableStyleableElement, mCellPositionWithinReptition, Boolean.TRUE);
                        if (splittedPrecedingCellElement != null) {
                            mPriorCellElement = splittedPrecedingCellElement;
                            // reset position to zero as cell was split
                            mCellPositionWithinReptition = 0;
                        } else {
                            mCellPositionWithinReptition++;
                        }
                    }
                    currentPos++;
                }
                // check this cell, with the cloned one
            }
            // ADJUST FORMULA
        }

// An empty cell element will not show formatting
// NEW CELLS HAVE TO BE CREATED/ADDED
//		if (elementBeforeInsertion instanceof TableTableCellElement) {
//					ensureContent((TableTableCellElement) elementBeforeInsertion, mXMLDoc);
//		}
//		JsonOperationConsumer.addStyle(mFormatProps, (OdfStylableElement) cellElement, mXMLDoc);
    }

    private void adaptStyleAndInsert(OdfStylableElement priorNode, Node followingNode, boolean adaptStyles) {
        OdfStylableElement newElement;
        OdfElement parent;
        // Better to add after the preceding due if nextSibling is null, the node will be appended
        if (followingNode == null) {
            parent = ((OdfElement) priorNode.getParentNode());
        } else {
            parent = ((OdfElement) followingNode.getParentNode());
        }

        if (priorNode != null) {
            newElement = (OdfStylableElement) priorNode.cloneNode(false);
        } else {
            newElement = (OdfStylableElement) ((OdfFileDom) ((OdfElement) followingNode).getOwnerDocument()).newOdfElement(((OdfElement) followingNode).getClass());
        }
        if (mNewColumnCount > 1) {
            newElement.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:number-columns-repeated", String.valueOf(mNewColumnCount));
        } else {
            newElement.removeAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated");
        }
        // Better to add after the preceding due if nextSibling is null, the node will be appended
        parent.insertBefore(newElement, followingNode);
        if (adaptStyles) { // works only for cells
            adaptFormat((TableTableCellElementBase) newElement, (TableTableCellElementBase) followingNode, 0, Boolean.FALSE);
        }
    }

    private OdfStylableElement receiveNewContainerElement(OdfStylableElement tableStyleableElement, String localName, String qualifiedName, int newCount, Class containerClass) {
        OdfStylableElement newContainerElement = null;
        if (mPrecedingRowElement != null) {
            // clone all cells with styles
            newContainerElement = (OdfStylableElement) mPrecedingRowElement.cloneNode(1);
            OdfElement parent = ((OdfElement) mPrecedingRowElement.getParentNode());
            // Better to add after the preceding due if nextSibling is null, the node will be appended
            parent.insertBefore(newContainerElement, mPrecedingRowElement.getNextSibling());
        } else { // create a default element from scratch as there was no preceding element
            if (mXMLDoc == null) {
                mXMLDoc = (OdfFileDom) tableStyleableElement.getOwnerDocument();
            }
            newContainerElement = (OdfStylableElement) mXMLDoc.newOdfElement(containerClass);
        }
        // ADJUST REPETITION
        if (newCount > 1) { // add repeated
            newContainerElement.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), qualifiedName, String.valueOf(newCount));
        } else { // remove repeated
            newContainerElement.removeAttributeNS(OdfDocumentNamespace.TABLE.getUri(), localName);
        }
        return newContainerElement;
    }

    private static OdfStylableElement receivePrecedingElement(OdfStylableElement tableStyleableElement, String localName, String qualifiedName, int currentPos, int endNo, int newCount) {
        if (tableStyleableElement.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), localName)) {
            int repetition = tableStyleableElement.getRepetition();

            // the current element is representing the preceding and following at the same time cellsRepeated can be enhanced for column, but not for rows
            if (repetition > 1) {
                // if there is no style (as later border deletion would change the prior as well), no content (for rows as it would be repeated..
                // If it is a table and has no chilrden OR if the element before and after are part of the current element
                if ((tableStyleableElement instanceof TableTableRowElement && tableStyleableElement.hasChildNodes())) {
                    // 1) FOLLOWING PART - clone existing with children
                    OdfStylableElement followingPart = (OdfStylableElement) tableStyleableElement.cloneElement();
                    int newRepetition = (endNo - currentPos);
                    followingPart.removeAttributeNS(OdfDocumentNamespace.XML.getUri(), "id");
                    followingPart.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), qualifiedName, String.valueOf(newRepetition));

                    // 2) NEW PART - clone existing without cell children, without cell mNewCellValue attributes
                    OdfStylableElement newPart = (OdfStylableElement) tableStyleableElement.cloneNode(1);
                    newPart.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), qualifiedName, String.valueOf(newCount));

                    // 3) FIRST PART - only the cellsRepeated is no longer valid.
                    if (repetition - newRepetition < 2) {
                        tableStyleableElement.removeAttributeNS(OdfDocumentNamespace.TABLE.getUri(), localName);
                    } else {
                        tableStyleableElement.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), qualifiedName, String.valueOf(repetition - newRepetition));
                    }
                } else { // FOR COLUMNS AND NONE-EMPTY ROWS
                    // the cellsRepeated can be incremented
                    tableStyleableElement.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), qualifiedName, String.valueOf(newCount + repetition));
                }
            }
        }
        // do nothing aside to memorize, as we need the following element to insert in DOM
        return tableStyleableElement;
    }

    /**
     * Get the next available cell element of the row
     */
    private static TableTableCellElementBase getNextTableCell(TableTableRowElement parentRow, TableTableCellElementBase precedingCellElement) {
        TableTableCellElementBase nextCell = null;
        Node cellCandidate = null;
        if (precedingCellElement == null) {
            cellCandidate = parentRow.getFirstChildElement();
        } else {
            cellCandidate = precedingCellElement.getNextSibling();
        }
        // make sure a cell is being taken
        while (cellCandidate != null && !(cellCandidate instanceof TableTableCellElementBase)) {
            cellCandidate = cellCandidate.getNextSibling();
        }
        if (cellCandidate instanceof TableTableCellElementBase) {
            nextCell = (TableTableCellElementBase) cellCandidate;
        }
        return nextCell;
    }

    /**
     * The styles of the changableCell are being adapted to fit to the style
     * behavior being used at Excel. Formulas are never copied. Drawings and
     * merged cells, which are overlapping the insertion point (i.e. the
     * following and preceding cell) are being extended. Cell borders are
     * overtaken for a row/column addition only if, the same side (right, left,
     * top or bottom) are identical for the preceding and following cell.
     *
     * @param changableCell the new cell being added, with the formatting of the
     * prior cell
     * @param comparableCell the cell following the insertion, required to
     * identify the styles of the new cells
     * @return if the preceding element had a cellsRepeated and was split, the
     * return is the following element to be compared next, otherwise null.
     */
    private static TableTableCellElementBase adaptFormat(TableTableCellElementBase changableCell, TableTableCellElementBase comparableCell, int currentPos, boolean isRowInsertion) {
        TableTableCellElementBase followUpCell = null;

        // Formulas are never copied.
        changableCell.removeAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula");

        // remove the xml:id
        changableCell.removeAttributeNS(OdfDocumentNamespace.XML.getUri(), "id");

        // adapt borders
        OdfStyle newStyle = changableCell.getAutomaticStyle();
        StyleTableCellPropertiesElement newCellProps = null;
        StyleTableCellPropertiesElement followingCellProps = null;
        if (newStyle != null) {
            newCellProps = (StyleTableCellPropertiesElement) newStyle.getPropertiesElement(OdfStylePropertiesSet.TableCellProperties);
            OdfStyle followingStyle = comparableCell.getAutomaticStyle();
            if (followingStyle != null) {
                followingCellProps = (StyleTableCellPropertiesElement) followingStyle.getPropertiesElement(OdfStylePropertiesSet.TableCellProperties);
            }
            // Only the similar borders will remain
            // SIMPLIFICATION: SHOULD be applied on "absolute" styles, without any further inheritance, but not used in apps and no utiltiy fkt :(
            if (newCellProps != null) {
                OdfOfficeAutomaticStyles autoStyles = changableCell.getAutomaticStyles();
                // defines what borders should be removed
                byte borderRemoval;
                byte diagonalCrossRemoval;
                if (followingCellProps == null) {
                    // remove all borders
                    borderRemoval = StyleTableCellPropertiesElement.BORDER_ALL;
                    diagonalCrossRemoval = StyleTableCellPropertiesElement.DIAGONAL_CROSS_ALL;
                } else {
                    //ToDo: currently assuming two borders of the same application are equal in measure (e.g. both cm)
                    borderRemoval = StyleTableCellPropertiesElement.findDifferentBorders(newCellProps, followingCellProps);
                    diagonalCrossRemoval = StyleTableCellPropertiesElement.findDifferentDiagonalCross(newCellProps, followingCellProps);
                }
                // if you need to adapt the border format
                if (borderRemoval != StyleTableCellPropertiesElement.BORDER_NONE || diagonalCrossRemoval != StyleTableCellPropertiesElement.DIAGONAL_CROSS_NONE) {
                    // if the changableCell had a repeating attribute split the cell, so only the current cell will be influenced..
                    if (isRowInsertion) {
                        changableCell = (TableTableCellElementBase) changableCell.split(currentPos);
                        int repeatComparable = comparableCell.getRepetition();
                        int repeatChangable = changableCell.getRepetition();
                        if (repeatComparable != repeatChangable && repeatChangable > 1) {
                            followUpCell = (TableTableCellElementBase) changableCell.split(1);
                        } else {
                            followUpCell = changableCell;
                        }
                    }
                    // clone the existing automatic styles, so not further users of that style will be influenced..
                    OdfStyle styleToBeChanged = autoStyles.makeStyleUnique(newStyle);
                    changableCell.setStyleName(styleToBeChanged.getStyleNameAttribute());
                    newCellProps = (StyleTableCellPropertiesElement) styleToBeChanged.getPropertiesElement(OdfStylePropertiesSet.TableCellProperties);
                    if (borderRemoval != StyleTableCellPropertiesElement.BORDER_NONE) {
                        newCellProps.removeBorder(borderRemoval);
                    }
                    if (diagonalCrossRemoval != StyleTableCellPropertiesElement.DIAGONAL_CROSS_NONE) {
                        newCellProps.removeDiagonalCross(diagonalCrossRemoval);
                    }
                }
            }
        }
        return followUpCell;
    }

    /**
     * ToDo: PossibleRefactoring: This format Method might be moved to
     * TableTableCell element
     */
    void formatCell(TableTableCellElement cellElement, JSONObject formatProps) {
        if (mXMLDoc == null) {
            mXMLDoc = (OdfFileDom) cellElement.getOwnerDocument();
        }
        // An empty cell element will not show formatting
        ensureContent(cellElement, mXMLDoc);
        JsonOperationConsumer.addStyle(formatProps, (OdfStylableElement) cellElement, mXMLDoc);
    }

    /**
     * ToDo: PossibleRefactoring: This format Method might be moved to
     * TableTableCell element
     */
    void changeCell(TableTableCellElement cellElement) {
    }

    /**
     * An empty cell element will not show formatting. At least an empty
     * paragraph will be ensured.
     */
    private static void ensureContent(TableTableCellElement cellElement, OdfFileDom document) {
        if (!cellElement.hasChildNodes()) {
            cellElement.appendChild(document.createElementNS(TextPElement.ELEMENT_NAME));
        }
    }

    /**
     * ToDo: PossibleRefactoring: This format Method might be moved to
     * TableTableRow element
     */
    void formatRow(TableTableRowElement rowElement, JSONObject formatProps) {
        if (mXMLDoc == null) {
            mXMLDoc = (OdfFileDom) rowElement.getOwnerDocument();
        }
        JsonOperationConsumer.addStyle(formatProps, (OdfStylableElement) rowElement, mXMLDoc);
        JSONObject rowProps = formatProps.optJSONObject("row");
        if (rowProps != null) {
            Boolean changeVisibility = rowProps.optBoolean("visible", Boolean.TRUE);
            rowElement.setVisiblity(changeVisibility);
        }
    }

    /**
     * ToDo: PossibleRefactoring: This format Method might be moved to
     * TableTableColumn element
     */
    void formatColumn(TableTableColumnElement columnElement, JSONObject formatProps) {
        if (mXMLDoc == null) {
            mXMLDoc = (OdfFileDom) columnElement.getOwnerDocument();
        }
        JsonOperationConsumer.addStyle(formatProps, (OdfStylableElement) columnElement, mXMLDoc);
        JSONObject columnProps = formatProps.optJSONObject("column");
        if (columnProps != null) {
            Boolean changeVisibility = columnProps.optBoolean("visible", Boolean.TRUE);
            columnElement.setVisiblity(changeVisibility);
        }
    }

    public boolean isColumnDeletion() {
        return mIsColumnDeletion;
    }

    public boolean effectsColumns() {
        return hasColumnEffect;
    }

    public boolean effectsRows() {
        return hasRowEffect;
    }

    public boolean effectsCells() {
        return hasCellEffect;
    }

    @Override
    public String toString() {
        return "formatChange: " + mFormatProps.toString();
    }

    public void setColumnStart(int columnStartNo) {
        mColumnStartNo = columnStartNo;
    }

    public void setColumnEnd(int columnEndNo) {
        mColumnEndNo = columnEndNo;
    }

    public void setRowStart(int rowStartNo) {
        mRowStartNo = rowStartNo;
    }

    public void setRowEnd(int rowEndNo) {
        mRowEndNo = rowEndNo;
    }
}
