/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.dom.OdfContentDom;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.office.OfficeBodyElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeDocumentContentElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeSpreadsheetElement;
import org.odftoolkit.odfdom.dom.element.table.TableConsolidationElement;
import org.odftoolkit.odfdom.dom.element.table.TableDataPilotTablesElement;
import org.odftoolkit.odfdom.dom.element.table.TableDatabaseRangeElement;
import org.odftoolkit.odfdom.dom.element.table.TableDatabaseRangesElement;
import org.odftoolkit.odfdom.dom.element.table.TableDdeLinksElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterAndElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterConditionElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterOrElement;
import org.odftoolkit.odfdom.dom.element.table.TableFilterSetItemElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.component.OperationConstants;
import org.w3c.dom.Node;

class AutoFilterColumn {
    //TODO: To support different operators over different columns this definitions needs to be changed
    public String mOperator = ""; //initially set in table:filter-condition element, overwritten by filter-set-items which always use operator '='
    public String mValue = "";    //   - "" -
    public ArrayList<String> mFilterConditions = new ArrayList<String>(); //entries in changeTableColumns
}

//a table:database-range can be container of AutoFilter settings
class CachedDBRange extends CachedComponent {
    private static final long serialVersionUID = 1L;
    public String name = null;
    private Integer mSheetIndex = null;
    private int mStartX = 0;
    private int mStartY = 0;
    private int mEndX = 0;
    private int mEndY = 0;
    boolean displayFilterButtons = false;
    private HashMap< Integer, AutoFilterColumn> mFilterColumns = null; //entries in changeTableColumns
    private int mCurrentFilterConditionColumn = -1;
    private final Map<String, Integer> mTableNames;
    
    public CachedDBRange(final Map<String, Integer> tableNames){
        mTableNames = tableNames;
    }
    
    public void createOperations() {
        if(mSheetIndex != null && name.startsWith("__Anonymous_Sheet_DB__")){
            List<Integer> startArray = new ArrayList<Integer>();
            startArray.add(mStartX);
            startArray.add(mStartY);
            List<Integer> endArray = new ArrayList<Integer>();
            endArray.add(mEndX);
            endArray.add(mEndY);
            Map<String, Object> attributes = new HashMap<String, Object>();
            CachedOperation newOp = new CachedOperation(OperationConstants.AUTOFILTER, startArray, attributes, endArray, new Integer(mSheetIndex));
            add(newOp);
            if(mFilterColumns != null) {
                for( Integer column : mFilterColumns.keySet() ) {
                    AutoFilterColumn  conditions = mFilterColumns.get(column);
                    if(conditions.mOperator.equals("=")) {
                        List<Integer> start = new ArrayList<Integer>();
                        start.add(column);
                        ArrayList<String> filterConditions = conditions.mFilterConditions;
                        if(filterConditions.isEmpty()){
                            filterConditions.add(conditions.mValue);
                        }
                        CachedOperation newColOp = new CachedOperation(OperationConstants.AUTOFILTERCOLUMN, start, null, new Integer(mSheetIndex), filterConditions);
                        add(newColOp);
                    }
                }
            }
        }
    }
    private void convertStringToAddress(String address, int[] columnRow) {
        int index = 0;
        address = address.toLowerCase();
        int col = 0;
        while(index < address.length()){
            char character = address.charAt(index++);
            if(character <= 'z' && character >= 'a'){
                col *= 26;
                col = character - 'a';
            } else {
                columnRow[0] = col;
                String intPart = address.substring(index - 1);
                columnRow[1] = Integer.parseInt(intPart) - 1;
                break;
            }
        }
    }
    public void setRangeAddress(String value) {
        int colonIndex = value.indexOf(':');
        String left = colonIndex > 0 ? value.substring(0, colonIndex) : value;
        String right = colonIndex > 0 ? value.substring(colonIndex + 1) : value;
        int sheetIndex = left.indexOf('.');
        String sheetName = value.substring(0, sheetIndex);
        mSheetIndex = mTableNames.get(sheetName);
        if(mSheetIndex != null) {
            String leftCell = left.substring(sheetIndex +1 );
            int columnRow[] = new int[2];
            columnRow[0] = -1;
            convertStringToAddress(leftCell, columnRow);
            if(columnRow[0] >= 0){
                mStartX = columnRow[0];
                mStartY = columnRow[1];
            }
            if(colonIndex > 0){
                sheetIndex = right.indexOf('.');
                String rightCell = right.substring(sheetIndex + 1);
                columnRow[0] = -1;
                convertStringToAddress(rightCell, columnRow);
                if(columnRow[0] >= 0){
                    mEndX = columnRow[0];
                    mEndY = columnRow[1];
                }
            } else {
                mEndX = mStartX;
                mEndY = mStartY;
            }
        }
    }
    public void createFilterCondition(int column, String operator, String value) {
        if( mFilterColumns == null) {
            mFilterColumns = new HashMap< Integer, AutoFilterColumn>();
        }
        if(mFilterColumns.containsKey(column)) {
            AutoFilterColumn filterColumn = mFilterColumns.get(column);
            if(filterColumn.mOperator.equals( operator ) && operator.equals("=")) {
                if(filterColumn.mFilterConditions.isEmpty() && filterColumn.mValue != null && !filterColumn.mValue.isEmpty()) {
                    filterColumn.mFilterConditions.add(filterColumn.mValue);
                    filterColumn.mValue = "";
                }
                filterColumn.mFilterConditions.add(value);
            }
        } else {
            AutoFilterColumn filterColumn = new AutoFilterColumn();
            filterColumn.mOperator = operator;
            filterColumn.mValue = value;
            mFilterColumns.put(column, filterColumn);
        }
        mCurrentFilterConditionColumn = column;
    }
    public void addFilterConditionItem(String value) {
        //  possible operators:
        // =, !=, >, <, >=, <=,  empty, !empty, top values, top percent,
        // "begins-with", "does-not-begin-with", "ends-with", "does-not-end-with", "contains", "does not contain",
        // "bottom percent", "bottom values"
        AutoFilterColumn column = mFilterColumns.get(mCurrentFilterConditionColumn);
        if(column != null) {
            column.mFilterConditions.add(value);
            column.mOperator = "=";
            column.mValue = "";
        }
    }
}

class AutoFilter {
    /**
     * returns the database ranges element if available of if create is true
     * @param contentDom
     * @param create create database ranges
     * @return found or created database ranges element or null
     */
    private static Node findDatabaseRanges(OdfContentDom contentDom, boolean create ) {

        OdfElement contentElement = OdfElement.findFirstChildNode(OfficeDocumentContentElement.class, contentDom);
        OdfElement bodyElement = OdfElement.findFirstChildNode(OfficeBodyElement.class, contentElement);
        OdfElement spreadsheetElement = OdfElement.findFirstChildNode(OfficeSpreadsheetElement.class, bodyElement);
        OdfElement databaseRangesElement = OdfElement.findFirstChildNode(TableDatabaseRangesElement.class, spreadsheetElement);
        if( databaseRangesElement == null && create) {
            databaseRangesElement = new TableDatabaseRangesElement(contentDom);
            Node refChild = null;
//            OdfElement namedExpressions = OdfElement.findFirstChildNode(TableNamedExpressionsElement.class, spreadsheetElement);
            OdfElement dataPilot = OdfElement.findFirstChildNode(TableDataPilotTablesElement.class, spreadsheetElement);
            if(dataPilot != null ) {
                refChild = dataPilot;
            } else {
                OdfElement consolidation = OdfElement.findFirstChildNode(TableConsolidationElement.class, spreadsheetElement);
                if(consolidation != null) {
                    refChild = consolidation;
                } else {
                    OdfElement ddeLinks = OdfElement.findFirstChildNode(TableDdeLinksElement.class, spreadsheetElement);
                    if(ddeLinks != null) {
                        refChild = ddeLinks;
                    }
                }
            }
            spreadsheetElement.insertBefore(databaseRangesElement, refChild);
        }
        return databaseRangesElement;
    }
    private static String findSheetName(OdfContentDom contentDom, int sheetNo) {
        String name = null;
        OdfElement contentElement = OdfElement.findFirstChildNode(OfficeDocumentContentElement.class, contentDom);
        OdfElement bodyElement = OdfElement.findFirstChildNode(OfficeBodyElement.class, contentElement);
        OdfElement spreadsheetElement = OdfElement.findFirstChildNode(OfficeSpreadsheetElement.class, bodyElement);
        OdfElement tableElement = OdfElement.findFirstChildNode(TableTableElement.class, spreadsheetElement);
        int index = 0;
        while(tableElement != null) {
            if(index == sheetNo) {
                name = tableElement.getAttribute("table:name");
                break;
            }
            ++index;
            tableElement = (OdfElement)tableElement.getNextSibling();
        }
        return name;
    }
    private static String arrayToAddress(JSONArray colRow) throws JSONException {
        StringBuilder ret = new StringBuilder();
        int col = colRow.getInt(0);
        int letterVal = col / 676;
        if( letterVal > 0 ) {
            col -= (letterVal * 676);
            char character = 'A';
            character += letterVal;
            ret.append(character);
        }
        letterVal = col / 26;
        if(letterVal > 0) {
            col -= (letterVal * 26);
            char character = 'A';
            character += letterVal;
            ret.append(character);
        }
        char character = 'A';
        character += col;
        ret.append(character);
        ret.append(colRow.getInt(1) + 1);
        return ret.toString();
    }
    private static class RangeAndFreeIndex {
        public TableDatabaseRangeElement mRange = null;
        public int mFreeIndex = 0;
        public String  mSheetName = "";
    }
    private static RangeAndFreeIndex findDBRange(OdfContentDom contentDom, Node dbRanges, int sheetNo) {
        RangeAndFreeIndex ret = new RangeAndFreeIndex();
        ret.mSheetName = findSheetName(contentDom, sheetNo);
        Node range = dbRanges.getFirstChild();
        int freeIndex = sheetNo;
        while( range != null ) {
            TableDatabaseRangeElement rangeElement = (TableDatabaseRangeElement)range;
            String tblName = rangeElement.getAttribute("table:name");
            if(tblName.startsWith("__Anonymous_Sheet_DB__")) {
                try {
                    int tblNameIndex = Integer.parseInt(tblName.substring("__Anonymous_Sheet_DB__".length()));
                    if(tblNameIndex >= freeIndex ) {
                        freeIndex = tblNameIndex + 1;
                    }
                } catch( NumberFormatException e) {
                    //no int value - no handling required
                }
                String targetRange = rangeElement.getAttribute("table:target-range-address");
                if( targetRange.substring(0, targetRange.indexOf('.')).equals(ret.mSheetName)) {
                    ret.mRange = (TableDatabaseRangeElement)range;
                    break;
                }
            }
            range = range.getNextSibling();
        }
        return ret;
    }
    public static void insertAutoFilter(OdfContentDom contentDom, int sheetNo, JSONArray start, JSONArray end, JSONObject attrs) throws JSONException {
        Node dbRanges = findDatabaseRanges(contentDom, true);
        if( dbRanges != null ) {
            RangeAndFreeIndex found = findDBRange(contentDom, dbRanges, sheetNo);

            if( found.mRange != null) {
                dbRanges.removeChild(found.mRange); //existing filter ranges might have been ignored as unsupported while loading 
            }
            TableDatabaseRangeElement newRange = new TableDatabaseRangeElement(contentDom);
            newRange.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:name", "__Anonymous_Sheet_DB__" + found.mFreeIndex);
            String startString = arrayToAddress(start);
            String endString = arrayToAddress(end);
            newRange.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:target-range-address", found.mSheetName + "." + startString + ":" + found.mSheetName + "." + endString);
            newRange.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:display-filter-buttons", "true");
            dbRanges.insertBefore(newRange, null);
        }

    }
    public static void deleteAutoFilter(OdfContentDom contentDom, int sheetNo) {
        Node dbRanges = findDatabaseRanges(contentDom, true);
        if( dbRanges != null ) {
            RangeAndFreeIndex found = findDBRange(contentDom, dbRanges, sheetNo);
            if(found.mRange != null) {
                dbRanges.removeChild(found.mRange);
                if( dbRanges.getFirstChild() == null ) {
                    dbRanges.getParentNode().removeChild(dbRanges); // remove ranges if empty
                }
            } else {            
                //TODO: throw exception
            }
        }
    }
    public static void findAndRemoveColumn(OdfContentDom contentDom, TableDatabaseRangeElement range, int col) {
        Node tableFilter = OdfElement.findFirstChildNode(TableFilterElement.class, range);
        if( tableFilter != null) {
            // if it contains a TableFilterAnd -> remove it, it breaks our AutoFilter support ATM
            Node filterChild = tableFilter.getFirstChild();
            while(filterChild != null) {
                Node nextChild = filterChild.getNextSibling();
                if(filterChild instanceof TableFilterAndElement ) {
                    tableFilter.removeChild(filterChild);
                } else {
                    TableFilterOrElement orElement = (TableFilterOrElement)filterChild;
                    Node orChild = orElement.getFirstChild();
                    if(orChild instanceof TableFilterAndElement) {
                        tableFilter.removeChild(filterChild);
                    } else {
                        TableFilterConditionElement condition = (TableFilterConditionElement)orChild;
                        String column = condition.getAttribute("table:field-number");
                        if( Integer.parseInt(column) == col ) {
                            tableFilter.removeChild(filterChild);
                        }
                    }
                }
                
                filterChild = nextChild;
            }
        }
    }
    public static void changeAutoFilterColumn(OdfContentDom contentDom, int sheetNo, int col, JSONObject attrs) {
        Node dbRanges = findDatabaseRanges(contentDom, false);
        if( dbRanges != null ) {
            RangeAndFreeIndex found = findDBRange(contentDom, dbRanges, sheetNo);
            if(found.mRange != null) {
                findAndRemoveColumn(contentDom, found.mRange, col);
                List<String> entries = null;
                JSONObject filter = attrs.optJSONObject("filter");
                if( filter != null ) {
                    String type = filter.optString("type");
                    if( type != null && type.equals("discrete")) {
                        JSONArray entriesArray = filter.optJSONArray("entries");
                        if(entriesArray != null && entriesArray.length() > 0) {
                            entries = new ArrayList<String>();
                            Iterator<Object> entryIt = entriesArray.iterator();
                            while(entryIt.hasNext()) {
                                entries.add((String)entryIt.next());
                            }
                        }
                    }
                }
                Node tableFilter = found.mRange.getFirstChild();
                if(entries != null) {
                    if( tableFilter == null ) {
                        //insert new filter
                        tableFilter = new TableFilterElement(contentDom);
                        found.mRange.insertBefore(tableFilter, null);
                    }
                    //insert new FilterOr with FilterCondition
                    TableFilterOrElement tableFilterOr = new TableFilterOrElement(contentDom);
                    tableFilter.insertBefore(tableFilterOr, null);
                    TableFilterConditionElement tableFilterCondition = new TableFilterConditionElement(contentDom);
                    tableFilterOr.insertBefore(tableFilterCondition, null);
                    tableFilterCondition.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:field-number", Integer.toString(col) );
                    tableFilterCondition.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:value", entries.get(0) );
                    tableFilterCondition.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:operator", "=" );
                    ListIterator<String> entriesIt = entries.listIterator();
                    while(entriesIt.hasNext()) {
                        TableFilterSetItemElement filterSetItem = new TableFilterSetItemElement(contentDom); 
                        tableFilterCondition.insertBefore(filterSetItem, null);
                        filterSetItem.setAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "table:value", entriesIt.next() );
                    }
                }
            } else {
                //TODO: throw exception
            }
        }
    }
    
}