/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.io.File;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;

import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineConfig;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;

//=============================================================================
public class CalcEngineClientPerformanceMeasurement implements ICalcEngineClient
{
    //-------------------------------------------------------------------------
    private static final Log LOG = LogFactory.getJclLog(CalcEngineClientPerformanceMeasurement.class);

    //-------------------------------------------------------------------------
    private CalcEngineClientPerformanceMeasurement ()
        throws Exception
    {}
    
    //-------------------------------------------------------------------------
    public synchronized static CalcEngineClientPerformanceMeasurement create(final ICalcEngineClient iWrapped)
        throws Exception
    {
    	CalcEngineClientPerformanceMeasurement aClient = new CalcEngineClientPerformanceMeasurement ();
    	aClient.m_iWrapped = iWrapped;
    	return aClient;
    }

    //-------------------------------------------------------------------------
    @Override
    public String createDocument()
        throws Exception
    {
    	final long nStart = System.currentTimeMillis();
    	try
    	{
	    	final String sHandle = m_iWrapped.createDocument();
	    	return sHandle;
    	}
    	finally
    	{
    		final long nEnd  = System.currentTimeMillis();
    		final long nTime = nEnd - nStart;
    		
    		MEASURE("create-document", nTime);
    	}
    }
    
    //-------------------------------------------------------------------------
    @Override
    public void destroyDocument(final String sHandle)
        throws Exception
    {
    	final long nStart = System.currentTimeMillis();
    	try
    	{
    		m_iWrapped.destroyDocument(sHandle);
    	}
    	finally
    	{
    		final long nEnd  = System.currentTimeMillis();
    		final long nTime = nEnd - nStart;
    		
    		MEASURE("destroy-document", nTime);
    	}
    }

    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError executeOperation(final String       sHandle       ,
                                   			 final String       sOperationJSON,
                                   			 final StringBuffer sResultJSON   )
        throws Exception
    {
    	final long nStart = System.currentTimeMillis();
    	try
    	{
	    	final ECalcEngineError eError = m_iWrapped.executeOperation(sHandle, sOperationJSON, sResultJSON);
	    	return eError;
    	}
    	finally
    	{
    		final long nEnd  = System.currentTimeMillis();
    		final long nTime = nEnd - nStart;
    		
    		MEASURE("execute-operation", nTime);
    	}
    }
    
    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError restoreDocument (final String    sHandle             ,
			 				                 final String... lPagedOperationJSONs)
        throws Exception
    {
    	final ECalcEngineError eError = m_iWrapped.restoreDocument(sHandle, lPagedOperationJSONs);
    	return eError;
    }

    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError copy(final String                   sSourceDocHandle,
								 final CalcEngineClipBoardEvent aEvent          )
		throws Exception
	{
    	final ECalcEngineError eError = m_iWrapped.copy(sSourceDocHandle, aEvent);
    	return eError;
	}

    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError paste(final String                   sTargetDocHandle,
								  final CalcEngineClipBoardEvent aEvent          )
		throws Exception
	{
    	final ECalcEngineError eError = m_iWrapped.paste(sTargetDocHandle, aEvent);
    	return eError;
	}

    //-------------------------------------------------------------------------
    @Override
    public String getVersion()
    {
        return "Calc Engine Performance Measurement 1.0 :-)";
    }

    //-------------------------------------------------------------------------
    @Override
    public void setLogLevel (final ELogLevel eLevel) 
    	throws Exception
    {
    	m_iWrapped.setLogLevel(eLevel);
    }

    //-------------------------------------------------------------------------
    @Override
    public void dispose ()
    	throws Exception
    {
    	if (m_iWrapped != null)
    		m_iWrapped.dispose ();
    	m_iWrapped = null;
    }

    //-------------------------------------------------------------------------
    private static synchronized void MEASURE (final String sOp  ,
    		 			                      final long   nTime)
    {
    	try
    	{
    		final Map< String, String > lStamps = mem_Stamps ();
	    	      String                sStamp  = lStamps.get(sOp);
	    	
	    	if (sStamp == null)
	    	{
	    		sStamp = Long.toString(nTime);
	    	}
	    	else
	    	{
	    		sStamp += ",";
	    		sStamp += Long.toString(nTime);
	    	}
	    	
	    	lStamps.put (sOp, sStamp);
	    	FileUtils.writeStringToFile(mem_PerformanceFile (), impl_toString (lStamps), "utf-8");
    	}
    	catch (final Throwable ex)
    	{
    		// logged but silently ignored to not disturb the real product !
    		LOG.error (ex.getMessage (), ex);
    	}
    }
    
    //-------------------------------------------------------------------------
    private static String impl_toString (final Map< String, String > lStamps)
        throws Exception
    {
    	final StringBuffer sStamps = new StringBuffer (256);
    	final Iterator< String > lKeys = lStamps.keySet().iterator();
    	while (lKeys.hasNext())
    	{
    		final String sOp = lKeys.next ();
    		final String sStamp = lStamps.get(sOp);
    		sStamps.append (sOp);
    		sStamps.append (",");
    		sStamps.append (sStamp);
    		sStamps.append ("\n");
    	}
    	return sStamps.toString ();
    }
    
    //-------------------------------------------------------------------------
    private static File mem_PerformanceFile ()
        throws Exception
    {
    	if (m_aPerformanceFile == null)
    	{
    		final CalcEngineConfig aConfig   = CalcEngineConfig.get();
    		final String           sMode     = aConfig.getMode().name();
    		final File             aTempPath = FileUtils.getTempDirectory();
    		final String           sPerfFile = "ce-performance-"+sMode+".csv";
	    	final File             aPerfFile = new File (aTempPath, sPerfFile);
	    	
	    	LOG.info("place CE-Performance-File at '"+aPerfFile.getAbsolutePath()+"' ...");

	    	m_aPerformanceFile = aPerfFile;
    	}
    	return m_aPerformanceFile;
    }

    //-------------------------------------------------------------------------
    private static Map< String, String > mem_Stamps ()
        throws Exception
    {
    	if (m_lStamps == null)
	    	m_lStamps = new HashMap< String, String > ();
    	return m_lStamps;
    }

    //-------------------------------------------------------------------------
    private ICalcEngineClient m_iWrapped = null;

    //-------------------------------------------------------------------------
    private static File m_aPerformanceFile = null;

    //-------------------------------------------------------------------------
    private static Map< String, String > m_lStamps = null;
}
