/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.net.ConnectException;
import java.util.Map;
import java.util.Set;

import javax.ws.rs.core.UriBuilder;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;

import com.openexchange.office.calcengine.CalcEngineConst;
import com.openexchange.office.calcengine.client.CalcEngineClipBoardEvent;
import com.openexchange.office.calcengine.client.CalcEngineConfig;
import com.openexchange.office.calcengine.client.CalcEngineDescriptor;
import com.openexchange.office.calcengine.client.CalcEngineHandleGenerator;
import com.openexchange.office.calcengine.client.CalcEngineHttpEntity;
import com.openexchange.office.calcengine.client.CalcEngineHttpEntityReader;
import com.openexchange.office.calcengine.client.CalcEngineHttpEntityWriter;
import com.openexchange.office.calcengine.client.ECalcEngineError;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.logging.ContextAwareLogHelp;
import com.openexchange.office.tools.logging.ELogLevel;
import com.openexchange.office.tools.logging.LogFactory;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;

//=============================================================================
public class CalcEngineClientHTTP implements ICalcEngineClient
{
    //-------------------------------------------------------------------------
    private static final Log LOG = LogFactory.getJclLog(CalcEngineClientHTTP.class);

    //-------------------------------------------------------------------------
    private CalcEngineClientHTTP ()
        throws Exception
    {}
    
    //-------------------------------------------------------------------------
    public synchronized static CalcEngineClientHTTP create()
        throws Exception
    {
    	return new CalcEngineClientHTTP ();
    }

    //-------------------------------------------------------------------------
    public void describeEnvironment (final CalcEngineDescriptor aDesc)
        throws Exception
    {
    	m_aDescriptor = aDesc;
    }
    
    //-------------------------------------------------------------------------
    @Override
    public String createDocument()
        throws Exception
    {
        ClientResponse aResponse = mem_Service().path  (CalcEngineConst.CALCENGINE_CONTEXTPATH)
                                                .get   (ClientResponse.class                  );

        boolean bOK = (aResponse.getClientResponseStatus() == ClientResponse.Status.OK);
        if ( ! bOK)
            throw new Exception ("Failed with status : "+aResponse.getStatus());

        final CalcEngineHttpEntity aEntity = aResponse.getEntity(CalcEngineHttpEntity.class);
        final ECalcEngineError     eError  = aEntity.getError();
              String               sHandle = null;

        if (eError == ECalcEngineError.E_NONE)
        {
        	final String sJSONResult = aEntity.getJSONPayload();
        	             sHandle     = CalcEngineHandleGenerator.fromJSON(sJSONResult);
        }
        else
        	throw new Exception ("Failed with error '"+eError+"'.");

        return sHandle;
    };
    
    //-------------------------------------------------------------------------
    @Override
    public void destroyDocument(final String sHandle)
        throws Exception
    {
    	Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

        ClientResponse aResponse = mem_Service().path  (CalcEngineConst.CALCENGINE_CONTEXTPATH)
                                                .path  (sHandle                               )
                                                .delete(ClientResponse.class                  );

        boolean bOK = (aResponse.getClientResponseStatus() == ClientResponse.Status.OK);
        if ( ! bOK)
            throw new Exception ("Failed with status : "+aResponse.getStatus());
        
        // TODO do we need to check ECalcEngineError here too ?!
    };

    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError executeOperation(final String       sHandle       ,
                                   			 final String       sOperationJSON,
                                   			 final StringBuffer sResultJSON   )
        throws Exception
    {
    	Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

        ClientResponse aResponse = mem_Service().path  (CalcEngineConst.CALCENGINE_CONTEXTPATH               )
                                                .path  (sHandle                                              )
                                                .path  (CalcEngineConst.RELPATH_EXECUTE                      )
                                                .entity(sOperationJSON, CalcEngineConst.MEDIATYPE_4_STREAMING)
                                                .post  (ClientResponse.class                                 );

        boolean bOK = (aResponse.getClientResponseStatus() == ClientResponse.Status.OK);
        if ( ! bOK)
            throw new Exception ("Failed with status : "+aResponse.getStatus());
        
        final CalcEngineHttpEntity aEntity = aResponse.getEntity(CalcEngineHttpEntity.class);
        final ECalcEngineError     eError  = aEntity.getError();

        if (eError == ECalcEngineError.E_NONE)
        {
        	final String sJSONResult = aEntity.getJSONPayload();
	        sResultJSON.setLength (0);
	        sResultJSON.append    (sJSONResult);
        }
        
        return eError;
    };
    
    //-------------------------------------------------------------------------
    @Override
    public ECalcEngineError restoreDocument (final String    sHandle             ,
			 				                 final String... lPagedOperationJSONs)
        throws Exception
    {
    	Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

        for (String sOperationJSON : lPagedOperationJSONs)
        {
	        ClientResponse aResponse = mem_Service().path  (CalcEngineConst.CALCENGINE_CONTEXTPATH               )
	                                                .path  (sHandle                                              )
	                                                .path  (CalcEngineConst.RELPATH_RESTORE                      )
	                                                .entity(sOperationJSON, CalcEngineConst.MEDIATYPE_4_STREAMING)
	                                                .post  (ClientResponse.class                                 );
	
	        boolean bOK = (aResponse.getClientResponseStatus() == ClientResponse.Status.OK);
	        if ( ! bOK)
	            throw new Exception ("Failed with status : "+aResponse.getStatus());

	        final CalcEngineHttpEntity aEntity = aResponse.getEntity(CalcEngineHttpEntity.class);
	        final ECalcEngineError     eError  = aEntity.getError();

	        if (eError != ECalcEngineError.E_NONE)
	        	return eError;
        }

        return ECalcEngineError.E_NONE;
    }

    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError copy(final String                   sHandle,
								 final CalcEngineClipBoardEvent aEvent )
		throws Exception
	{
    	Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

    	final CalcEngineHttpEntity aRequestEntity = CalcEngineHttpEntity.OK(null, aEvent);
    	
        ClientResponse aResponse = mem_Service().path  (CalcEngineConst.CALCENGINE_CONTEXTPATH               )
                                                .path  (sHandle                                              )
                                                .path  (CalcEngineConst.RELPATH_COPY                         )
                                                .entity(aRequestEntity, CalcEngineConst.MEDIATYPE_4_STREAMING)
                                                .post  (ClientResponse.class                                 );

        boolean bOK = (aResponse.getClientResponseStatus() == ClientResponse.Status.OK);
        if ( ! bOK)
            throw new Exception ("Failed with status : "+aResponse.getStatus());
        
        final CalcEngineHttpEntity aResponseEntity = aResponse.getEntity(CalcEngineHttpEntity.class);
        final ECalcEngineError     eError          = aResponseEntity.getError();

        if (eError == ECalcEngineError.E_NONE)
        {
        	final CalcEngineClipBoardEvent aResult = aResponseEntity.getClipboardEvent();
        	aEvent.takeOver(aResult); // in/out param !
        }
        
        return eError;
	}

    //-------------------------------------------------------------------------
	@Override
	public ECalcEngineError paste(final String                   sHandle,
								  final CalcEngineClipBoardEvent aEvent )
		throws Exception
	{
    	Validate.isTrue(CalcEngineHandleGenerator.validateHandle(sHandle), "'"+sHandle+"' is an invalid handle.");

    	final CalcEngineHttpEntity aRequestEntity = CalcEngineHttpEntity.OK(null, aEvent);
    	
        ClientResponse aResponse = mem_Service().path  (CalcEngineConst.CALCENGINE_CONTEXTPATH               )
                                                .path  (sHandle                                              )
                                                .path  (CalcEngineConst.RELPATH_PASTE                        )
                                                .entity(aRequestEntity, CalcEngineConst.MEDIATYPE_4_STREAMING)
                                                .post  (ClientResponse.class                                 );

        boolean bOK = (aResponse.getClientResponseStatus() == ClientResponse.Status.OK);
        if ( ! bOK)
            throw new Exception ("Failed with status : "+aResponse.getStatus());
        
        final CalcEngineHttpEntity aResponseEntity = aResponse.getEntity(CalcEngineHttpEntity.class);
        final ECalcEngineError     eError          = aResponseEntity.getError();

        if (eError == ECalcEngineError.E_NONE)
        {
        	final CalcEngineClipBoardEvent aResult = aResponseEntity.getClipboardEvent();
        	aEvent.takeOver(aResult); // in/out param !
        }
        
        return eError;
	}

    //-------------------------------------------------------------------------
    @Override
    public String getVersion()
        throws Exception
    {
        return "";
    }

    //-------------------------------------------------------------------------
    @Override
    public void setLogLevel (final ELogLevel eLevel) 
    	throws Exception
    {
    	final String               sLevelJSON     = ELogLevel.toJSON (eLevel);
    	final CalcEngineHttpEntity aRequestEntity = CalcEngineHttpEntity.OK(sLevelJSON, null);
    	
        ClientResponse aResponse = mem_Service().path  (CalcEngineConst.CALCENGINE_CONTEXTPATH               )
                                                .path  (CalcEngineConst.RELPATH_SETLOGLEVEL                  )
                                                .entity(aRequestEntity, CalcEngineConst.MEDIATYPE_4_STREAMING)
                                                .post  (ClientResponse.class                                 );

        boolean bOK = (aResponse.getClientResponseStatus() == ClientResponse.Status.OK);
        if ( ! bOK)
            throw new Exception ("Failed with status : "+aResponse.getStatus());
    }

    //-------------------------------------------------------------------------
    @Override
    public void dispose ()
    	throws Exception
    {
    	// TODO
    }

    //-------------------------------------------------------------------------
    public enum EAliveState
    {
    	E_YES,
    	E_NO ,
    	E_DONTKNOW;
    }
    
    //-------------------------------------------------------------------------
    public EAliveState isAlive()
        throws Exception
    {
		final ContextAwareLogHelp aLog = mem_Log ();
		try
    	{
    		LOG.info(aLog.forLevel(ELogLevel.E_INFO)
    				     .toLog   ("check if worker is alive ..."));

	        ClientResponse aResponse = mem_Service().path (CalcEngineConst.CALCENGINE_CONTEXTPATH)
	                							    .path (CalcEngineConst.RELPATH_ISALIVE       )
	                							    .post (ClientResponse.class                  );
	
			boolean bOK = (aResponse.getClientResponseStatus() == ClientResponse.Status.OK);
			if (bOK)
			{
	    		LOG.info(aLog.forLevel(ELogLevel.E_INFO)
   				             .toLog   ("... yes : worker is alive"));
				return EAliveState.E_YES;
			}
			else
			{
	    		LOG.error(aLog.forLevel(ELogLevel.E_ERROR)
   				              .toLog   ("... no : worker is not alive"));
			    return EAliveState.E_NO;
			}
		}
		catch (ClientHandlerException exClient)
		{
			final Throwable exCause = exClient.getCause();
			if (exCause instanceof java.net.ConnectException)
				return EAliveState.E_NO;

    		LOG.error(aLog.forLevel(ELogLevel.E_ERROR)
			              .toLog   ("Client handler exception on isAlive :"), exClient);
		}
    	catch (Throwable ex)
    	{
    		LOG.error(aLog.forLevel(ELogLevel.E_ERROR)
		                  .toLog   ("General exception on isAlive :"), ex);
    	}

		LOG.warn(aLog.forLevel(ELogLevel.E_WARNING)
		             .toLog   ("... ??? : dont know if worker is alive - got unknown exception"));
		return EAliveState.E_DONTKNOW;
    }

    //-------------------------------------------------------------------------
    public boolean shutdown()
        throws Exception
    {
		final ContextAwareLogHelp aLog = mem_Log ();
    	try
    	{
    		LOG.info(aLog.forLevel(ELogLevel.E_INFO)
				         .toLog   ("trigger shutdown on worker ..."));
	        ClientResponse aResponse = mem_Service().path (CalcEngineConst.CALCENGINE_CONTEXTPATH)
	                							    .path (CalcEngineConst.RELPATH_SHUTDOWN      )
	                							    .post (ClientResponse.class                  );

	        final ClientResponse.Status aState = aResponse.getClientResponseStatus();
	        final boolean               bOK    = (aState == ClientResponse.Status.OK);
			if (bOK == false)
			{
	    		LOG.error(aLog.forLevel(ELogLevel.E_ERROR)
				              .toLog   ("... shutdown of worker failed. response was ["+aState+"]"));
				return false;
			}
		}
    	catch (final ClientHandlerException ex)
    	{
    		final Throwable exCause = ex.getCause();
    		if (exCause instanceof java.net.SocketTimeoutException)
    		{
	    		LOG.error(aLog.forLevel(ELogLevel.E_ERROR)
			                  .toLog   ("... shutdown of worker timed out"));
    			return false;
    		}
    		else
    		if (exCause instanceof java.net.ConnectException)
    		{
	    		LOG.warn(aLog.forLevel(ELogLevel.E_WARNING)
			                 .toLog   ("... shutdown of worker .. didnt found any worker (will be ignored silently)"));
    			return true;
    		}

    		throw ex;
    	}
    	
		LOG.info(aLog.forLevel(ELogLevel.E_INFO)
		             .toLog   ("... check if shutdown was successfully"));

		final long nStart    = System.currentTimeMillis();
		final long nTimeOver = nStart + 1000;

		while (true)
		{
			final EAliveState eState = isAlive();
			if (eState == EAliveState.E_NO)
			{
				LOG.info(aLog.forLevel(ELogLevel.E_INFO)
			                 .toLog   ("... shutdown was successfully"));
				return true;
			}
			
			final long nNow = System.currentTimeMillis();
			if (nNow > nTimeOver)
			{
				LOG.error(aLog.forLevel(ELogLevel.E_ERROR)
			                  .toLog   ("... shutdown needs to long ?!"));
				return false;
			}

			synchronized (this)
			{
				wait (100);
			}
		}
    }

    //-------------------------------------------------------------------------
    public void kill()
        throws Exception
    {
		final ContextAwareLogHelp aLog = mem_Log ();

    	try
    	{
    		LOG.info(aLog.forLevel(ELogLevel.E_INFO)
			             .toLog   ("trigger kill on worker ..."));
	        mem_Service().path (CalcEngineConst.CALCENGINE_CONTEXTPATH)
	                	 .path (CalcEngineConst.RELPATH_KILL          )
	                	 .post (ClientResponse.class                  );
	
			synchronized (this)
			{
				wait (500);
			}
    	}
		catch (ClientHandlerException exIgnore)
		{
			// IGNORE those jersey exception here !
			// Killing the process will produce a ConnectionRefusedException always .-)
		}
    }

    //-------------------------------------------------------------------------
    private ContextAwareLogHelp mem_Log ()
        throws Exception
    {
    	if (m_aLog == null)
    	{
    		final ContextAwareLogHelp aLog  = new ContextAwareLogHelp (LOG);
    		final String              sHost = m_aDescriptor.m_sHost;
  	      	final int                 nPort = m_aDescriptor.m_nPort;
  	      	
  	      	aLog.enterContext(CalcEngineLogContextConst.CONTEXT_WORKER_HOST,                  sHost );
  	      	aLog.enterContext(CalcEngineLogContextConst.CONTEXT_WORKER_PORT, Integer.toString(nPort));

  	      	m_aLog = aLog;
    	}
    	return m_aLog;
    }
    
    //-------------------------------------------------------------------------
    private ClientConfig mem_ClientConfig ()
        throws Exception
    {
        if (m_aClientConfig == null)
        {
        	final CalcEngineConfig    aCEConfig       = CalcEngineConfig.get();
        	final int                 nRequestTimeout = aCEConfig.getRequestTimeout();
            final DefaultClientConfig aConfig         = new DefaultClientConfig ();

            // ensure OUR own entity reader/writer pair is know inside Jersey runtime .-)
            Set< Class< ? > > lClasses = aConfig.getClasses();
            lClasses.add(CalcEngineHttpEntityWriter.class);
            lClasses.add(CalcEngineHttpEntityReader.class);
            
            Map< String, Object > lProps = aConfig.getProperties();
            lProps.put(ClientConfig.PROPERTY_CONNECT_TIMEOUT, nRequestTimeout);
            lProps.put(ClientConfig.PROPERTY_READ_TIMEOUT   , nRequestTimeout);

            m_aClientConfig = aConfig;
        }
        return m_aClientConfig;
    }

    //-------------------------------------------------------------------------
    private Client mem_Client ()
        throws Exception
    {
        if (m_aClient == null)
        {
            Client      aClient = Client.create(mem_ClientConfig ());
            m_aClient = aClient;
        }
        return m_aClient;
    }
 
    //-------------------------------------------------------------------------
    private WebResource mem_Service ()
        throws Exception
    {
        if (m_aService == null)
        {
    		final ContextAwareLogHelp aLog    = mem_Log ();
        	final CalcEngineConfig    aConfig = CalcEngineConfig.get ();
        	      String              sHost   = m_aDescriptor.m_sHost;
        	      int                 nPort   = m_aDescriptor.m_nPort;

        	if (StringUtils.isEmpty(sHost))
        		sHost = aConfig.getServerHost(1);

        	if (nPort < 1)
        		nPort = aConfig.getServerPort(1);
        	
        	Validate.notEmpty(sHost  , "No host configured.");
        	Validate.isTrue  (nPort>0, "No port configured.");

        	final String sBaseURI  = "http://"+sHost+":"+nPort;
        	LOG.info(aLog.forLevel(ELogLevel.E_INFO)
        			     .toLog   ("use calc engine remote at '"+sBaseURI+"' ..."));
            m_aService = mem_Client().resource(UriBuilder.fromUri(sBaseURI).build());
        }
        return m_aService;
    }
    
    //-------------------------------------------------------------------------
    private ContextAwareLogHelp m_aLog = null;
    
    //-------------------------------------------------------------------------
    private CalcEngineDescriptor m_aDescriptor = null;

    //-------------------------------------------------------------------------
    private ClientConfig m_aClientConfig = null;

    //-------------------------------------------------------------------------
    private Client m_aClient = null;

    //-------------------------------------------------------------------------
    private WebResource m_aService = null;
}