/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;

import com.hazelcast.core.HazelcastInstance;
import com.openexchange.office.tools.ClassContextAwareSerializationUtils;
import com.openexchange.office.tools.cache.HazelcastCache;
import com.openexchange.office.tools.cache.ICache;
import com.openexchange.office.tools.logging.LogFactory;
import com.openexchange.server.ServiceLookup;

//=============================================================================
public class CalcEngineClipBoard
{
	//-------------------------------------------------------------------------
	private static final Log LOG = LogFactory.getJclLog(CalcEngineClipBoard.class);
	
	//-------------------------------------------------------------------------
	private CalcEngineClipBoard ()
	{}

	//-------------------------------------------------------------------------
	public static synchronized CalcEngineClipBoard forUser (final String sUserId)
	    throws Exception
	{
		Validate.notEmpty(sUserId, "Invalid parameter 'userId'. Has not to be empty nor null.");
		
		final Map< String, CalcEngineClipBoard > aRegistry  = mem_ClipBoardRegistry ();
		      CalcEngineClipBoard                aClipBoard = aRegistry.get  (sUserId);
		      
		if (aClipBoard == null)
		{
			aClipBoard           = new CalcEngineClipBoard ();
			aClipBoard.m_sUserId = sUserId;
			aRegistry.put(sUserId, aClipBoard);
		}

		return aClipBoard;
	}

	//-------------------------------------------------------------------------
	public static String newHandle ()
	    throws Exception
	{
		final String sHandle = UUID.randomUUID().toString();
		LOG.info ("new handle = '"+sHandle+"'");
		return sHandle;
	}
	
	//-------------------------------------------------------------------------
	public static boolean isHandleValid (final String sHandle)
	    throws Exception
	{
		if (StringUtils.isEmpty(sHandle))
			return false;
		
		try
		{
			UUID.fromString(sHandle);
		}
		catch (IllegalArgumentException exInvalid)
		{
			return false;
		}
		
		return true;
	}

	//-------------------------------------------------------------------------
	public synchronized String getUserId ()
	    throws Exception
	{
		return m_sUserId;
	}
	
	//-------------------------------------------------------------------------
	public synchronized void setData (final String                   sHandle,
									  final CalcEngineClipBoardEvent aData  )
	    throws Exception
	{
		Validate.isTrue (isHandleValid (sHandle), "Invalid argument 'handle'");
		
		LOG.info("cache data '"+aData+"' for handle ["+sHandle+"].");
		
		final CalcEngineClipBoardEvent aDataClone = new CalcEngineClipBoardEvent(aData);
		final CalcEngineClipboardItem  aItem      = CalcEngineClipboardItem.create(sHandle, aDataClone);

		final ICache iCache = mem_Cache ();
		final String sItem  = ClassContextAwareSerializationUtils.serializeObject(aItem);
		iCache.setCacheItem(m_sUserId, sItem);
	}

	//-------------------------------------------------------------------------
	public synchronized CalcEngineClipBoardEvent getData (final String sHandle)
	    throws Exception
	{
		Validate.isTrue (isHandleValid (sHandle), "Invalid argument 'handle'");
		
		final ICache                  iCache = mem_Cache ();
		final String                  sItem  = iCache.getCacheItem(m_sUserId);
		final CalcEngineClipboardItem aItem  = ClassContextAwareSerializationUtils.deserializeObject(sItem, this);

		if (aItem == null)
			throw new RuntimeException ("No clip board data available.");

		final String                   sCachedHandle = aItem.getHandle();
		final CalcEngineClipBoardEvent aCachedEvent  = aItem.getData  ();
		
		Validate.notNull (aItem                                     , "Invalid handle 'handle'. No data available ..."    );
		Validate.isTrue  (StringUtils.equals(sCachedHandle, sHandle), "Miss match. No data for this handle : 'handle' ...");
		Validate.notNull (aCachedEvent                              , "Valid handle 'handle' - but no data available ..." );

		LOG.info("return cached data '"+aCachedEvent+"' for handle ["+sCachedHandle+"].");
		return aCachedEvent;
	}

	//-------------------------------------------------------------------------
	private static synchronized Map< String, CalcEngineClipBoard > mem_ClipBoardRegistry ()
	    throws Exception
	{
		if (m_lClipBoardRegistry == null)
			m_lClipBoardRegistry = new HashMap< String, CalcEngineClipBoard > ();
		return m_lClipBoardRegistry;
	}
	
	//-------------------------------------------------------------------------
	private ICache mem_Cache ()
	    throws Exception
	{
		if (m_iCache == null)
		{
			final CalcEngineContext aCEContext = CalcEngineContext.get();
			final ServiceLookup     aSmgr      = aCEContext.accessServiceLookup();
			final HazelcastInstance aHZInst    = aSmgr.getService(HazelcastInstance.class);

			Validate.notNull(aHZInst, "Miss service of type 'HazelcastInstance'.");

			m_iCache = HazelcastCache.create(aHZInst);
		}
		return m_iCache;
	}
	
	//-------------------------------------------------------------------------
	private static Map< String, CalcEngineClipBoard > m_lClipBoardRegistry = null;
	
	//-------------------------------------------------------------------------
	private String m_sUserId = null;

	//-------------------------------------------------------------------------
	private ICache m_iCache = null;
}
