/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.jms;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.apache.commons.lang.mutable.MutableInt;

//=============================================================================
/** Must be used to 'manage' a set of queues for a defined work flow.
 *  E.g. those you can manage a combination of request/response queues;
 *  handle stickiness for queues; define error or timeout queues etcpp.
 *  
 *  If you provide queue performance data to this manager ...
 *  it can be used as load balancer mapping new or sticky requests
 *  to queues.
 * 
 *  @note	These manager handle queue names only !
 *  		It's not (directly) bound to queues ...
 */
public class JMSQueueManager
{
    //-------------------------------------------------------------------------
	/*
	 	LAYOUT :

		client-01              				->	QUEUE_WORKER_01_IN	-> worker-01
		client-01	<-	QUEUE_CLIENT_01_IN  <-						   worker-01
												QUEUE_WORKER_01_ERR	<- worker-01 (???)

		client-02              				->	QUEUE_WORKER_01_IN	-> worker-01
		client-02	<-	QUEUE_CLIENT_02_IN  <-						   worker-01
												QUEUE_WORKER_01_ERR	<- worker-01 (???)

		client-03              				->	QUEUE_WORKER_02_IN	-> worker-02
		client-03	<-	QUEUE_CLIENT_03_IN  <-						   worker-02
												QUEUE_WORKER_02_ERR	<- worker-02 (???)

	 	=> CONSUMER-MAP :

		QUEUE_CLIENT_01_IN	->	client-01
		QUEUE_CLIENT_02_IN	->	client-02
		QUEUE_CLIENT_03_IN	->	client-03

		QUEUE_WORKER_01_IN	->	worker-01
		QUEUE_WORKER_02_IN	->	worker-02

	 	=> PRODUCER-MAP :

		will be dynamic ... and so not registered !

		=> STICKY-MAP :

		dochandle-01	->		QUEUE_WORKER_01_IN
		dochandle-02	->		QUEUE_WORKER_01_IN
		dochandle-03	->		QUEUE_WORKER_02_IN

	 */

    //-------------------------------------------------------------------------
	/** made private to force using of our static factory method !
	 * 
	 * @see {@link create}
	 */
	private JMSQueueManager ()
		throws Exception
	{}

    //-------------------------------------------------------------------------
	/** create new manager instances ...
	 *  for the given queue set.
	 *  
	 *  A queue set define is nothing more then a 'name space' for queues.
	 *  All queues handled by this manager are relative to that name space.
	 * 
	 *  @param	sQueueSetNamespace [IN]
	 *  		the name space to be used by the manager returned.
	 *          Can be null or empty to use the 'global' name space.
	 * 
	 *  @return a manager for the given name space.
	 */
	public static synchronized JMSQueueManager create (final String sQueueSetNamespace)
		throws Exception
	{
		final Map< String, JMSQueueManager > aManagerRegistry = mem_Registry ();
		      JMSQueueManager                aManager         = null;
		
		if (aManagerRegistry.containsKey(sQueueSetNamespace))
			aManager = aManagerRegistry.get(sQueueSetNamespace);
		
		if (aManager == null)
		{
			aManager = new JMSQueueManager ();
			aManager.m_sQueueSetNamespace = sQueueSetNamespace;
			aManagerRegistry.put(sQueueSetNamespace, aManager);
		}
		
		return aManager;
	}

	//-------------------------------------------------------------------------
	public /* synchronized */ String newConsumerIDForType (final String sType)
	    throws Exception
	{
		final String sID         = UUID.randomUUID().toString();
		final String sConsumerID = sType + "-" + sID;
		return sConsumerID;
	}
	
	//-------------------------------------------------------------------------
	public String getQueueNameAbsolute (final String sQueue)
	    throws Exception
	{
		String sName = null;

		if (StringUtils.isEmpty(m_sQueueSetNamespace))
			sName = sQueue;
		else
			sName = buildQueueName (m_sQueueSetNamespace, sQueue);

		return sName;
	}

	//-------------------------------------------------------------------------
	public static String buildQueueName (final String...lParts)
	    throws Exception
	{
		final StringBuffer sName         = new StringBuffer (256);
		      boolean      bAddSeparator = false;

		for (String sPart : lParts)
		{
			if (bAddSeparator)
				sName.append ("-");
			else
				bAddSeparator = true;
			
			sName.append (sPart);
		}
		return sName.toString ();
	}

	//-------------------------------------------------------------------------
	public synchronized void bindQueue2Type (final String  sQueue,
										     final String  sType )
	    throws Exception
	{
		final Map< String, List< String > > aMap            = mem_QueueTypeMap ();
 		      List< String >                aQueueList4Type = aMap.get (sType);

 		if (aQueueList4Type == null)
 		{
 			aQueueList4Type = new ArrayList< String > ();
 			aMap.put(sType, aQueueList4Type);
 		}
 		
 		if ( ! aQueueList4Type.contains(sQueue))
 			aQueueList4Type.add (sQueue);
	}
	
	//-------------------------------------------------------------------------
	public synchronized List< String > getQueues4Type (final String sType)
	    throws Exception
	{
		final Map< String, List< String > > aMap            = mem_QueueTypeMap ();
	          List< String >                aQueueList4Type = aMap.get (sType);
		
	    // return empty list instead of null ...
        // can be handled by outside code better :-)
		if (aQueueList4Type == null)
			aQueueList4Type = new ArrayList< String > ();
		
		return aQueueList4Type;
	}
	
	//-------------------------------------------------------------------------
	public synchronized String getBalancedQueue4Type (final String sType)
	    throws Exception
	{
		Validate.notEmpty(sType, "Invalid argument 'type'.");
		
		final Map< String, List< String > > aMap            = mem_QueueTypeMap ();
	          List< String >                aQueueList4Type = aMap   .get (sType);
		
	    // a) no queues for type ... no balanced queue for return

		if (CollectionUtils.isEmpty(aQueueList4Type))
			return null;

		// b) one queue only .. no load balancing needed - nor possible .-)

		if (aQueueList4Type.size() == 1)
			return aQueueList4Type.get(0);
		
		// c) multiple queues available ...
		//    use round robin counter to select next queue.

		final Map< String, MutableInt > aRRBMap     = m_aRoundRobinBalancerMap4Types ();
		      MutableInt				aRRBPointer = aRRBMap.get (sType);
		  
		if (aRRBPointer == null)
		{
			aRRBPointer = new MutableInt (0);
			aRRBMap.put(sType, aRRBPointer);
		}

		final int    nMaxIndex = aQueueList4Type.size()-1;
		final int    nIndex    = aRRBPointer.intValue();
		final String sQueue    = aQueueList4Type.get(nIndex);
		
		if (nIndex < nMaxIndex)
			aRRBPointer.increment();
		else
			aRRBPointer.setValue(0);
		
		return sQueue;
	}

	//-------------------------------------------------------------------------
	public synchronized void bindConsumer2Queue (final String sConsumerID,
											     final String sQueue     )
	    throws Exception
	{
		//throw new UnsupportedOperationException ("not implemented yet");
	}

	//-------------------------------------------------------------------------
	public void dump ()
	    throws Exception
	{
		final StringBuffer sDump = new StringBuffer (256);
		
		sDump.append ("QUEUE MANAGER DUMP :\n");
		
		sDump.append ("\tTYPE-MAP :\n");
		
		final Map< String, List< String > >                aTypeMap = mem_QueueTypeMap ();
		final Iterator< Entry < String, List< String > > > pIt      = aTypeMap.entrySet().iterator();
		while (pIt.hasNext())
		{
			final Entry < String, List< String > > aEntry  = pIt.next();
			final String                           sType   = aEntry.getKey();
			final List< String >                   lQueues = aEntry.getValue();
			
			sDump.append ("\ttype='"+sType+"' => queues={"+lQueues+"}\n");
		}
		
		sDump.append ("\n");
		
		System.out.println (sDump.toString ());
	}
	
	//-------------------------------------------------------------------------
	private static Map< String, JMSQueueManager > mem_Registry ()
	    throws Exception
	{
		if (m_aRegistry == null)
			m_aRegistry = new HashMap< String, JMSQueueManager > ();
		return m_aRegistry;
	}
	
    //-------------------------------------------------------------------------
	private Map< String, List< String > > mem_QueueTypeMap ()
	    throws Exception
	{
		if (m_aQueueTypeMap == null)
			m_aQueueTypeMap = new HashMap< String, List< String > > ();
		return m_aQueueTypeMap;
	}
	
	//-------------------------------------------------------------------------
	private Map< String, MutableInt > m_aRoundRobinBalancerMap4Types ()
	    throws Exception
	{
		if (m_aRoundRobinBalancerMap4Types == null)
			m_aRoundRobinBalancerMap4Types = new HashMap< String, MutableInt > ();
		return m_aRoundRobinBalancerMap4Types;
	}
	
    //-------------------------------------------------------------------------
	private static Map< String, JMSQueueManager > m_aRegistry = null;
	
    //-------------------------------------------------------------------------
	private String m_sQueueSetNamespace = null;
	
    //-------------------------------------------------------------------------
	private Map< String, List< String > > m_aQueueTypeMap = null;

	//-------------------------------------------------------------------------
	private Map< String, MutableInt > m_aRoundRobinBalancerMap4Types = null;
}
