package com.openexchange.office.tools;

import org.apache.commons.lang.Validate;
import com.openexchange.context.ContextService;
import com.openexchange.groupware.contexts.Context;
import com.openexchange.groupware.ldap.User;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.user.UserService;

//=============================================================================
/** helper to handle seesions and it's values more comfortable.
 */
public class SessionUtils
{
	//-------------------------------------------------------------------------
	/** helper with static functions only do not need a ctor .-)
	 */
	private SessionUtils ()
	{}

	//-------------------------------------------------------------------------
	/** extract unique session ID from given session.
	 *
	 *  @param	aSession [IN]
	 *  		the session where those session ID has to be retrieved from.
	 *
	 *  @return the unique session ID.
	 *
	 *  @throws an exception in case session is not valid ...
	 *  		or session ID could not be retrieved successfully.
	 */
	public static String getSessionId (final Session aSession)
	    throws Exception
	{
		Validate.notNull(aSession, "Invalid argument 'session'.");
		final String sId = aSession.getSessionID();
		Validate.notEmpty(sId, "Invalid session ID.");
		return sId;
	}

	//-------------------------------------------------------------------------
	/** extract unique user ID from given session.
	 *
	 *  @param	aSession [IN]
	 *  		the session where those user ID has to be retrieved from.
	 *
	 *  @return the unique user ID.
	 *
	 *  @throws an exception in case session is not valid ...
	 *  		or user ID could not be retrieved successfully.
	 */
	public static int getUserId (final Session aSession)
	    throws Exception
	{
		Validate.notNull(aSession, "Invalid argument 'session'.");
		final int nUserId = aSession.getUserId();
		return nUserId;
	}

	//-------------------------------------------------------------------------
	/** extract unique (user) context ID from given session.
	 *
	 *  @param	aSession [IN]
	 *  		the session where those context ID has to be retrieved from.
	 *
	 *  @return the unique context ID.
	 *
	 *  @throws an exception in case session is not valid ...
	 *  		or context ID could not be retrieved successfully.
	 */
	public static int getContextId (final Session aSession)
	    throws Exception
	{
		Validate.notNull(aSession, "Invalid argument 'session'.");
		final int nContextId = aSession.getContextId();
		return nContextId;
	}

	//-------------------------------------------------------------------------
	/** extract unique user ID from given session.
	 *  and return it's string representation.
	 *
	 *  @param	aSession [IN]
	 *  		the session where those user ID has to be retrieved from.
	 *
	 *  @return the unique user ID as string.
	 *
	 *  @throws an exception in case session is not valid ...
	 *  		or user ID could not be retrieved successfully.
	 */
	public static String getUserIdString (final Session aSession)
	    throws Exception
	{
		Validate.notNull(aSession, "Invalid argument 'session'.");
		final int    nUserId = SessionUtils.getUserId(aSession);
		final String sUserId = Integer.toString(nUserId);
		return sUserId;
	}

	//-------------------------------------------------------------------------
	/** extract unique (user) context ID from given session
	 *  and return it's string representation.
	 *
	 *  @param	aSession [IN]
	 *  		the session where those context ID has to be retrieved from.
	 *
	 *  @return the unique context ID as string.
	 *
	 *  @throws an exception in case session is not valid ...
	 *  		or context ID could not be retrieved successfully.
	 */
	public static String getContextIdString (final Session aSession)
	    throws Exception
	{
		Validate.notNull(aSession, "Invalid argument 'session'.");
		final int    nContextId = SessionUtils.getContextId(aSession);
		final String sContextId = Integer.toString(nContextId);
		return sContextId;
	}

	//-------------------------------------------------------------------------
	/** analyze the given session and return an User object which
	 *  describe the user more in it's details.
	 *
	 *  @param	aSession [IN]
	 *  		the session where those user info has to be retrieved from.
	 *
	 *  @return the user info object.
	 *
	 *  @throws an exception in case session is not valid ...
	 *  		or user info could not be retrieved successfully.
	 */
	public static User getUserInfo (final Session       aSession      ,
								    final ServiceLookup aServiceLookup)
	    throws Exception
	{
		Validate.notNull(aSession      , "Invalid argument 'session'."      );
		Validate.notNull(aServiceLookup, "Invalid argument 'serviceLookup'.");

    	final UserService    aUserService    = aServiceLookup.getService(UserService   .class);
    	final ContextService aContextService = aServiceLookup.getService(ContextService.class);

		Validate.notNull(aUserService   , "Miss service '"+UserService   .class.getName ()+"'.");
		Validate.notNull(aContextService, "Miss service '"+ContextService.class.getName ()+"'.");

		final int  nUserId    = SessionUtils.getUserId    (aSession);
		final int  nContextId = SessionUtils.getContextId (aSession);
		final User aUserInfo  = aUserService.getUser(nUserId, nContextId);

		return aUserInfo;
	}

	/**
     *  Analyze the given session and return an User object which
     *  describe the user more in it's details.
	 *
     * @param  aSession [IN]
     *         the session where those user info has to be retrieved from.
     *
	 * @param nUserId [IN]
	 *         a user id that should be used to retrieve the user data.
	 *
	 * @param aServiceLookup
	 *         the reference to be used to create needed services.
	 *
     * @return the user info object.
     *
     * @throws an exception in case session is not valid ...
     *         or user info could not be retrieved successfully.
	 */
    public static User getUserInfo (final Session       aSession      ,
                                    final int           nUserId       ,
                                    final ServiceLookup aServiceLookup)
        throws Exception
    {
        Validate.notNull(aSession      , "Invalid argument 'session'."      );
        Validate.notNull(aServiceLookup, "Invalid argument 'serviceLookup'.");

        final UserService    aUserService    = aServiceLookup.getService(UserService   .class);
        final ContextService aContextService = aServiceLookup.getService(ContextService.class);

        Validate.notNull(aUserService   , "Miss service '"+UserService   .class.getName ()+"'.");
        Validate.notNull(aContextService, "Miss service '"+ContextService.class.getName ()+"'.");

        final int  nContextId = SessionUtils.getContextId (aSession);
        final User aUserInfo  = aUserService.getUser(nUserId, nContextId);

        return aUserInfo;
    }

    //-------------------------------------------------------------------------
	/** analyze the given session and return an Context object which
	 *  describe the user context more in it's details.
	 *
	 *  @param	aSession [IN]
	 *  		the session where those context info has to be retrieved from.
	 *
	 *  @return the context info object.
	 *
	 *  @throws an exception in case session is not valid ...
	 *  		or user context could not be retrieved successfully.
	 */
	public static Context getContextInfo (final Session       aSession      ,
								          final ServiceLookup aServiceLookup)
	    throws Exception
	{
		Validate.notNull(aSession      , "Invalid argument 'session'."      );
		Validate.notNull(aServiceLookup, "Invalid argument 'serviceLookup'.");

    	final ContextService aContextService = aServiceLookup.getService(ContextService.class);

		Validate.notNull(aContextService, "Miss service '"+ContextService.class.getName ()+"'.");

		final int     nContextId = SessionUtils.getContextId (aSession  );
		final Context aContext   = aContextService.getContext(nContextId);

		return aContext;
	}

	//-------------------------------------------------------------------------
	/** extract users name from given session to be used as visible information
	 *  e.g. at the UI.
	 *
	 *  @param	aSession [IN]
	 *  		the session where those user name has to be retrieved from.
	 *
	 *  @return the user name.
	 *
	 *  @throws an exception in case session is not valid ...
	 *  		or user name could not be retrieved successfully.
	 */
	public static String getUserDisplayName (final Session       aSession      ,
								             final ServiceLookup aServiceLookup)
	    throws Exception
	{
		Validate.notNull(aSession      , "Invalid argument 'session'."      );
		Validate.notNull(aServiceLookup, "Invalid argument 'serviceLookup'.");

		final User   aUser = SessionUtils.getUserInfo(aSession, aServiceLookup);
		final String sName = aUser.getDisplayName();

		Validate.notEmpty(sName, "Was not able to retrieve users display name from given session.");
		return sName;
	}

	//-------------------------------------------------------------------------
    /** dump given session to string for debug purposes.
    *
    *   @param	aSession [IN]
    *  			the session to be analyzed here
    *
    *   @return the string representation of those session.
    */
    public static String dumpSessionToString (final Session aSession)
		// no exception here please .-)
    {
		final StringBuffer sDump = new StringBuffer (256);

	   		sDump.append ("#### =================================================\n");
	   		sDump.append ("#### session dump :                                   \n");
	   	try{sDump.append ("#### session      : "+aSession                       +"\n");}catch(final Throwable ignore){}; // catch errors line by line ! we want to get info's as much as possible :-)
	   	try{sDump.append ("#### login        : "+aSession.getLogin()            +"\n");}catch(final Throwable ignore){};
	   	try{sDump.append ("#### login name   : "+aSession.getLoginName()        +"\n");}catch(final Throwable ignore){};
	   	try{sDump.append ("#### user login   : "+aSession.getUserlogin()        +"\n");}catch(final Throwable ignore){};
	   	try{sDump.append ("#### user id      : "+aSession.getUserId()           +"\n");}catch(final Throwable ignore){};
	   	try{sDump.append ("#### context id   : "+aSession.getContextId()        +"\n");}catch(final Throwable ignore){};
	   		sDump.append ("#### =================================================\n");

	   	return sDump.toString ();
    }
}
