/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.encoding.Base64;

/**
 * {@link ResourceManager}
 *
 * @author <a href="mailto:sven.jacobi@open-xchange.com">Sven Jacobi</a>
 */
public class ResourceManager {

    /**
     * Initializes a new {@link ResourceManager}.
     *
     * @param services
     */
    public ResourceManager(ServiceLookup services) {
        m_services = services;
    }

    /**
     *
     */
    void touch() {
        synchronized (m_resources) {
            for (final Iterator<Resource> resourceIter = m_resources.values().iterator(); resourceIter.hasNext();) {
                resourceIter.next().touch();
            }
        }
    }

    /**
     * @param lockResources
     */
    public void lockResources(boolean lock) {
        m_resourceLocker.lockResourceManager(this, lock);
    }

    /**
     * @param base64data
     * @return
     */
    public long addBase64(String base64Data) {
        long uid = 0;

        final String basePattern = "base64,";
        int pos = base64Data.indexOf(basePattern);
        if ((pos != -1) && ((pos += basePattern.length()) < (base64Data.length() - 1))) {
            final byte[] byteData = Base64.decode(base64Data.substring(pos));
            if (byteData != null && byteData.length > 0) {
                uid = addResource(byteData);
            }
        }
        return uid;
    }

    /**
     * @param resdata
     * @return
     */
    public long addResource(byte[] resourceData) {
        return addResource(new Resource(m_services, resourceData));
    }

    /**
     * @param crc32 crc32 of resdata
     * @param resdata
     * @return the unique key: crc32 | (vector index << 32)
     */
    public long addResource(int crc32, byte[] resourceData) {
        return addResource(new Resource(m_services, crc32, resourceData));
    }

    /**
     * @param resource
     * @return the unique key: crc32 | (vector index << 32)
     */
    public long addResource(Resource resource) {
        final Long uidObject = (null != resource) ? Long.valueOf(resource.getUID()) : null;

        if (null != uidObject) {
            synchronized (m_resources) {
                if (!m_resources.containsKey(uidObject)) {
                    m_resources.put(uidObject, resource);
                }
            }
        }

        return ((null != uidObject) ? uidObject.longValue() : 0);
    }

    /**
     * @param uid
     * @return true, if the resource for the given uid is contained within this instance
     */
    public boolean containsResource(long uid) {
        return m_resources.containsKey(Long.valueOf(uid));
    }

    /**

    /**
     * @param uid
     * @return the resource for the given uid or null if the resource couldn't be found
     */
    public Resource getResource(long uid) {
        return m_resources.get(Long.valueOf(uid));
    }

    /**
     * @param id
     * @return the resource for the given ManagedFile id or null if the resource couldn't be found
     */
    public Resource createManagedResource(String managedResourceId) {
        return Resource.createFrom(m_services, managedResourceId);
    }

    /**
     * @return
     */
    public Hashtable<Long, byte[]> getByteArrayTable() {
        Hashtable<Long, byte[]> ret = null;

        synchronized (m_resources) {
            ret = new Hashtable<Long, byte[]>(m_resources.size());

            for (final Long curKey : m_resources.keySet()) {
                final Resource curResource = m_resources.get(curKey);

                if (null != curResource) {
                    ret.put(curKey, curResource.getBuffer());
                }
            }
        }

        return ret;
    }

    // - Members ---------------------------------------------------------------

    static private final ResourceLocker m_resourceLocker = new ResourceLocker();

    private final HashMap<Long, Resource> m_resources = new HashMap<Long, Resource>();

    private ServiceLookup m_services = null;
}
