package com.openexchange.office.tools;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.logging.Log;

import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.File.Field;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.FileStorageFileAccess.SortDirection;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.iterator.SearchIterator;
import com.openexchange.tx.TransactionAwares;


/**
 * A collection of low-level file helper functions.
 *
 * @author Carsten Driesner
 *
 */
public class FileHelper {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(FileHelper.class);

    /**
     * Gets the base name, minus the full path and extension, from a full
     * filename.
     *
     * @param fileName
     *  The file name where the base name should be retrieved from. The
     *  fileName can be null.
     *
     * @return
     *  The base name of the given filename or null if fileName is null.
     */
    static public String getBaseName(String fileName) {
        return FilenameUtils.getBaseName(fileName);
    }

    /**
     * Gets the file name including the extension, minus the full path, from
     * a full path filename.
     *
     * @param fullPathFileName
     *  The filename with  the full path including the extension. The
     *  fullPathFileName can be null.
     *
     * @return
     *  The file name of the given full path file name or null if the
     *  fullPathFileName is null.
     */
    static public String getFileName(String fullPathFileName) {
        return FilenameUtils.getName(fullPathFileName);
    }

    /**
     * Retrieves the extension of a file name.
     *
     * @param fileName
     *  The file name, the extension should be retrieved from.
     *
     * @return
     *  The extension of the given filename or an empty string in case of
     *  (null == fileName) or if no extension is set at all
     */
    static public String getExtension(String fileName) {
        String extension = "";

        if (null != fileName) {
            int beginIndex = fileName.lastIndexOf(".");
            int endIndex = beginIndex;

            if (beginIndex > -1) {
                while (++endIndex < fileName.length()) {
                    int c = fileName.codePointAt(endIndex);

                    if (!(((c >= 'a') && (c <= 'z')) || ((c >= 'A') && (c <= 'Z'))))
                        break;
                }
            }

            if ((beginIndex > -1) && (endIndex <= fileName.length()) && (beginIndex < (endIndex - 1))) {
                extension = fileName.substring(beginIndex + 1, endIndex);
            }
        }

        return extension.toLowerCase();
    }

    /**
     * Creates a OX Drive file and writes the provided input stream to it.
     *
     * @param session
     *  The session of the user who requested to create a new file.
     *
     * @param fileAccess
     *  The file access instance to be used by this method. Can be null
     *  where the method creates it's own file access.
     *
     * @param file
     *  The new file instance, which is correctly initialized.
     *
     * @param inputStream
     *  The input stream of the document to be written. Attention: The stream
     *  is not closed by the method. So the caller is responsible to close
     *  the stream.
     *
     * @return
     *  The error code of the operation which is ErrorCode.NO_ERROR on success
     *  or any other code in case of an error.
     */
    public static ErrorCode createFileAndWriteStream(final ServiceLookup services, final Session session, final IDBasedFileAccess fileAccess, final File file, final InputStream inputStream) {
        IDBasedFileAccess myFileAccess = fileAccess;
        if (null == myFileAccess) {
            final IDBasedFileAccessFactory fileFactory = services.getService(IDBasedFileAccessFactory.class);
            myFileAccess = fileFactory.createAccess(session);
        }
        return writeFileStream(myFileAccess, file, inputStream);
    }

    /**
     *
     * @param fileAccess
     * @param folder_id
     * @param fileName
     * @param mimeType
     * @param inputStream
     * @return
     */
    public static ErrorCode createFileWithStream(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName, final String mimeType, final InputStream inputStream) {
        // create default file for creating new file
        final DefaultFile metaData = new DefaultFile();

        // set new file attributes - using old file as template
        metaData.setId(FileStorageFileAccess.NEW);
        metaData.setFolderId(folder_id);
        metaData.setFileName(fileName);
        metaData.setFileMIMEType(mimeType);

        return writeFileStream(fileAccess, metaData, inputStream);
    }

    /**
     *
     * @param fileAccess
     * @param fileName
     * @param sourceFile
     * @param inputStream
     * @return
     */
    public static ErrorCode createFileCopyWithStream(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName, final File sourceFile, final InputStream inputStream, boolean copyMetaData) {
        // create default file for creating new file
        final DefaultFile metaData = new DefaultFile();

        // set new file attributes - using old file as template
        metaData.setId(FileStorageFileAccess.NEW);
        metaData.setFolderId(folder_id);
        metaData.setFileName(fileName);
        metaData.setFileMIMEType(sourceFile.getFileMIMEType());
        // take over modification, creator and dates from the source file
        if (copyMetaData) {
            metaData.setCreatedBy(sourceFile.getCreatedBy());
            metaData.setModifiedBy(sourceFile.getModifiedBy());
            metaData.setCreated(sourceFile.getCreated());
            metaData.setLastModified(sourceFile.getLastModified());
        }

        return writeFileStream(fileAccess, metaData, inputStream);
    }

    /**
     *
     * @param fileAccess
     * @param file
     * @param inputStream
     * @return
     */
    public static ErrorCode writeStreamToFile(final IDBasedFileAccess fileAccess, final File targetFile, final InputStream inputStream, final File copyMetaData) {
        final DefaultFile metaData = new DefaultFile();

        // set necessary meta data to overwrite an existing file
        metaData.setId(targetFile.getId());
        metaData.setFolderId(targetFile.getFolderId());
        metaData.setFileName(targetFile.getFileName());
        metaData.setFileMIMEType(targetFile.getFileMIMEType());
        // take over modification, creator and dates from the source file
        if (null != copyMetaData) {
            metaData.setCreatedBy(copyMetaData.getCreatedBy());
            metaData.setModifiedBy(copyMetaData.getModifiedBy());
            metaData.setCreated(copyMetaData.getCreated());
            metaData.setLastModified(copyMetaData.getLastModified());
        }

        return writeFileStream(fileAccess, metaData, inputStream);
    }

    /**
     *
     * @param fileAccess
     * @param folder_id
     * @param fileName
     * @return
     */
    public static SearchIterator<File> getMetaDataFromFileName(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName) throws OXException {
        final List<Field> fields = new ArrayList<Field>();
        fields.add(Field.ID);
        fields.add(Field.FOLDER_ID);
        fields.add(Field.FILENAME);
        fields.add(Field.FILE_MIMETYPE);
        fields.add(Field.VERSION);

        return getMetaDataFromFileName(fileAccess, folder_id, fileName, fields);
    }

    /**
     *
     * @param fileAccess
     * @param folder_id
     * @param fileName
     * @param fields
     * @return
     * @throws OXException
     */
    public static SearchIterator<File> getMetaDataFromFileName(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName, final List<Field> fields) throws OXException {
        return fileAccess.search(fileName, fields, folder_id, null, SortDirection.DEFAULT, FileStorageFileAccess.NOT_SET, FileStorageFileAccess.NOT_SET);
    }

    /**
     *
     * @param fileAccess
     * @param folder_id
     * @param fileName
     * @return
     * @throws OXException
     */
    public static boolean fileNameExists(final IDBasedFileAccess fileAccess, final String folder_id, final String fileName) throws OXException {
        final List<Field> fields = new ArrayList<Field>();
        fields.add(Field.ID);
        fields.add(Field.FILENAME);

        SearchIterator<File> iter = fileAccess.search(fileName, fields, folder_id, null, SortDirection.DEFAULT, -1, -1);
        return (iter != null) && (!iter.hasNext());
    }

    /**
     * Create a file name with postfix string part.
     *
     * @param fileName
     *  A file name with extension and without path part.
     *
     * @return
     *  A file name with consists of: <filename><postFix>.<ext>
     */
    public static String createFilenameWithPostfix(final String fileName, final String postFix) {
        final StringBuffer tmp = new StringBuffer(64);
        tmp.append(FileHelper.getBaseName(fileName));
        tmp.append(postFix);
        tmp.append(".");
        tmp.append(getExtension(fileName));
        return tmp.toString();
    }

    /**
     * Writes the stream to the provided file.
     *
     * @param fileAccess
     *   The FileAccess instance writing the file.
     *
     * @param newFile
     *   The new meta file instance. Must be filled with the necessary
     *   properties.
     *
     * @param inputStream
     *   The data to be written.
     *
     * @return
     */
    private static ErrorCode writeFileStream(final IDBasedFileAccess fileAccess, final File fileMetaData, final InputStream inputStream) {
        ErrorCode errorCode = ErrorCode.NO_ERROR;
        boolean rollback = true;

        try {
            LOG.trace("RT connection: [writeFileStream] saveDocument using meta data: " + ((null != fileMetaData) ? fileMetaData.toString() : "null"));
            fileAccess.startTransaction();
            rollback = true;
            fileAccess.saveDocument(fileMetaData, inputStream, FileStorageFileAccess.DISTANT_FUTURE, new ArrayList<Field>());
            fileAccess.commit();
            rollback = false;
        } catch (final Exception e) {
            errorCode = ErrorCode.GENERAL_PERMISSION_CREATE_MISSING_ERROR;
            if (e instanceof OXException) {
                errorCode = ExceptionToErrorCode.map((OXException)e, errorCode, false);
            }
            LOG.error(errorCode.getDescription(), e);
        } finally {
            // Roll-back (if needed) and finish
            if (rollback) {
                TransactionAwares.rollbackSafe(fileAccess);
            }

            TransactionAwares.finishSafe(fileAccess);
        }

        return errorCode;
    }
}
