/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.tools.logging;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.ext.LoggerWrapper;

//=============================================================================
@Aspect
public class LoggingAspect
{
	//-------------------------------------------------------------------------
	public static final String FQCN = LoggingAspect.class.getName();
	
	//-------------------------------------------------------------------------
	public static String LOG_LEVEL = "debug";
	
	//-------------------------------------------------------------------------
	@Pointcut("execution(*.new()) || execution(*.new(..))")  
    private void anyPublicCtor() {}  

	//-------------------------------------------------------------------------
	@Pointcut("execution(public * *(..))")  
    private void anyPublicOperation() {}  
      
	//-------------------------------------------------------------------------
    @Pointcut("within(com.openxchange..*) || within(com.openexchange..*)")  
    private void inOpenXchangePackage() {}  
      
	//-------------------------------------------------------------------------
    @Pointcut("anyPublicCtor() && inOpenXchangePackage()")  
    private void publicOpenXchangeCtors() {}

	//-------------------------------------------------------------------------
    @Pointcut("anyPublicOperation() && inOpenXchangePackage()")  
    private void publicOpenXchangeOperations() {}

    //-------------------------------------------------------------------------
	@Around("publicOpenXchangeCtors()")
	public Object logNewInst(ProceedingJoinPoint aJoinPoint)
		throws Throwable
	{
		final Object aThis = aJoinPoint.getThis();
		final Logger aLog  = impl_getLoggerForLogPoint (aJoinPoint);
		impl_log(aLog, "new instance '"+aThis+"' created ...");
		return aJoinPoint.proceed();
	}

	//-------------------------------------------------------------------------
	@Around("publicOpenXchangeOperations()")
	@SuppressWarnings("deprecation")
	public Object logAround(ProceedingJoinPoint aJoinPoint)
		throws Throwable
	{
		final String sCall = impl_formatMethodCall     (aJoinPoint);
		final Logger aLog  = impl_getLoggerForLogPoint (aJoinPoint);
		
		Object aResult = null;
		try
		{
			impl_log(aLog, sCall);
			aResult = aJoinPoint.proceed ();
			impl_log(aLog, "... return value = ["+ObjectUtils.toString(aResult)+"]");
		}
		catch (Throwable ex)
		{
			aLog.error (ex.getMessage (), ex);
			throw ex;
		}
		
		return aResult;
	}

	//-------------------------------------------------------------------------
	private void impl_log (final Logger aLog,
						   final String sMsg)
		throws Exception 
	{
		if (StringUtils.equals (LOG_LEVEL, "trace"))
			aLog.trace(sMsg);
		else
		if (StringUtils.equals (LOG_LEVEL, "debug"))
			aLog.debug(sMsg);
		else
		if (StringUtils.equals (LOG_LEVEL, "info"))
			aLog.info(sMsg);
		else
		if (StringUtils.equals (LOG_LEVEL, "warn"))
			aLog.warn(sMsg);
		else
		if (StringUtils.equals (LOG_LEVEL, "error"))
			aLog.error(sMsg);
		else
			throw new UnsupportedOperationException ("No support for log level '"+LOG_LEVEL+"' implemented yet.");
	}
	
	//-------------------------------------------------------------------------
	private String impl_formatMethodCall (final JoinPoint aJoinPoint)
		throws Exception
	{
		final Signature aSignature = aJoinPoint.getSignature();
		final String    sClass     = aSignature.getDeclaringTypeName();
		final String    sMethod    = aSignature.getName();
		final String    sArgs      = ArrayUtils.toString(aJoinPoint.getArgs());
		
		final StringBuffer sCall = new StringBuffer (256);
		sCall.append (sClass );
		sCall.append ("."    );
		sCall.append (sMethod);
		sCall.append ("("    );
		sCall.append (sArgs  );
		sCall.append (")"    );
		return sCall.toString ();
	}

	//-------------------------------------------------------------------------
	private Logger impl_getLoggerForLogPoint (final JoinPoint aLogPoint)
		throws Exception
	{
		final String                sType     = impl_getLogPointType (aLogPoint);
		final Map< String, Logger > aRegistry = m_aLoggerRegistry ();
		
		Logger aWrapLogger = aRegistry.get(sType);
		if (aWrapLogger == null)
		{
			Logger aOriginalLogger = LoggerFactory.getLogger (sType);
			       aWrapLogger     = new LoggerWrapper (aOriginalLogger, FQCN);
			aRegistry.put(sType, aWrapLogger);
		}
		
		return aWrapLogger;
	}
	
	//-------------------------------------------------------------------------
	private String impl_getLogPointType (final JoinPoint aLogPoint)
    	throws Exception
    {
		final Signature aSignature = aLogPoint .getSignature        ();
		final String    sClass     = aSignature.getDeclaringTypeName();
    	return sClass;
    }
        
	//-------------------------------------------------------------------------
	private Map< String, Logger > m_aLoggerRegistry ()
	    throws Exception
	{
		if (m_aLoggerRegistry == null)
			m_aLoggerRegistry = new HashMap< String, Logger > ();
		return m_aLoggerRegistry;
	}
	
	//-------------------------------------------------------------------------
	private Map< String, Logger > m_aLoggerRegistry = null; 
}
