/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/celleditmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/popup/tooltip',
    'io.ox/office/tk/popup/listmenu',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/view/popup/signaturetooltip',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, KeyCodes, Forms, ToolTip, ListMenu, Color, HyperlinkUtils, SheetUtils, PaneUtils, CellCollection, TokenArray, SignatureToolTip, gt) {

    'use strict';

    // mix-in class CellEditMixin =============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides extensions for
     * the cell in-place edit mode, and for showing and manipulating
     * highlighted cell ranges based on formula token arrays.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function CellEditMixin(app) {

        var // self reference (spreadsheet view instance)
            self = this,

            // the spreadsheet model, and other model objects
            docModel = null,
            undoManager = null,
            documentStyles = null,
            fontCollection = null,
            numberFormatter = null,

            // the underlay node containing highlighting for formulas
            textUnderlayNode = $('<div>').addClass('underlay'),
            // the text area control
            textAreaNode = $(Forms.createTextAreaMarkup()),
            // the container element for the text underlay node and the text area control
            textAreaContainer = $('<div>', { 'data-focus-role': 'textarea' }).addClass('textarea-container').append(textUnderlayNode, textAreaNode),

            // tooltip attached to the text area for function/parameter help
            signatureToolTip = new SignatureToolTip(app, textAreaNode),
            // drop-down list attached to the text area (for function names)
            textAreaListMenu = new ListMenu({ anchor: textAreaNode, autoFocus: false, focusableNodes: textAreaNode }),
            // tooltip attached to the text area drop-down list (for the selected function)
            textAreaListItemToolTip = new ToolTip({ anchor: textAreaListMenu.getNode(), anchorBorder: 'right left', anchorAlign: 'center', anchorAlignBox: getTextAreaListButtonNode }),

            // whether the in-place cell edit mode is active
            editActive = false,
            // the index of the sheet containing the edited cell (active sheet may change during edit mode)
            editSheet = null,
            // the cell address of the edited cell
            editAddress = null,
            // the grid pane instance where in-place cell edit mode is active
            editGridPane = null,
            // the descriptor of the edited cell (result, attributes, etc.)
            editCellData = null,
            // the token array representing the current formula in the text area
            editTokenArray = null,
            // current character attributes for the text area
            editAttributes = null,

            // local undo/redo information while cell edit mode is active
            undoStack = null,
            // pointer into the local undo/redo stack
            undoIndex = 0,

            // initial text at the beginning of the edit mode
            originalText = null,
            // the value at the time the text area has been updated the last time
            lastValue = null,
            // last committed cell range address, and additional data (e.g. circular reference detected)
            lastCommitData = null,

            // quick edit mode (true), or text cursor mode (false)
            quickEditMode = null,
            // the formula expression, if the input string starts with specific characters
            formulaString = null,
            // selection offset into the formula string (without leading equality sign)
            formulaOffset = 0,
            // whether a matrix formula is currently edited
            matrixFormula = false,
            // whether to show auto-completion list for a function or name
            autoComplete = false,
            // token info for function auto-completion
            autoTokenInfo = null,
            // how many closing parenthesis entered manually will be ignored
            autoCloseData = [],
            // the position of the current range address while in range selection mode
            rangeSelection = null,

            // the token arrays currently used for highlighted ranges
            highlightTokenArrays = [],
            // all options used to highlight cell ranges
            highlightOptions = null;

        // private methods ----------------------------------------------------

        /**
         * Updates the CSS formatting of the text area control used for cell
         * in-place edit mode according to the current formatting of the active
         * cell.
         */
        function updateTextAreaStyle() {

            var // the address of the active cell
                address = editAddress,
                // the cell attributes of the cell
                cellAttributes = null,
                // the character attributes of the cell
                charAttributes = null;

            // applies a transformation to the passed color, and returns the CSS color value
            function getModifiedCssColor(color, type, value) {
                if (Color.isAutoColor(color)) { return ''; }
                color = _.copy(color, true);
                Color.appendTransformation(color, type, value);
                return documentStyles.getCssColor(color);
            }

            // receive cell contents from cell collection
            editCellData = _.copy(self.getCellCollection().getCellEntry(address), true);
            cellAttributes = editCellData.attributes.cell;
            charAttributes = editCellData.attributes.character;

            // overwrite with current edit attributes
            _.extend(cellAttributes, editAttributes.cell);
            _.extend(charAttributes, editAttributes.character);

            // update line height and font size according to current zoom factor
            charAttributes.lineHeight = self.getEffectiveLineHeight(charAttributes);
            charAttributes.fontSize = Math.max(6, self.getEffectiveFontSize(charAttributes.fontSize));

            // set CSS attributes according to formatting of active cell
            textAreaContainer.children().css({
                padding: '0 ' + self.getEffectiveCellPadding() + 'px',
                lineHeight: charAttributes.lineHeight + 'px',
                fontFamily: documentStyles.getCssFontFamily(charAttributes.fontName),
                fontSize: charAttributes.fontSize + 'pt',
                fontWeight: charAttributes.bold ? 'bold' : 'normal',
                fontStyle: charAttributes.italic ? 'italic' : 'normal',
                textDecoration: documentStyles.getCssTextDecoration(charAttributes),
                textAlign: PaneUtils.getCssTextAlignment(editCellData, true)
            });

            // set cell text color and fill color to the appropriate nodes
            textAreaNode.css('color', getModifiedCssColor(charAttributes.color, 'shade', 30000));
            textUnderlayNode.css('background-color', getModifiedCssColor(cellAttributes.fillColor, 'tint', 30000));
        }

        /**
         * Updates the position and size of the text area control for the cell
         * in-place edit mode, according to the current length of the text in
         * the text area control, and the scroll position of the grid pane.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, the position of the text area will be updated
         *      immediately. By default, updating the text area will be done
         *      debounced.
         *  @param {Boolean} [options.size=true]
         *      If set to false, the size of the text area will not be changed,
         *      only its position. Used in situations where the content of the
         *      text area does not change (e.g. while scrolling).
         */
        var updateTextAreaPosition = (function () {

            var // whether updating the position is pending
                updatePosition = false,
                // whether updating the size is pending
                updateSize = false;

            // direct callback, invoked immediately when calling updateEditTokenArray()
            function directCallback(options) {

                // set the pending update flags for the deferred callback
                updatePosition = true;
                updateSize = updateSize || Utils.getBooleanOption(options, 'size', true);

                // invoke deferred callback immediately if specified, or if the text has caused a new line break
                if (Utils.getBooleanOption(options, 'direct', false) || (textAreaNode.outerHeight() < textAreaNode[0].scrollHeight)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                // do nothing if in-place edit mode is not active
                if (!editActive || (!updatePosition && !updateSize)) { return; }

                var // the position of the edited cell
                    position = null,
                    // the visible area of this grid pane
                    visiblePosition = null,
                    // the current character attributes (with zoomed font size)
                    charAttributes = editCellData.attributes.character,
                    // whether the text in the cell wraps automatically
                    isWrapped = CellCollection.isWrappedText(editCellData),
                    // the effective horizontal CSS text alignment
                    textAlign = textAreaNode.css('text-align'),
                    // round up width to multiples of the line height
                    blockWidth = Math.max(20, charAttributes.lineHeight),
                    // the new width and height of the text area
                    targetWidth = 0, targetHeight = 0;

                // returns required width needed to show all text lines in a single line
                function getMinimumWidth() {

                    var // required width for all text lines
                        requiredWidth = _.reduce(textAreaNode.val().split(/\n/), function (memo, textLine) {
                            return Math.max(memo, fontCollection.getTextWidth(textLine, charAttributes));
                        }, 0);

                    // round up to block width, enlarge to cell width
                    return Math.max(position.width, Utils.roundUp(requiredWidth, blockWidth) + blockWidth + Utils.SCROLLBAR_WIDTH);
                }

                // returns maximum width according to horizontal alignment and width of visible area
                function getMaximumWidth() {

                    var // the distance from left visible border to right cell border (for right alignment)
                        leftWidth = Utils.minMax(position.left + position.width - visiblePosition.left, 0, visiblePosition.width),
                        // the distance from left cell border to right visible border (for left alignment)
                        rightWidth = Utils.minMax(visiblePosition.left + visiblePosition.width - position.left, 0, visiblePosition.width),
                        // effective available width
                        availWidth = 0;

                    // get available width according to text alignment
                    switch (textAlign) {
                    case 'right':
                        availWidth = leftWidth;
                        break;
                    case 'center':
                        availWidth = Math.min(leftWidth, rightWidth) * 2 - position.width;
                        break;
                    default:
                        availWidth = rightWidth;
                    }

                    // enlarge to at least half of the width of the visible area
                    return Math.max(availWidth, Math.round(visiblePosition.width / 2));
                }

                // get horizontal and vertical cell position and size
                position = self.getCellRectangle(editAddress, { expandMerged: true });
                // reduce width by one pixel to exclude the grid line
                position.width = Math.max(blockWidth, position.width - 1);
                // reduce height by one pixel to exclude the grid line
                position.height -= 1;

                // get position and size of the visible area in the grid pane
                visiblePosition = editGridPane.getVisibleRectangle();

                if (Utils.CHROME_ON_ANDROID) {
                    // on small android devices softkeyboard can overlay complete gridpane,
                    // then the automatic height calculation has zero height,
                    // after closing the softkeyboard it still keeps this height and that looks really wrong
                    visiblePosition.height = Math.max(visiblePosition.height, charAttributes.lineHeight);
                }

                // calculate and set the new size of the text area
                if (updateSize) {

                    var // find best size, stick to cell width in auto-wrapping mode
                        minWidth = isWrapped ? position.width : getMinimumWidth(),
                        maxWidth = isWrapped ? visiblePosition.width : getMaximumWidth();

                    // calculate width and height of the text area
                    targetWidth = Math.min(minWidth, maxWidth);
                    fontCollection.getCustomFontMetrics(charAttributes, function (helperNode) {

                        var // the text whose height will be determined
                            text = textAreaNode.val();

                        // use NBSP instead of empty text; bug 35143: add NBSP to empty last line
                        if (/(^|\n)$/.test(text)) { text = text + '\xa0'; }

                        // prepare the helper node for multi-line text
                        helperNode.css({
                            width: targetWidth,
                            padding: '0 ' + self.getEffectiveCellPadding() + 'px',
                            lineHeight: charAttributes.lineHeight + 'px',
                            whiteSpace: 'pre-wrap',
                            wordWrap: 'break-word'
                        }).text(text);

                        targetHeight = Math.min(Math.max(helperNode.height(), position.height), visiblePosition.height);
                    });

                    textAreaNode.css({ width: targetWidth, height: targetHeight });

                } else {
                    targetWidth = textAreaNode.outerWidth();
                    targetHeight = textAreaNode.outerHeight();
                }

                // calculate and set the new position of the text area
                if (updatePosition) {

                    var // left and top position of the text area
                        leftOffset = position.left,
                        topOffset = position.top;

                    // calculate horizontal position according to alignment
                    switch (textAlign) {
                    case 'right':
                        leftOffset += position.width - targetWidth;
                        break;
                    case 'center':
                        leftOffset += Math.round((position.width - targetWidth) / 2);
                        break;
                    }

                    // move text area into visible area, set effective position
                    leftOffset = Utils.minMax(leftOffset, visiblePosition.left, visiblePosition.left + visiblePosition.width - targetWidth);
                    topOffset = Utils.minMax(topOffset, visiblePosition.top, visiblePosition.top + visiblePosition.height - targetHeight);
                    textAreaContainer.css({ left: leftOffset - visiblePosition.left, top: topOffset - visiblePosition.top });
                }

                updatePosition = updateSize = false;
            }

            // return the debounced method updateTextAreaPosition()
            return self.createDebouncedMethod(directCallback, deferredCallback, { delay: 250, maxDelay: 500 });
        }());

        /**
         * Returns the selected list item node in the drop-down menu element
         * for the text area.
         *
         * @returns {jQuery}
         *  The selected menu item element in the text area drop-down menu.
         */
        function getTextAreaListButtonNode() {
            return textAreaListMenu.getSelectedItemNodes().first();
        }

        /**
         * Updates and shows or hides the tooltip attached to the text area,
         * containing the name, parameter list, and description of the current
         * function.
         */
        function updateFunctionDescriptionToolTip() {

            var // the selected button element in the auto-completion menu
                buttonNode = getTextAreaListButtonNode(),
                // the name and type of the selected list item
                itemValue = Forms.getButtonValue(buttonNode),
                // the help descriptor for the selected function (nothing for defined names)
                functionHelp = (_.isObject(itemValue) && (itemValue.type === 'func')) ? app.getFunctionHelp(itemValue.name) : null;

            // only show tooltips for supported functions (not for defined names)
            if (functionHelp) {
                textAreaListItemToolTip.setText(functionHelp.description).show();
            } else {
                textAreaListItemToolTip.hide();
            }
        }

        /**
         * Hides the function signature tooltip and the function
         * auto-completion pop-up menu.
         */
        function hideTextAreaPopups() {
            signatureToolTip.hide();
            textAreaListMenu.hide();
        }

        /**
         * Shows the function signature tooltip or the function auto-completion
         * pop-up menu if possible.
         */
        function updateTextAreaPopups() {

            var // the cursor position in the text area (exclude the equality sign)
                textPos = textAreaNode[0].selectionStart - formulaOffset,
                // token info for function auto-completion
                tokenInfo = null,
                // the text to be used for function auto-completion
                autoText = null,
                // function data for the cursor position
                functionInfo = null;

            // show pop-ups only in formula mode, do not show them while dragging
            // highlight ranges, or selecting new ranges for the formula
            if (!_.isString(formulaString) ||
                _.isNumber(self.getSheetViewAttribute('highlightIndex')) ||
                _.isObject(self.getSheetViewAttribute('activeSelection'))
            ) {
                hideTextAreaPopups();
                return;
            }

            // prefer function auto-completion over function tooltip
            autoTokenInfo = null;

            // auto completion of names: cursor must be located exactly at the end of a function or name token
            if (autoComplete && (tokenInfo = editTokenArray.getTokenAtPosition(textPos)) && (tokenInfo.end === textPos)) {

                // get the text entered before the text cursor (may be a name token or a function token)
                if (tokenInfo.token.isType(/^(name|func)$/) && !tokenInfo.token.hasSheetRef()) {
                    // Special case: do not show a pop-up menu when a literal number precedes the name.
                    // Otherwise, e.g. entering scientific numbers will show a pop-up for functions starting with 'E'.
                    if ((tokenInfo.index === 0) || !editTokenArray.getToken(tokenInfo.index - 1).isType('lit')) {
                        autoText = tokenInfo.token.getText();
                    }
                }

                // store token info (needed to replace the entered text with the full function name)
                autoTokenInfo = tokenInfo;
            }

            // fill pop-up menu for auto-completion
            if (_.isString(autoText)) {

                // prepare the menu sections
                textAreaListMenu.clearContents();
                textAreaListMenu.createSectionNode('funcs');
                textAreaListMenu.createSectionNode('names');

                // insert all matching function names
                autoText = autoText.toUpperCase();
                _.each(app.getLocalizedFunctionNames(), function (funcName) {
                    if (funcName.substr(0, autoText.length) === autoText) {
                        textAreaListMenu.createItemNode({ name: funcName, type: 'func' }, { section: 'funcs', label: funcName });
                    }
                });

                // show the menu, if it contains at least one entry
                if (textAreaListMenu.getItemNodes().length > 0) {
                    signatureToolTip.hide();
                    textAreaListMenu.show().selectItem(0, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return;
                }
            }

            // try to show a tooltip for the current function in a formula
            hideTextAreaPopups();
            if ((functionInfo = editTokenArray.getFunctionAtPosition(textPos))) {
                signatureToolTip.updateSignature(functionInfo.name, functionInfo.paramIndex);
            }
        }

        /**
         * Updates the function tooltip, and other settings according to the
         * text selection in the text area control.
         */
        function updateTextAreaAfterSelection() {

            // do not open the auto-completion menu even if the new cursor position is valid
            autoComplete = false;

            // do not ignore closing parentheses entered manually anymore (function auto-complete)
            autoCloseData = [];

            // defer execution to let the browser process pending key events
            self.executeDelayed(function () {
                // fail-safe check that cell edit mode is still active
                if (editActive) { updateTextAreaPopups(); }
            });
        }

        /**
         * Updates the highlighting of cell references in the current formula
         * while cell in-place edit mode is active.
         */
        function updateTextAreaHighlighting() {

            var // info objects for all ranges in the formula
                rangeInfos = editTokenArray.extractRanges({
                    filterSheet: self.getActiveSheet(),
                    refAddress: editAddress,
                    targetAddress: editAddress,
                    resolveNames: true
                }),
                // unified token indexes of all ranges
                rangeIndexes = _.chain(rangeInfos).pluck('index').unique().value(),
                // the mark-up of the underlay node (span elements containing token texts)
                markup = '';

            // generate the new mark-up for the underlay node
            if (rangeIndexes.length > 0) {
                markup += '<span>' + ((formulaOffset === 0) ? '' : '=') + '</span>';
                editTokenArray.iterateTokens(function (token, index) {
                    var styleIndex = _.indexOf(rangeIndexes, index, true);
                    markup += '<span data-token-type="' + token.getType() + '"';
                    if (styleIndex >= 0) { markup += ' data-style="' + ((styleIndex % Utils.SCHEME_COLOR_COUNT) + 1) + '"'; }
                    markup += '>' + Utils.escapeHTML(token.getText()) + '</span>';
                });
            }
            textUnderlayNode[0].innerHTML = markup;
        }

        /**
         * Updates the formula token array according to the current contents of
         * the text area control used for cell in-place edit mode. The token
         * array will trigger a 'change:tokens' event that will cause to redraw
         * the highlighted ranges in all grid panes.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, the token array and all other dependent
         *      settings will be updated immediately. By default, updating the
         *      token array will be debounced.
         */
        var updateEditTokenArray = (function () {

            // direct callback, invoked immediately when calling updateEditTokenArray()
            function directCallback(options) {

                var // the current input string in the text area
                    value = textAreaNode.val();

                // update the formula expression immediately
                formulaString = /^[-+=]/.test(value) ? value.replace(/^=/, '') : null;
                formulaOffset = (value === formulaString) ? 0 : 1;

                // invoke deferred callback immediately if specified
                if (Utils.getBooleanOption(options, 'direct', false)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                // the cell edit mode may have been finished already
                if (!editActive) { return; }

                // parse the formula string
                if (_.isString(formulaString)) {
                    editTokenArray.parseFormula(formulaString);
                } else {
                    editTokenArray.clear();
                }

                // insert highlighting mark-up for the underlay node
                updateTextAreaHighlighting();

                // update pop-up nodes and other settings dependent on the text selection
                updateTextAreaPopups();
            }

            // return the debounced method updateEditTokenArray()
            return self.createDebouncedMethod(directCallback, deferredCallback, { delay: 100, maxDelay: 300 });
        }());

        /**
         * Changes the contents and selection of the text area, and updates its
         * position and size.
         *
         * @param {String} text
         *  The new contents of the text area.
         *
         * @param {Number} [start]
         *  The new start position of the selection. If omitted, the text
         *  cursor will be moved to the end of the new text.
         *
         * @param {Number} [end=start]
         *  The new end position of the selection. If omitted, a simple text
         *  cursor will be shown at the specified start position.
         */
        function setTextAreaContents(text, start, end) {
            textAreaNode.val(text);
            lastValue = text;
            Forms.setInputSelection(textAreaNode, _.isNumber(start) ? start : text.length, end);
            updateTextAreaPosition({ direct: true });
            updateEditTokenArray();
        }

        /**
         * Replaces the token before the text cursor with the selected entry
         * from the auto-complete menu.
         */
        function applyAutoCompleteText(itemValue) {

            var // the text to be inserted into the formula, and the new cursor position
                newText = null,
                newStart = 0;

            // always hide the pop-up menu
            textAreaListMenu.hide();

            // fail-safety check
            if (!autoTokenInfo || !_.isObject(itemValue)) { return; }

            // get new text and new cursor position
            newText = itemValue.name;
            newStart = autoTokenInfo.start + newText.length + 1;

            // insert parentheses after function name (unless the existing token is
            // a function token which means it already has an opening parenthesis)
            if (itemValue.type === 'func') {
                if (autoTokenInfo.token.getType() === 'name') {
                    newText += '()';
                    // ignore the next closing parenthesis entered manually
                    autoCloseData.push(true);
                }
                // move cursor behind the opening parenthesis
                newStart += 1;
            }

            // insert the new text into the formula string
            setTextAreaContents(Utils.replaceSubString(textAreaNode.val(), autoTokenInfo.start + 1, autoTokenInfo.end + 1, newText), newStart);
            saveUndoState({ update: true });
        }

        /**
         * Changes the quick edit mode while in-place cell edit mode is active,
         * and shows a status label at the bottom of the application pane.
         *
         * @param {Boolean} newQuickEdit
         *  The new value of the quick edit mode.
         */
        function setQuickEditMode(newQuickEditMode) {

            // change the quick edit state
            if (quickEditMode === newQuickEditMode) { return; }
            quickEditMode = newQuickEditMode;

            // update the status label
            self.setStatusLabel(quickEditMode ?
                //#. Special edit mode in spreadsheet document: quickly type text into cell, use cursor keys to move to next cell
                gt('Insert mode') :
                //#. Standard edit mode in spreadsheet document: edit existing cell text, cursor keys move text cursor inside cell
                gt('Edit mode')
            );
        }

        /**
         * Returns an object that represents the current state of the cell
         * in-place edit mode, intended to be inserted into the undo stack.
         */
        function createUndoStackEntry(fixed) {
            return {
                text: textAreaNode.val(),
                sel: Forms.getInputSelection(textAreaNode),
                attrs: _.copy(editAttributes, true),
                fixed: fixed
            };
        }

        /**
         * Saves the current state of the cell edit mode on top of the undo
         * stack. If the current undo action is not on top of the stack, all
         * following actions will be deleted first.
         */
        function saveUndoState(options) {

            var // whether to try to update the current stack entry with current text
                update = Utils.getBooleanOption(options, 'update', false);

            if (update && (undoIndex + 1 === undoStack.length) && !undoStack[undoIndex].fixed) {
                undoStack[undoIndex].text = textAreaNode.val();
                undoStack[undoIndex].sel = Forms.getInputSelection(textAreaNode);
            } else {
                undoStack.splice(undoIndex + 1);
                undoStack.push(createUndoStackEntry(!update));
            }

            undoIndex = undoStack.length - 1;
            self.trigger('celledit:change');
        }

        /**
         * Restores the current state in the undo stack.
         */
        function restoreUndoState() {

            var // the entry from the undo/redo stack to be restored
                entry = undoStack[undoIndex];

            editAttributes = _.copy(entry.attrs, true);
            updateTextAreaStyle();
            setTextAreaContents(entry.text, entry.sel.start, entry.sel.end);
            self.trigger('celledit:change');
        }

        /**
         * Removes the activated ranges from the view that have been inserted
         * into the formula while cell edit mode is active.
         */
        function resetActiveSelection() {
            self.setSheetViewAttribute('activeSelection', null);
        }

        /**
         * Updates the position of the text area control after the grid pane
         * has been scrolled (or modified in any other way, e.g. zoom).
         */
        function gridPaneScrollHandler() {
            updateTextAreaPosition({ direct: true, size: false });
        }

        /**
         * Handles 'keydown' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyDownHandler(event) {

            var // the list item currently selected in the auto-completion pop-up menu
                itemNode = textAreaListMenu.getSelectedItemNodes().first(),
                // the value of the item node
                itemValue = Forms.getButtonValue(itemNode);

            // first, try to cancel active tracking mode on ESCAPE key
            if ((event.keyCode === KeyCodes.ESCAPE) && $.cancelTracking()) {
                return false;
            }

            // short-cuts in visible pop-up menu for auto-completion
            if (textAreaListMenu.isVisible()) {

                // close pop-up menu on ESCAPE key
                if (event.keyCode === KeyCodes.ESCAPE) {
                    textAreaListMenu.hide();
                    return false;
                }

                // insert selected list item into the formula
                if (KeyCodes.matchKeyCode(event, 'TAB') || KeyCodes.matchKeyCode(event, 'ENTER')) {

                    // Bug 31999: For performance reasons, the token array and the auto-completion pop-up
                    // menu will be updated debounced. To prevent that ENTER/TAB keys insert a function
                    // name unintentionally, the pop-up menu must be updated before applying a list item.
                    updateEditTokenArray({ direct: true });

                    // list menu may have been hidden by the updateTextAreaPopups() call,
                    // do not apply any list item in this case, but leave cell edit mode
                    if (textAreaListMenu.isVisible()) {
                        applyAutoCompleteText(itemValue);
                        return false;
                    }
                }

                // otherwise, try to move the selected list item
                itemNode = textAreaListMenu.getItemNodeForKeyEvent(event, itemNode);
                if (itemNode.length > 0) {
                    textAreaListMenu.selectItemNode(itemNode, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return false;
                }
            }

            // quit in-place edit mode on ESCAPE key regardless of any modifier keys
            if (event.keyCode === KeyCodes.ESCAPE) {
                self.cancelCellEditMode();
                return false;
            }

            // toggle quick edit mode
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                setQuickEditMode(!quickEditMode);
                // forget last cell range selected in the sheet
                resetActiveSelection();
                return false;
            }

            // ENTER without modifier keys (but ignoring SHIFT): set value, move cell down/up
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                if (self.leaveCellEditMode('cell', { validate: true })) {
                    return; // let key event bubble up to move the cell cursor
                }
                return false;
            }

            // ENTER with CTRL or META key: fill value to all cells (no SHIFT), or insert matrix formula (SHIFT)
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null, ctrlOrMeta: true })) {
                self.leaveCellEditMode(event.shiftKey ? 'matrix' : 'fill', { validate: true });
                // do not let key event bubble up (no change of selection)
                return false;
            }

            // ENTER with ALT key without any other modifier keys: insert line break
            if (KeyCodes.matchKeyCode(event, 'ENTER', { alt: true })) {
                Forms.replaceTextInInputSelection(textAreaNode, '\n');
                textAreaInputHandler();
                return false;
            }

            // ENTER with any other modifier keys: discard the entire event
            if (event.keyCode === KeyCodes.ENTER) {
                return false;
            }

            // TAB key without modifier keys (but ignoring SHIFT): set value, move cell to next/previous cell
            if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                if (self.leaveCellEditMode('cell', { validate: true })) {
                    return; // let key event bubble up to move the cell cursor
                }
                return false;
            }

            // any cursor navigation key
            if (_.contains([KeyCodes.LEFT_ARROW, KeyCodes.RIGHT_ARROW, KeyCodes.UP_ARROW, KeyCodes.DOWN_ARROW, KeyCodes.PAGE_UP, KeyCodes.PAGE_DOWN, KeyCodes.HOME, KeyCodes.END], event.keyCode)) {
                if (quickEditMode) {
                    // quick edit mode: do not let the text area process the key events
                    event.preventDefault();
                } else {
                    // standard edit mode: do not process the key events (no change of cell selection)
                    event.stopPropagation();
                    // update pop-up nodes and other settings dependent on the new text selection
                    updateTextAreaAfterSelection();
                    // forget last cell range selected in the sheet
                    resetActiveSelection();
                }
                // let key event bubble up to move the cell cursor, or text cursor in the text area
                return;
            }

            // all other events will be processed by the text field
        }

        /**
         * Handles 'keypress' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyPressHandler(event) {

            // the entered Unicode character
            switch (String.fromCharCode(event.charCode)) {

            // Register an opening parenthesis entered manually. The related closing parenthesis
            // will be inserted even if it occurs among other closing parentheses inserted via
            // function auto-completion (see below).
            case '(':
                autoCloseData.push(false);
                break;

            // When a function name has been inserted from the auto-completion pop-up menu, skip
            // a closing parenthesis because it has been inserted already by the auto-completion).
            // But the text cursor needs to be moved one character forward.
            case ')':
                if (autoCloseData.pop() === true) {
                    Forms.setInputSelection(textAreaNode, Forms.getInputSelection(textAreaNode).start + 1);
                    updateTextAreaPopups();
                    return false;
                }
                break;
            }
        }

        /**
         * Handles any changes in the text area.
         */
        function textAreaInputHandler() {

            var // the current value of the text area
                currValue = textAreaNode.val(),
                // the text selection in the text area control
                textSelection = Forms.getInputSelection(textAreaNode),
                // whether the selection in the text area is a cursor only
                isCursor = textSelection.start === textSelection.end,
                // whether to show type-ahead suggestion in the text area
                typeAhead = (lastValue.length < currValue.length) && isCursor && (textSelection.end === currValue.length);

            // do nothing if the edit text did not change at all
            if (lastValue === currValue) { return; }

            // try to show drop-down list with function names and defined names
            autoComplete = isCursor;

            // update text area settings (position, formula range highlighting)
            lastValue = currValue;
            updateTextAreaPosition();
            updateEditTokenArray();
            saveUndoState({ update: true });

            // Mobile devices may add auto-correction text which modifies the selection after a text change
            // event has been processed. Recalculate the autoComplete flag after the browser has set the final selection.
            self.executeDelayed(function () {
                var textSelection = Forms.getInputSelection(textAreaNode);
                autoComplete = textSelection.start === textSelection.end;
            });

            // forget last cell range selected in the sheet
            resetActiveSelection();

            /* Start-of autocompleteText */
            // TODO rework needed
            var columnText = [], suggestions = [];

            if (typeAhead && !_.isString(formulaString)) { //if user writes(adds) new letter -> auto suggest new text

                var address = self.getActiveCell(),
                    boundRange = { start: [address[0], Math.max(0, address[1] - 256)], end: [address[0], Math.min(docModel.getMaxRow(), address[1] + 256)] };

                self.getCellCollection().iterateCellsInLine(address, 'up down', function (cellData) {
                    if (CellCollection.isText(cellData) && (cellData.result.length > 0)) {
                        columnText.push(cellData.result);
                    }
                }, { type: 'content', hidden: 'all', boundRange: boundRange, skipStartCell: true });

                //if first letter in currValue is LowerCase
                if (currValue.charAt(0) === currValue.charAt(0).toLowerCase() && currValue.charAt(0) !== currValue.charAt(0).toUpperCase()) {
                   //charAt() - returns the character at the specified index in a string.

                    for (var i = 0; i < columnText.length; i++) {
                        columnText[i] = columnText[i].charAt(0).toLowerCase() + columnText[i].substring(1); //make first letter in columText LowerCase
                    }
                    //search user input for matching element in array "columnText"
                    for (var i = 0; i < columnText.length; i++) {
                        if (columnText[i].indexOf(currValue) === 0) { //loop through each element in "columnText" array, to see if it begins with the text in textArea. indexOf() returns 0 if this is the case.
                            suggestions.push(columnText[i]); //fill an array "suggestions" with all possible suggestions
                        }
                    }
                } //end-of if LowerCase
                else {
                    for (var i = 0; i < columnText.length; i++) {
                        columnText[i] = columnText[i].charAt(0).toUpperCase() + columnText[i].substring(1); //make first letter in columText UpperCase
                    }
                    //search user input for matching element in array "columnText"
                    for (var i = 0; i < columnText.length; i++) {
                        if (columnText[i].indexOf(currValue) === 0) {
                            suggestions.push(columnText[i]);
                        }
                    }
                } //end-of else

                if (suggestions.length === 0) { //for mix of LowerCase/UpperCase suggestion
                    for (var i = 0; i < columnText.length; i++) {
                        columnText[i] = columnText[i].toLowerCase();
                    }
                    for (var i = 0; i < columnText.length; i++) {
                        if (columnText[i].indexOf(currValue) === 0) {
                            suggestions.push(columnText[i]);
                        }
                    }
                }
            }

            //sort suggestions by length-ascending
            suggestions.sort(function (a, b) {
                return a.length - b.length;
            });

            //make sure there is at least one suggestion in array "suggestions"
            if (suggestions.length > 0) {
                textAreaNode.focus();
                // suggest new text to user, first best only
                setTextAreaContents(suggestions[0], currValue.length, suggestions[0].length);
                saveUndoState({ update: true });
            }
        }

        /**
         * Handles mousedown or touchstart events in the text area.
         */
        function textAreaMouseTouchHandler() {
            // leave quick-edit mode (cursor keys will move the text cursor)
            setQuickEditMode(false);
            // update pop-up nodes and other settings dependent on the new text selection
            updateTextAreaAfterSelection();
            // forget last cell range selected in the sheet
            resetActiveSelection();
        }

        /**
         * Updates the scroll position of the underlay node, after the scroll
         * position of the text area node has been changed.
         */
        function textAreaScrollHandler() {
            textUnderlayNode.scrollTop(textAreaNode.scrollTop());
        }

        /**
         * Hides the pop-up nodes if the focus leaves the text area control,
         * unless the auto-completion menu is being clicked.
         */
        var textAreaFocusHandler = (function () {

            var // whether to ignore focus event in the text area (no update of pop-up nodes)
                ignoreFocusEvent = false;

            return function (event) {
                switch (event.type) {

                // text area focused: rebuild all active pop-up nodes
                case 'focus':
                    textAreaContainer.addClass(Forms.FOCUSED_CLASS);
                    if (!ignoreFocusEvent) {
                        updateTextAreaPopups();
                    }
                    break;

                // text area lost focus: hide pop-up nodes (unless the auto-complete menu
                // has been clicked, in this case, move focus back to text area immediately)
                case 'blur':
                    self.executeDelayed(function () {
                        if (editActive && textAreaListMenu.hasFocus()) {
                            ignoreFocusEvent = true;
                            textAreaNode.focus();
                            // IE sends focus event deferred
                            self.executeDelayed(function () { ignoreFocusEvent = false; });
                        } else {
                            textAreaContainer.removeClass(Forms.FOCUSED_CLASS);
                            hideTextAreaPopups();
                        }
                    });
                }
            };
        }());

        /**
         * Inserts the name clicked in the auto-completion pop-up menu into the
         * current formula text.
         */
        function textAreaListClickHandler(event) {
            var itemValue = Forms.getButtonValue(event.currentTarget);
            applyAutoCompleteText(itemValue);
        }

        /**
         * Returns whether the current text selection in the text area control
         * is valid to start selecting new ranges for the current formula. The
         * text area must contain a formula string (starting with an equality
         * sign), and the start position of the text selection must be located
         * behind the leading equality sign, an operator, a list separator, or
         * an opening parenthesis).
         */
        function isValidRangeSelectionPosition() {

            var // the cursor position in the token array (exclude leading equality sign)
                textPos = textAreaNode[0].selectionStart - formulaOffset,
                // the information for the token at the cursor position
                tokenInfo = null;

            // invalid if not in cell in-place edit mode with a formula
            if (!editActive || !_.isString(formulaString)) { return false; }

            // existing view attribute, or cursor at the beginning of the formula is always valid
            if ((textPos === 0) || _.isObject(self.getSheetViewAttribute('activeSelection'))) { return true; }

            // get data of the token located before the text cursor
            tokenInfo = editTokenArray.getTokenAtPosition(textPos);

            // cursor must be located between tokens; leading token must be of correct
            // type (or cursor must be located at the beginning of the formula)
            return tokenInfo && (textPos === tokenInfo.end) && tokenInfo.token.isType(/^(op|sep|open)$/);
        }

        /**
         * When in cell edit mode with a formula, returns the active selection
         * which will be used for keyboard navigation and mouse/touch tracking.
         */
        function getCellSelectionHandler() {

            var // the current value of the 'activeSelection' view attribute
                activeSelection = self.getSheetViewAttribute('activeSelection');

            // always return existing view attribute
            if (_.isObject(activeSelection)) { return activeSelection; }

            // check the text selection in the text area control
            if (!isValidRangeSelectionPosition()) { return null; }

            // start with an empty active selection
            return { ranges: [], activeRange: 0, activeCell: [0, 0] };
        }

        /**
         * When in cell edit mode with a formula, selects a single range and
         * inserts the range address into the formula string.
         */
        function setCellSelectionHandler(selection) {

            var // a helper token array to generate reference addresses for the active ranges
                tokenArray = null,
                // the string representation of the active ranges
                references = null,
                // the new formula text
                formulaText = null;

            // do nothing if ranges cannot be selected with the current text selection
            if (!isValidRangeSelectionPosition()) { return false; }

            // store current text selection, if no range are activated yet
            // (selected text will be replaced with initial range addresses)
            if (!_.isObject(self.getSheetViewAttribute('activeSelection'))) {
                rangeSelection = Forms.getInputSelection(textAreaNode);
            }

            // build the new formula term to be inserted into the text
            tokenArray = new TokenArray(app, self.getSheetModel(), { trigger: 'never', grammar: 'ui' });
            tokenArray.appendRangeList(selection.ranges);
            references = tokenArray.getFormula();
            tokenArray.destroy();
            formulaText = Utils.replaceSubString(textAreaNode.val(), rangeSelection.start, rangeSelection.end, references);

            // insert the new formula text, set cursor behind the range (no real selection)
            rangeSelection.end = rangeSelection.start + references.length;
            setTextAreaContents(formulaText, rangeSelection.end);
            saveUndoState({ update: true });

            // update the view attribute
            self.setSheetViewAttribute('activeSelection', selection);

            // do not perform standard sheet selection
            return true;
        }

        /**
         * Handles change events from any of the token arrays, and notifies the
         * own event listeners.
         */
        function changeTokenArrayHandler() {

            var // options passed to the extractRanges method
                extractOptions = Utils.extendOptions(highlightOptions, { filterSheet: self.getActiveSheet() }),
                // array of arrays of range info objects from the token arrays
                allRangeInfos = _.invoke(highlightTokenArrays, 'extractRanges', extractOptions),
                // whether dragging the cell ranges is enabled
                draggable = Utils.getBooleanOption(highlightOptions, 'draggable', false),
                // the flattened highlighted range addresses
                highlightRanges = [];

            _.each(allRangeInfos, function (rangeInfos, arrayIndex) {
                _.each(rangeInfos, function (rangeInfo) {
                    highlightRanges.push(_.extend({
                        // the range identifier containing the index of the token array
                        id: arrayIndex + ',' + rangeInfo.index,
                        // ranges of defined names are not draggable
                        draggable: draggable && (rangeInfo.type === 'ref')
                    }, rangeInfo.range));
                });
            });

            // set the highlighted ranges at the model instance of the active sheet
            self.setSheetViewAttribute('highlightRanges', highlightRanges);
        }

        /**
         * Registers or unregisters the change listener at all current token
         * arrays.
         */
        function registerTokenArrayListeners(type) {
            _.invoke(highlightTokenArrays, type, 'triggered', changeTokenArrayHandler);
        }

        /**
         * Cancels cell edit mode after the application loses edit rights.
         */
        function editModeHandler(event, editMode) {
            // leave cell in-place edit mode, if document changes to read-only mode
            if (!editMode && app.isImportFinished()) {
                self.cancelCellEditMode();
            }
        }

        /**
         * Updates the text area for a changed zoom factor.
         */
        function changeSheetViewAttributesHandler(event, attributes) {
            if (editActive && ('zoom' in attributes)) {
                updateTextAreaStyle();
                updateTextAreaPosition({ direct: true });
            }
        }

        /**
         * Shows an alert box for circular references in new formulas.
         */
        function showCircularWarning() {
            self.yell({
                type: 'info',
                headline: /*#. title for a warning message: a formula addresses its own cell, e.g. =A1 in cell A1 */ gt('Circular Reference'),
                message: gt('A reference in the formula is dependent on the result of the formula cell.')
            });
        }

        /**
         * Processes 'docs:update:cells' events of the application, triggered
         * from a server notification. If the event contains a cell-related
         * error code, such as a circular reference error, and the error was
         * caused by a local change after inplace cell edit mode, an alert box
         * will be shown to the user.
         */
        function updateCellsHandler(changedData, allChangedRanges, errorCode) {

            // check whether all required data objects exist
            if (_.isObject(changedData) && _.isString(errorCode) && _.isObject(lastCommitData) && (lastCommitData.sheet in changedData) && _.isArray(lastCommitData.ranges)) {

                var // the notified changed ranges in the last edited sheet
                    changedRanges = changedData[lastCommitData.sheet].changedRanges;

                // the last committed cells must opverlap with the notified changed ranges
                if (_.isArray(changedRanges) && SheetUtils.rangesOverlapRanges(changedRanges, lastCommitData.ranges)) {

                    // show an appropriate alert for the error code
                    if (errorCode === 'circular') {
                        // do not show another circular reference warning, if it has been detected locally
                        if (!lastCommitData.circular) { showCircularWarning(); }
                    } else {
                        Utils.warn('CellEditMixin.updateCellsHandler(): unknown error code "' + errorCode + '"');
                    }
                }
            }

            lastCommitData = null;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the initial text for the in-place cell edit mode. Returns
         * the formula definition for formula cells, otherwise the cell value
         * formatted with an appropriate default number format.
         *
         * @param {Object} cellData
         *  The descriptor of a cell whose initial edit text will be returned.
         *
         * @returns {String}
         *  The initial text for the in-place cell edit mode.
         */
        this.getTextForCellEditMode = function (cellData) {

            var // default for undefined cells
                editText = '';

            if (_.isString(cellData.formula)) {
                // hide formula if the cell contains the hidden attribute
                editText = (this.isSheetLocked() && cellData.attributes.cell.hidden) ? '' : cellData.formula;
            } else {
                // use appropriate representation according to number format category
                editText = numberFormatter.formatValueForEditMode(cellData.result, cellData.format.cat);
            }

            return editText;
        };

        /**
         * Returns whether the in-place cell edit mode is currently active.
         *
         * @returns {Boolean}
         *  Whether the in-place cell edit mode is currently active.
         */
        this.isCellEditMode = function () {
            return editActive;
        };

        /**
         * Starts in-place cell edit mode in the active grid pane.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {String} [options.text]
         *      The initial text to insert into the edit area. If omitted, the
         *      current value of the active cell will be inserted.
         *  @param {Number} [options.pos]
         *      The initial position of the text cursor. If omitted, the text
         *      cursor will be placed after the entire text.
         *  @param {Boolean} [options.quick=false]
         *      If set to true, the quick edit mode will be initiated. All
         *      cursor navigation keys will not move the text cursor in the
         *      text area control but will behave as in the normal sheet
         *      selection mode.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.enterCellEditMode = function (options) {

            var // the initial text for the text area
                initialText = Utils.getStringOption(options, 'text'),
                // the initial cursor position for the text area
                initialPos = Utils.getIntegerOption(options, 'pos');

            // first, check document edit mode
            if (!this.requireEditMode()) { return this; }

            // check whether in-place edit mode is already active
            if (editActive) { return this; }

            // cancel current tracking
            $.cancelTracking();

            // store location of edited cell (sheet may change during range selection mode for formulas)
            editAddress = this.getActiveCell();
            editSheet = this.getActiveSheet();
            editGridPane = this.getActiveGridPane();
            editAttributes = {};

            // initialize CSS formatting of the text area (this initializes the variable editCellData)
            updateTextAreaStyle();

            // scroll to the active cell (before checking edit mode); on touch devices,
            // scroll the cell to the top border of the grid pane (due to virtual keyboard)
            editGridPane.scrollToCell(editAddress, { forceTop: Modernizr.touch });

            // BUG #34366
            // Touch devices scroll the window out of visible area, so that neither the bars (tool and tab)
            // are visible, nor the editable cell. Root cause is the soft-keyboard.
            // However, scroll back to 0,0 and everything is fine. (Delay of 1000 necessary for ios)
            if (Modernizr.touch) {
                this.executeDelayed(function () { window.scrollTo(0, 0); }, 1000);
            }

            // give user some info and quit if cell is locked
            if (this.isSheetLocked() && !editCellData.attributes.cell.unlocked) {
                this.trigger('celledit:reject', 'cells:locked');
                return this;
            }

            // initial state of quick edit mode and other settings
            editActive = true;
            setQuickEditMode(Utils.getBooleanOption(options, 'quick', false));

            // TODO: matrix formula support
            matrixFormula = false;

            // the token array stores the parsed formula for range highlighting
            editTokenArray = new TokenArray(app, this.getSheetModel(), { grammar: 'ui' });
            this.startRangeHighlighting(editTokenArray, {
                draggable: true,
                refAddress: editAddress,
                targetAddress: editAddress,
                resolveNames: true
            });

            // update formula string in text area while dragging the highlighted ranges
            editTokenArray.on('change:token', function (event, token, index) {
                var tokenPos = editTokenArray.getTokenPosition(index);
                setTextAreaContents(textAreaNode.val()[0] + editTokenArray.getFormula(), tokenPos.end + 1);
                saveUndoState({ update: true });
                // immediately update the highlighted cell references
                updateTextAreaHighlighting();
                resetActiveSelection();
            });

            // insert the text area control into the DOM of the active grid pane, and activate it
            editGridPane.getNode().append(textAreaContainer);
            editGridPane.initializeCellEditMode();
            editGridPane.on('change:scrollpos', gridPaneScrollHandler);

            originalText = this.getTextForCellEditMode(editCellData);
            textUnderlayNode.empty();

            // set initial contents, position, size, and selection of the text area
            setTextAreaContents(_.isString(initialText) ? initialText : originalText, initialPos);

            // initialize local undo/redo stack
            undoStack = [{ text: originalText, sel: _.isString(initialText) ? { start: 0, end: originalText.length } : Forms.getInputSelection(textAreaNode), attrs: {}, fixed: true }];
            undoIndex = 0;

            // selection handler that changes cell ranges in formaulas in cell edit mode
            this.registerCellSelectionHandlers(getCellSelectionHandler, setCellSelectionHandler);

            // set last value and invoke input handler to handle custom initial text
            lastValue = originalText;
            textAreaInputHandler();

            // notify listeners
            return this.trigger('celledit:enter');
        };

        /**
         * Cancels in-place cell edit mode, without committing the current edit
         * text.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.cancelCellEditMode = function () {

            // nothing to do, if in-place edit mode is not active
            if (!editActive) { return this; }

            // deinitialize range highlighting
            editActive = false;
            this.endRangeHighlighting();
            this.unregisterCellSelectionHandlers();
            resetActiveSelection();
            editTokenArray.destroy();
            editTokenArray = null;

            // back to cell selection mode
            quickEditMode = null;
            formulaString = null;
            hideTextAreaPopups();

            // remove the text area control from the DOM of the active grid pane
            editGridPane.off('change:scrollpos', gridPaneScrollHandler);
            editGridPane.cleanupCellEditMode();
            editGridPane = null;
            textAreaContainer.detach();

            // notify listeners
            this.setStatusLabel(null);
            this.trigger('celledit:leave');
            return this;
        };

        /**
         * Leaves in-place cell edit mode, and optionally commits the current
         * text. If the current text is a formula expression, it will be
         * validated before committing.
         *
         * @param {String} commitMode
         *  Specifies how to leave the cell edit mode. The following values are
         *  supported:
         *  - 'cell': The current value will be committed to the active cell of
         *      the selection.
         *  - 'fill': The current value will be filled to all cells of the
         *      selection.
         *  - 'matrix': The current value will be inserted as matrix formula
         *      into the selected cell range.
         *  - 'auto': Preserves the current formula type of the cell: either
         *      'cell' or 'matrix'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.validate=false]
         *      If set to true, a formula expression will be velidated. If it
         *      is invalid, a notification alert will be shown, and the cell
         *      edit mode will not be left.
         *
         * @returns {Boolean}
         *  Whether the cell edit mode has actually been left. Returns false,
         *  if the option 'validate' have been set to true, and the current
         *  formula expression is invalid.
         */
        this.leaveCellEditMode = function (commitMode, options) {

            var // the new cell text
                editText = textAreaNode.val(),
                // whether to validate the formula
                validate = Utils.getBooleanOption(options, 'validate', false),
                // the trimmed URL extracted from the value
                url = HyperlinkUtils.checkForHyperlink(editText),
                // additional properties for the operation
                properties = { parse: true };

            function triggerReject(cause) {
                self.trigger('celledit:reject', cause);
                // switch to regular edit mode
                setQuickEditMode(false);
            }

            function addAttribute(family, name, value) {
                editAttributes[family] = editAttributes[family] || {};
                editAttributes[family][name] = value;
            }

            // nothing to do, if in-place edit mode is not active
            if (!editActive) { return true; }

            // check validity of commit mode (e.g. disallow 'cell' in matrix formulas)
            switch (commitMode) {
            case 'cell':
            case 'fill':
                if (matrixFormula) {
                    // TODO: define behavior (update entire matrix automatically?)
                    triggerReject('formula:matrix');
                    return false;
                }
                break;
            case 'matrix':
                // TODO: check that selection matches range of matrix formula
                break;
            case 'auto':
                commitMode = matrixFormula ? 'matrix' : 'cell';
                break;
            default:
                Utils.error('CellEditMixin.leaveCellEditMode(): unknown commit mode: ' + commitMode);
                commitMode = null;
            }

            // do nothing else, if the document does not have edit rights anymore
            if (!docModel.getEditMode() || !commitMode) {
                this.cancelCellEditMode();
                return true;
            }

            // prepare cached commit data (needed to evaluate server notifications with error codes)
            lastCommitData = { sheet: editSheet };

            // handle specific strings not to be recognized as formulas
            if (/^[-+=]+$/.test(editText)) {
                editText = '\'' + editText;
            } else {
                // validate formula
                updateEditTokenArray({ direct: true });
                if (_.isString(formulaString)) {

                    var // TODO: get value/matrix result depending on commit mode
                        result = editTokenArray.interpretFormula('val', editSheet, editAddress);

                    if (validate) {
                        switch (result.type) {
                        case 'error':
                            triggerReject('formula:invalid');
                            return false;
                        case 'warn':
                            if (result.value === 'circular') {
                                showCircularWarning(); // TODO: yes/no dialog?
                                lastCommitData.circular = true;
                            }
                        }
                    }

                    // insert the formula result value into the operation
                    properties.result = null;
                    if (result.type === 'result') {
                        // convert error code objects to string representation, add leading apostrophe to strings if necessary
                        if (SheetUtils.isErrorCode(result.value)) {
                            properties.result = result.value.code;
                        } else if (_.isString(result.value)) {
                            properties.result = result.value.replace(/^(['#])/, '\'$1');
                        } else {
                            properties.result = result.value;
                        }
                    }
                }
            }

            // deinitialize cell edit mode
            this.cancelCellEditMode();

            // add auto-wrapping attribute, if the text contains an explicit line break
            if (editText.indexOf('\n') >= 0) {
                addAttribute('cell', 'wrapText', true);
            }

            // add hyperlink specific attributes
            if (_.isString(url) && (url.length > 0)) {
                addAttribute('character', 'url', url);
                addAttribute('character', 'underline', true);
                addAttribute('character', 'color', { type: 'scheme', value: 'hyperlink' });
            } else if (!editText) {
                //delete explicit the url, if text is cleared
                addAttribute('character', 'url', null);
            }

            // do not pass empty attributes object
            if (_.isEmpty(editAttributes)) { editAttributes = null; }

            // send current value to document model after returning to selection mode
            switch (docModel.getEditMode() ? commitMode : 'discard') {
            case 'cell':
                // single cell mode: do not send operations if text and formatting did not change
                if ((originalText !== editText) || _.isObject(editAttributes)) {
                    this.setCellContents(editText, editAttributes, properties);
                    lastCommitData.ranges = [{ start: editAddress, end: editAddress }];
                }
                break;
            case 'fill':
                // fill mode: always fill all cells in the selection
                this.fillCellRanges(editText, editAttributes, properties);
                lastCommitData.ranges = this.getSelectedRanges();
                break;
            case 'matrix':
                // TODO: matrix formulas
                Utils.warn('CellEditMixin.leaveCellEditMode(): matrix formulas not implemented');
                lastCommitData.ranges = this.getSelectedRanges();
                break;
            }

            return true;
        };

        /**
         * Returns the additional formatting attributes used in in-place cell
         * edit mode.
         *
         * @return {Object|Null}
         *  The new attributes applied while the in-place cell edit mode is
         *  active.
         */
        this.getCellEditAttributes = function () {
            return editActive ? _.copy(editAttributes, true) : null;
        };

        /**
         * Modifies formatting attributes while the in-place cell edit mode is
         * active.
         *
         * @param {Object} [attributes]
         *  The new attributes to be applied to the text area. If omitted, the
         *  call of this method will be ignored.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.setCellEditAttributes = function (attributes) {

            function extendAttributeFamily(family) {
                if (_.isObject(attributes[family])) {
                    _.extend(editAttributes[family] || (editAttributes[family] = {}), attributes[family]);
                }
            }

            if (editActive && _.isObject(attributes)) {

                // extend the edit attribute set
                extendAttributeFamily('cell');
                extendAttributeFamily('character');

                // update the formatting of the text area, and the undo/redo stack
                updateTextAreaStyle();
                updateTextAreaPosition({ direct: true });
                saveUndoState();
            }

            return this;
        };

        // extended undo/redo -------------------------------------------------

        /**
         * Returns whether at least one undo action is available on the undo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * are changed currently.
         *
         * @returns {Boolean}
         *  Whether at least one undo action is available on the stack.
         */
        this.isUndoAvailable = function () {
            return editActive ? (undoIndex > 0) : (undoManager.getUndoCount() > 0);
        };

        /**
         * Applies the topmost undo action on the undo stack. In cell in-place
         * edit mode, restores the original contents of the cell.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the undo action has been
         *  applied.
         */
        this.undo = function () {
            if (editActive) {
                if (undoIndex > 0) {
                    undoIndex -= 1;
                    restoreUndoState();
                }
                return $.when();
            }
            return undoManager.undo(1);
        };

        /**
         * Returns whether at least one redo action is available on the redo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * have been changed, but are not changed anymore (after undo).
         *
         * @returns {Boolean}
         *  Whether at least one redo action is available on the stack.
         */
        this.isRedoAvailable = function () {
            return editActive ? (undoIndex + 1 < undoStack.length) : (undoManager.getRedoCount() > 0);
        };

        /**
         * Applies the topmost redo action on the redo stack. In cell in-place
         * edit mode, restores the last changed contents of the cell, after
         * these changes have been undone.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the redo action has been
         *  applied.
         */
        this.redo = function () {
            if (editActive) {
                if (undoIndex + 1 < undoStack.length) {
                    undoIndex += 1;
                    restoreUndoState();
                }
                return $.when();
            }
            return undoManager.redo(1);
        };

        // range highlighting -------------------------------------------------

        /**
         * Registers token arrays used to render highlighted cell ranges in the
         * active sheet, and updates the 'highlightRanges' view property of the
         * active sheet whenever the token arrays are changing.
         *
         * @param {Array|TokenArray} tokenArrays
         *  An array of token arrays, or a single token array instance.
         *
         * @param {Boolean} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.draggable=false]
         *      If set to true, the cell ranges representing reference tokens
         *      will be marked draggable in the 'highlightRanges' sheet view
         *      attribute.
         *  @param {Number[]} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Number[]} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated range addresses that are located
         *      outside the sheet range will be wrapped at the sheet borders.
         *  @param {Boolean} [options.resolveNames=false]
         *      If set to true, the cell ranges contained in defined names will
         *      be highlighted too.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.startRangeHighlighting = function (tokenArrays, options) {

            // store tracking options for usage in event handlers
            highlightOptions = _.clone(options);

            // unregister event handlers at old token arrays
            registerTokenArrayListeners('off');

            // store the token arrays
            highlightTokenArrays = _.chain(tokenArrays).getArray().clone().value();
            registerTokenArrayListeners('on');

            // notify initial ranges to listeners
            changeTokenArrayHandler();
            return this;
        };

        /**
         * Unregisters all token arrays currently registered.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.endRangeHighlighting = function () {
            return this.startRangeHighlighting([]);
        };

        /**
         * Modifies the cell range address of a single reference token inside a
         * single token array currently registered for range highlighting.
         *
         * @param {String} tokenId
         *  The unique identifier of the reference token to be modified, as
         *  contained in the 'highlightRanges' view attribute.
         *
         * @param {Object} range
         *  The new logical range address to be inserted into the reference
         *  token. The absolute/relative state of the reference components will
         *  not be changed.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.modifyReferenceToken = function (tokenId, range) {

            var // the token array index, and the token index
                indexes = _.map(tokenId.split(/,/), function (id) { return parseInt(id, 10); }),
                // the token array instance
                tokenArray = (indexes.length === 2) ? highlightTokenArrays[indexes[0]] : null;

            // modify the reference token, if the indexes are valid
            if (tokenArray) {
                tokenArray.modifyToken(indexes[1], 'ref', function (refToken) {
                    return refToken.setRange(range);
                });
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.onInit(function () {

            // resolve reference to document model objects
            docModel = app.getModel();
            undoManager = docModel.getUndoManager();
            documentStyles = docModel.getDocumentStyles();
            fontCollection = docModel.getFontCollection();
            numberFormatter = docModel.getNumberFormatter();

            // cancel cell edit mode after the application loses edit rights
            docModel.on('change:editmode', editModeHandler);

            // update zoom settings during edit mode
            self.on('change:sheet:viewattributes', changeSheetViewAttributesHandler);
        });

        // events for in-place edit mode
        textAreaNode.on({
            keydown: textAreaKeyDownHandler,
            keypress: textAreaKeyPressHandler,
            input: textAreaInputHandler,
            'mousedown mouseup touchstart touchend touchcancel': textAreaMouseTouchHandler,
            scroll: textAreaScrollHandler,
            'focus blur': textAreaFocusHandler
        });

        // handle click events in the auto-completion pop-up menu
        textAreaListMenu.getNode().on('click', Forms.BUTTON_SELECTOR, textAreaListClickHandler);

        // hide function description tooltip when closing the auto-completion menu
        textAreaListMenu.on('popup:hide', function () { textAreaListItemToolTip.hide(); });

        // post-process changed cells (e.g. for circular references alert box)
        this.listenTo(app, 'docs:update:cells', updateCellsHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {

            registerTokenArrayListeners('off');

            textAreaContainer.remove();
            signatureToolTip.destroy();
            textAreaListMenu.destroy();
            textAreaListItemToolTip.destroy();

            app = self = docModel = undoManager = documentStyles = fontCollection = numberFormatter = null;
            textAreaContainer = textAreaNode = textUnderlayNode = null;
            signatureToolTip = textAreaListMenu = textAreaListItemToolTip = null;
            undoStack = highlightTokenArrays = highlightOptions = null;
        });

    } // class CellEditMixin

    // exports ================================================================

    return CellEditMixin;

});
