/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/discretefiltergroup', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/checkgroup',
    'io.ox/office/spreadsheet/model/formula/tokenutils',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, CheckGroup, TokenUtils, gt) {

    'use strict';

    var // GUI label of the filter entry representing blank cells
        BLANK_CELLS_LABEL =
            //#. Represents a list entry in spreadsheet data filters that selects all blank (empty) cells in a column
            gt.pgettext('filter', 'Blank cells');

    // class DiscreteFilterGroup ==============================================

    /**
     * A check-list group control that shows all entries of a discrete filter
     * (all different values in a table column).
     *
     * @constructor
     *
     * @extends CheckGroup
     */
    function DiscreteFilterGroup(app) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        CheckGroup.call(this, { matcher: matcher, sorted: sortComparator, boxed: true, checkAll: true });

        // private methods ----------------------------------------------------

        /**
         * A case-insensitive list item matcher.
         */
        function matcher(value, entry) {
            return _.isString(value) && (value.toLowerCase() === entry.toLowerCase());
        }

        /**
         * Compares the passed sort order descriptors stored at the list
         * entries of this control.
         */
        function sortComparator(sortIndex1, sortIndex2) {
            var result = TokenUtils.compareValues(sortIndex1.result, sortIndex2.result);
            return (result === 0) ? TokenUtils.compareStrings(sortIndex1.display, sortIndex2.display) : result;
        }

        // public methods -----------------------------------------------------

        /**
         * Collects and inserts the cell entries of a table column into this
         * control.
         *
         * @param {TableModel} tableModel
         *  The table model.
         *
         * @param {Number} tableCol
         *  The zero-based column index, relative to the table range.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  control has been initialized.
         */
        this.initialize = function (tableModel, tableCol) {

            var // promise representing the result of this method
                promise = null;

            this.clearOptionButtons();

            // unfiltered table column: show entries of all visible rows; filtered table column: show
            // entries for all data rows that are not filtered by other columns (but that may be hidden)
            if (tableModel.isColumnFiltered(tableCol)) {
                promise = tableModel.queryFilterResult({ excludeTableCol: tableCol });
            } else {
                promise = $.when(tableModel.getVisibleDataRows());
            }

            // receive source data from document
            promise = promise.then(function (rowIntervals) {
                return tableModel.queryColumnData(tableCol, rowIntervals);
            });

            // fill the menu with the values
            promise = promise.then(function (cellArray) {

                var // each display string will be inserted once only (case-insensitive, always use first occurrence)
                    usedEntries = {},
                    // the filter attributes of the table column
                    attributes = tableModel.getColumnAttributes(tableCol).filter;

                // insert the new entries into the list
                _.each(cellArray, function (cellData) {

                    var // the label displayed in the list entry (preserve case)
                        // bug 36264: trim space characters but no other whitespace
                        label = cellData.display.replace(/^ +| +$/g, ''),
                        // the case-insensitive key used to detect duplicates
                        key = label.toLowerCase(),
                        // options for the list entry
                        options = null;

                    if (!(key in usedEntries)) {
                        usedEntries[key] = true;
                        options = (key.length === 0) ?
                            options = { label: BLANK_CELLS_LABEL, labelStyle: 'font-style:italic;', sortIndex: { result: null, display: '' } } :
                            options = { label: _.noI18n(label), sortIndex: { result: cellData.result, display: key } };
                        self.createOptionButton(key, options);
                    }
                });

                // initialize list entries according to current filter rules
                self.setValue((attributes.type === 'discrete') ? attributes.entries : _.keys(usedEntries));
            });

            return promise;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = self = null;
        });

    } // class DiscreteFilterGroup

    // exports ================================================================

    // derive this class from class CheckGroup
    return CheckGroup.extend({ constructor: DiscreteFilterGroup });

});
