/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/documentstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/editframework/utils/color',
     'io.ox/office/editframework/utils/border',
     'io.ox/office/editframework/utils/attributeutils',
     'io.ox/office/editframework/model/fontcollection',
     'io.ox/office/editframework/model/themecollection'
    ], function (Utils, TriggerObject, Color, Border, AttributeUtils, FontCollection, ThemeCollection) {

    'use strict';

    // class DocumentStyles ===================================================

    /**
     * Provides the style collections for all attribute families used in a
     * document, and other custom collections containing further special
     * formatting information for the document. Creates a font collection (an
     * instance of class FontCollection) assigned to the custom collection key
     * 'fonts', and a themes collection (an instance of class ThemeCollection)
     * assigned to the custom collection key 'themes'.
     *
     * Triggers the following events:
     * - 'change:attributes': After the document attributes have been changed
     *      via the method 'DocumentStyles.setDocumentAttributes()'. The event
     *      handlers receive the new attribute map, and the old attribute map.
     * - 'change:defaults': After the default values of formatting attributes
     *      have been changed.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {EditApplication} app
     *  The root application instance.
     *
     * @param {Object} attributeDefaultValues
     *  Default values of all document attributes supported by the current
     *  document.
     */
    function DocumentStyles(app, attributeDefaultValues) {

        var // self reference
            self = this,

            // the collection of document fonts
            fontCollection = new FontCollection(app),

            // the collection of document themes
            themeCollection = new ThemeCollection(app),

            // style collections mapped by attribute family
            styleCollections = {},

            // other custom collections
            customCollections = {},

            // global document attributes
            documentAttributes = _.copy(attributeDefaultValues),

            // attribute definitions registry (mapped by attribute family)
            definitionsRegistry = {},

            // default values for all supported attribute families
            defaultAttributeSet = {};

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // public methods -----------------------------------------------------

        /**
         * Returns the global attributes of the document.
         *
         * @param {Object}
         *  A deep copy of the global document attributes.
         */
        this.getDocumentAttributes = function () {
            return _.copy(documentAttributes, true);
        };

        /**
         * Changes the global attributes of the document.
         *
         * @param {Object} attributes
         *  The new document attributes.
         *
         * @returns {DocumentStyles}
         *  A reference to this instance.
         */
        this.setDocumentAttributes = function (attributes) {

            var // copy of the old attributes for change listeners
                oldAttributes = this.getDocumentAttributes(),
                // whether any attribute has been changed
                changed = false;

            // update all valid attributes
            _.each(attributes, function (value, name) {
                if ((name in documentAttributes) && !_.isEqual(documentAttributes[name], value)) {
                    documentAttributes[name] = _.copy(value, true);
                    changed = true;
                }
            });

            // notify all change listeners
            if (changed) {
                this.trigger('change:attributes', this.getDocumentAttributes(), oldAttributes);
            }

            return this;
        };

        /**
         * Registers formatting attributes for a specific attribute family.
         *
         * @param {String} family
         *  The name of the attribute family the new formatting attributes are
         *  associated with.
         *
         * @param {Object} definitions
         *  The attribute definitions map. Contains attribute definition
         *  objects for all attributes supported by the specified attribute
         *  family, mapped by the names of the attributes. This method may be
         *  called repeatedly for the same attribute family to add new
         *  attributes, or to overwrite existing attribute definitions. Each
         *  definition object contains the following properties:
         *  - {Number|String|Boolean|RegExp|Object} definition.def
         *      Specifies the default value of the attribute which will be used
         *      if neither a style sheet nor an explicit value exists for the
         *      attribute.
         *  - {String} [definition.scope='any']
         *      Specifies where the attribute can be used. If set to 'element',
         *      the attribute can be set at explicitly only (not in style
         *      sheets). If set to 'style', the attribute can be contained in
         *      style sheets only. By default, the attribute can be set
         *      explicitly as well as in style sheets.
         *  - {Function} [definition.merge]
         *      A function that will be called while merging attribute sets,
         *      for example while collecting attributes from style sheets and
         *      explicit attribute sets. Will be called to merge two existing
         *      values of this attribute, where the second value has to
         *      overwrite the first value in some way. Will be called in the
         *      context of this instance. The function receives the 'old'
         *      attribute value in the first parameter, and the 'new' attribute
         *      value in the second parameter. By default, the new value wins
         *      and the first value will be overwritten completely.
         *  The attribute definitions may contain any other user-defined
         *  properties.
         *
         * @returns {DocumentStyles}
         *  A reference to this instance.
         */
        this.registerAttributeDefinitions = function (family, definitions) {

            var // get existing or create a new entry in the global registry
                definitionsMap = definitionsRegistry[family] || (definitionsRegistry[family] = {}),
                // the container for the default attribute values
                defaultAttributes = defaultAttributeSet[family] || (defaultAttributeSet[family] = {});

            // store definitions, and the default attribute values
            _.each(definitions, function (definition, name) {
                if (_.isNull(definition.def) || _.isUndefined(definition.def) || _.isFunction(definition.def)) {
                    Utils.error('DocumentStyles.registerAttributeDefinitions(): invalid attribute default value, family=' + family + ', name=' + name + ', value=', definition.def);
                } else {
                    definitionsMap[name] = _.clone(definition);
                    defaultAttributes[name] = definition.def;
                }
            });

            // reset all old settings for child attributes
            _.each(definitionsMap, function (definition) { delete definition.childAttrs; });

            // collect settings for child attributes
            _.each(definitionsMap, function (definition, name) {
                var parentName = Utils.getStringOption(definition.parent, 'name');
                if (_.isString(parentName) && (parentName in definitionsMap)) {
                    var parentDef = definitionsMap[parentName],
                        childAttrs = parentDef.childAttrs || (parentDef.childAttrs = []);
                    childAttrs.push({ name: name, value: definition.parent.value });
                }
            });

            return this;
        };

        /**
         * Returns the static definitions map of all attributes registered for
         * the specified attribute family.
         *
         * @param {String} family
         *  The attribute family whose attribute definitions will be returned.
         *
         * @returns {Object|Null}
         *  A reference to the static attribute definitions map, or null if the
         *  passed attribute family is invalid.
         */
        this.getAttributeDefinitions = function (family) {
            return (family in definitionsRegistry) ? definitionsRegistry[family] : null;
        };

        /**
         * Returns the default attribute values of the specified attribute
         * families.
         *
         * @param {String|Array} families
         *  A single attribute family, or an array of attribute families.
         *
         * @returns {Object}
         *  The attribute set containing all default values of the specified
         *  attribute families.
         */
        this.getDefaultAttributes = function (families) {

            var // the attribute set containing all specified families
                attributeSet = {};

            _.chain(families).getArray().each(function (family) {
                if (family in defaultAttributeSet) {
                    attributeSet[family] = _.copy(defaultAttributeSet[family], true);
                }
            });

            return attributeSet;
        };

        /**
         * Changes the default values for various formatting attributes. These
         * values override the defaults of the attribute definitions passed in
         * the method 'DocumentStyles.registerAttributeDefinitions()', and will
         * be used before the values of any style sheet attributes and explicit
         * element attributes will be resolved.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set with new attribute default values.
         *
         * @returns {DocumentStyles}
         *  A reference to this instance.
         */
        this.setDefaultAttributes = function (attributeSet) {

            // insert passed values into the default attribute set
            _.each(attributeSet, function (attributes, family) {
                var defaultAttributes = defaultAttributeSet[family];
                if (_.isObject(attributes) && _.isObject(defaultAttributes)) {
                    _.each(attributes, function (value, name) {
                        if ((name in defaultAttributes) && !_.isNull(value) && !_.isUndefined(value)) {
                            defaultAttributes[name] = _.copy(value, true);
                        }
                    });
                }
            });

            // notify listeners
            this.trigger('change:defaults');
            return this;
        };

        /**
         * Builds an attribute set containing all formatting attributes of the
         * specified attribute families, set to the null value.
         *
         * @param {String|Array} styleFamilies
         *  The name of a single attribute family, or an array with names of
         *  attribute families.
         *
         * @returns {Object}
         *  An attribute set with null values for all attributes registered for
         *  the specified attribute families.
         */
        this.buildNullAttributes = function (families) {

            var // the resulting attribute set
                attributeSet = {};

            _.each(_.getArray(families), function (family) {

                var // the attribute definitions of the current family
                    definitions = self.getAttributeDefinitions(family),
                    // attribute values of the current family
                    attributes = attributeSet[family] = {};

                // add null values for all registered attributes
                _.each(definitions, function (definition, name) {
                    if (!definition.special) {
                        attributes[name] = null;
                    }
                });
            });

            return attributeSet;
        };

        /**
         * Extends the passed attribute set with the existing values of the
         * second attribute set. If the definitions of a specific attribute
         * contain a merger function, and both attribute sets contain an
         * attribute value, uses that merger function to merge the values from
         * both attribute sets, otherwise the value of the second attribute set
         * will be copied to the first attribute set.
         *
         * @param {Object} attributeSet1
         *  (in/out) The first attribute set that will be extended in-place, as
         *  map of attribute value maps (name/value pairs), keyed by attribute
         *  family.
         *
         * @param {Object} attributeSet2
         *  The second attribute set, as map of attribute value maps
         *  (name/value pairs), keyed by attribute family, whose attribute
         *  values will be inserted into the first attribute set.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result set.
         *
         * @returns {Object}
         *  A reference to the first passed and extended attribute set.
         */
        this.extendAttributes = function (attributeSet1, attributeSet2, options) {

            // add attributes existing in attributeSet2 to attributeSet1
            _.each(attributeSet2, function (attributes2, family) {

                var // the attribute definitions of the current family
                    definitions = this.getAttributeDefinitions(family);

                // copy style identifier directly (may be string or null)
                if (family === 'styleId') {
                    attributeSet1.styleId = attributes2;
                    return;
                }

                // restrict to valid attribute families
                if (!definitions || !_.isObject(attributes2)) { return; }

                // insert attribute values of attributes2
                var attributes1 = attributeSet1[family] || (attributeSet1[family] = {});
                _.each(attributes2, function (value, name) {
                    if (AttributeUtils.isRegisteredAttribute(definitions, name, options)) {

                        var // try to find merger function from attribute definition
                            merger = definitions[name].merge;

                        // either set return value from merger, or copy the attribute directly
                        if ((name in attributes1) && !_.isNull(attributes1[name]) && _.isFunction(merger)) {
                            attributes1[name] = merger.call(this, attributes1[name], value);
                        } else {
                            attributes1[name] = value;
                        }
                    }
                });
            }, this);

            return attributeSet1;
        };

        /**
         * Adds a style collection to the document.
         *
         * @internal
         *  Called from constructor functions of derived classes.
         *
         * @param {StyleCollection} styleCollection
         *  The new style collection to be inserted into the document.
         *
         * @returns {DocumentStyles}
         *  A reference to this instance.
         */
        this.addStyleCollection = function (styleCollection) {
            styleCollections[styleCollection.getStyleFamily()] = styleCollection;
            return this;
        };

        /**
         * Returns the style collection for the specified attribute family.
         *
         * @param {String} family
         *  The name of the attribute family.
         *
         * @returns {StyleCollection|Null}
         *  The specified style collection if existing, otherwise null.
         */
        this.getStyleCollection = function (family) {
            return (family in styleCollections) ? styleCollections[family] : null;
        };

        /**
         * Adds a custom formatting collection to the document.
         *
         * @internal
         *  Called from constructor functions of derived classes.
         *
         * @param {String} key
         *  The unique key of the custom collection.
         *
         * @param {ModelObject} collection
         *  The new custom formatting collection to be inserted.
         *
         * @returns {DocumentStyles}
         *  A reference to this instance.
         */
        this.addCustomCollection = function (key, collection) {
            customCollections[key] = collection;
            return this;
        };

        /**
         * Returns the specified custom formatting collection.
         *
         * @param {String} key
         *  The unique key of the custom collection.
         *
         * @returns {ModelObject|Null}
         *  The specified custom collection if existing, otherwise null.
         */
        this.getCustomCollection = function (key) {
            return (key in customCollections) ? customCollections[key] : null;
        };

        /**
         * Returns the collection with all registered fonts.
         *
         * @returns {FontCollection}
         *  The collection with all registered fonts.
         */
        this.getFontCollection = function () {
            return fontCollection;
        };

        /**
         * Returns the collection with all registered themes.
         *
         * @returns {ThemeCollection}
         *  The collection with all registered themes.
         */
        this.getThemeCollection = function () {
            return themeCollection;
        };

        /**
         * Returns whether the current theme contains scheme color definitions.
         *
         * @returns {Boolean}
         *  Whether the current theme contains scheme color definitions.
         */
        this.hasSchemeColors = function () {
            return themeCollection.getTheme().hasSchemeColors();
        };

        /**
         * Returns the value of the CSS 'font-family' attribute containing the
         * specified font name and all alternative font names.
         *
         * @param {String} fontName
         *  The name of of the font (case-insensitive).
         *
         * @returns {String}
         *  The value of the CSS 'font-family' attribute containing the
         *  specified font name and all alternative font names.
         */
        this.getCssFontFamily = function (fontName) {
            return fontCollection.getCssFontFamily(fontName);
        };

        /**
         * Converts the passed color attribute object to a color descriptor
         * with details about the resulting RGB color. Scheme colors will be
         * resolved by using the current theme.
         *
         * @param {Object} color
         *  The color object as used in operations.
         *
         * @param {String|Object} autoColor
         *  Additional information needed to resolve the automatic color. See
         *  method Color.getColorDetails() for details about this parameter.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  Color.getColorDetails().
         *
         * @returns {Object}
         *  The color descriptor constructed from the passed color object. See
         *  method Color.getColorDetails() for details about this object.
         */
        this.getColorDetails = function (color, autoColor, options) {
            // use the static helper function from module Color, pass current theme
            return Color.getColorDetails(color, autoColor, themeCollection.getTheme(), options);
        };

        /**
         * Converts the passed color attribute object to a CSS color value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} color
         *  The color object as used in operations.
         *
         * @param {String|Object} autoColor
         *  Additional information needed to resolve the automatic color. See
         *  method Color.getColorDetails() for details about this parameter.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  Color.getCssColor().
         *
         * @returns {String}
         *  The CSS color value converted from the passed color object.
         */
        this.getCssColor = function (color, autoColor, options) {
            // use the static helper function from module Color, pass current theme
            return Color.getCssColor(color, autoColor, themeCollection.getTheme(), options);
        };

        /**
         * Resolves the passed text color to an explicit color if it is set to
         * 'auto', according to the specified fill colors.
         *
         * @param {Object} textColor
         *  The source text color object, as used in operations.
         *
         * @param {Object[]} fillColors
         *  The source fill color objects, from outermost to innermost level.
         *
         * @returns {String}
         *  The CSS text color value converted from the passed color objects.
         */
        this.getCssTextColor = function (textColor, fillColors) {

            var // effective fill color
                effectiveFillColor = null;

            if (!_.isObject(textColor) || Color.isAutoColor(textColor)) {

                // find last non-transparent fill color in the passed array
                // TODO: merge partly-transparent colors?
                _.each(fillColors, function (fillColor) {
                    if (fillColor && !Color.isAutoColor(fillColor)) {
                        effectiveFillColor = fillColor;
                    }
                });

                // convert fill color to RGB (still transparent: fall-back to white)
                effectiveFillColor = this.getCssColor(effectiveFillColor || Color.WHITE, 'fill');

                // set text color to black or white according to fill color brightness
                textColor = Color.isDark(effectiveFillColor) ? Color.WHITE : Color.BLACK;
            }

            // convert to CSS color string
            return this.getCssColor(textColor, 'text');
        };

        /**
         * Returns the effective CSS attributes for the passed border value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} border
         *  The border object as used in operations.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preview=false]
         *      If set to true, the border will be rendered for a preview
         *      element in the GUI. The border width will be restricted to two
         *      pixels in that case.
         *  @param {Object} [options.autoColor]
         *      If specified, a replacement color to be used in case the border
         *      object contains the automatic color.
         *
         * @returns {Object}
         *  A map with CSS border attributes. Contains the property 'style'
         *  with the effective CSS border style as string; the property 'width'
         *  with the effective border width in pixels (as number, rounded to
         *  entire pixels); and the property 'color' with the effective CSS
         *  color (as string with leading hash sign).
         */
        this.getCssBorderAttributes = function (border, options) {
            // use the static helper function from module Border, pass current theme
            return Border.getCssBorderAttributes(border, themeCollection.getTheme(), options);
        };

        /**
         * Converts the passed border attribute object to a CSS border value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} border
         *  The border object as used in operations.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clearNone=false]
         *      If set to true, the return value for invisible borders will be
         *      the empty string instead of the keyword 'none'.
         *  @param {Boolean} [options.preview=false]
         *      If set to true, the border will be rendered for a preview
         *      element in the GUI. The border width will be restricted to two
         *      pixels in that case.
         *  @param {Object} [options.autoColor]
         *      If specified, a replacement color to be used in case the border
         *      object contains the automatic color.
         *
         * @returns {String}
         *  The CSS border value converted from the passed border object.
         */
        this.getCssBorder = function (border, options) {
            // use the static helper function from module Border, pass current theme
            return Border.getCssBorder(border, themeCollection.getTheme(), options);
        };

        /**
         * Returns the value of the CSS property 'text-decoration' for the
         * passed character attributes.
         *
         * @param {Object} charAttributes
         *  A map of character attributes.
         *
         * @returns {String}
         *  The resulting value of the CSS property 'text-decoration'.
         */
        this.getCssTextDecoration = function (charAttributes) {

            var result = '';

            function addToken(token) {
                if (result) { result += ' '; }
                result += token;
            }

            if (_.isBoolean(charAttributes.underline) && charAttributes.underline) { addToken('underline'); }
            if (_.isString(charAttributes.strike) && (charAttributes.strike !== 'none')) { addToken('line-through'); }

            return result || 'none';
        };

        // initialization -----------------------------------------------------

        // register own collections for public access
        this.addCustomCollection('fonts', fontCollection)
            .addCustomCollection('themes', themeCollection);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(customCollections, 'destroy');
            _.invoke(styleCollections, 'destroy');
            styleCollections = customCollections = null;
        });

    } // class DocumentStyles

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: DocumentStyles });

});
