/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/utils/drawingutils', [
    'io.ox/office/tk/utils'
], function (Utils) {

    'use strict';

    // static class DrawingUtils ==============================================

    /**
     * Contains common static helper functions to handle drawing objects.
     */
    var DrawingUtils = {};

    // methods ----------------------------------------------------------------

    /**
     * Sorts the passed array of logical drawing positions in-place. The
     * positions of embedded drawings will be kept behind the positions of
     * their parent drawings.
     *
     * @param {Array} positions
     *  (in/out) An array with logical drawing positions. Each drawing position
     *  (each array element) MUST be an array with integers. The passed array
     *  will be sorted in-place.
     *
     * @returns {Array}
     *  A reference to the passed original array that has been sorted in-place.
     */
    DrawingUtils.sortDrawingPositions = function (positions) {
        positions.sort(Utils.compareNumberArrays);
        return positions;
    };

    /**
     * Sorts the passed array of logical drawing positions in-place, and
     * removes the positions of all embedded drawings whose parent drawing
     * positions are contained in the array too.
     *
     * @param {Array} positions
     *  (in/out) An array with logical drawing positions. Each drawing position
     *  (each array element) MUST be an array with integers. The passed array
     *  will be sorted and modified in-place.
     *
     * @returns {Array}
     *  A reference to the passed original array that has been sorted and
     *  modified in-place.
     */
    DrawingUtils.optimizeDrawingPositions = function (positions) {

        var // array index for deleting positions from the array
            index = 0,
            // array elements while deleting positions
            pos1 = null, pos2 = null;

        // first, sort the positions
        DrawingUtils.sortDrawingPositions(positions);

        // remove positions of embedded drawings whose parent position is contained too
        while (index + 1 < positions.length) {
            pos1 = positions[index];
            pos2 = positions[index + 1];
            if ((pos1.length < pos2.length) && _.isEqual(pos1, pos2.slice(0, pos1.length))) {
                positions.splice(index + 1, 1);
            } else {
                index += 1;
            }
        }

        return positions;
    };

    // exports ================================================================

    return DrawingUtils;

});
