/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Edy Haryono <edy.haryono@open-xchange.com>
 */
define('io.ox/office/text/utils/textutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/text/dom',
    'io.ox/office/text/format/characterstyles'
], function (Utils, DOM, CharacterStyles) {

    'use strict';

    // static class TextUtils ================================================

    // copy tk/utils methods
    var TextUtils = _.extend({}, Utils);

    // methods ----------------------------------------------------------------

    /**
    * Returns the absolute CSS position of the passed DOM.Point, relative to a
    * specific root node and zoom factor.
    *
    * Info: Please use 'Position.getPixelPositionFromDomPoint' from the OX Text
    * pixel API.
    *
    * @param {DOM.Point} point
    *  the point object, from which the CSS position should be calculated
    *
    * @param {jQuery| HTMLElement} rootNode
    *  the calculated CSS positions are relative to this root node.
    *
    * @param {Number} zoomFactor
    *  the current active zoom factor of the application view.
    *
    * @returns {Object}
    *  with the following CSS position properties:
    *  - {Number} top
    *  - {Number} left
    */
    TextUtils.getCSSPositionFromPoint = function (point, rootNode, zoomFactor) {

        if (!point) { return null; }

        var caretSpan = $('<span>').text('|'),
            cursorElement = $(point.node.parentNode),
            cursorElementLineHeight = parseFloat(cursorElement.css('line-height')),
            zoom = zoomFactor / 100;

        // break cursor element on the text offset
        if (point.offset === 0) {
            cursorElement.before(caretSpan);
        } else {
            if (DOM.isSpan(cursorElement)) {
                DOM.splitTextSpan(cursorElement, point.offset, {append: true});
            }
            cursorElement.after(caretSpan);
        }

        // create caret overlay and calculate its position
        var caretTop = (caretSpan.offset().top - rootNode.offset().top) / zoom  - cursorElementLineHeight  + caretSpan.outerHeight(),
            caretLeft = (caretSpan.offset().left - rootNode.offset().left) / zoom;

        // restore original state of document
        caretSpan.remove();

        if (point.offset > 0) { CharacterStyles.mergeSiblingTextSpans(cursorElement, true); }

        return { top: caretTop, left: caretLeft};
    };

    /**
     * Returns whether the passed intervals overlap with at least one index.
     *
     * @param {Object} interval1
     *  The first index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @param {Object} interval2
     *  The second index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Boolean}
     *  Whether the passed intervals are overlapping.
     */
    TextUtils.intervalOverlapsInterval = function (interval1, interval2) {
        return (interval1.first <= interval2.last) && (interval2.first <= interval1.last);
    };


    // exports ================================================================

    return TextUtils;

});