/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/statuspane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/baseframework/view/toolpane',
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/controls',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, Forms, ToolPane, ToolBar, Labels, Controls, gt) {

    'use strict';

    // class StatusPane =======================================================

    /**
     * The status pane in spreadsheet documents containing the sheet tabs and
     * the subtotal results of the current selection.
     *
     * @constructor
     *
     * @extends ToolPane
     */
    function StatusPane(app) {

        var // self reference
            self = this,

            // application view instance
            docView = app.getView(),

            // the tool bar for all sheet controls
            sheetToolBar = null,

            // the sheet selector controls
            sheetGroup = null,
            largeSheetList = null,
            smallSheetList = null,
            moreMenuGroup = null,

            // label control for the formula in the active cell
            formulaLabel = null,
            // old visible boolean for hiding this bar when soft keyboard is open on Android devices
            visibleOverwrite = null;

        // base constructor ---------------------------------------------------

        ToolPane.call(this, app, { position: 'bottom', classes: 'status-pane standard-design' });

        // private methods ----------------------------------------------------

        /**
         * Refreshes the layout of the sheet selectors according to the current
         * width of the sheet tool bar.
         */
        function refreshSheetToolBar() {

            var // the node of the pane
                paneNode = self.getNode(),
                // total width of pane
                paneWidth = paneNode.width(),

                // the node of the sheet tool bar
                sheetNode = sheetToolBar.getNode(),
                // the width of the center node
                avaSheetWidth = Utils.getFloorNodeSize(sheetNode.closest('.float-wrapper')).width,

                // max width of the info node in percent
                minSheetWidth = 240;

            function getToolBarWidth() {
                return Utils.getCeilNodeSize(sheetNode).width;
            }

            // calculate and set remaining space for the specified sheet selector
            function setRemainingWidth(group) {
                var remainingWidth = group.getNode().outerWidth() + avaSheetWidth - getToolBarWidth();
                group.setWidth(remainingWidth);
            }

            // hide the sheet list selectors, show the sheet tab selector with automatic width
            largeSheetList.hide();
            smallSheetList.hide();
            sheetGroup.show();
            sheetGroup.setFullWidth();

            // add leading padding with the width of the row header panes
            sheetNode.css({ width: 'auto', paddingLeft: docView.getHeaderWidth() + 'px' });

            if (avaSheetWidth < getToolBarWidth()) {
                sheetNode.css({ paddingLeft: '' });

                if (avaSheetWidth < 380) {
                    // check min width for sheets
                    if (avaSheetWidth < minSheetWidth) {
                        // set fix min width for sheets
                        avaSheetWidth = minSheetWidth;
                        // set remaining width (incl. 20px padding) to formula
                        formulaLabel.setWidth(paneWidth - minSheetWidth - 20);
                    } else {
                        // reset width of formula
                        formulaLabel.setWidth('auto');
                    }
                    // show large sheet drop-down list for small devices
                    sheetGroup.hide();
                    largeSheetList.show();
                    // calculate and set remaining space for the sheet drop-down button
                    largeSheetList.setWidth(avaSheetWidth);
                    setRemainingWidth(largeSheetList);
                } else {
                    // show additional sheet selector controls
                    smallSheetList.show();
                    // calculate and set remaining space for the sheet tabs
                    setRemainingWidth(sheetGroup);
                }
            } else {
                // reset width of formula
                formulaLabel.setWidth('auto');
            }

            // reset sheet component to its current width
            sheetNode.css({ width: avaSheetWidth });
        }

        // initialization -----------------------------------------------------

        // create the 'More' drop-down menu with all standard actions for sheets
        moreMenuGroup = new Controls.CompoundButton(app, { label: Labels.MORE_LABEL, tooltip: gt('More sheet operations') })
            .addGroup('sheet/rename/dialog', new Controls.Button({ label: Labels.RENAME_SHEET_LABEL }))
            .addGroup('sheet/copy/dialog', new Controls.Button({ label: Labels.COPY_SHEET_LABEL }))
            .addGroup('sheet/delete', new Controls.Button({ label: Labels.DELETE_SHEET_LABEL }))
            .addGroup('sheet/visible', new Controls.Button({ label: Labels.HIDE_SHEET_LABEL, value: false }))
            .addSeparator()
            .addGroup('sheet/reorder/dialog', new Controls.Button({ label: Labels.REORDER_SHEETS_LABEL }))
            .addGroup('sheet/showall', new Controls.Button({ label: Labels.SHOW_SHEETS_LABEL }));

        this.addViewComponent(sheetToolBar = new ToolBar(app)
            .addGroup('view/sheet/active', largeSheetList = new Controls.ActiveSheetList(app, { showNames: true }))
            .addGap()
            .addGroup('view/sheet/active', smallSheetList = new Controls.ActiveSheetList(app))
            .addGap()
            .addGroup('view/sheet/active', sheetGroup = new Controls.ActiveSheetGroup(app))
            .addGap()
            .addGroup('sheet/insert', new Controls.Button({ icon: 'fa-plus', tooltip: gt('Insert sheet') }))
            .addGroup('view/sheet/more', moreMenuGroup)
        );

        this.addViewComponent(new ToolBar(app)
            .addGroup('view/status', new Controls.DynamicLabel(), { visibleKey: 'view/status' })
            .addGroup('view/subtotals/type', new Controls.SubtotalList(app), { visibleKey: 'view/subtotals/type' })
            .addGroup('view/formula', formulaLabel = new Controls.DynamicLabel({ classes: 'formula-text', setToolTip: true }), { visibleKey: 'view/formula'} ),
        { targetArea: 'trailing' });

        // refresh sheet tool bar after sheet tabs have been inserted, deleted, or changed
        this.listenTo(sheetGroup, 'change:sheets', refreshSheetToolBar);
        this.on('pane:layout', refreshSheetToolBar);

        // always hide the status pane when an internal application error occurs
        this.listenTo(app, 'docs:state:error', function () { self.hide(); });

        if (_.browser.Android) {
            //".window-container" is oriented in screen with "top left right bottom" set to zero,
            //when the soft keyboard on Android devices pops up, the status bar stays over the keyboard
            //this makes no sense and looks ugly, so we detect the open keyboard and hide the status pane
            app.registerWindowResizeHandler(function () {
                var softKeyboard = Utils.isSoftKeyboardOpen();
                if (softKeyboard) {
                    if (visibleOverwrite === null && self.isVisible()) {
                        visibleOverwrite = true;
                        self.hide();
                    }
                } else {
                    if (visibleOverwrite !== null) {
                        self.toggle(visibleOverwrite);
                    }
                    visibleOverwrite = null;
                }
            } );
        }

        // destroy class members on destruction
        this.registerDestructor(function () {
            app = self = docView = null;
            sheetToolBar = sheetGroup = largeSheetList = smallSheetList = moreMenuGroup = formulaLabel = null;
        });

    } // class StatusPane

    // exports ================================================================

    // derive this class from class ToolPane
    return ToolPane.extend({ constructor: StatusPane });

});
