/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/headercontextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, Button, ContextMenu, SheetUtils, gt) {

    'use strict';

    // class HeaderContextMenu ================================================

    /**
     * A context menu for header panes. Provides menu actions to manipulate
     * entire columns/rows in the active sheet.
     *
     * @constructor
     *
     * @extends ContextMenu
     */
    function HeaderContextMenu(app, headerPane, columns) {

        var // self reference
            self = this,

            // the document model and view
            docModel = app.getModel(),
            docView = app.getView();

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, app, headerPane.getNode(), {
            selector: '.cell', // only on cell header nodes (not on resizers)
            mouseDown: true, // open context menu on 'mousedown' instead of 'mouseup'
            // do not show context menu at all in locked sheets, or during cell edit mode
            enableKey: ['sheet/unlocked', 'view/cell/editmode/off']
        });

        // private methods ----------------------------------------------------

        /**
         * Inserts column-specific menu entries into the context menu.
         */
        function initColumnMenu() {
            self.addGroup('column/insert', new Button({ label: gt('Insert columns') }))
                .addGroup('column/delete', new Button({ label: gt('Delete columns') }))
                .addSeparator()
                .addGroup('column/width/optimal', new Button({ label: gt('Set optimal column width') }))
                .addGroup('column/hide', new Button({ label: gt('Hide columns') }))
                .addGroup('column/show', new Button({ label: gt('Show hidden columns') }));
        }

        /**
         * Inserts row-specific menu entries into the context menu.
         */
        function initRowMenu() {
            self.addGroup('row/insert', new Button({ label: gt('Insert rows') }))
                .addGroup('row/delete', new Button({ label: gt('Delete rows') }))
                .addSeparator()
                .addGroup('row/height/optimal', new Button({ label: gt('Set optimal row height') }))
                .addGroup('row/hide', new Button({ label: gt('Hide rows') }))
                .addGroup('row/show', new Button({ label: gt('Show hidden rows') }));
        }

        /**
         * Selects the column/row that has been clicked with the right mouse
         * button, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {

            var // the source event that caused to open the context menu
                sourceEvent = event.sourceEvent,
                // the index of the column/row to be selected
                index = Utils.getElementAttributeAsInteger(sourceEvent.currentTarget, 'data-index', -1);

            // functor to check whether the column/row index is contained in a range address
            function rangeContainsIndex(range) {
                return docModel.isFullRange(range, columns) && SheetUtils.rangeContainsIndex(range, index, columns);
            }

            // Do nothing if no valid cell node has been clicked, or when the event
            // is of type 'contextmenu' (otherwise, the context menu will be reopened
            // after moving the mouse with pressed right mouse button).
            if ((index < 0) || (sourceEvent.type === 'contextmenu')) {
                event.preventDefault();
                return;
            }

            // select the clicked column/row if it is not contained in the selection
            if (!_.any(docView.getSelectedRanges(), rangeContainsIndex)) {
                docView.grabFocus();
                headerPane.trigger('select:start', index, 'select');
                headerPane.trigger('select:end', index, true);
            }
        }

        // initialization -----------------------------------------------------

        // suppress system context menu on other nodes (resizers etc.)
        headerPane.getNode().on('contextmenu', ':not(.cell)', false);

        // lazy initialization of the menu items according to the orientation of the header pane
        this.one('popup:beforeshow', columns ? initColumnMenu : initRowMenu);

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // destroy all class members
        this.registerDestructor(function () {
            app = headerPane = self = docModel = docView = null;
        });

    } // class HeaderContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: HeaderContextMenu });

});
