/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/mergecellspicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet'
], function (Utils, RadioList, SheetUtils, gt) {

    'use strict';

    var // the label or tooltip for this control
        TOOLTIP_LABEL = gt('Merge or unmerge cells'),

        // icon names for merge states
        MERGE_ICON = 'docs-merged-cells-on',
        MERGE_HOR_ICON = 'docs-merged-cells-horizontal',
        MERGE_VERT_ICON = 'docs-merged-cells-vertical',
        UNMERGE_ICON = 'docs-merged-cells-off';

    // class MergeCellsPicker =================================================

    /**
     * The selector for merged ranges, as drop-down list with split button.
     *
     * @constructor
     *
     * @extends RadioList
     */
    function MergeCellsPicker(app, initOptions) {

        var // self reference
            self = this,

            // whether to show the 'merge' menu item
            showMergeItem = false,
            // whether to show the 'merge horizontal' menu item
            showHMergeItem = false,
            // whether to show the 'merge vertical' menu item
            showVMergeItem = false,
            // whether to show the 'unmerge' menu item
            showUnmergeItem = false;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: MERGE_ICON,
            tooltip: TOOLTIP_LABEL,
            dropDownVersion: { label: TOOLTIP_LABEL }
        }, initOptions, {
            highlight: _.identity,
            splitValue: 'toggle',
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Updates the icon according to the merged state of the selection.
         */
        function updateHandler() {
            self.setIcon(showUnmergeItem ? UNMERGE_ICON : MERGE_ICON);
        }

        /**
         * Updates the visibility flags of the list items after the selection
         * in the active sheet has been changed, or after the merge collection
         * of the active sheet has changed.
         */
        function updateVisibilityFlags() {

            var // the collection of merged ranges in the active sheet
                mergeCollection = app.getView().getMergeCollection(),
                // the selected cell ranges
                ranges = app.getView().getSelectedRanges();

            // show the 'merge' list item, if at least one range consists of more than a cell or a merged range
            showMergeItem = _.any(ranges, function (range) {
                var mergedRange = (SheetUtils.getCellCount(range) > 1) ? mergeCollection.getMergedRange(range.start) : null;
                return !mergedRange || !SheetUtils.rangeContainsRange(mergedRange, range);
            });

            // show the 'horizontal' list item, if at least one range does not consist entirely of horizontally merged cells
            showHMergeItem = _.any(ranges, function (range) {
                return (range.start[0] < range.end[0]) && (range.start[1] < range.end[1]) && !mergeCollection.isHorizontallyMerged(range);
            });

            // show the 'vertical' list item, if at least one range does not consist entirely of vertically merged cells
            showVMergeItem = _.any(ranges, function (range) {
                return (range.start[0] < range.end[0]) && (range.start[1] < range.end[1]) && !mergeCollection.isVerticallyMerged(range);
            });

            // show the 'unmerge' list item, if at least one range overlaps with a merged range
            showUnmergeItem = mergeCollection.rangesOverlapAnyMergedRange(ranges);
        }

        // initialization -----------------------------------------------------

        // create all drop-down menu items
        this.createOptionButton('merge',      { icon: MERGE_ICON,      label: gt('Merge cells'),              visible: function () { return showMergeItem; } })
            .createOptionButton('horizontal', { icon: MERGE_HOR_ICON,  label: gt('Merge cells horizontally'), visible: function () { return showHMergeItem; } })
            .createOptionButton('vertical',   { icon: MERGE_VERT_ICON, label: gt('Merge cells vertically'),   visible: function () { return showVMergeItem; } })
            .createOptionButton('unmerge',    { icon: UNMERGE_ICON,    label: gt('Unmerge cells'),            visible: function () { return showUnmergeItem; } });

        // update the icon in the drop-down button
        this.registerUpdateHandler(updateHandler);

        // update state flags used for visibility of list items (to keep the 'visible' callback functions simple and fast)
        this.listenTo(app.getImportPromise(), 'done', function () {
            updateVisibilityFlags();
            self.listenTo(app.getView(), 'change:selection insert:merged delete:merged', updateVisibilityFlags);
        });

    } // class MergeCellsPicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: MergeCellsPicker });

});
