/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/sheetmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/model/namecollection',
    'io.ox/office/spreadsheet/model/colrowcollection',
    'io.ox/office/spreadsheet/model/mergecollection',
    'io.ox/office/spreadsheet/model/cellcollection',
    'io.ox/office/spreadsheet/model/tablecollection',
    'io.ox/office/spreadsheet/model/validationcollection',
    'io.ox/office/spreadsheet/model/drawing/drawingcollection',
    'io.ox/office/spreadsheet/model/viewsettingsmixin',
    'io.ox/office/spreadsheet/model/sheetoperationsgenerator'
], function (Utils, TimerMixin, Color, Border, AttributedModel, DrawingUtils, Config, Operations, SheetUtils, PaneUtils, NameCollection, ColRowCollection, MergeCollection, CellCollection, TableCollection, ValidationCollection, SheetDrawingCollection, ViewSettingsMixin, SheetOperationsGenerator) {

    'use strict';

    var // definitions for sheet view attributes
        SHEET_VIEW_ATTRIBUTES = {

            /**
             * Sheet selection (cells and drawing objects), as object with the
             * following properties:
             * - {Array} ranges
             *      The addresses of all selected cell ranges. MUST NOT be an
             *      empty array.
             * - {Number} activeRange
             *      Element index of the active range in the 'ranges' property.
             * - {Number[]} activeCell
             *      The address of the active cell in the active range.
             * - {Array} drawings
             *      The positions of all selected drawing frames in the sheet,
             *      as integer arrays (without leading sheet index).
             */
            selection: {
                def: {
                    ranges: [{ start: [0, 0], end: [0, 0] }],
                    activeRange: 0,
                    activeCell: [0, 0],
                    drawings: []
                },
                validate: function (selection) {
                    return _.isObject(selection) ? selection : Utils.BREAK;
                }
            },

            /**
             * Zoom factor in the sheet, as floating-point number. The value 1
             * represents a zoom factor of 100%.
             */
            zoom: {
                def: 1,
                validate: function (zoom) {
                    return _.isNumber(zoom) ? Utils.minMax(zoom, SheetUtils.MIN_ZOOM, SheetUtils.MAX_ZOOM) : Utils.BREAK;
                }
            },

            /**
             * Whether to display the grid lines between all cells.
             */
            showGrid: {
                def: true,
                validate: function (show) { return !!show; }
            },

            /**
             * The color of the grid lines drawn between all cells.
             */
            gridColor: {
                def: Color.AUTO,
                validate: function (color) { return _.isObject(color) ? color : null; }
            },

            /**
             * The type of the current view split. Has no effect, if both view
             * properties 'splitWidth' and 'splitHeight' are set to the value
             * zero (view is not split at all). Supports the following values:
             * - 'split': Dynamic split mode (movable split lines).
             * - 'frozen': Frozen split mode (a fixed number of columns and/or
             *      rows are frozen, split lines are not movable). When leaving
             *      frozen split mode, the split lines will be hidden.
             * - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
             *      mode). When leaving frozen split mode, the split lines
             *      remain visible and become movable (back to 'split' mode).
             */
            splitMode: {
                def: 'split',
                validate: /^(split|frozen|frozenSplit)$/
            },

            /**
             * The inner width of the left panes in view split mode. In dynamic
             * split mode, this value is interpreted in 1/100 of millimeters.
             * In frozen split mode, this value specifies the number of columns
             * shown in the frozen panes. Must not be negative. If this value
             * is zero, the view is not split into left and right panes (but
             * may be split into upper and lower panes).
             */
            splitWidth: {
                def: 0,
                validate: function (splitPos) {
                    return _.isNumber(splitPos) ? Math.max(splitPos, 0) : 0;
                }
            },

            /**
             * The inner height of the upper panes in view split mode. In
             * dynamic split mode, this value is interpreted in 1/100 of
             * millimeters. In frozen split mode, this value specifies the
             * number of rows shown in the frozen panes. Must not be negative.
             * If this value is zero, the view is not split into top and bottom
             * panes (but may be split into left and right panes).
             */
            splitHeight: {
                def: 0,
                validate: function (splitPos) {
                    return _.isNumber(splitPos) ? Math.max(splitPos, 0) : 0;
                }
            },

            /**
             * The identifier of the active pane in split mode. Must be one of
             * 'topLeft', 'bottomLeft', 'topRight', or 'bottomRight'.
             */
            activePane: {
                def: 'bottomRight',
                validate: /^(top|bottom)(Left|Right)$/
            },

            /**
             * The identifier of an entire active pane side. Must be one of the
             * values null, 'left', 'right', 'top', or 'bottom'. Used for
             * example while selecting entire columns or rows.
             */
            activePaneSide: {
                def: null,
                validate: function (paneSide) {
                    return (/^(left|right|top|bottom)$/).test(paneSide) ? paneSide : null;
                }
            },

            /**
             * The horizontal scroll position in the left panes. Must be an
             * object with the properties 'index' (index of the first column
             * visible in the panes), and 'ratio' (offset inside the column, as
             * ratio of the current column width).
             */
            anchorLeft: {
                def: { index: 0, ratio: 0 },
                validate: function (anchor) { return _.isObject(anchor) ? anchor : null; }
            },

            /**
             * The horizontal scroll position in the right panes. Must be an
             * object with the properties 'index' (index of the first column
             * visible in the panes), and 'ratio' (offset inside the column, as
             * ratio of the current column width).
             */
            anchorRight: {
                def: { index: 0, ratio: 0 },
                validate: function (anchor) { return _.isObject(anchor) ? anchor : null; }
            },

            /**
             * The vertical scroll position in the upper panes. Must be an
             * object with the properties 'index' (index of the first row
             * visible in the panes), and 'ratio' (offset inside the row, as
             * ratio of the current row height).
             */
            anchorTop: {
                def: { index: 0, ratio: 0 },
                validate: function (anchor) { return _.isObject(anchor) ? anchor : null; }
            },

            /**
             * The vertical scroll position in the lower panes. Must be an
             * object with the properties 'index' (index of the first row
             * visible in the panes), and 'ratio' (offset inside the row, as
             * ratio of the current row height).
             */
            anchorBottom: {
                def: { index: 0, ratio: 0 },
                validate: function (anchor) { return _.isObject(anchor) ? anchor : null; }
            },

            /**
             * Highlighted ranges beside the regular cell selection, used for
             * example in formula edit mode to visualize the ranges used in the
             * formula. Allowed values are null (no highlighting), or an array
             * of range addresses. Each range address object in this array may
             * contain the following additional properties:
             * - {String} [range.id]
             *      An arbitrary identifier for the range.
             * - {Boolean} [range.draggable=false]
             *      Whether the range is intended to be movable and resizable
             *      with mouse or touch gestures.
             */
            highlightRanges: {
                def: null,
                validate: function (ranges) { return _.isArray(ranges) ? ranges : null; }
            },

            /**
             * The array index of the highlighted range currently tracked.
             */
            highlightIndex: {
                def: null,
                validate: function (index) { return _.isNumber(index) ? index : null; }
            },

            /**
             * Specific cell ranges that are in a special activated state, used
             * for example for ranges that will be inserted into a formula
             * while in formula edit mode. These ranges will be rendered in a
             * special way over the regular cell selection. Allowed values are
             * null (no active ranges available), or a selection object as
             * described for the 'selection' view attribute (see above), but
             * with the following differences:
             * - The property 'drawings' must not be contained in this object.
             * - The array property 'ranges' may be an empty array.
             */
            activeSelection: {
                def: null,
                validate: function (data) { return _.isObject(data) ? data : null; }
            },

            /**
             * Additional data for the current cell range while auto-fill
             * tracking is active. Allowed values are null (auto-fill tracking
             * not active), or an object with the following properties:
             * - {String} border
             *      Which border of the selected range will be modified by the
             *      auto-fill. Allowed values are 'left', 'right', 'top', or
             *      'bottom'.
             * - {Number} count
             *      The number of columns/rows to extend the selected range
             *      with (positive values), or to shrink the selected range
             *      (negative values).
             * This attribute will be ignored, if the current sheet selection
             * consists of more than one cell range.
             */
            autoFillData: {
                def: null,
                validate: function (data) { return _.isObject(data) ? data : null; }
            },

            /**
             * Data for the active table range and table column that will be
             * used to apply filtering and sorting settings via the application
             * controller from arbitrary GUI elements. Allowed values are null
             * (no active table available), or an object with the following
             * properties:
             * - {TableModel} tableModel
             *      The model instance of the active table range.
             * - {Number} tableCol
             *      The zero-based index of the active table column, relative
             *      to the cell range covered by the table.
             */
            activeTableData: {
                def: null,
                validate: function (data) { return _.isObject(data) ? data : null; }
            }
        };

    // global private functions ===============================================

    /**
     * Returns a rectangle descriptor for the passed column and row entry.
     */
    function getRectangle(colPosition, rowPosition) {
        return {
            leftHmm: colPosition.offsetHmm,
            topHmm: rowPosition.offsetHmm,
            widthHmm: colPosition.sizeHmm,
            heightHmm: rowPosition.sizeHmm,
            left: colPosition.offset,
            top: rowPosition.offset,
            width: colPosition.size,
            height: rowPosition.size
        };
    }

    // class SheetModel =======================================================

    /**
     * Represents a single sheet in the spreadsheet document.
     *
     * Additionally to the events triggered by the base class AttributedModel,
     * an instance of this class triggers the following events:
     * - 'change:viewattributes'
     *      After at least one view attribute has been changed. Event handlers
     *      receive an incomplete attribute map containing all changed view
     *      attributes with their new values.
     * - 'insert:columns', 'change:columns', 'delete:columns'
     *      The respective '*:entries' event forwarded from the column
     *      collection of this sheet. See class ColRowCollection for details.
     * - 'insert:rows', 'change:rows', 'delete:rows'
     *      The respective '*:entries' event forwarded from the column
     *      collection of this sheet. See class ColRowCollection for details.
     * - 'insert:merged', 'delete:merged'
     *      The respective event forwarded from the merge collection of this
     *      sheet. See class MergeCollection for details.
     * - 'insert:table', 'change:table', 'delete:table'
     *      The respective event forwarded from the table collection of this
     *      sheet. See class TableCollection for details.
     * - 'change:cells', 'change:usedarea'
     *      The respective event forwarded from the cell collection of this
     *      sheet. See class CellCollection for details.
     * - 'change:drawing', 'insert:drawing', 'delete:drawing'
     *      The respective event forwarded from the drawing collection of this
     *      sheet. See class SheetDrawingCollection for details.
     *
     * @constructor
     *
     * @extends AttributedModel
     * @extends TimerMixin
     * @extends ViewSettingsMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application instance with the document model containing this sheet.
     *
     * @param {String|SheetModel} sheetType
     *  The type identifier of this sheet. Supported values are:
     *  - 'worksheet': A regular worksheet with cells and formulas.
     *  - 'chartsheet': A sheet that consists entirely of a chart object.
     *  - 'macrosheet': A sheet with cells and formulas that contains sheet
     *      macros. This sheet type is a legacy feature of MS Excel. These
     *      sheets will not be shown in the user interface.
     *  - 'dialogsheet': A sheet that consists entirely of a user dialog. This
     *      sheet type is a legacy feature of MS Excel. These sheets will not
     *      be shown in the user interface.
     *  Alternatively, a source sheet model, if this constructor will be used
     *  as copy constructor.
     *
     * @param {Object} [initialAttrs]
     *  An attribute set with initial formatting attributes for the sheet.
     */
    function SheetModel(app, sheetType, initialAttrs) {

        var // self reference
            self = this,

            // the document model and other model collections
            docModel = app.getModel(),
            documentStyles = docModel.getDocumentStyles(),
            styleCollection = docModel.getStyleCollection('cell'),

            // the collection of defined names in this sheet
            nameCollection = null,

            // the column and row collections (size and formatting of all columns/rows)
            colCollection = null,
            rowCollection = null,

            // the collection of merged cells in this sheet
            mergeCollection = null,

            // the collection of table ranges (filter, sorting)
            tableCollection = null,

            // the collection of cell ranges with validation settings
            validationCollection = null,

            // the cell collection
            cellCollection = null,

            // the collection of drawing objects contained in this sheet
            drawingCollection = null,

            // handlers to adjust range addresses after inserting/deleting columns/rows
            transformationHandlers = [];

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, initialAttrs, { trigger: 'always', families: 'sheet column row' });
        TimerMixin.call(this);
        ViewSettingsMixin.call(this, app, SHEET_VIEW_ATTRIBUTES);

        // private methods ----------------------------------------------------

        /**
         * Updates the passed selection according to the current merged ranges
         * in this sheet.
         */
        function getExpandedSelection(selection) {
            selection = _.copy(selection, true);
            _.each(selection.ranges, function (range, index) {
                // expand range to merged ranges, unless an entire column/row is selected
                if (!docModel.isColRange(range) && !docModel.isRowRange(range)) {
                    selection.ranges[index] = mergeCollection.expandRangeToMergedRanges(range);
                }
            });
            return selection;
        }

        /**
         * Builds an attribute set containing all cell formatting attributes
         * set to the null value, but without the 'styleId' property.
         */
        function buildNullAttributes() {
            var attributes = styleCollection.buildNullAttributes();
            delete attributes.styleId;
            delete attributes.character.url;
            return attributes;
        }

        /**
         * Returns whether additional columns or rows can be inserted into the
         * sheet.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals.
         *
         * @param {Boolean} columns
         *  True for columns, or false for rows.
         *
         * @returns {Boolean}
         *  Whether additional columns or rows can be inserted, according to
         *  the passed intervals and the number of used columns/rows in the
         *  sheet.
         */
        function intervalsInsertable(intervals, columns) {

            var // the number of inserted columns/rows
                insertCount = Utils.getSum(SheetUtils.getUnifiedIntervals(intervals), SheetUtils.getIntervalSize),
                // the number of used columns/rows
                usedCount = columns ? cellCollection.getUsedCols() : cellCollection.getUsedRows(),
                // the available number of columns/rows in the sheet
                maxCount = docModel.getMaxIndex(columns) + 1;

            // the number of used columns/rows, and the number of inserted columns/rows
            // must not exceed the total number of columns/rows in the sheet
            return (0 < insertCount) && ((usedCount + insertCount) <= maxCount);
        }

        /**
         * Returns whether the specified columns or rows can be deleted from
         * the sheet.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals.
         *
         * @param {Boolean} columns
         *  True for columns, or false for rows.
         *
         * @returns {Boolean}
         *  Whether the columns can be deleted.
         */
        function intervalsDeleteable(intervals, columns) {

            var // the number of deleted columns/rows
                deleteCount = Utils.getSum(intervals, SheetUtils.getIntervalSize),
                // the available number of columns/rows in the sheet
                maxCount = docModel.getMaxIndex(columns) + 1;

            // prevent deleting all columns/rows (entire sheet) at once
            return (0 < deleteCount) && (deleteCount < maxCount);
        }

        /**
         * Checks the passed ranges that are about to be changed by a specific
         * formatting operation, and returns an error code, if a limit has been
         * exceeded.
         *
         * @param {Array} ranges
         *  The cell ranges to be checked.
         *
         * @returns {String}
         *  The empty string, if all passed counts are inside the respective
         *  limits, otherwise one of the following error codes:
         *  - 'cols:overflow': Too many entire columns in the ranges.
         *  - 'rows:overflow': Too many entire rows in the ranges.
         *  - 'cells:overflow': Too many cells in the ranges.
         */
        function checkCellLimits(ranges) {

            var // number of changed entire columns
                cols = 0,
                // number of changed entire rows
                rows = 0,
                // number of changed cells
                cells = 0;

            // count columns, rows, and cells in all ranges
            _.each(ranges, function (range) {
                if (docModel.isColRange(range)) {
                    cols += SheetUtils.getColCount(range);
                } else if (docModel.isRowRange(range)) {
                    rows += SheetUtils.getRowCount(range);
                } else {
                    cells += SheetUtils.getCellCount(range);
                }
            });

            // check all limits
            return (cells > SheetUtils.MAX_FILL_CELL_COUNT) ? 'cells:overflow' :
                (rows > SheetUtils.MAX_CHANGE_ROWS_COUNT) ? 'rows:overflow' :
                (cols > SheetUtils.MAX_CHANGE_COLS_COUNT) ? 'cols:overflow' : '';
        }

        /**
         * Returns the column/row interval shown in the leading panes of the
         * split view.
         *
         * @param {Boolean} columns
         *  Whether to return the column interval of the left sheet area
         *  (true), or the row interval of the top sheet area (false).
         *
         * @returns {Object|Null}
         *  The column/row interval shown in the leading panes. If the view is
         *  not split in the specified direction, this method returns null.
         */
        function getSplitInterval(columns) {

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // the value of the split size view attribute (number of columns/rows in frozen split)
                splitSize = self.getViewAttribute(columns ? 'splitWidth' : 'splitHeight'),
                // start and end scroll anchor in the leading sheet area
                startAnchor = null, endAnchor = null,
                // the resulting column/row interval
                interval = null;

            // no split: return null
            if (splitSize === 0) { return null; }

            // build column/row interval in frozen split mode directly
            startAnchor = self.getViewAttribute(columns ? 'anchorLeft' : 'anchorTop');
            if (self.hasFrozenSplit()) {
                return { first: startAnchor.index, last: startAnchor.index + splitSize - 1 };
            }

            // get index of last visible column/row, and build the resulting interval
            endAnchor = collection.getScrollAnchorByOffset(collection.convertScrollAnchorToHmm(startAnchor) + splitSize);
            interval = { first: startAnchor.index, last: endAnchor.index };

            // exclude last column/row, if visible less than half of its size
            if ((interval.first < interval.last) && (endAnchor.ratio < 0.5)) {
                interval.last -= 1;
            }

            // exclude first column/row, if visible less than half of its size
            if ((interval.first < interval.last) && (startAnchor.ratio > 0.5)) {
                interval.first += 1;
            }

            return interval;
        }

        /**
         * Returns the width or height of the leading area in a sheet in split
         * mode, in 1/100 of millimeters, or in pixels according to the current
         * sheet zoom factor. In frozen split mode, returns the total size of
         * the frozen columns/rows.
         *
         * @param {Boolean} columns
         *  Whether to return the width of the left sheet area (true), or the
         *  height of the top sheet area (false).
         *
         * @param {Boolean} pixel
         *  Whether to return pixels (true) or 1/100 of millimeters (false).
         *
         * @returns {Number}
         *  The width or height of the leading area if the sheet is in split
         *  mode, otherwise zero.
         */
        function getSplitSize(columns, pixel) {

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // the column interval in frozen split mode
                interval = null,
                // the size in dynamic split mode
                splitSize = 0;

            // return total size of frozen columns/rows
            if (self.hasFrozenSplit()) {
                interval = getSplitInterval(columns);
                return interval ? collection.getIntervalPosition(interval)[pixel ? 'size' : 'sizeHmm'] : 0;
            }

            // return size of dynamic split
            splitSize = self.getViewAttribute(columns ? 'splitWidth' : 'splitHeight');
            return pixel ? self.convertHmmToPixel(splitSize) : splitSize;
        }

        /**
         * Updates the frozen split attributes after columns or rows have been
         * inserted into or deleted from this sheet.
         */
        function transformFrozenSplit(interval, insert, columns) {

            var // attribute names
                LEADING_ANCHOR_NAME = columns ? 'anchorLeft' : 'anchorTop',
                TRAILING_ANCHOR_NAME = columns ? 'anchorRight' : 'anchorBottom',
                SPLIT_SIZE_NAME = columns ? 'splitWidth' : 'splitHeight',
                // the frozen column/row interval (will be null if not frozen)
                frozenInterval = self.hasFrozenSplit() ? getSplitInterval(columns) : null,
                // the number of inserted columns/rows
                insertSize = 0,
                // intersection of frozen interval and deleted interval
                intersectInterval = null,
                // the new view attributes
                attributes = {};

            // no frozen columns/rows available
            if (!frozenInterval) { return; }

            if (insert) {

                // the number of inserted columns/rows
                insertSize = SheetUtils.getIntervalSize(interval);

                // inserted before frozen interval: update scroll anchor
                if (interval.first < frozenInterval.first) {
                    attributes[LEADING_ANCHOR_NAME] = { index: frozenInterval.first + insertSize, ratio: 0 };
                    attributes[TRAILING_ANCHOR_NAME] = { index: frozenInterval.last + 1 + insertSize, ratio: 0 };
                }

                // inserted inside frozen interval: update size of frozen interval
                else if (interval.first <= frozenInterval.last) {
                    attributes[SPLIT_SIZE_NAME] = SheetUtils.getIntervalSize(frozenInterval) + insertSize;
                    attributes[TRAILING_ANCHOR_NAME] = { index: frozenInterval.last + 1 + insertSize, ratio: 0 };
                }

            } else {

                // deleted inside frozen interval: update size of frozen interval
                if ((intersectInterval = SheetUtils.getIntersectionInterval(interval, frozenInterval))) {
                    attributes[SPLIT_SIZE_NAME] = SheetUtils.getIntervalSize(frozenInterval) - SheetUtils.getIntervalSize(intersectInterval);
                }

                // deleted before frozen interval: update leading scroll anchor
                if (interval.first < frozenInterval.first) {
                    attributes[LEADING_ANCHOR_NAME] = { index: interval.first, ratio: 0 };
                }

                // deleted before or inside frozen interval: update trailing scroll anchor
                if (interval.first <= frozenInterval.last) {
                    attributes[TRAILING_ANCHOR_NAME] = { index: interval.first, ratio: 0 };
                }
            }

            // commit the new view attributes
            self.setViewAttributes(attributes);
        }

        /**
         * Invokes all registered range transformation handlers, after columns
         * or rows have been inserted into or deleted from this sheet, and
         * triggers an appropriate event.
         */
        function invokeTransformationHandlers(eventType, interval, insert, columns) {

            // invoke all registered transformation handlers
            _.each(transformationHandlers, function (handler) {
                handler.call(self, interval, insert, columns);
            });

            // notify all listeners
            self.trigger(eventType, interval);
        }

        /**
         * Updates the selection after a drawing object has been inserted into
         * this sheet.
         */
        function insertDrawingHandler(event, drawingModel, insertPosition) {

            var // deep copy of the current selection
                selection = self.getViewAttribute('selection'),
                // the length of the position array
                length = insertPosition.length;

            if (selection.drawings.length === 0) { return; }

            // update position of all drawings following the inserted drawing
            _.each(selection.drawings, function (position) {

                // nothing to do, if the selected drawing is a parent object
                if (position.length < length) { return; }
                // nothing to do, if the selected drawing is located in another parent object
                if (Utils.compareNumberArrays(insertPosition, position, length - 1) !== 0) { return; }

                // inserted drawing is a predecessor in the same parent
                if (insertPosition[length - 1] <= position[length - 1]) {
                    position[length - 1] += 1;
                }
            });

            self.setViewAttribute('selection', selection);
        }

        /**
         * Updates the selection after a drawing object has been deleted from
         * this sheet.
         */
        function deleteDrawingHandler(event, drawingModel, deletePosition) {

            var // deep copy of the current selection
                selection = self.getViewAttribute('selection'),
                // the length of the position array
                length = deletePosition.length;

            if (selection.drawings.length === 0) { return; }

            // update position of all drawings following the deleted drawing
            Utils.iterateArray(selection.drawings, function (position, index) {

                // nothing to do, if the selected drawing is a parent object
                if (position.length < length) { return; }
                // nothing to do, if the selected drawing is located in another parent object
                if (Utils.compareNumberArrays(deletePosition, position, length - 1) !== 0) { return; }

                // deleted drawing or any of its children has been deleted
                if (deletePosition[length - 1] === position[length - 1]) {
                    selection.drawings.splice(index, 1);
                }
                // deleted drawing is a predecessor in the same parent
                else if (deletePosition[length - 1] < position[length - 1]) {
                    position[length - 1] -= 1;
                }
            }, { reverse: true });

            self.setViewAttribute('selection', selection);
        }

        // operation generators -----------------------------------------------

        /**
         * Invokes the passed callback function with activated custom 'action
         * processing mode' in the document model which causes the sheet cell
         * renderers to block until all operations have been generated and
         * applied. Intended to be used to generate document operations without
         * applying them but simply sending them silently to the server. This
         * method creates a new operations generator, invokes the callback
         * function, and then sends all operations contained in the generator
         * to the server, but does not apply the operations. The passed
         * callback function must implement all necessary updates in the
         * document model by itself. Additionally, registers an attribute
         * change listener and a delete listener at the drawing collection of
         * this sheet. All changes made directly or indirectly at the drawing
         * collection (insert, delete, or modify columns or rows) will be
         * inserted as drawing operations into the operations generator
         * afterwards. After the callback function has returned, the event
         * handlers will be removed from the drawing collection.
         *
         * @param {Function} callback
         *  The callback function that will be invoked with activated action
         *  processing mode. Receives an empty sheet operations generator as
         *  first parameter. May return the promise of a Deferred object to
         *  defer the action processing mode until the promise has been
         *  resolved or rejected.
         *
         * @returns {jQuery.Promise}
         *  The promise that will be resolved or rejected according to the
         *  result of the callback function.
         */
        function createAndSendSheetOperations(callback) {

            // simulate actions processing mode (blocks the cell renderer)
            return self.createAndApplyOperations(function (generator) {

                var // the own sheet index
                    sheet = self.getIndex(),
                    // positions of deleted drawing objects
                    deletedDrawings = [],
                    // changed drawing anchor attributes, mapped by position
                    changedDrawingAttrs = {},
                    // the promise returned by this method
                    promise = null;

                // collects deleted drawing objects (only on sheet level)
                function deleteDrawingHandler(event, drawingModel, position) {
                    if (position.length === 1) {
                        deletedDrawings.push(position);
                    }
                }

                // collects changed anchor attributes of moved drawing objects (only on sheet level)
                function changeDrawingHandler(event, drawingModel, position, type, newAttributes, oldAttributes) {
                    if ((type === 'attributes') && (position.length === 1)) {
                        var key = JSON.stringify(position);
                        if (!(key in changedDrawingAttrs)) { changedDrawingAttrs[key] = { oldValues: oldAttributes.drawing }; }
                        changedDrawingAttrs[key].newValues = _.copy(newAttributes.drawing, true);
                    }
                }

                // invoke the callback function with registered drawing event handlers
                drawingCollection.on('delete:drawing', deleteDrawingHandler);
                drawingCollection.on('change:drawing', changeDrawingHandler);

                // invoke passed callback function
                promise = $.when(callback.call(self, generator, sheet));

                // always unregister the temporary drawing event handlers
                promise.always(function () {
                    drawingCollection.off('delete:drawing', deleteDrawingHandler);
                    drawingCollection.off('change:drawing', changeDrawingHandler);
                });

                // process the changed drawings, unless the callback has returned a rejected promise
                promise.done(function () {

                    // remove entries for deleted drawings from 'changedAttributes' map
                    _.each(deletedDrawings, function (position) {
                        delete changedDrawingAttrs[JSON.stringify(position)];
                    });

                    // create 'setDrawingAttributes' operations for all changed drawings
                    _.each(changedDrawingAttrs, function (attributes, key) {
                        var position = JSON.parse(key);
                        // delete unchanged attributes
                        _.each(attributes.newValues, function (value, name) {
                            if (_.isEqual(attributes.oldValues[name], value)) { delete attributes.newValues[name]; }
                        });
                        // generate operations if some values really have been changed
                        if (!_.isEmpty(attributes.newValues)) {
                            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { scope: 'filter', attrs: { drawing: attributes.newValues } });
                        }
                    });

                    // create 'deleteDrawing' operations for all deleted drawings (reverse order!)
                    DrawingUtils.sortDrawingPositions(deletedDrawings);
                    Utils.iterateArray(deletedDrawings, function (position) {
                        generator.generateDrawingOperation(Operations.DELETE_DRAWING, position, { scope: 'filter' });
                    }, { reverse: true });
                });

                return promise;

            }, { filter: false }); // do not apply operations, only send to server
        }

        /**
         * Generates all operations to insert new columns or rows into the
         * sheet. The columns/rows will be inserted into the sheet, and all
         * operations needed to refresh drawing objects will be generated too.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals.
         *
         * @param {Boolean} columns
         *  Either true to insert new columns into the sheet, or false to
         *  insert new rows into the sheet.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  operations have been generated successfully.
         */
        function generateInsertIntervalOperations(generator, intervals, columns) {

            var // the name of the insert operation
                OPERATION_NAME = columns ? Operations.INSERT_COLUMNS : Operations.INSERT_ROWS,
                // the column or row collection
                collection = columns ? colCollection : rowCollection;

            // unify and sort the passed intervals
            intervals = SheetUtils.getUnifiedIntervals(intervals);

            // create the operations for all intervals in reverse order (!), and insert
            // the columns/rows already (causes generation of the indirect drawing operations)
            return self.iterateArraySliced(intervals, function (interval) {
                generator.generateIntervalOperation(OPERATION_NAME, interval);
                collection.insertEntries(interval);
            }, { reverse: true });
        }

        /**
         * Generates all operations to delete columns or rows from the sheet.
         * The columns/rows will be deleted from the sheet, and all operations
         * needed to refresh drawing objects will be generated too.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals.
         *
         * @param {Boolean} columns
         *  Either true to delete columns from the sheet, or false to delete
         *  rows from the sheet.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  operations have been generated successfully.
         */
        function generateDeleteIntervalOperations(generator, intervals, columns) {

            var // the name of the delete operation
                OPERATION_NAME = columns ? Operations.DELETE_COLUMNS : Operations.DELETE_ROWS,
                // the column or row collection
                collection = columns ? colCollection : rowCollection;

            // unify and sort the passed intervals
            intervals = SheetUtils.getUnifiedIntervals(intervals);

            // create the operations for all intervals in reverse order (!), and delete
            // the columns/rows already (causes generation of the indirect drawing operations)
            return self.iterateArraySliced(intervals, function (interval) {
                generator.generateIntervalOperation(OPERATION_NAME, interval);
                collection.deleteEntries(interval);
            }, { reverse: true });
        }

        /**
         * Returns a reduced copy of the passed column/row intervals, that will
         * contain only the formatting attributes that would change the current
         * column/row formatting.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals. The array
         *  MUST be sorted, the intervals MUST NOT overlap each other. Each
         *  interval object MUST contain an additional object property 'attrs'
         *  containing the (incomplete) column/row attributes, as simple object
         *  (NOT mapped as a 'column'/'row' sub-object).
         *
         * @param {Boolean} columns
         *  Whether the passed intervals are column intervals (true), or row
         *  intervals (false).
         *
         * @returns {Array}
         *  The resulting intervals (sub set of the passed intervals) that will
         *  contain only the formatting attributes that would change the
         *  current column/row formatting.
         */
        function reduceAttributedIntervals(intervals, columns) {

            var // attribute family (column or row)
                FAMILY_NAME = columns ? 'column' : 'row',
                // the column or row collection
                collection = columns ? colCollection : rowCollection,
                // the remaining intervals with different attributes compared to current formatting
                resultIntervals = [];

            // remove attributes and intervals that are equal to the current formatting
            // (regardless whether set explicitly or derived from sheet default attributes)
            collection.iterateEntries(intervals, function (entry, uniqueInterval, origInterval) {

                var // the current merged column/row attributes of the visited entry
                    mergedAttrs = entry.attributes[FAMILY_NAME],
                    // the current explicit column/row attributes of the visited entry
                    explicitAttrs = entry.explicit[FAMILY_NAME],
                    // the new column/row attributes to be set at the current interval
                    newAttrs = _.copy(origInterval.attrs, true);

                // remove all attributes that are about to be changed but already have that
                // value, or that are about to be deleted but do not exist in the interval
                _.each(newAttrs, function (value, name) {
                    if (_.isNull(value) ? !(name in explicitAttrs) : _.isEqual(value, mergedAttrs[name])) {
                        delete newAttrs[name];
                    }
                });

                // ignore intervals that do not change anything (no remaining new attributes)
                if (!_.isEmpty(newAttrs)) {
                    uniqueInterval.attrs = newAttrs;
                    resultIntervals.push(uniqueInterval);
                }
            }, { hidden: 'all', unique: true });

            // merge adjacent intervals in the reduced array of intervals
            Utils.iterateArray(resultIntervals, function (interval, index) {

                var // the preceding interval
                    prevInterval = resultIntervals[index - 1];

                // merge with preceding interval, if (remaining) attributes are equal
                if (prevInterval && (prevInterval.last + 1 === interval.first) && _.isEqual(prevInterval.attrs, interval.attrs)) {
                    prevInterval.last = interval.last;
                    resultIntervals.splice(index, 1);
                }
            }, { reverse: true });

            return resultIntervals;
        }

        function mergeAttributedIntervals(sourceIntervals, targetIntervals) {

            var // the resulting intervals returned by this method
                resultIntervals = [];

            // arrays will be modified in-place for simplicity, work on clones of cloned objects
            sourceIntervals = _.map(sourceIntervals, _.clone);
            targetIntervals = _.map(targetIntervals, _.clone);

            // merge the source intervals array over the target intervals array
            Utils.iterateArray(sourceIntervals, function (sourceInterval, sourceIndex) {

                var // index of last target interval completely preceding the current source interval
                    lastIndex = Utils.findLastIndex(targetIntervals, function (targetInterval) {
                        return targetInterval.last < sourceInterval.first;
                    }, { sorted: true });

                // move all preceding target intervals into the result
                if (lastIndex >= 0) {
                    resultIntervals.push(targetIntervals.splice(0, lastIndex + 1));
                }

                // merge the current interval over all covered target intervals
                while (true) {

                    // no more intervals available in target array: append all remaining source intervals and exit the loop
                    if (targetIntervals.length === 0) {
                        // push current interval separately (may be modified, see below)
                        resultIntervals.push(sourceInterval);
                        resultIntervals = resultIntervals.concat(sourceIntervals.slice(sourceIndex + 1));
                        return Utils.BREAK;
                    }

                    var // first remaining interval in the target array
                        targetInterval = targetIntervals[0];

                    // entire source interval fits into the gap before the current target interval
                    if (sourceInterval.last < targetInterval.first) {
                        resultIntervals.push(sourceInterval);
                        // continue the loop with next source interval
                        return;
                    }

                    // one interval starts before the other interval: split the leading interval
                    if (sourceInterval.first < targetInterval.first) {
                        resultIntervals.push(_.extend({}, sourceInterval, { last: targetInterval.first - 1 }));
                        sourceInterval.first = targetInterval.first;
                    } else if (targetInterval.first < sourceInterval.first) {
                        resultIntervals.push(_.extend({}, targetInterval, { last: sourceInterval.first - 1 }));
                        targetInterval.first = sourceInterval.first;
                    }

                    // both intervals start at the same position now: merge their shared part
                    resultIntervals.push({
                        first: sourceInterval.first,
                        last: Math.min(sourceInterval.last, targetInterval.last),
                        attrs: _.extend({}, sourceInterval.attrs, targetInterval.attrs)
                    });

                    // source and target intervals have equal size: continue with next source and target interval
                    if (sourceInterval.last === targetInterval.last) {
                        targetIntervals.shift();
                        return;
                    }

                    // source interval ends before target interval: shorten the target interval
                    if (sourceInterval.last < targetInterval.last) {
                        targetInterval.first = sourceInterval.last + 1;
                        return;
                    }

                    // otherwise: shorten source interval, continue with next target interval
                    sourceInterval.first = targetInterval.last + 1;
                    targetIntervals.shift();
                }
            });

            // append remaining target intervals
            return resultIntervals.concat(targetIntervals);
        }

        /**
         * Generates all operations to modify the attributes of columns or rows
         * in the sheet. The columns/rows will be formatted, and all operations
         * needed to refresh drawing objects will be generated too.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {Object|Array} intervals
         *  A single index interval, or an array of index intervals. The array
         *  MUST be sorted, the intervals MUST NOT overlap each other. Each
         *  interval object MUST contain an additional object property 'attrs'
         *  containing the (incomplete) column/row attributes, as simple object
         *  (NOT mapped as a 'column'/'row' sub-object).
         *
         * @param {Boolean} columns
         *  Either true to modify the column attributes, or false to modify the
         *  row attributes.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  operations have been generated successfully, or that will be
         *  rejected with one of the following error codes:
         *  - 'cols:overflow': Too many columns in the passed intervals (only
         *      returned, if parameter 'columns' is true).
         *  - 'rows:overflow': Too many rows in the passed intervals (only
         *      returned, if parameter 'columns' is false).
         */
        function generateChangeIntervalOperations(generator, intervals, columns) {

            var // reduce intervals to attributes different to current formatting
                changeIntervals = reduceAttributedIntervals(intervals, columns);

            // nothing to do, if passed intervals would not change anything
            if (changeIntervals.length === 0) { return $.when(); }

            var // attribute family (column or row)
                FAMILY_NAME = columns ? 'column' : 'row',
                // maximum number of columns/rows to be changed at the same time
                MAX_CHANGE_COUNT = columns ? SheetUtils.MAX_CHANGE_COLS_COUNT : SheetUtils.MAX_CHANGE_ROWS_COUNT,
                // the name of the attribute operation
                OPERATION_NAME = columns ? Operations.SET_COLUMN_ATTRIBUTES : Operations.SET_ROW_ATTRIBUTES,
                // the column or row collection
                collection = columns ? colCollection : rowCollection,
                // the null attributes that will be applied for all columns/rows when changing sheet defaults
                nullAttributes = null,
                // the intervals containing the attributes to be restored after changing the sheet defaults
                restoreIntervals = null;

            // try to find sheet default formatting for very large intervals (but not when deleting attributes)
            if (!_.any(changeIntervals, function (interval) { return _.any(interval.attrs, _.isNull); })) {

                var // group all remaining intervals into a map of arrays, keyed by equal attributes (to find largest intervals)
                    intervalMap = _.groupBy(changeIntervals, function (interval) { return JSON.stringify(interval.attrs); }),
                    // find the largest array of intervals in the attribute map (sum of all interval sizes per map element)
                    maxIntervals = _.reduce(intervalMap, function (memoEntry, mapEntry) {
                        mapEntry.size = Utils.getSum(mapEntry, SheetUtils.getIntervalSize);
                        return (memoEntry && (memoEntry.size >= mapEntry.size)) ? memoEntry : mapEntry;
                    }, null);

                // set very large intervals as sheet default attributes
                if (maxIntervals && (maxIntervals.size > collection.getMaxIndex() - MAX_CHANGE_COUNT)) {

                    var // the column/row attributes to be set as sheet default attributes
                        sheetAttrs = maxIntervals[0].attrs,
                        // the column/row intervals that are not affected by the changed sheet default attributes
                        remainingIntervals = SheetUtils.getRemainingIntervals(collection.getFullInterval(), maxIntervals);

                    // build the attribute set with null attributes intended to be set at all columns/rows
                    nullAttributes = Utils.mapProperties(sheetAttrs, _.constant(null));

                    // create interval array to restore current attributes for the remaining intervals
                    // that will not be affected by the changed sheet default attributes
                    restoreIntervals = [];
                    collection.iterateEntries(remainingIntervals, function (entry, uniqueInterval) {

                        var // the current merged column/row attributes of the visited entry
                            mergedAttrs = entry.attributes[FAMILY_NAME];

                        // insert current values of the attributes that will be changed at the sheet
                        uniqueInterval.attrs = {};
                        _.each(sheetAttrs, function (value, name) {
                            uniqueInterval.attrs[name] = mergedAttrs[name];
                        });
                        restoreIntervals.push(uniqueInterval);

                    }, { hidden: 'all', unique: true });

                    // generate a 'setSheetAttributes' operation and immediately update the sheet attributes
                    // (the column/row collection, and the drawing objects will update themselves)
                    sheetAttrs = Utils.makeSimpleObject(FAMILY_NAME, sheetAttrs);
                    generator.generateSheetOperation(Operations.SET_SHEET_ATTRIBUTES, { attrs: sheetAttrs });
                    self.setAttributes(sheetAttrs);

                    // reduce the passed intervals again according to the new sheet defaults
                    changeIntervals = reduceAttributedIntervals(intervals, columns);
                }
            }

            // bug 34641: restrict maximum number of modified columns/rows
            if (Utils.getSum(changeIntervals, SheetUtils.getIntervalSize) > MAX_CHANGE_COUNT) {
                return $.Deferred().reject(columns ? 'cols:overflow' : 'rows:overflow');
            }

            // merge the intervals that will restore the original formatting after changing
            // the sheet default attributes with the new attributes passed to this method
            if (restoreIntervals) {
                changeIntervals = mergeAttributedIntervals(changeIntervals, restoreIntervals);
                changeIntervals = reduceAttributedIntervals(changeIntervals, columns);
            }

            // add an interval over all columns/rows with null attributes, if sheet defaults have been changed
            if (nullAttributes) {
                changeIntervals.unshift(collection.getFullInterval());
                changeIntervals[0].attrs = nullAttributes;
            }

            // generate the column/row operations, and simultaneously update the column/row collection
            return self.iterateArraySliced(changeIntervals, function (interval) {
                var attributes = Utils.makeSimpleObject(FAMILY_NAME, interval.attrs);
                generator.generateIntervalOperation(OPERATION_NAME, interval, { attrs: attributes });
                collection.setAttributes(interval, attributes);
            });
        }

        /**
         * Invokes the passed callback function with activated custom 'action
         * processing mode' in the document model. This method receives the
         * name of a table range, and passed the table model to the callback
         * function, otherwise behaves similarly to the method
         * createAndSendTableOperations().
         *
         * @param {String} tableName
         *  The name of an existing table range. The empty string addresses the
         *  anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet. If the table range does not
         *  exist, this method will exit with the error code 'table:missing'.
         *
         * @param {Function} callback
         *  The callback function that will be invoked with activated action
         *  processing mode. Receives the following parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      An empty sheet operations generator.
         *  (2) {TableModel} tableModel
         *      A reference to the table model specified with the parameter
         *      'tableName' passed to this method.
         *  May return the promise of a Deferred object to defer the action
         *  processing mode until the promise has been resolved or rejected.
         *
         * @returns {jQuery.Promise}
         *  The promise that will be resolved or rejected according to the
         *  result of the callback function.
         */
        function createAndSendTableOperations(tableName, callback) {

            var // the table model
                tableModel = tableCollection.getTable(tableName);

            // check existence of the table range
            if (!tableModel) { return $.Deferred().reject('table:missing'); }

            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return callback.call(this, generator, tableModel);
            });
        }

        /**
         * Generates all needed operations to refresh the passed table model.
         * Evaluates the current filter settings, and updates the visibility of
         * all rows covered by the table range. Also generates all implicit
         * operations for drawing objects.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {TableModel} tableModel
         *  The model of the table range to be refreshed.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  table has been refreshed successfully, or rejected with one of the
         *  following error codes:
         *  - 'rows:overflow': The current filter settings would modify (show
         *      or hide) too many rows at once.
         */
        function generateRefreshTableOperations(generator, tableModel) {

            var // start with the entire data range
                dataRange = tableModel.getDataRange();

            // do nothing, if no data rows are available
            if (!dataRange) { return $.when(); }

            var // the entire table range
                tableRange = tableModel.getRange(),
                // the effective table range, e.g. after expanding auto-filter data range
                effectiveRange = SheetUtils.getBoundingRange(tableRange, dataRange);

            // update the effective table range (data range may have been expanded)
            if (!_.isEqual(tableRange, effectiveRange)) {
                generator.generateTableOperation(Operations.CHANGE_TABLE, tableModel.getName(), effectiveRange);
                tableModel.setRange(effectiveRange);
            }

            // update the visibility state of all filter rows
            return tableModel.queryFilterResult().then(function (rowIntervals) {

                var // the entire row interval of the data range
                    dataRowInterval = SheetUtils.getRowInterval(dataRange),
                    // the new visible row intervals
                    visibleIntervals = _.map(rowIntervals, function (rowInterval) {
                        return { first: rowInterval.first + dataRowInterval.first, last: rowInterval.last + dataRowInterval.first };
                    }),
                    // the new hidden row intervals
                    hiddenIntervals = SheetUtils.getRemainingIntervals(dataRowInterval, visibleIntervals),
                    // all intervals in a single array
                    intervals = null;

                // add visibility property to the modified interval arrays, before merging and sorting the arrays
                _.each(visibleIntervals, function (interval) { interval.attrs = { visible: true }; });
                _.each(hiddenIntervals, function (interval) { interval.attrs = { visible: false }; });

                // create a single sorted array of row intervals
                intervals = visibleIntervals.concat(hiddenIntervals);
                intervals.sort(function (interval1, interval2) { return interval1.first - interval2.first; });

                // create operations and update all affected rows
                return generateChangeIntervalOperations(generator, intervals, false);
            });
        }

        /**
         * Generates an attribute operation for the specified cell range,
         * without changing the content values of the cells. If the range
         * covers entire columns or entire rows, generates the appropriate
         * operation automatically.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator that will be filled with the operations.
         *
         * @param {Object} range
         *  The address of the cell range to generate the operation for.
         *
         * @param {Object} attributes
         *  The formatting attribute set to be set at the cell range.
         *
         * @param {Object} [properties]
         *  Additional properties to be inserted into the generated operation.
         */
        function generateFillRangeOperation(generator, range, attributes, properties) {

            // add the attributes to the operation
            properties = _.extend({ attrs: attributes }, properties);

            // decide whether to generate a column, row, or cell range operation
            if (docModel.isColRange(range)) {
                generator.generateIntervalOperation(Operations.SET_COLUMN_ATTRIBUTES, SheetUtils.getColInterval(range), properties);
            } else if (docModel.isRowRange(range)) {
                properties.attrs.row = { customFormat: true };
                generator.generateIntervalOperation(Operations.SET_ROW_ATTRIBUTES, SheetUtils.getRowInterval(range), properties);
            } else {
                generator.generateRangeOperation(Operations.FILL_CELL_RANGE, range, properties);
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Creates and returns a clone of this sheet model instance.
         *
         * @param {Number} targetIndex
         *  The zero-based index of the new cloned sheet.
         *
         * @returns {SheetModel}
         *  A clone of this sheet model with the same sheet type, with cloned
         *  instances of all collections this sheet model contains.
         */
        this.clone = function (targetIndex) {
            // construct a new sheet model with hidden parameters that cause cloning from this sheet
            return new SheetModel(app, sheetType, this.getExplicitAttributes(true), this, targetIndex);
        };

        /**
         * Returns the type of this sheet.
         *
         * @returns {String}
         *  The type identifier of this sheet.
         */
        this.getType = function () {
            return sheetType;
        };

        /**
         * Returns the current sheet index of this sheet model.
         *
         * @returns {Number}
         *  The current zero-based sheet index of this sheet model.
         */
        this.getIndex = function () {
            return docModel.getSheetIndexOfModel(this);
        };

        /**
         * Returns the name of this sheet model.
         *
         * @returns {String}
         *  The name of this sheet model.
         */
        this.getName = function () {
            return docModel.getSheetName(this.getIndex());
        };

        /**
         * Returns the collection of all defined names contained in this sheet.
         *
         * @returns {NameCollection}
         *  The collection of all defined names in this sheet.
         */
        this.getNameCollection = function () {
            return nameCollection;
        };

        /**
         * Returns the collection of all columns in this sheet, containing
         * their size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all columns in this sheet.
         */
        this.getColCollection = function () {
            return colCollection;
        };

        /**
         * Returns the collection of all rows in this sheet, containing their
         * size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all rows in this sheet.
         */
        this.getRowCollection = function () {
            return rowCollection;
        };

        /**
         * Returns the collection of all merged cells, containing their ranges.
         *
         * @returns {MergeCollection}
         *  The collection of all merged cells in this sheet.
         */
        this.getMergeCollection = function () {
            return mergeCollection;
        };

        /**
         * Returns the collection of all known cells in this sheet.
         *
         * @returns {CellCollection}
         *  The collection of all known cells in this sheet.
         */
        this.getCellCollection = function () {
            return cellCollection;
        };

        /**
         * Returns the collection of all table ranges in this sheet.
         *
         * @returns {TableCollection}
         *  The collection of all table ranges in this sheet.
         */
        this.getTableCollection = function () {
            return tableCollection;
        };

        /**
         * Returns the collection of all cell ranges with validation settings.
         *
         * @returns {ValidationCollection}
         *  The collection of all cell ranges with validation settings in this
         *  sheet.
         */
        this.getValidationCollection = function () {
            return validationCollection;
        };

        /**
         * Returns the drawing collection of this sheet, containing the models
         * of all drawing objects contained in this sheet.
         *
         * @returns {DrawingCollection}
         *  The collection of all drawing models in this sheet.
         */
        this.getDrawingCollection = function () {
            return drawingCollection;
        };

        /**
         * Registers a callback function that will be invoked after columns or
         * rows have been inserted into or deleted from this sheet, but before
         * this sheet model triggers the according insert/delete events.
         *
         * @param {Function} handler
         *  The callback handler function. Will be invoked in the context of
         *  this sheet model instance. Receives the following parameters:
         *  (1) {Object} interval
         *      The column/row interval of the insert/delete operation, with
         *      the index properties 'first' and 'last'.
         *  (2) {Boolean} insert
         *      Whether the specified column/row interval has been inserted
         *      into this sheet (true), or deleted from this sheet (false).
         *  (3) {Boolean} columns
         *      Whether the specified interval is a column interval (true), or
         *      a row interval (false).
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.registerTransformationHandler = function (handler) {
            transformationHandlers.push(handler);
            return this;
        };

        /**
         * Unregisters a callback function that has been registered with the
         * method SheetModel.registerTransformationHandler() before.
         *
         * @param {Function} handler
         *  The callback handler function to be unregistered.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.unregisterTransformationHandler = function (handler) {
            transformationHandlers = _.without(transformationHandlers, handler);
            return this;
        };

        // operations generator -----------------------------------------------

        /**
         * Creates a new operations generator associated to this sheet with
         * additional helper methods.
         *
         * @returns {SheetOperationsGenerator}
         *  A new operations generator associated to this sheet.
         */
        this.createOperationsGenerator = function () {
            return new SheetOperationsGenerator(app, this.getIndex());
        };

        /**
         * Creates a new operations generator associated to this sheet, invokes
         * the callback function, and then applies all operations contained in
         * the generator, and sends them to the server. This method works the
         * same as the method EditModel.createAndApplyOperations() but creates
         * a dedicated operations generator for this sheet (see method
         * SheetModel.createOperationsGenerator() for details).
         *
         * @param {Function} callback
         *  The callback function. Receives a new operations generator for this
         *  sheet (instance of the class SheetOperationsGenerator) as first
         *  parameter. May return the promise of a Deferred object to defer
         *  applying and sending the operations until the promise has been
         *  resolved. Will be called in the context of this instance.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  EditModel.createAndApplyOperations() except the option 'generator'.
         *
         * @returns {jQuery.Promise}
         *  The promise that will be resolved after the operations have been
         *  applied and sent successfully (with the result of that promise); or
         *  rejected, if the callback has returned a rejected promise (with the
         *  result of that promise), or if sending the operations has failed
         *  for internal reasons (with the string value 'internal').
         */
        this.createAndApplyOperations = function (callback, options) {
            options = Utils.extendOptions(options, { generator: this.createOperationsGenerator() });
            return docModel.createAndApplyOperations(_.bind(callback, this), options);
        };

        // dimensions ---------------------------------------------------------

        /**
         * Converts the passed length in pixels to a length in 1/100 of
         * millimeters, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in pixels.
         *
         * @returns {Number}
         *  The converted length in 1/100 of millimeters.
         */
        this.convertPixelToHmm = function (length) {
            return Utils.convertLengthToHmm(length / this.getEffectiveZoom(), 'px');
        };

        /**
         * Converts the passed length in 1/100 of millimeters to a length in
         * pixels, according to the current sheet zoom factor.
         *
         * @param {Number} length
         *  The length in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The converted length in pixels.
         */
        this.convertHmmToPixel = function (length) {
            return Utils.convertHmmToLength(length * this.getEffectiveZoom(), 'px', 1);
        };

        /**
         * Returns the absolute position of the passed cell range in the sheet,
         * in 1/100 of millimeters, as well as in pixels according to the
         * current sheet zoom factor.
         *
         * @param {Object} range
         *  The range address.
         *
         * @returns {Object}
         *  The absolute position and size of the range in the sheet, in the
         *  following properties:
         *  - {Number} leftHmm
         *      The left offset of the cell range, in 1/100 of millimeters.
         *  - {Number} topHmm
         *      The top offset of the cell range, in 1/100 of millimeters.
         *  - {Number} widthHmm
         *      The total width of the cell range, in 1/100 of millimeters.
         *  - {Number} heightHmm
         *      The total height of the cell range, in 1/100 of millimeters.
         *  - {Number} left
         *      The left offset of the cell range, in pixels.
         *  - {Number} top
         *      The top offset of the cell range, in pixels.
         *  - {Number} width
         *      The total width of the cell range, in pixels.
         *  - {Number} height
         *      The total height of the cell range, in pixels.
         */
        this.getRangeRectangle = function (range) {

            var // the position and size of the column interval
                colPosition = colCollection.getIntervalPosition(SheetUtils.getColInterval(range)),
                // the position and size of the row interval
                rowPosition = rowCollection.getIntervalPosition(SheetUtils.getRowInterval(range));

            return getRectangle(colPosition, rowPosition);
        };

        /**
         * Returns the absolute position of the specified cell in the sheet, in
         * 1/100 of millimeters, as well as in pixels according to the current
         * sheet zoom factor.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.expandMerged=false]
         *      If set to true, and the cell is part of a merged range, the
         *      position of the entire merged range will be returned, and the
         *      address of the merged range will be stored in the property
         *      'mergedRange' of the result object returned by this method.
         *
         * @returns {Object}
         *  The absolute position and size of the cell in the sheet, in the
         *  following properties:
         *  - {Number} leftHmm
         *      The left offset of the cell, in 1/100 of millimeters.
         *  - {Number} topHmm
         *      The top offset of the cell, in 1/100 of millimeters.
         *  - {Number} widthHmm
         *      The total width of the cell, in 1/100 of millimeters.
         *  - {Number} heightHmm
         *      The total height of the cell, in 1/100 of millimeters.
         *  - {Number} left
         *      The left offset of the cell, in pixels.
         *  - {Number} top
         *      The top offset of the cell, in pixels.
         *  - {Number} width
         *      The total width of the cell, in pixels.
         *  - {Number} height
         *      The total height of the cell, in pixels.
         *  - {Object} [mergedRange]
         *      The address of the entire merged range the specified cell is
         *      part of, if the option 'expandMerged' has been set, AND the
         *      passed address is part of a merged cell range.
         */
        this.getCellRectangle = function (address, options) {

            var // whether to expand to merged ranges
                expandMerged = Utils.getBooleanOption(options, 'expandMerged', false),
                // a merged range covering the cell position
                mergedRange = expandMerged ? mergeCollection.getMergedRange(address, 'all') : null;

            // try to extend to a merged range
            if (mergedRange) {
                return _.extend(this.getRangeRectangle(mergedRange), { mergedRange: mergedRange });
            }

            // calculate position of the single cell
            return getRectangle(colCollection.getEntry(address[0]), rowCollection.getEntry(address[1]));
        };

        /**
         * Returns whether the specified range consists of a single cell, or
         * whether it is a single merged range.
         *
         * @param {Object} range
         *  A range address.
         *
         * @returns {Boolean}
         *  Whether the cell is the only (merged) cell in the range.
         */
        this.isSingleCellInRange = function (range) {
            return (SheetUtils.getCellCount(range) === 1) || mergeCollection.isMergedRange(range);
        };

        /**
         * If the passed range represents a single cell, or exactly covers a
         * merged range, the surrounding content range will be returned.
         * Otherwise, the passed range will be returned.
         *
         * @param {Object} range
         *  A range address.
         *
         * @returns {Object}
         *  The content range of a single cell, otherwise the passed range.
         */
        this.getContentRangeForCell = function (range) {
            return this.isSingleCellInRange(range) ? cellCollection.findContentRange(range) : range;
        };

        /**
         * Returns the visible parts of the passed cell ranges.
         *
         * @param {Object|Array} ranges
         *  A cell range address, or an array of range addresses.
         *
         * @returns {Array}
         *  The visible parts of the passed cell ranges, in the same order. If
         *  a range is split into multiple parts (hidden columns or rows inside
         *  the range), the range parts are ordered row-by-row. If the entire
         *  range list is located in hidden columns or rows, the returned array
         *  will be empty.
         */
        this.getVisibleRanges = function (ranges) {

            var // the resulting visible ranges
                visibleRanges = [];

            // process each range in the passed range list
            _.each(_.getArray(ranges), function (range) {

                var // the visible column intervals
                    colIntervals = colCollection.getVisibleIntervals(SheetUtils.getColInterval(range)),
                    // the visible row intervals
                    rowIntervals = rowCollection.getVisibleIntervals(SheetUtils.getRowInterval(range));

                // push visible ranges to the result
                visibleRanges = visibleRanges.concat(SheetUtils.makeRangesFromIntervals(colIntervals, rowIntervals));
            });

            return visibleRanges;
        };

        /**
         * Returns the address of a visible cell located as close as possible
         * to the passed cell address.
         *
         * @param {Number[]} address
         *  The address of the cell.
         *
         * @param {String} method
         *  Specifies how to look for another cell, if the specified cell is
         *  hidden. The following lookup methods are supported:
         *  - 'exact':
         *      Does not look for other columns/rows. If the specified cell is
         *      hidden, returns null.
         *  - 'next':
         *      Looks for a visible column and/or row following the cell.
         *  - 'prev':
         *      Looks for a visible column and/or row preceding the cell.
         *  - 'nextPrev':
         *      First, looks for visible column and/or row following the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *  - 'prevNext':
         *      First, looks for visible column and/or row preceding the cell.
         *      If there are none available, looks for a visible column and/or
         *      row preceding the cell. It may happen that a preceding column,
         *      but a following row will be found, and vice versa.
         *
         * @returns {Number[]|Null}
         *  The address of a visible cell near the passed cell address; or null
         *  if all columns or all rows are hidden.
         */
        this.getVisibleCell = function (address, method) {

            var // collection entries of visible column/row
                colEntry = colCollection.getVisibleEntry(address[0], method),
                rowEntry = rowCollection.getVisibleEntry(address[1], method);

            return (colEntry && rowEntry) ? [colEntry.index, rowEntry.index] : null;
        };

        /**
         * Shrinks the passed range address to the visible area, if the leading
         * or trailing columns/rows of the range are hidden.
         *
         * @param {Object} range
         *  The address of the original cell range.
         *
         * @returns {Object|Null}
         *  The range address of the visible area of the passed cell range; or
         *  null, if the range is completely hidden. The first and last column,
         *  and the first and last row of the range are visible, but some inner
         *  columns/rows in the range may be hidden.
         */
        this.shrinkToVisibleRange = function (range) {

            var // shrunken visible column interval
                colInterval = colCollection.shrinkToVisibleInterval(SheetUtils.getColInterval(range)),
                // shrunken visible row interval
                rowInterval = rowCollection.shrinkToVisibleInterval(SheetUtils.getRowInterval(range));

            return (colInterval && rowInterval) ? SheetUtils.makeRangeFromIntervals(colInterval, rowInterval) : null;
        };

        /**
         * Shrinks the passed range addresses to their visible areas, if the
         * leading or trailing columns/rows of a range are hidden.
         *
         * @param {Object|Array} ranges
         *  The address of a cell range, or an array of cell range addresses.
         *
         * @returns {Array}
         *  The range addresses of the visible areas of the passed cell ranges.
         *  If one of the ranges is completely hidden, nothing will be inserted
         *  into the result array for that range.
         */
        this.shrinkToVisibleRanges = function (ranges) {

            var // the resulting ranges
                visibleRanges = [];

            _.each(_.getArray(ranges), function (range) {
                range = self.shrinkToVisibleRange(range);
                if (range) { visibleRanges.push(range); }
            });

            return visibleRanges;
        };

        // split settings -----------------------------------------------------

        /**
         * Returns whether the view split mode is enabled for the columns in
         * this sheet, regardless of the specific split mode (dynamic or frozen
         * mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the columns in this
         *  sheet.
         */
        this.hasColSplit = function () {
            return this.getViewAttribute('splitWidth') > 0;
        };

        /**
         * Returns whether the view split mode is enabled for the rows in this
         * sheet, regardless of the specific split mode (dynamic or frozen
         * mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the rows in this sheet.
         */
        this.hasRowSplit = function () {
            return this.getViewAttribute('splitHeight') > 0;
        };

        /**
         * Returns whether the view split mode is enabled for this sheet,
         * regardless of the specific split mode (dynamic or frozen mode).
         *
         * @returns {Boolean}
         *  Whether the view split mode is enabled for the sheet.
         */
        this.hasSplit = function () {
            return this.hasColSplit() || this.hasRowSplit();
        };

        /**
         * Returns the current view split mode of this sheet.
         *
         * @returns {String}
         *  The current view split mode. The empty string indicates that the
         *  sheet is not split at all. Otherwise, one of the following values
         *  will be returned:
         *  - 'split': Dynamic split mode (movable split lines).
         *  - 'frozen': Frozen split mode (a fixed number of columns and/or
         *      rows are frozen, split lines are not movable). When leaving
         *      frozen split mode, the split lines will be hidden.
         *  - 'frozenSplit': Frozen split mode (same behavior as 'frozen'
         *      mode). When leaving frozen split mode, the split lines remain
         *      visible and become movable (dynamic 'split' mode).
         */
        this.getSplitMode = function () {
            return this.hasSplit() ? this.getViewAttribute('splitMode') : '';
        };

        /**
         * Returns whether this sheet is currently split, and dynamic split
         * mode is activated. Returns also true, if the split is currently
         * frozen, but will thaw to dynamic split mode ('frozenSplit' mode).
         *
         * @returns {Boolean}
         *  Whether dynamic split mode is activated (split modes 'split' or
         *  'frozenSplit').
         */
        this.hasDynamicSplit = function () {
            return (/^(split|frozenSplit)$/).test(this.getSplitMode());
        };

        /**
         * Returns whether this sheet is currently split, and the frozen split
         * mode is activated.
         *
         * @returns {Boolean}
         *  Whether frozen split mode is activated (split modes 'frozen' or
         *  'frozenSplit').
         */
        this.hasFrozenSplit = function () {
            return (/^(frozen|frozenSplit)$/).test(this.getSplitMode());
        };

        /**
         * Returns the width of the left area in a sheet in split mode, in
         * 1/100 of millimeters. In frozen split mode, returns the total size
         * of the frozen columns.
         *
         * @returns {Number}
         *  The width of the left area if the sheet is in split mode, otherwise
         *  zero.
         */
        this.getSplitWidthHmm = function () {
            return getSplitSize(true, false);
        };

        /**
         * Returns the height of the upper area in a sheet in split mode, in
         * 1/100 of millimeters. In frozen split mode, returns the total size
         * of the frozen rows.
         *
         * @returns {Number}
         *  The height of the upper area if the sheet is in split mode,
         *  otherwise zero.
         */
        this.getSplitHeightHmm = function () {
            return getSplitSize(false, false);
        };

        /**
         * Returns the width of the left area in a sheet in split mode, in
         * pixels according to the current sheet zoom factor. In frozen split
         * mode, returns the total size of the frozen columns.
         *
         * @returns {Number}
         *  The width of the left area if the sheet is in split mode, otherwise
         *  zero.
         */
        this.getSplitWidth = function () {
            return getSplitSize(true, true);
        };

        /**
         * Returns the height of the upper area in a sheet in split mode, in
         * pixels according to the current sheet zoom factor. In frozen split
         * mode, returns the total size of the frozen rows.
         *
         * @returns {Number}
         *  The height of the upper area if the sheet is in split mode,
         *  otherwise zero.
         */
        this.getSplitHeight = function () {
            return getSplitSize(false, true);
        };

        /**
         * Returns the column interval shown in the left area of the sheet. If
         * the first or last visible columns are visible less than half of
         * their width, they will not be included in the interval, with one
         * exception: If the resulting interval would be empty, it will contain
         * the first visible column instead, regardless how much of its width
         * is visible.
         *
         * @returns {Object|Null}
         *  The column interval shown in the left area of the sheet if it is in
         *  split mode, otherwise null.
         */
        this.getSplitColInterval = function () {
            return getSplitInterval(true);
        };

        /**
         * Returns the row interval shown in the upper area of the sheet. If
         * the first or last visible rows are visible less than half of their
         * height, they will not be included in the interval, with one
         * exception: If the resulting interval would be empty, it will contain
         * the first visible row instead, regardless how much of its height
         * is visible.
         *
         * @returns {Object|Null}
         *  The row interval shown in the upper area of the sheet if it is in
         *  split mode, otherwise null.
         */
        this.getSplitRowInterval = function () {
            return getSplitInterval(false);
        };

        /**
         * Sets the split size for dynamic split mode, and updates the scroll
         * anchor view attributes, if the horizontal or vertical split will be
         * enabled.
         *
         * @param {Number} splitWidth
         *  The new value for the 'splitWidth' view attribute, in 1/100 of
         *  millimeters. The value zero will disable the split between left and
         *  right sheet area.
         *
         * @param {Number} splitHeight
         *  The new value for the 'splitHeight' view attribute, in 1/100 of
         *  millimeters. The value zero will disable the split between upper
         *  and lower sheet area.
         *
         * @param {Object} [attributes]
         *  Additional view attributes to be set at this sheet model.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setDynamicSplit = function (splitWidth, splitHeight, attributes) {

            var // the old split size
                oldSplitWidth = this.getSplitWidthHmm(),
                oldSplitHeight = this.getSplitHeightHmm();

            // the new view attributes for split mode
            attributes = _.isObject(attributes) ? _.copy(attributes, true) : {};
            _.extend(attributes, { splitMode: 'split', splitWidth: splitWidth, splitHeight: splitHeight });

            // Update left/right scroll positions. Copy from left to right, if
            // split has been disabled, or copy from right to left, if split
            // has been enabled.
            if ((oldSplitWidth > 0) && (attributes.splitWidth === 0)) {
                attributes.anchorRight = this.getViewAttribute('anchorLeft');
            } else if ((oldSplitWidth === 0) && (attributes.splitWidth > 0)) {
                attributes.anchorLeft = this.getViewAttribute('anchorRight');
            }

            // Update top/bottom scroll positions. Copy from top to bottom, if
            // split has been disabled, or copy from bottom to top, if split
            // has been enabled.
            if ((oldSplitHeight > 0) && (attributes.splitHeight === 0)) {
                attributes.anchorBottom = this.getViewAttribute('anchorTop');
            } else if ((oldSplitHeight === 0) && (attributes.splitHeight > 0)) {
                attributes.anchorTop = this.getViewAttribute('anchorBottom');
            }

            // set the new attributes
            this.setViewAttributes(attributes);
            return this;
        };

        /**
         * Sets the split size for frozen split mode, and updates the scroll
         * anchor view attributes.
         *
         * @param {Object|Null} colInterval
         *  If set to an index interval, the new column interval to be shown in
         *  frozen split mode. The value null will disable the split between
         *  left and right sheet area.
         *
         * @param {Object|Null} rowInterval
         *  If set to an index interval, the new row interval to be shown in
         *  frozen split mode. The value null will disable the split between
         *  top and bottom sheet area.
         *
         * @param {Object} [attributes]
         *  Additional view attributes to be set at this sheet model.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setFrozenSplit = function (colInterval, rowInterval, attributes) {

            // the new view attributes for split mode
            attributes = _.isObject(attributes) ? _.copy(attributes, true) : {};
            attributes.splitMode = this.hasDynamicSplit() ? 'frozenSplit' : 'frozen';

            // set column interval if specified
            if (_.isObject(colInterval)) {
                attributes.splitWidth = SheetUtils.getIntervalSize(colInterval);
                attributes.anchorLeft = { index: colInterval.first, ratio: 0 };
                attributes.anchorRight = { index: colInterval.last + 1, ratio: 0 };
            } else {
                attributes.splitWidth = 0;
            }

            // set row interval if specified
            if (_.isObject(rowInterval)) {
                attributes.splitHeight = SheetUtils.getIntervalSize(rowInterval);
                attributes.anchorTop = { index: rowInterval.first, ratio: 0 };
                attributes.anchorBottom = { index: rowInterval.last + 1, ratio: 0 };
            } else {
                attributes.splitHeight = 0;
            }

            // set the new attributes
            this.setViewAttributes(attributes);
            return this;
        };

        /**
         * Resets all split attributes, effectively disabling any split view
         * that is currently active.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.clearSplit = function () {
            return this.setDynamicSplit(0, 0);
        };

        // sheet operations ---------------------------------------------------

        /**
         * Returns whether this sheet is locked. In locked sheets it is not
         * possible to insert, delete, or modify columns and rows, to edit
         * protected cells, or to manipulate drawing objects.
         *
         * @returns {Boolean}
         *  Whether this sheet is locked.
         */
        this.isLocked = function () {
            return this.getMergedAttributes().sheet.locked;
        };

        /**
         * Returns the effective zoom factor to be used for this sheet,
         * according to the value of the 'zoom' view attribute, and the type of
         * the current device. On touch devices, the effective zoom factor will
         * be increased for better usability.
         *
         * @returns {Number}
         *  The effective zoom factor for this sheet.
         */
        this.getEffectiveZoom = function () {
            var zoom = this.getViewAttribute('zoom');
            return Modernizr.touch ? (zoom * 1.25) : zoom;
        };

        /**
         * Calculates the effective font size for the passed font size,
         * according to the current zoom factor of this sheet.
         *
         * @param {Number} fontSize
         *  The original font size, in points.
         *
         * @returns {Number}
         *  The effective font size, in points.
         */
        this.getEffectiveFontSize = function (fontSize) {
            return Utils.round(fontSize * this.getEffectiveZoom(), 0.1);
        };

        /**
         * Calculates the effective line height to be used in spreadsheet cells
         * for the passed character attributes, according to the current zoom
         * factor of this sheet.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height.
         *  @param {String} charAttributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} charAttributes.fontSize
         *      The original font size, in points.
         *  @param {Boolean} [charAttributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [charAttributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @returns {Number}
         *  The line height for the passed character attributes, in pixels.
         */
        this.getEffectiveLineHeight = function (charAttributes) {
            return this.convertHmmToPixel(documentStyles.getLineHeight(charAttributes));
        };

        /**
         * Returns the effective horizontal padding between cell grid lines and
         * the text contents of the cell, according to the current zoom factor
         * of this sheet.
         *
         * @returns {Number}
         *  The effective horizontal cell content padding, in points.
         */
        this.getEffectiveCellPadding = function () {
            // 2 pixels in 100% zoom, round up early
            return Math.floor(2 * this.getEffectiveZoom() + 0.75);
        };

        /**
         * Returns the effective grid color of this sheet. The automatic color
         * will be replaced with medium gray.
         *
         * @param {Object} autoColor
         *  The color to be returned as replacement for the automatic color.
         *
         * @returns {Object}
         *  The effective grid color of this sheet.
         */
        this.getEffectiveGridColor = function (autoColor) {
            var gridColor = this.getViewAttribute('gridColor');
            // fall back to black for automatic grid color
            return Color.isAutoColor(gridColor) ? autoColor : gridColor;
        };

        /**
         * On first invocation, initializes the view attributes for this sheet
         * from the imported sheet view attributes.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.initializeViewAttributes = _.once(_.bind(function () {

            var // the merged sheet attributes
                sheetAttributes = self.getMergedAttributes().sheet,
                // all selected ranges
                selectedRanges = Utils.getArrayOption(sheetAttributes, 'selectedRanges', []),
                // the array index of the active range (bug 34549: filter may send invalid index)
                activeRange = Utils.minMax(sheetAttributes.activeIndex, 0, selectedRanges.length - 1),
                // selection, built from multiple sheet attributes, expanded to merged ranges
                selection = getExpandedSelection({
                    ranges: selectedRanges,
                    activeRange: activeRange,
                    activeCell: sheetAttributes.activeCell,
                    drawings: sheetAttributes.selectedDrawings
                }),
                // the resulting view attributes
                viewAttributes = {
                    selection: selection,
                    zoom: sheetAttributes.zoom,
                    showGrid: sheetAttributes.showGrid,
                    gridColor: sheetAttributes.gridColor,
                    splitMode: sheetAttributes.splitMode,
                    splitWidth: sheetAttributes.splitWidth,
                    splitHeight: sheetAttributes.splitHeight,
                    activePane: sheetAttributes.activePane,
                    anchorLeft: { index: sheetAttributes.scrollLeft, ratio: 0 },
                    anchorRight: { index: sheetAttributes.scrollRight, ratio: 0 },
                    anchorTop: { index: sheetAttributes.scrollTop, ratio: 0 },
                    anchorBottom: { index: sheetAttributes.scrollBottom, ratio: 0 }
                };

            // show warnings for invalid selection settings (bug 34549)
            if (selectedRanges.length === 0) {
                Utils.warn('SheetModel.initializeViewAttributes(): sheet=' + this.getIndex() + ', missing ranges in selection');
                selection.ranges = [{ start: [0, 0], end: [0, 0] }];
                selection.activeRange = 0;
                selection.activeCell = [0, 0];
                selection = getExpandedSelection(selection);
            } else if (activeRange !== sheetAttributes.activeIndex) {
                Utils.warn('SheetModel.initializeViewAttributes(): sheet=' + this.getIndex() + ', invalid range index in selection');
            }

            // Sheet view attributes in operations use the top-left grid pane as default pane
            // which is always visible without active view split. Internally, the view uses the
            // bottom-right grid pane as the default pane instead. According to the split settings,
            // the 'activePane' attribute, and the scroll attributes have to be translated.

            // no horizontal split: map all left pane settings to right pane settings
            if (sheetAttributes.splitWidth === 0) {
                viewAttributes.activePane = PaneUtils.getNextPanePos(viewAttributes.activePane, 'right');
                viewAttributes.anchorRight = viewAttributes.anchorLeft;
                delete viewAttributes.anchorLeft;
            }

            // no vertical split: map all top pane settings to bottom pane settings
            if (sheetAttributes.splitHeight === 0) {
                viewAttributes.activePane = PaneUtils.getNextPanePos(viewAttributes.activePane, 'bottom');
                viewAttributes.anchorBottom = viewAttributes.anchorTop;
                delete viewAttributes.anchorTop;
            }

            this.setViewAttributes(viewAttributes);
            return this;
        }, this));

        /**
         * Returns all changed sheet attributes as used in 'setSheetAttributes'
         * operations according to the current runtime view settings.
         *
         * @returns {Object}
         *  All changed sheet attributes matching the current view settings of
         *  this sheet.
         */
        this.getChangedViewAttributes = function () {

            var // the current merged sheet attributes
                oldSheetAttributes = this.getMergedAttributes().sheet,
                // the current view settings
                viewAttributes = this.getViewAttributes(),
                // the new sheet attributes according to the view settings
                newSheetAttributes = {
                    selectedRanges: viewAttributes.selection.ranges,
                    activeIndex: viewAttributes.selection.activeRange,
                    activeCell: viewAttributes.selection.activeCell,
                    selectedDrawings: viewAttributes.selection.drawings,
                    zoom: viewAttributes.zoom,
                    showGrid: viewAttributes.showGrid,
                    gridColor: viewAttributes.gridColor,
                    splitMode: viewAttributes.splitMode,
                    splitWidth: viewAttributes.splitWidth,
                    splitHeight: viewAttributes.splitHeight,
                    activePane: viewAttributes.activePane,
                    scrollLeft: getScrollIndex(viewAttributes.anchorLeft, true),
                    scrollRight: getScrollIndex(viewAttributes.anchorRight, true),
                    scrollTop: getScrollIndex(viewAttributes.anchorTop, false),
                    scrollBottom: getScrollIndex(viewAttributes.anchorBottom, false)
                };

            // converts the passed scroll anchor to a simple column/row index
            function getScrollIndex(scrollAnchor, columns) {
                return Math.min(Math.round(scrollAnchor.index + scrollAnchor.ratio), docModel.getMaxIndex(columns));
            }

            // Sheet view attributes in operations use the top-left grid pane as default pane
            // which is always visible without active view split. Internally, the view uses the
            // bottom-right grid pane as the default pane instead. According to the split settings,
            // the 'activePane' attribute, and the scroll attributes have to be translated.

            // no horizontal split: map all right pane settings to left pane settings
            if (newSheetAttributes.splitWidth === 0) {
                newSheetAttributes.activePane = PaneUtils.getNextPanePos(newSheetAttributes.activePane, 'left');
                newSheetAttributes.scrollLeft = newSheetAttributes.scrollRight;
                delete newSheetAttributes.scrollRight;
            }

            // no vertical split: map all bottom pane settings to top pane settings
            if (newSheetAttributes.splitHeight === 0) {
                newSheetAttributes.activePane = PaneUtils.getNextPanePos(newSheetAttributes.activePane, 'top');
                newSheetAttributes.scrollTop = newSheetAttributes.scrollBottom;
                delete newSheetAttributes.scrollBottom;
            }

            // reduce new sheet attribute to changed ones
            _.each(newSheetAttributes, function (value, name) {
                if (_.isEqual(oldSheetAttributes[name], value)) {
                    delete newSheetAttributes[name];
                }
            });

            return newSheetAttributes;
        };

        /**
         * Returns whether this sheet contains changed view attributes compared
         * to the sheet view attributes imported from the document file.
         *
         * @returns {Boolean}
         *  Whether this sheet contains changed view attributes.
         */
        this.hasChangedViewAttributes = function () {

            var // the new sheet attributes according to the view settings
                sheetAttributes = this.getChangedViewAttributes();

            // delete 'unimportant' view attributes (selection, scroll positions)
            delete sheetAttributes.selectedRanges;
            delete sheetAttributes.activeIndex;
            delete sheetAttributes.activeCell;
            delete sheetAttributes.selectedDrawings;
            delete sheetAttributes.activePane;
            delete sheetAttributes.scrollLeft;
            delete sheetAttributes.scrollRight;
            delete sheetAttributes.scrollTop;
            delete sheetAttributes.scrollBottom;

            // return whether there are changed view attributes left
            return !_.isEmpty(sheetAttributes);
        };

        // column operations --------------------------------------------------

        /**
         * Returns whether additional columns can be inserted into the sheet.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals.
         *
         * @returns {Boolean}
         *  Whether additional columns can be inserted, according to the passed
         *  column intervals and the number of used columns in the sheet.
         */
        this.canInsertColumns = function (intervals) {
            return intervalsInsertable(intervals, true);
        };

        /**
         * Generates and applies all operations to insert new columns into the
         * sheet.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.insertColumns = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateInsertIntervalOperations(generator, intervals, true);
            });
        };

        /**
         * Returns whether the specified columns can be deleted from the sheet.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals.
         *
         * @returns {Boolean}
         *  Whether the columns can be deleted.
         */
        this.canDeleteColumns = function (intervals) {
            return intervalsDeleteable(intervals, true);
        };

        /**
         * Generates and applies all operations to delete columns from the
         * sheet.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.deleteColumns = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateDeleteIntervalOperations(generator, intervals, true);
            });
        };

        /**
         * Returns the mixed formatting attributes of all columns covered by
         * the passed column intervals.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals.
         *
         * @returns {Object}
         *  The mixed attributes of all columns covered by the passed
         *  intervals, as simple object (NOT mapped as a 'column' sub-object).
         *  All attributes that could not be resolved unambiguously will be set
         *  to the value null.
         */
        this.getColumnAttributes = function (intervals) {
            return colCollection.getMixedAttributes(intervals);
        };

        /**
         * Generates and applies all operations to modify the specified column
         * attributes in the sheet.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals. Each
         *  interval object MUST contain an additional object property 'attrs'
         *  containing the (incomplete) column attributes, as simple object
         *  (NOT mapped as a 'column' sub-object).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'cols:overflow': Too many columns in the passed intervals.
         *  - 'internal': Internal error while applying the operations.
         */
        this.setColumnAttributes = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateChangeIntervalOperations(generator, intervals, true);
            });
        };

        // row operations -----------------------------------------------------

        /**
         * Returns whether additional rows can be inserted into the sheet.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals.
         *
         * @returns {Boolean}
         *  Whether additional rows can be inserted, according to the passed
         *  row intervals and the number of used rows in the sheet.
         */
        this.canInsertRows = function (intervals) {
            return intervalsInsertable(intervals, false);
        };

        /**
         * Generates and applies all operations to insert new rows into the
         * sheet.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.insertRows = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateInsertIntervalOperations(generator, intervals, false);
            });
        };

        /**
         * Returns whether the specified rows can be deleted from the sheet.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals.
         *
         * @returns {Boolean}
         *  Whether the rows can be deleted.
         */
        this.canDeleteRows = function (intervals) {
            return intervalsDeleteable(intervals, false);
        };

        /**
         * Generates and applies all operations to delete rows from the sheet.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.deleteRows = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateDeleteIntervalOperations(generator, intervals, false);
            });
        };

        /**
         * Returns the mixed formatting attributes of all rows covered by the
         * passed row intervals.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals.
         *
         * @returns {Object}
         *  The mixed attributes of all rows covered by the passed intervals,
         *  as simple object (NOT mapped as a 'row' sub-object). All attributes
         *  that could not be resolved unambiguously will be set to the value
         *  null.
         */
        this.getRowAttributes = function (intervals) {
            return rowCollection.getMixedAttributes(intervals);
        };

        /**
         * Generates and applies all operations to modify the specified row
         * attributes in the sheet.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals. Each interval
         *  object MUST contain an additional object property 'attrs'
         *  containing the (incomplete) row attributes, as simple object (NOT
         *  mapped as a 'row' sub-object).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'rows:overflow': Too many rows in the passed intervals.
         *  - 'internal': Internal error while applying the operations.
         */
        this.setRowAttributes = function (intervals) {
            // send operations without applying them (done manually during creation)
            return createAndSendSheetOperations(function (generator) {
                return generateChangeIntervalOperations(generator, intervals, false);
            });
        };

        // cell operations ----------------------------------------------------

        /**
         * Generates and applies a 'setCellContents' operation that will insert
         * the specified cell contents (values and/or formatting) into the
         * sheet.
         *
         * @param {Number[]} start
         *  The address of the first cell to be modified.
         *
         * @param {Array} contents
         *  A two-dimensional array containing the values and attribute sets
         *  for the cells.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI locale.
         *
         * @returns {String}
         *  The empty string, if the operation has been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.setCellContents = function (start, contents, options) {

            var // the generator for the 'setCellContents' operation
                generator = this.createOperationsGenerator(),
                // additional properties for the operation
                properties = { contents: contents };

            // add the parse property (bug 32624: need to insert 'en_US' for unparsed contents)
            properties.parse = Utils.getBooleanOption(options, 'parse', false) ? Config.LOCALE : 'en_US';

            // create and apply the operation
            generator.generateCellOperation(Operations.SET_CELL_CONTENTS, start, properties);
            return docModel.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies a 'setCellContents' operation that will insert
         * the specified contents (values and/or formatting) into a single cell
         * in the sheet.
         *
         * @param {Number[]} address
         *  The address of the cell to be modified.
         *
         * @param {Any} [value]
         *  The new value for the active cell. If omitted, only the cell
         *  formatting will be changed. If set to null or an empty string, the
         *  cell value will be deleted.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI locale.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell while applying the new attributes.
         *  @param {Any} [options.result]
         *      The result of a formula, as calculated by the local formula
         *      interpreter.
         *
         * @returns {String}
         *  The empty string, if the operation has been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.setSingleCellContents = function (address, value, attributes, options) {

            var // the cell contents object (value and attributes)
                contents = { attrs: attributes },
                // the formula result passed in the options
                result = Utils.getOption(options, 'result');

            // add cell value, if passed (use null to delete the cell value)
            if (_.isString(value) && (value.length === 0)) {
                contents.value = null;
                options = Utils.extendOptions(options, { parse: false });
            } else if (!_.isUndefined(value)) {
                contents.value = value;
            } else {
                options = Utils.extendOptions(options, { parse: false });
            }

            // add all attributes to be cleared
            if (Utils.getBooleanOption(options, 'clear', false)) {
                contents.attrs = Utils.extendOptions(buildNullAttributes(), contents.attrs);
            }

            // do nothing, if neither value nor attributes are present
            if (!_.isObject(contents.attrs) || _.isEmpty(contents.attrs)) { delete contents.attrs; }
            if (_.isEmpty(contents)) { return ''; }

            // add the result property for formulas
            if (_.isString(value) && !_.isUndefined(result)) {
                contents.result = result;
            }

            // send the 'setCellContents' operation (operation expects two-dimensional array)
            return this.setCellContents(address, [[contents]], options);
        };

        /**
         * Generates and applies 'fillCellRange' operations that will insert
         * the specified contents (values and/or formatting) into a range of
         * cells in the sheet.
         *
         * @param {Object|Array} ranges
         *  The address of a single cell range, or an array of range addresses.
         *
         * @param {Any} [value]
         *  The new value for all cells in the passed ranges. If omitted, only
         *  the cell formatting will be changed. If set to null or an empty
         *  string, the cell values will be deleted.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cells.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the 'parse' property will be inserted into the
         *      operation, set to the name of the current UI locale.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell ranges while applying the new attributes.
         *  @param {Number[]} [options.ref]
         *      The address of the reference cell to be inserted into the
         *      operations. If omitted, the top-left cell of the first range
         *      passed to this method will be used.
         *  @param {String} [options.cat]
         *      The number format category the format code contained in the
         *      passed formatting attributes belongs to. Will be copied to all
         *      affected cells in the cell collection.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'cols:overflow': The cell ranges contain too many entire columns.
         *  - 'rows:overflow': The cell ranges contain too many entire rows.
         *  - 'cells:overflow': The cell ranges contain too many filled cells.
         *  - 'internal': Internal error while applying the operation.
         */
        this.fillCellRanges = function (ranges, value, attributes, options) {

            var // the generator for the 'fillCellRange' operations
                generator = this.createOperationsGenerator(),
                // additional properties for the 'fillCellRange' operations
                properties = {},
                // reference cell address
                ref = Utils.getArrayOption(options, 'ref'),
                // the new number format category
                category = Utils.getStringOption(options, 'cat', '');

            // convert ranges to unique array
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (ranges.length === 0) { return ''; }

            // use top-left cell as reference cell, if missing
            if (!_.isArray(ref)) { ref = ranges[0].start; }

            // treat empty string as 'clear value' function
            if (_.isString(value) && (value.length === 0)) {
                value = null;
            }

            // add value to the operation options
            if (!_.isUndefined(value)) {
                properties.value = value;
            }

            // prepare shared formulas, add the parse property
            if (_.isString(value)) {
                properties.shared = 0;
                properties.ref = ref;
                // bug 32624: need to insert 'en_US' for unparsed contents
                properties.parse = Utils.getBooleanOption(options, 'parse', false) ? Config.LOCALE : 'en_US';
            }

            // validate attributes
            if (!_.isObject(attributes) || _.isEmpty(attributes)) {
                attributes = null;
            }

            // add all attributes to be cleared
            if (Utils.getBooleanOption(options, 'clear', false)) {
                attributes = Utils.extendOptions(buildNullAttributes(), attributes);
            }

            //delete explicit the url, if text is cleared
            if (value === null) {
                attributes = Utils.extendOptions({ character: { url:null } }, attributes);
            }

            // create operation to insert a dirty cell style into the document
            if (attributes && _.isString(attributes.styleId)) {
                generator.generateMissingStyleSheetOperation('cell', attributes.styleId);
            }

            // always create 'fillCellRange' operations for cell values (and their attributes)
            if ('value' in properties) {

                // check number of modified cells (but do not limit number of cleared cells)
                if (attributes || !_.isNull(properties.value)) {
                    if (SheetUtils.getCellCountInRanges(ranges) > SheetUtils.MAX_FILL_CELL_COUNT) {
                        return 'cells:overflow';
                    }
                }

                // add attribute to operation
                if (attributes) {
                    properties.attrs = attributes;
                }

                // generate all range operations
                _.each(ranges, function (range) {
                    generator.generateRangeOperation(Operations.FILL_CELL_RANGE, range, properties);
                    // remove value and dependent properties (keep 'shared' property only)
                    delete properties.value;
                    delete properties.parse;
                    delete properties.ref;
                });
            }

            // create 'fillCellRange', 'setColumnAttributes', or 'setRowAttributes' operations for attributes only
            else if (_.isObject(attributes)) {

                // check column/row/cell limits
                var result = checkCellLimits(ranges);
                if (result.length > 0) { return result; }

                _.each(ranges, function (range) {
                    generateFillRangeOperation(generator, range, attributes);
                });
            }

            // copy number format category into the cell collection (bug 34021: before applying the operation)
            if (category.length > 0) {
                _.each(ranges, function (range) {
                    if (!docModel.isColRange(range) && !docModel.isRowRange(range)) {
                        cellCollection.setFormatCategory(range, category);
                    }
                });
            }

            // apply the operations
            return docModel.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies 'fillCellRange' operations that will change
         * the visibility of the specified borders in the cell ranges. The
         * border attributes will be applied at range level (inner and outer
         * borders) instead of per-cell level.
         *
         * @param {Object|Array} ranges
         *  The address of a single cell range, or an array of range addresses.
         *
         * @param {Object} borderAttributes
         *  An attribute map containing all border attributes to be changed in
         *  the current selection.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'cols:overflow': The cell ranges contain too many entire columns.
         *  - 'rows:overflow': The cell ranges contain too many entire rows.
         *  - 'cells:overflow': The cell ranges contain too many filled cells.
         *  - 'internal': Internal error while applying the operation.
         */
        this.setBorderAttributes = function (ranges, borderAttributes) {

            var // the generator for the 'fillCellRange' operations
                generator = this.createOperationsGenerator(),
                // additional properties for the operations
                properties = { attrs: { cell: borderAttributes }, rangeBorders: true };

            // generates a 'setColumnAttributes' or 'setRowAttributes' operation for the adjacent cells of a column
            function generateAdjacentIntervalOperations(index, newBorderName, columns) {
                var OPERATION_NAME = columns ? Operations.SET_COLUMN_ATTRIBUTES : Operations.SET_ROW_ATTRIBUTES,
                    opProperties = { attrs: { cell: Utils.makeSimpleObject(newBorderName, Border.NONE) }, rangeBorders: true };
                // do not send borders for rows without custom format
                if (columns || rowCollection.getEntry(index).attributes.row.customFormat) {
                    generator.generateIntervalOperation(OPERATION_NAME, { first: index, last: index }, opProperties);
                }
            }

            // generates a 'fillCellRange' operation for the adjacent cells of one border
            function generateAdjacentRangeOperations(firstCol, firstRow, lastCol, lastRow, newBorderName) {
                var range = { start: [firstCol, firstRow], end: [lastCol, lastRow] },
                    opProperties = { attrs: { cell: Utils.makeSimpleObject(newBorderName, Border.NONE) }, rangeBorders: true };
                generator.generateRangeOperation(Operations.FILL_CELL_RANGE, range, opProperties);
            }

            // reduce all ranges to a single column/row, if only a single outer line will be changed
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (_.size(borderAttributes) === 1) {
                if ('borderLeft' in borderAttributes) {
                    _.each(ranges, function (range) { range.end[0] = range.start[0]; });
                    ranges = SheetUtils.getUniqueRanges(ranges);
                } else if ('borderRight' in borderAttributes) {
                    _.each(ranges, function (range) { range.start[0] = range.end[0]; });
                    ranges = SheetUtils.getUniqueRanges(ranges);
                } else if ('borderTop' in borderAttributes) {
                    _.each(ranges, function (range) { range.end[1] = range.start[1]; });
                    ranges = SheetUtils.getUniqueRanges(ranges);
                } else if ('borderBottom' in borderAttributes) {
                    _.each(ranges, function (range) { range.start[1] = range.end[1]; });
                    ranges = SheetUtils.getUniqueRanges(ranges);
                }
            }

            // check column/row/cell limits
            var result = checkCellLimits(ranges);
            if (result.length > 0) { return result; }

            // send the 'fillCellRange' operations
            _.each(ranges, function (range) {
                // generate operation for the current range
                // setColumnAttributes, setRowAttributes or fillCellRange
                if (docModel.isColRange(range)) {
                    generator.generateIntervalOperation(Operations.SET_COLUMN_ATTRIBUTES, SheetUtils.getColInterval(range), properties);

                    // Generate operation for the adjacent cells of the range
                    if ((range.start[0] > 0) && ('borderLeft' in borderAttributes)) {
                        generateAdjacentIntervalOperations(range.start[0] - 1, 'borderRight', true);
                    }
                    if ((range.end[0] < docModel.getMaxCol()) && ('borderRight' in borderAttributes)) {
                        generateAdjacentIntervalOperations(range.end[0] + 1, 'borderLeft', true);
                    }
                } else if (docModel.isRowRange(range)) {
                    properties.attrs.row = { customFormat: true };
                    generator.generateIntervalOperation(Operations.SET_ROW_ATTRIBUTES, SheetUtils.getRowInterval(range), properties);

                    // Generate operation for the adjacent cells of the range
                    if ((range.start[1] > 0) && ('borderTop' in borderAttributes)) {
                        generateAdjacentIntervalOperations(range.start[1] - 1, 'borderBottom', false);
                    }
                    if ((range.end[1] < docModel.getMaxRow()) && ('borderBottom' in borderAttributes)) {
                        generateAdjacentIntervalOperations(range.end[1] + 1, 'borderTop', false);
                    }
                } else {
                    generator.generateRangeOperation(Operations.FILL_CELL_RANGE, range, properties);

                    // Deleting all borders at neighboring cells -> first approach: delete all neighboring borders, so that
                    // there is always only one valid border between two cells.
                    // Generate operation for the adjacent cells of the range
                    if ((range.start[0] > 0) && ('borderLeft' in borderAttributes)) {
                        generateAdjacentRangeOperations(range.start[0] - 1, range.start[1], range.start[0] - 1, range.end[1], 'borderRight');
                    }
                    if ((range.end[0] < docModel.getMaxCol()) && ('borderRight' in borderAttributes)) {
                        generateAdjacentRangeOperations(range.end[0] + 1, range.start[1], range.end[0] + 1, range.end[1], 'borderLeft');
                    }
                    if ((range.start[1] > 0) && ('borderTop' in borderAttributes)) {
                        generateAdjacentRangeOperations(range.start[0], range.start[1] - 1, range.end[0], range.start[1] - 1, 'borderBottom');
                    }
                    if ((range.end[1] < docModel.getMaxRow()) && ('borderBottom' in borderAttributes)) {
                        generateAdjacentRangeOperations(range.start[0], range.end[1] + 1, range.end[0], range.end[1] + 1, 'borderTop');
                    }
                }
            });

            // apply the operations
            return docModel.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies 'fillCellRange' operations that will change
         * the specified border properties (line style, line color, line width)
         * of all visible cell borders in a range of cells.
         *
         * @param {Object|Array} ranges
         *  The address of a single cell range, or an array of range addresses.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the cell range.
         *  Omitted border properties will not be changed.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.changeVisibleBorders = function (ranges, border) {

            var // the generator for the 'fillCellRange' operations
                generator = this.createOperationsGenerator(),
                // additional properties for the operations
                properties = { rangeBorders: true, visibleBorders: true };

            // generates an operation for the passed range for horizontal or vertical borders
            function generateRangeOperation(range, columns) {

                var // the passed range, that will be expanded at two borders
                    expandedRange = _.copy(range, true),
                    // the array index in cell addresses
                    addressIndex = columns ? 0 : 1,
                    // the border attributes inserted into the operation
                    borderAttributes = {};

                // add inner border property
                borderAttributes[columns ? 'borderInsideVert' : 'borderInsideHor'] = border;

                // expand range at leading border, or add outer border attribute at leading sheet border
                if (expandedRange.start[addressIndex] > 0) {
                    expandedRange.start[addressIndex] -= 1;
                } else {
                    borderAttributes[columns ? 'borderLeft' : 'borderTop'] = border;
                }

                // expand range at trailing border, or add outer border attribute at trailing sheet border
                if (expandedRange.end[addressIndex] < docModel.getMaxIndex(columns)) {
                    expandedRange.end[addressIndex] += 1;
                } else {
                    borderAttributes[columns ? 'borderRight' : 'borderBottom'] = border;
                }

                generateFillRangeOperation(generator, expandedRange, { cell: borderAttributes }, properties);
            }

            // convert ranges to unique array
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (ranges.length === 0) { return ''; }

            // check column/row/cell limits
            var result = checkCellLimits(ranges);
            if (result.length > 0) { return result; }

            // expand the ranges and generate operations for the border attributes
            _.each(ranges, function (range) {
                generateRangeOperation(range, true);
                generateRangeOperation(range, false);
            });

            // apply the operations
            return docModel.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies 'clearCellRange' operations that will delete
         * the values and formatting from a range of cells in the sheet.
         *
         * @param {Object|Array} ranges
         *  The address of a single cell range, or an array of range addresses.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.clearCellRanges = function (ranges) {

            var // the generator for the 'clearCellRange' operations
                generator = this.createOperationsGenerator();

            // convert ranges to unique array
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (ranges.length === 0) { return ''; }

            // the create 'clearCellRange' operations
            _.each(ranges, function (range) {
                generator.generateRangeOperation(Operations.CLEAR_CELL_RANGE, range);
            });

            // apply the operations
            return docModel.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies an 'autoFill' operation that will copy the
         * values and formatting from a range to the surrounding cells.
         *
         * @param {Object} range
         *  The address of a single cell range.
         *
         * @returns {String}
         *  The empty string, if the operation has been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.autoFill = function (range, border, count) {

            var // the generator for the 'clearCellRange' operations
                generator = this.createOperationsGenerator(),
                // the address of the target range
                target = null;

            // passed count must be positive
            if (count <= 0) { return 'internal'; }

            // calculate target address for the 'autoFill' operation
            switch (border) {
            case 'left':
                target = [range.start[0] - count, range.end[1]];
                break;
            case 'right':
                target = [range.end[0] + count, range.end[1]];
                break;
            case 'top':
                target = [range.end[0], range.start[1] - count];
                break;
            case 'bottom':
                target = [range.end[0], range.end[1] + count];
                break;
            default:
                return 'internal';
            }

            // check validity of the target address
            if (!docModel.isValidAddress(target)) { return 'internal'; }

            // generate the 'autoFill' operation
            generator.generateRangeOperation(Operations.AUTO_FILL, range, { target: target });

            // apply the operations
            return docModel.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies 'mergeCells' operations for all passed ranges.
         *
         * @param {Object|Array} ranges
         *  The address of a single cell range, or an array of range addresses.
         *
         * @param {String} type
         *  The merge type. Must be one of:
         *  - 'merge': Merges the entire ranges.
         *  - 'horizontal': Merges the single rows in all ranges.
         *  - 'vertical': Merges the single columns in all ranges.
         *  - 'unmerge': Removes all merged ranges covered by the ranges.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'merge:overlap': The passed cell ranges overlap each other.
         *  - 'merge:overflow': Tried to merge too many ranges at once.
         *  - 'internal': Internal error while applying the operation.
         */
        this.mergeRanges = function (ranges, type) {

            var // the generator for the 'mergeCells' operations
                generator = this.createOperationsGenerator(),
                // the total number of created merged ranges
                rangeCount = 0;

            // convert ranges to unique array
            ranges = SheetUtils.getUniqueRanges(ranges);
            if (ranges.length === 0) { return ''; }

            // do not allow to merge overlapping ranges
            if ((type !== 'unmerge') && SheetUtils.anyRangesOverlap(ranges)) {
                return 'merge:overlap';
            }

            // generate the 'mergeCells' operations
            _.each(ranges, function (range) {

                // count number of new merged ranges
                switch (type) {
                case 'merge':
                    rangeCount += 1;
                    break;
                case 'horizontal':
                    // Bug 30662: reduce range to used area for entire columns
                    if (docModel.isColRange(range)) {
                        range.end[1] = Math.max(0, cellCollection.getUsedRows() - 1);
                    }
                    rangeCount += SheetUtils.getRowCount(range);
                    break;
                case 'vertical':
                    // Bug 30662: reduce range to used area for entire rows
                    if (docModel.isRowRange(range)) {
                        range.end[0] = Math.max(0, cellCollection.getUsedCols() - 1);
                    }
                    rangeCount += SheetUtils.getColCount(range);
                    break;
                }

                // generate a single 'mergeCells' operation
                generator.generateRangeOperation(Operations.MERGE_CELLS, range, { type: type });
            });

            // overflow check
            if (rangeCount > SheetUtils.MAX_MERGED_RANGES_COUNT) { return 'merge:overflow'; }

            // apply the operations
            return docModel.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        // table/filter operations --------------------------------------------

        /**
         * Generates and applies a single 'insertTable' operation to insert a
         * new table range into this sheet.
         *
         * @param {String} tableName
         *  The name for the new table to be inserted. The empty string
         *  addresses the anonymous table range used to store filter settings
         *  for the standard auto filter of the sheet.
         *
         * @param {Object} range
         *  The address of the cell range covered by the new table.
         *
         * @param {Object} [attributes]
         *  The initial attribute set for the table.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'table:duplicate': The sheet already contains a table with the
         *      passed name.
         *  - 'table:overlap': The passed cell range covers an existing table.
         *  - 'internal': Internal error while applying the operation.
         */
        this.insertTable = function (tableName, range, attributes) {

            // check existence of the table range (TODO: table names must be unique in entire document)
            if (tableCollection.hasTable(tableName)) { return $.Deferred().reject('table:duplicate'); }

            // check that the range does not overlap with an existing table
            if (tableCollection.findTables(range).length > 0) { return $.Deferred().reject('table:overlap'); }

            // generate an 'insertTable' operation, and more dependent operations
            return this.createAndApplyOperations(function (generator) {

                // 'real' tables must not contain merged cells (neither in header, footer, nor data rows)
                if (tableName.length > 0) {
                    generator.generateRangeOperation(Operations.MERGE_CELLS, range, { type: 'unmerge' });
                }

                // create the 'insertTable' operation
                var properties = _.clone(range);
                if (_.isObject(attributes)) { properties.attrs = attributes; }
                generator.generateTableOperation(Operations.INSERT_TABLE, tableName, properties);

                // bug 36152: auto filter will hide drop-down buttons in header cells covered by
                // merged ranges (but only if the cells are merged before creating the auto filter)
                // - the drop-down button in the LAST column of a merged range remains visible
                if (tableName.length === 0) {
                    _.each(mergeCollection.getMergedRanges(SheetUtils.getHeaderRange(range)), function (mergedRange) {
                        var firstCol = Math.max(mergedRange.start[0], range.start[0]),
                            lastCol = Math.min(mergedRange.end[0], range.end[0]) - 1;
                        for (var col = firstCol; col <= lastCol; col += 1) {
                            generator.generateTableOperation(Operations.CHANGE_TABLE_COLUMN, tableName, { col: col - range.start[0], attrs: { filter: { hideMergedButton: true } } });
                        }
                    });
                }
            });
        };

        /**
         * Refreshes the specified table according to its current filter
         * settings, by generating operations for showing/hiding the affected
         * rows.
         *
         * @param {String} tableName
         *  The name of the table to be refreshed. The empty string addresses
         *  the anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'table:missing': The sheet does not contain a table with the
         *      passed name.
         *  - 'rows:overflow': The current filter settings of the table would
         *      modify (show or hide) too many rows at once.
         *  - 'internal': Internal error while applying the operation.
         */
        this.refreshTable = function (tableName) {
            // send operations without applying them (done manually during creation)
            return createAndSendTableOperations(tableName, generateRefreshTableOperations);
        };

        /**
         * Generates and applies a single 'changeTable' operation to modify the
         * specified table range in this sheet.
         *
         * @param {String} tableName
         *  The name of the table to be modified. The empty string addresses
         *  the anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @param {Object} attributes
         *  The new attributes to be set for this table.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'table:missing': The sheet does not contain a table with the
         *      passed name.
         *  - 'internal': Internal error while applying the operation.
         */
        this.changeTable = function (tableName, attributes) {

            // send operations without applying them (done manually during creation)
            return createAndSendTableOperations(tableName, function (generator, tableModel) {

                // generate a single 'changeTable' operation, and immediately modify the table
                generator.generateTableOperation(Operations.CHANGE_TABLE, tableName, { attrs: attributes });
                tableModel.setAttributes(attributes);

                // refresh the table with current filter settings
                return generateRefreshTableOperations(generator, tableModel);
            });
        };

        /**
         * Generates and applies a single 'deleteTable' operation to delete the
         * specified table range from this sheet. Additionally, if the table
         * contains active filter rules, all operations will be generated that
         * make the affected hidden rows visible again.
         *
         * @param {String} tableName
         *  The name of the table to be deleted. The empty string addresses the
         *  anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'table:missing': The sheet does not contain a table with the
         *      passed name.
         *  - 'internal': Internal error while applying the operation.
         */
        this.deleteTable = function (tableName) {

            // send operations without applying them (done manually during creation)
            return createAndSendTableOperations(tableName, function (generator, tableModel) {

                var // the resulting promise
                    promise = null;

                // silently deactivate the filters in the table, and refresh the
                // table (this makes all filtered rows visible)
                tableModel.setAttributes({ table: { filtered: false } }, { notify: 'never' });
                promise = generateRefreshTableOperations(generator, tableModel);

                // ignore 'rows:overflow' error (leave rows hidden, but delete the table anyways)
                promise = promise.then(null, function (errorCode) {
                    return (errorCode === 'rows:overflow') ? $.when() : errorCode;
                });

                // add the 'deleteTable' operation, and immediately delete the table
                promise.done(function () {
                    generator.generateTableOperation(Operations.DELETE_TABLE, tableName);
                    tableCollection.deleteTable(tableName);
                });

                return promise;
            });
        };

        /**
         * Generates and applies a 'changeTableColumn' operation with the
         * passed filtering and sorting attributes, and executes the new filter
         * settings by generating operations for showing/hiding the affected
         * rows.
         *
         * @param {String} tableName
         *  The name of the table to be modified. The empty string addresses
         *  the anonymous table range used to store filter settings for the
         *  standard auto filter of the sheet.
         *
         * @param {Number} tableCol
         *  The zero-based index of the table column to be modified, relative
         *  to the cell range covered by this table.
         *
         * @param {Object} attributes
         *  The attributes to be set for the specified table column.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  generated operations have been applied successfully, or that will
         *  be rejected with one of the following error codes:
         *  - 'table:missing': The sheet does not contain a table with the
         *      passed name.
         *  - 'rows:overflow': The passed filter settings would modify (show or
         *      hide) too many rows at once.
         *  - 'internal': Internal error while applying the operation.
         */
        this.changeTableColumn = function (tableName, tableCol, attributes) {

            // send operations without applying them (done manually during creation)
            return createAndSendTableOperations(tableName, function (generator, tableModel) {

                // store the new table column attributes
                generator.generateTableOperation(Operations.CHANGE_TABLE_COLUMN, tableName, { col: tableCol, attrs: attributes });
                tableModel.setColumnAttributes(tableCol, attributes);

                // always refresh the filter (also if no attributes have been changed
                // actually, the cell contents may have changed in the meantime)
                return generateRefreshTableOperations(generator, tableModel);
            });
        };

        // drawing operations -------------------------------------------------

        /**
         * Generates and applies an 'insertDrawing' operation that will insert
         * a new drawing object into this sheet.
         *
         * @param {Number[]} position
         *  The position of the new drawing object.
         *
         * @param {String} type
         *  The type identifier of the drawing object.
         *
         * @param {Object} attributes
         *  Initial attributes for the drawing object, especially the anchor
         *  attributes specifying the position of the drawing in the sheet.
         *
         * @param {Function} [generatorCallback]
         *  A callback function that will be invoked after the initial
         *  'insertDrawing' operation has been generated. Allows to create
         *  additional operations for the drawing object, before all these
         *  operations will be applied at once. Receives the following
         *  parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array} position
         *      The position of the new drawing object in the sheet.
         *
         * @returns {String}
         *  The empty string, if the operation has been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operation.
         */
        this.insertDrawing = function (position, type, attributes, generatorCallback) {

            var // the own sheet index
                sheet = self.getIndex(),
                // the generator for the operations
                generator = this.createOperationsGenerator();

            // add missing generic anchor attributes to the passed attribute set
            drawingCollection.addGenericAnchorAttributes(attributes);

            // create the 'insertDrawing' operation, invoke callback function for more operations
            generator.generateDrawingOperation(Operations.INSERT_DRAWING, position, { type: type, attrs: attributes });
            if (_.isFunction(generatorCallback)) {
                generatorCallback.call(this, generator, sheet, position);
            }

            // apply the operations
            return docModel.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        /**
         * Generates and applies one or more 'deleteDrawing' operations that
         * will delete the specified drawings from this sheet.
         *
         * @param {Array} positions
         *  An array with positions of the drawing objects to be deleted.
         *
         * @returns {String}
         *  The empty string, if the operations have been applied successfully;
         *  otherwise one of the following error codes:
         *  - 'internal': Internal error while applying the operations.
         */
        this.deleteDrawings = function (positions) {

            var // the generator for the operations
                generator = this.createOperationsGenerator();

            // sort by position, remove positions of embedded drawings whose parents will be deleted too
            DrawingUtils.optimizeDrawingPositions(positions);

            // create the 'deleteDrawing' operations, visit the positions backwards to prevent
            // changed positions of following drawings after deleting a previous drawing
            Utils.iterateArray(positions, function (position) {
                generator.generateDrawingOperation(Operations.DELETE_DRAWING, position);
            }, { reverse: true });

            // apply the operations
            return docModel.applyOperations(generator.getOperations()) ? '' : 'internal';
        };

        // initialization -----------------------------------------------------

        // class member construction
        (function (args) {

            var // the source sheet model and target sheet index used for copy construction
                sourceModel = args[SheetModel.length],
                targetIndex = args[SheetModel.length + 1];

            // existing source sheet model indicates copy construction
            if ((sourceModel instanceof SheetModel) && _.isNumber(targetIndex)) {

                // copy construction: add a temporary getIndex() method that does not try to
                // find the sheet in the collection of the document model (sheet has not been
                // inserted into the document yet, real getSheet() will fail to find the sheet)
                var getIndexMethod = self.getIndex;
                self.getIndex = function () { return targetIndex; };

                // create clones of all child collection instances
                nameCollection = sourceModel.getNameCollection().clone(self);
                colCollection = sourceModel.getColCollection().clone(self);
                rowCollection = sourceModel.getRowCollection().clone(self);
                mergeCollection = sourceModel.getMergeCollection().clone(self);
                tableCollection = sourceModel.getTableCollection().clone(self);
                validationCollection = sourceModel.getValidationCollection().clone(self);
                cellCollection = sourceModel.getCellCollection().clone(self);
                drawingCollection = sourceModel.getDrawingCollection().clone(self);
                self.setViewAttributes(sourceModel.getViewAttributes());

                // restore the real getIndex() method
                self.getIndex = getIndexMethod;

            } else {

                // regular construction
                nameCollection = new NameCollection(app, self);
                colCollection = new ColRowCollection(app, self, true);
                rowCollection = new ColRowCollection(app, self, false);
                mergeCollection = new MergeCollection(app, self);
                tableCollection = new TableCollection(app, self);
                validationCollection = new ValidationCollection(app, self);
                cellCollection = new CellCollection(app, self);
                drawingCollection = new SheetDrawingCollection(app, self);
            }
        }(arguments));

        // update frozen split size and position, after inserting/deleting rows or columns
        this.registerTransformationHandler(transformFrozenSplit);

        // invoke all registered transformation handlers, after inserting/deleting
        // columns or rows, and forward all column/row collection events
        colCollection.on({
            'insert:entries': function (event, interval) { invokeTransformationHandlers('insert:columns', interval, true, true); },
            'delete:entries': function (event, interval) { invokeTransformationHandlers('delete:columns', interval, false, true); },
            'change:entries': function () { self.trigger.apply(self, ['change:columns'].concat(_.toArray(arguments).slice(1))); }
        });
        rowCollection.on({
            'insert:entries': function (event, interval) { invokeTransformationHandlers('insert:rows', interval, true, false); },
            'delete:entries': function (event, interval) { invokeTransformationHandlers('delete:rows', interval, false, false); },
            'change:entries': function () { self.trigger.apply(self, ['change:rows'].concat(_.toArray(arguments).slice(1))); }
        });

        // update cell selection after inserting or deleting merged ranges, and forward the event
        mergeCollection.on('triggered', function () {
            var selection = getExpandedSelection(self.getViewAttribute('selection'));
            self.setViewAttribute('selection', selection);
            self.trigger.apply(self, _.toArray(arguments).slice(1));
        });

        // forward events of the table collection
        tableCollection.on('triggered', function () { self.trigger.apply(self, _.toArray(arguments).slice(1)); });

        // forward events of the cell collection
        cellCollection.on('triggered', function () { self.trigger.apply(self, _.toArray(arguments).slice(1)); });

        // update drawing selection after inserting or deleting drawing objects, forward events of the collection
        drawingCollection.on({ 'insert:drawing': insertDrawingHandler, 'delete:drawing': deleteDrawingHandler });
        drawingCollection.on('triggered', function () { self.trigger.apply(self, _.toArray(arguments).slice(1)); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            drawingCollection.destroy();
            cellCollection.destroy();
            validationCollection.destroy();
            tableCollection.destroy();
            mergeCollection.destroy();
            colCollection.destroy();
            rowCollection.destroy();
            nameCollection.destroy();
            app = docModel = documentStyles = styleCollection = null;
            nameCollection = colCollection = rowCollection = null;
            mergeCollection = tableCollection = validationCollection = null;
            cellCollection = drawingCollection = null;
            transformationHandlers = null;
        });

    } // class SheetModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: SheetModel });

});
