/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/compiler', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/model/formula/tokenutils',
    'io.ox/office/spreadsheet/model/formula/matrix'
], function (Utils, ModelObject, TokenUtils, Matrix) {

    'use strict';

    // class OperatorToken ====================================================

    /**
     * Represents an operator or function call in the compiled token array.
     *
     * @property {String} type
     *  The string 'op'.
     *
     * @property {Token} token
     *  A reference to the original parser token describing the operator or
     *  function call.
     *
     * @property {Number} params
     *  The number of operands associated to the operator.
     */
    function OperatorToken(token, params) {
        this.type = 'op';
        this.token = token;
        this.params = params;
    }

    OperatorToken.prototype.toString = function () {
        return 'op[' + this.token.getText()  + ']' + ':' + this.params;
    };

    // class OperandToken =====================================================

    /**
     * Represents an operand (literals, references, or names) in the compiled
     * token array.
     *
     * @property {String} type
     *  The string 'val'.
     *
     * @property {Token} token
     *  A reference to the original parser token describing the operand.
     */
    function OperandToken(token) {
        this.type = 'val';
        this.token = token;
    }

    OperandToken.prototype.toString = function () {
        return 'val[' + this.token.getType() + ':' + this.token.getText() + ']';
    };

    // class Compiler =========================================================

    /**
     * Compiles arrays of formula tokens from infix to prefix notation (a.k.a.
     * Polish Notation).
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function Compiler(app) {

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // public methods -----------------------------------------------------

        /**
         * Compiles the passed tokens from infix to prefix notation. Removes
         * all tokens that are not important for interpreting the formula, e.g.
         * white-space tokens, parentheses and parameter separators of function
         * calls, and all other regular parentheses.
         *
         * @param {Array} tokens
         *  The array of parser tokens, in infix notation, as received from the
         *  method Tokenizer.parseFormula().
         *
         * @param {Tokenizer} tokenizer
         *  The tokenizer instance used as token factory for additional tokens
         *  created during compilation.
         *
         * @returns {Object}
         *  The result descriptor of the compilation process, with the
         *  following properties:
         *  - {Array|Null} tokens
         *      An array of specific token descriptors in prefix notation:
         *      instances of class OperatorToken for operators, and instances
         *      of class OperandToken for operands; or null, if compilation has
         *      failed (see next property 'error').
         *  - {String|Null} error
         *      Null, if the token array has been compiled successfully.
         *      Otherwise, one of the following error codes:
         *      - 'missing': The end of the token array has been reached
         *          unexpectedly (e.g. incomplete formula).
         *      - 'unexpected': A token with an invalid type has been found.
         */
        this.compileTokens = function (tokens, tokenizer) {

            var // the current array index
                index = 0,
                // whether to stop subexpression at separator token (inside functions)
                stopAtSep = false,
                // the compiled array of token descriptors
                compiled = [],
                // result of the outer expression compilation (must be null for success)
                resultToken = null;

            /**
             * Pushes an operator descriptor to the compiled array.
             */
            function pushOperator(token, insert, params) {
                compiled.splice(insert, 0, new OperatorToken(token, params));
            }

            /**
             * Pushes an operand descriptor to the compiled array.
             */
            function pushOperand(token) {
                compiled.push(new OperandToken(token));
            }

            /**
             * Returns the next token, and increases the array index.
             *
             * @returns {Token|Null}
             *  The next available token; or null, if the end of the array has
             *  been reached.
             */
            function getNextRawToken() {
                var token = tokens[index];
                if (token) { index += 1; }
                return token;
            }

            /**
             * Returns the next non-white-space token, and increases the array
             * index.
             *
             * @param {Boolean} required
             *  If set to true, a following token must exist in the passed
             *  token array, otherwise an exception will be thrown (see below).
             *
             * @returns {Token|Null}
             *  The next available token; or null, if the end of the array has
             *  been reached (and the parameter 'required' is not set).
             *
             * @throws {String}
             *  The text 'missing', if the parameter 'required' is set, and no
             *  more tokens are available.
             */
            function getNextToken(required) {
                var token = getNextRawToken();
                while (token && token.isType('ws')) { token = getNextRawToken(); }
                if (required && !token) { throw 'missing'; }
                return token;
            }

            /**
             * Compiles a subexpression by invoking the passed function, and
             * afterwards, inserts the passed token into the compiled token
             * array.
             *
             * @param {Function} compilerFunc
             *  The callback function invoked with the next available token.
             *
             * @param {Token} opToken
             *  The operator token to be inserted into the token array.
             *
             * @param {Number} insert
             *  The target array index in the token array for the passed
             *  operator token.
             *
             * @param {Number} params
             *  The number of parameters expected by the operator token.
             *
             * @returns {Token|Null}
             *  The token returned by the passed callback function.
             */
            function compileTermAndPush(compilerFunc, opToken, insert, params) {
                var token = compilerFunc(getNextToken());
                pushOperator(opToken, insert, params);
                return token;
            }

            /**
             * Returns whether the token is an operator of the specified type.
             */
            function isOperatorToken(token, opType) {
                return _.isObject(token) && token.isType('op') && (token.getOpType() === opType);
            }

            /**
             * Compiles a subexpression starting at the passed token, until the
             * end of the formula, or the specified terminator token.
             */
            function compileExpression(token, newStopAtSep) {
                var oldStopAtSep = stopAtSep;
                stopAtSep = newStopAtSep;
                token = compileCompareTerm(token);
                stopAtSep = oldStopAtSep;
                return token;
            }

            /**
             * Compiles a comparison operator subexpression.
             */
            function compileCompareTerm(token) {
                var index = compiled.length;
                token = compileConcatTerm(token);
                while (isOperatorToken(token, '=')) {
                    token = compileTermAndPush(compileConcatTerm, token, index, 2);
                }
                return token;
            }

            /**
             * Compiles a string concatenation operator subexpression.
             */
            function compileConcatTerm(token) {
                var index = compiled.length;
                token = compileAddSubTerm(token);
                while (isOperatorToken(token, '&')) {
                    token = compileTermAndPush(compileAddSubTerm, token, index, 2);
                }
                return token;
            }

            /**
             * Compiles an addition or subtraction operator subexpression.
             */
            function compileAddSubTerm(token) {
                var index = compiled.length;
                token = compileMulDivTerm(token);
                while (isOperatorToken(token, '+')) {
                    token = compileTermAndPush(compileMulDivTerm, token, index, 2);
                }
                return token;
            }

            /**
             * Compiles a multiplication or division operator subexpression.
             */
            function compileMulDivTerm(token) {
                var index = compiled.length;
                token = compilePowTerm(token);
                while (isOperatorToken(token, '*')) {
                    token = compileTermAndPush(compilePowTerm, token, index, 2);
                }
                return token;
            }

            /**
             * Compiles a power operator subexpression.
             */
            function compilePowTerm(token) {
                var index = compiled.length;
                token = compileUnaryTerm(token);
                while (isOperatorToken(token, '^')) {
                    token = compileTermAndPush(compileUnaryTerm, token, index, 2);
                }
                return token;
            }

            /**
             * Compiles a unary prefix/postfix operator subexpression.
             */
            function compileUnaryTerm(token) {
                var index = compiled.length;
                if (isOperatorToken(token, '+')) {
                    token = compileTermAndPush(compileUnaryTerm, token, compiled.length, 1);
                } else {
                    token = compileListTerm(token);
                }
                while (isOperatorToken(token, '%')) {
                    pushOperator(token, index, 1);
                    token = getNextToken();
                }
                return token;
            }

            /**
             * Compiles a list operator subexpression.
             */
            function compileListTerm(token) {
                var index = compiled.length;
                token = compileIntersectTerm(token, index);
                while (token && token.isType('sep') && !stopAtSep) {
                    token = compileTermAndPush(compileIntersectTerm, tokenizer.createOperatorToken(',', ','), index, 2);
                }
                return token;
            }

            /**
             * Compiles an intersection operator subexpression.
             */
            function compileIntersectTerm(token) {
                var index = compiled.length;
                token = compileRangeTerm(token);
                while (isOperatorToken(token, '!')) {
                    token = compileTermAndPush(compileRangeTerm, token, index, 2);
                }
                return token;
            }

            /**
             * Compiles a range operator subexpression.
             */
            function compileRangeTerm(token) {
                var index = compiled.length;
                token = compileOperand(token);
                while (isOperatorToken(token, ':')) {
                    token = compileTermAndPush(compileOperand, token, index, 2);
                }
                return token;
            }

            /**
             * Compiles a single operand or function call in a subexpression.
             */
            function compileOperand(token) {

                if (!token) { throw 'missing'; }

                if (token.isType('lit')) {
                    pushOperand(token);
                    return getNextToken();
                }

                // bug 33526: reference and name tokens must contain valid sheet references
                if (token.isType(/^(ref|name)$/) && !token.hasSheetError()) {
                    pushOperand(token);
                    return getNextToken();
                }

                if (token.isType('open')) {
                    return compileParentheses();
                }

                if (token.isType('mat_open')) {
                    return compileMatrix();
                }

                if (token.isType('func')) {
                    return compileFunction(token);
                }

                throw 'unexpected';
            }

            /**
             * Compiles a complete subexpression in parentheses.
             */
            function compileParentheses() {
                var token = compileExpression(getNextToken(true));
                // accept missing closing parenthesis at end of formula, TODO: notify?
                if (token && !token.isType('close')) { throw 'unexpected'; }
                return getNextToken();
            }

            /**
             * Compiles a complete matrix literal.
             */
            function compileMatrix() {

                var // current state of the matrix processor
                    state = 'mat_open',
                    // current token
                    token = null,
                    // the two-dimensional array with the matrix elements
                    array = [];

                while (state !== 'end') {
                    switch (state) {
                    case 'mat_open':
                    case 'mat_row':
                        // start new row in the matrix
                        array.push([]);
                        /* falls through */
                    case 'mat_col':
                        // next token must be a literal
                        token = getNextToken(true);
                        if (!token.isType('lit')) { throw 'unexpected'; }
                        state = 'lit';
                        _.last(array).push(token.getValue());
                        break;
                    case 'lit':
                        // literal value: a separator or closing parenthesis must follow
                        token = getNextToken(true);
                        if (!token.isType(/^mat_(col|row|close)$/)) { throw 'unexpected'; }
                        state = token.getType();
                        break;
                    case 'mat_close':
                        // closing parenthesis: finalize the matrix literal (check equal length of all rows first)
                        if (_.chain(array).pluck('length').unique().value().length > 1) { throw 'unexpected'; }
                        pushOperand(tokenizer.createMatrixToken(new Matrix(array)));
                        state = 'end';
                        break;
                    }
                }

                return getNextToken();
            }

            /**
             * Compiles a complete function call.
             */
            function compileFunction(token) {

                var // the passed function token
                    funcToken = token,
                    // array index for the function token (cannot be pushed before parameter count is known)
                    index = compiled.length,
                    // current state of the function processor
                    state = 'func',
                    // number of parameters in the function
                    params = 0;

                while (state !== 'end') {
                    switch (state) {
                    case 'func':
                        // function start: next token must be an opening parenthesis
                        token = getNextToken(true);
                        if (!token.isType('open')) { throw 'unexpected'; }
                        state = 'open';
                        break;
                    case 'open':
                        // opening parenthesis: a closing parenthesis may follow directly
                        token = getNextToken();
                        // accept missing closing parenthesis after opening parenthesis at end of formula, TODO: notify?
                        state = (!token || token.isType('close')) ? 'close' : 'param';
                        break;
                    case 'param':
                        // start of parameter subexpression
                        params += 1;
                        // accept missing closing parenthesis after separator at end of formula, TODO: notify?
                        if (!token || token.isType(/^(sep|close)$/)) {
                            // empty parameter, e.g. in SUM(1,,2)
                            pushOperand(tokenizer.createLiteralToken(null));
                        } else {
                            // compile the parameter subexpression (stop at next separator token)
                            token = compileExpression(token, true);
                            // accept missing closing parenthesis after parameter at end of formula, TODO: notify?
                            if (token && !token.isType(/^(sep|close)$/)) { throw 'unexpected'; }
                        }
                        state = token ? token.getType() : 'close';
                        break;
                    case 'sep':
                        // parameter separator: a valid token must follow
                        token = getNextToken();
                        state = 'param';
                        break;
                    case 'close':
                        // closing parenthesis: finalize the function call
                        pushOperator(funcToken, index, params);
                        state = 'end';
                        break;
                    }
                }

                return getNextToken();
            }

            try {
                TokenUtils.info('Compiler.compileTokens()');
                TokenUtils.logTokens('  compiling', tokens);
                // compile the entire token array as subexpression
                resultToken = compileExpression(getNextToken(true));
                // double-check the compiled tokens
                if (compiled.length === 0) { throw 'missing'; }
                // error, when tokens are left in the passed token array
                if (_.isObject(resultToken) || (index < tokens.length)) { throw 'unexpected'; }
                TokenUtils.logTokens('  compiled', compiled);
            } catch (error) {
                if (_.isString(error)) {
                    TokenUtils.warn('Compiler.compileTokens(): error: ' + error + ' token, index=' + index);
                    return { tokens: null, error: error };
                }
                throw error;
            }

            return { tokens: compiled, error: null };
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = null;
        });

    } // class Compiler

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: Compiler });

});
