/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/editmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/logger',
     'io.ox/office/baseframework/model/basemodel',
     'io.ox/office/editframework/utils/operations'
    ], function (Utils, Logger, BaseModel, Operations) {

    'use strict';

    var // operations logger
        logger = new Logger({ enable: 'office:log-ops', prefix: 'OP' });

    // class EditModel ========================================================

    /**
     * The base class for editable document models. Adds generic support for
     * operations and undo action management.
     *
     * Triggers the following events:
     * - 'operations:before'
     *      Before an array of operations will be applied via the method
     *      EditModel.applyOperations(). The event handler receives the
     *      operations array, and the external state passed to the method. If a
     *      single operation has been passed to that method, it will be
     *      converted to an array with one element before calling the event
     *      handler.
     * - 'operations:after'
     *      After an array of operations have been applied via the method
     *      EditModel.applyOperations(), regardless if successfully or not. The
     *      event handler receives the external state passed to the method
     *      EditModel.applyOperations().
     * - 'operations:success'
     *      Directly after the event 'operations:after', if the operations have
     *      been applied successfully. The event handler receives the
     *      operations array, and the external state passed to the method
     *      EditModel.applyOperations(). If a single operation has been passed
     *      to that method, it will be converted to an array with one element
     *      before calling the event handler.
     * - 'operations:error'
     *      Directly after the event 'operations:after', if applying the
     *      operations has failed. The event handler receives the external
     *      state passed to the method EditModel.applyOperations().
     * - 'change:editmode'
     *      When the edit mode of the document model has been changed. Event
     *      handlers receive the new state of the edit mode.
     *
     * @constructor
     *
     * @extends BaseModel
     *
     * @param {EditApplication} app
     *  The application containing this document model instance.
     *
     * @param {Function} OperationsGeneratorClass
     *  The constructor function of the operations generator. Must be a sub
     *  class of class OperationsGenerator.
     *
     * @param {UndoManager} undoManager
     *  The undo manager used by the document model to store undo and redo
     *  actions.
     *
     * @param {DocumentStyles} documentStyles
     *  Global collection with the style sheet containers and custom formatting
     *  containers of a document.
     */
    function EditModel(app, OperationsGeneratorClass, undoManager, documentStyles) {

        var // self reference for local functions
            self = this,

            // Deferred object for current asynchronous actions, will remain in
            // resolved/rejected state as long as no new actions will be applied
            actionsDef = $.when(),

            // maps all operation names to operation handler functions
            operationHandlers = {},

            // whether operations are currently applied (prevent recursive calls)
            processingOperations = false,

            // edit mode, received dynamically from server (one editing user at a time)
            editMode = null,

            // the total number of operations applied successfully
            totalOperations = 0,

            // a counter for the operations, that adds all single operations, taking care of merged
            // operations. So this number must be equal on every client, independent from merge
            // processes on client or server side.
            operationStateNumber = -1,

            // collecting the last two operations in cache
            operationsCache = { previous: null, current: null };

        // base constructors --------------------------------------------------

        BaseModel.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Switches this document model into 'action processing mode', and
         * invokes the passed callback function. As long as processing mode is
         * active, the method EditModel.getActionsPromise() returns a pending
         * promise. The callback function may return a promise to control the
         * duration of the action processing mode. This method MUST NOT be
         * called recursively! The methods EditModel.applyOperations() and
         * EditModel.applyActions() must not be used (these methods will enter
         * the action processing mode by themselves).
         *
         * @param {Function} callback
         *  The callback function that will be invoked with activated action
         *  processing mode. Must return either a Boolean value that represents
         *  the result of the callback function, or the promise of a Deferred
         *  object to extend the processing mode until the promise has been
         *  resolved or rejected. Will be called in the context of this
         *  instance.
         *
         * @returns {Any}
         *  The return value of the callback function.
         */
        function enterProcessActionsMode(callback) {

            var // the result of the callback function
                result = null;

            // prevent recursive invocations
            if (actionsDef.state() === 'pending') {
                Utils.error('EditModel.enterProcessActionsMode(): recursive call');
                return false;
            }

            // create an new Deferred object which remains in pending state as long
            // as the callback function runs (and the promise it returns is pending)
            actionsDef = $.Deferred().always(function () {
                logger.log('EditModel.enterProcessActionsMode(): ' + actionsDef.state().replace(/ed$/, 'ing') + ' promise...');
            });

            // invoke the callback function
            result = callback.call(self);

            // resolve the actions promise
            if (_.isObject(result) && _.isFunction(result.promise)) {
                result.progress(_.bind(actionsDef.notify, actionsDef))
                    .done(_.bind(actionsDef.resolve, actionsDef))
                    .fail(_.bind(actionsDef.reject, actionsDef));
            } else if (result === true) {
                actionsDef.resolve();
            } else {
                actionsDef.reject();
            }

            return result;
        }

        /**
         * Extracts and returns the operation filter callback function from the
         * passed options.
         */
        function getFilterCallback(options) {
            var filterFunc = Utils.getFunctionOption(options, 'filter');
            return _.isFunction(filterFunc) ? filterFunc : _.constant(Utils.getBooleanOption(options, 'filter', true));
        }

        /**
         * Called before an operations array will be applied at the document
         * model. Triggers an 'operations:before' event.
         *
         * @param {Array} operations
         *  The operations to be applied. Will be passed to the listeners of
         *  the 'operations:before' event.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the 'operations:before' event.
         *
         * @returns {Boolean}
         *  False if this method has been called recursively, i.e. the model
         *  already processes operations. This is considered a fatal error.
         */
        function beginProcessOperations(operations, external) {

            if (processingOperations) {
                Utils.error('EditModel.beginProcessOperations(): recursive call - currently processing operations.');
                return false;
            }

            if (app.getState() === 'error') {
                Utils.error('EditModel.beginProcessOperations(): illegal call - application is in error state.');
                return false;
            }

            self.trigger('operations:before', operations, external);
            processingOperations = true;
            return true;
        }

        /**
         * Called after an operations array has been applied at the document
         * model. Triggers an 'operations:after' event, and one of the
         * 'operations:success' or 'operations:error' events.
         *
         * @param {Boolean} success
         *  Whether applying the operations has succeeded.
         *
         * @param {Array} operations
         *  The operations that have been applied. Will be passed to the
         *  listeners of the operations events.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the operations events.
         */
        function endProcessOperations(success, operations, external) {
            processingOperations = false;
            self.trigger('operations:after', operations, external);
            if (success) {
                self.trigger('operations:success', operations, external);
            } else {
                self.setEditMode(false);
                self.trigger('operations:error', operations, external);
            }
        }

        /**
         * Handles a number for the operation state to find problems
         * in the communication between different clients earlier. This
         * is currently only used in debug mode.
         *
         * @param {Object} operation
         *  A single operation to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         */
        function handleOperationStateNumber(operation, external) {

            // checking the correct operation state number of an external operation
            // The operation state number must be identical on every client.
            // The operation state number is sent from server, when the document is loaded
            if (external) {

                // During loading the document, the initial operation state number
                // is sent by the server. If we don't have a valid value use it.
                if ((operation.osn >= 0) && (operationStateNumber === -1)) {
                    operationStateNumber = operation.osn;
                }

                // check for the correct operation state number of an external operation
                if (_.isNumber(operation.osn) && (operationStateNumber !== operation.osn)) {
                    // If this client is already synchronized -> there is a conflict in the operation state number
                    return false;
                }
            } else {
                // Preparing internal operations, so that remote clients can control the correct operation state number.
                // A. setting current operation state number as required base for this operation
                operation.osn = operationStateNumber;
                // B. setting length of this operation -> this might be increased by operation merge processes
                operation.opl = 1;
            }

            if (operationStateNumber >= 0) {
                // increasing the operation state number of the document
                operationStateNumber = operationStateNumber + (operation.opl || 1);
            }

            return true;
        }

        /**
         * Executes the handler function for the passed operation.
         *
         * @param {Object} operation
         *  A single operation to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Function} filter
         *  A predicate function that specifies whether to invoke the operation
         *  callback handler.
         *
         * @returns {Boolean}
         *  Whether the operation has been applied successfully.
         */
        function applyOperation(operation, external, filter) {

            var // the operation scope
                scope = Utils.getStringOption(operation, 'scope', 'all'),
                // the function that executes the operation
                operationHandler = null,
                // success value of operation
                success = false;

            function failAndReturn(message) {
                message = 'EditModel.applyOperation(): ' + message;
                Utils.error(message);
                app.sendLogMessage(message);
                return false;
            }

            // log operation
            logger.log('i=' + totalOperations + ', op=', operation);

            // check operation
            if (!_.isObject(operation) || !_.isString(operation.name)) {
                return failAndReturn('Expecting operation object.');
            }

            // Handle an operation state number. This number needs to be synchronized
            // between all clients and has then to be identical for all clients.
            if (!handleOperationStateNumber(operation, external)) {
                return failAndReturn('Wrong operation state number: ' + operationStateNumber + ' (expected: ' + operation.osn + ').');
            }

            // get and check operation handler
            operationHandler = operationHandlers[operation.name];
            if (!_.isFunction(operationHandler)) {
                return failAndReturn('Invalid operation name "' + operation.name + '".');
            }

            // cache the last two operations
            operationsCache.previous = operationsCache.current;
            operationsCache.current = operation;

            // execute the operation handler (unless the operation scope is 'filter' for operations intended
            // to be executed in the filter component only, or the filter callback function returns false)
            try {
                success = (scope === 'filter') || (filter.call(self, operation) !== true) || (operationHandler.call(self, operation, external) !== false);
            } catch (ex) {
                Utils.exception(ex);
            }

            // log error and return with failure
            if (!success) {
                return failAndReturn((totalOperations + 1) + (external ? ' E ' : ' I ') + JSON.stringify(operation) + ' failed.');
            }

            totalOperations += 1;
            return true;
        }

        /**
         * Executes the handler functions for all passed operations. If the
         * undo manager is active, all applied operations will be embedded into
         * a single undo group action.
         *
         * @param {Array} operations
         *  The operations to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Function} filter
         *  A predicate function that specifies whether to invoke the operation
         *  callback handler.
         *
         * @returns {Boolean}
         *  Whether applying all operations was successful.
         */
        function applyOperationsSync(operations, external, filter) {

            var // the boolean result for synchronous mode
                success = false;

            // if the operation's origin is not an external client and
            // we don't have edit rights then we skip the operations silently.
            if ((!external && !editMode) || (operations.length === 0)) {
                return true;
            }

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external)) {
                // apply all operations at once, group into an undo action
                undoManager.enterUndoGroup(function () {
                    success = _.all(operations, function (operation) {
                        return applyOperation(operation, external, filter);
                    });
                });
            }

            // post-processing (trigger end events, leave operations mode),
            // also if beginProcessOperations() has failed (notify the error)
            endProcessOperations(success, operations, external);
            return success;
        }

        /**
         * Executes the handler functions for all passed operations in an
         * asynchronous background loop. If the undo manager is active, all
         * applied operations will be embedded into a single undo group action.
         *
         * @param {Array} operations
         *  The operations to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Function} filter
         *  A predicate function that specifies whether to invoke the operation
         *  callback handler.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after all
         *  operations have been applied successfully, or rejected immediately
         *  after applying an operation has failed. The Promise regularly sends
         *  progress updates while applying the operations.
         */
        function applyOperationsAsync(operations, external, filter) {

            var // the resulting Promise
                promise = null,
                // the global index of the first applied operation
                startIndex = totalOperations;

            // immediate success on empty operations array
            if (operations.length === 0) { return $.when(); }

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external)) {

                // apply all operations in a background task, group into an undo action
                promise = undoManager.enterUndoGroup(function () {

                    // return the Deferred to enterUndoGroup() to defer the open undo group
                    return self.iterateArraySliced(operations, function (operation) {

                        // if the operation's origin is not an external client and
                        // we don't have edit rights then we skip the operations silently
                        if (!external && !editMode) {
                            promise.abort();
                            return Utils.BREAK;
                        }

                        // break the loop and reject the promise if an operation has failed
                        if (!applyOperation(operation, external, filter)) {
                            return $.Deferred().reject({ cause: 'operation' });
                        }
                    }, { delay: 'immediate' });
                });

            } else {
                // currently applying operations (called recursively)
                promise = $.Deferred().reject({ cause: 'recursive' });
            }

            // post-processing (trigger end events, leave operations mode)
            return promise.done(function () {
                // all passed operations applied successfully
                endProcessOperations(true, operations, external);
            }).fail(function (result) {
                if (result === 'abort') {
                    // manually aborted: notify applied operations only
                    endProcessOperations(true, operations.slice(0, totalOperations - startIndex), external);
                } else {
                    // operations failed: notify all operations
                    endProcessOperations(false, operations, external);
                }
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Registers a function that will be executed when an operation with
         * the specified name will be applied.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be executed for the operation. All operation
         *  handler functions will be called in the context of this document
         *  model instance. Receives the operation object as first parameter,
         *  and the external state that has been passed to the method
         *  EditModel.applyOperations() as second parameter. If the operation
         *  handler returns the boolean value false, further processing of
         *  operations will be stopped immediately, and an 'operations:error'
         *  event will be triggered. Every other return value (also undefined)
         *  indicates that the operation has been applied successfully.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.registerOperationHandler = function (name, handler) {
            operationHandlers[name] = handler;
            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether the 'action processing mode' of this document model
         * is currently active (whether this model currently generates,
         * applies, or sends operation actions, either synchronously or
         * asynchronously).
         *
         * @returns {Boolean}
         *  Whether the 'action processing mode' is currently active.
         */
        this.isProcessingActions = function () {
            return actionsDef.state() === 'pending';
        };

        /**
         * Returns the promise of a Deferred object representing the state of
         * the 'action processing mode'. If in pending state, this model
         * currently generates, applies, or sends operation actions (either
         * synchronously or asynchronously). If in resolved state, the last
         * operations have been applied successfully. If in rejected state, the
         * last operations have not been applied successfully.
         *
         * @attention
         *  The promise MUST NOT be cached externally. Its solely purpose is to
         *  register done/fail callbacks from operation handlers or other code
         *  depending on operations. Whenever new operations will be applied, a
         *  new promise will be created representing the new operations (the
         *  old promises returned by this method before will remain in
         *  resolved/rejected state forever).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object representing the state of the
         *  action processing mode.
         */
        this.getActionsPromise = function () {
            return actionsDef.promise();
        };

        /**
         * Returns whether operations are currently processed.
         *
         * @returns {Boolean}
         *  Whether the model processes operations.
         */
        this.isProcessingOperations = function () {
            return processingOperations;
        };

        /**
         * Sends the passed operations to the server without applying them
         * locally. In special situations, a client may choose to apply the
         * operations manually before they will be sent to the server. In this
         * case, the registered operation handlers must not be called.
         *
         * @param {Object|Array} operations
         *  A single operation, or an array with operations to be sent to the
         *  server.
         *
         * @returns {Boolean}
         *  Whether registering all operations for sending was successful.
         */
        this.sendOperations = logger.profileMethod('EditModel.sendOperations()', function (operations) {
            return applyOperationsSync(_.getArray(operations), false, _.constant(false));
        });

        /**
         * Executes the handler functions for all passed operations. If the
         * undo manager is active, all applied operations will be embedded into
         * a single undo group action.
         *
         * @param {Object|Array} operations
         *  A single operation, or an array with operations to be applied at
         *  the document model.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Function|Boolean} [options.filter]
         *      A predicate function that will be invoked for each operation
         *      contained in the passed actions. Receives the operation as
         *      first parameter, and must return a Boolean specifying whether
         *      to invoke the operation handler. Can be set to the Boolean
         *      value false to prevent applying any operation locally (only
         *      send them to the server).
         *
         * @returns {Boolean}
         *  Whether applying all operations was successful.
         */
        this.applyOperations = logger.profileMethod('EditModel.applyOperations()', function (operations, options) {

            // apply the operations synchronously, but enter action processing mode anyway,
            // operation handler may register themselves at the actions promise
            return enterProcessActionsMode(function () {
                return applyOperationsSync(_.getArray(operations), false, getFilterCallback(options));
            });
        });

        /**
         * Executes the handler functions for all operations contained in the
         * passed actions. The operations of each action will be enclosed into
         * distinct undo groups automatically.
         *
         * @param {Object|Array} actions
         *  A single action, or an array with actions containing operations to
         *  be applied at the document model.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.external=false]
         *      If set to true, the actions have been received from an external
         *      sender. Otherwise, the actions have been generated by the
         *      document model itself.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the actions will be applied asynchronously in a
         *      browser timeout loop. The Promise returned by this method will
         *      be resolved or rejected after applying all actions, and will be
         *      notified about the progress of the operation.
         *  @param {Function} [options.filter]
         *      A predicate function that will be invoked for each operation
         *      contained in the passed actions. Receives the operation as
         *      first parameter, and must return a Boolean specifying whether
         *      to invoke the operation handler. Can be set to the Boolean
         *      value false to prevent applying any operation locally (only
         *      send them to the server).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved after all
         *  actions have been applied successfully, or rejected immediately
         *  after applying an action has failed. The caller has to wait for the
         *  promise before new actions can be applied. The promise regularly
         *  sends progress updates while applying the actions. In synchronous
         *  mode (option 'async' not set), the returned promise is already
         *  resolved or rejected when this method finishes. In asynchronous
         *  mode, the promise will contain an additional method 'abort()' that
         *  can be used to stop applying the operations at any time. When
         *  rejected, the promise will pass a result object with the following
         *  properties:
         *  - {String} result.cause
         *      The error code. Will be set to 'operation', if an operation
         *      callback handler has failed; or to 'recursive', if this method
         *      has been invoked recursively.
         */
        this.applyActions = logger.profileAsyncMethod('EditModel.applyActions()', function (actions, options) {

            var // total number of operations
                operationCount = 0;

            // convert parameter to an array, filter for actions containing
            // at least one operation, get total number of operations
            actions = _.chain(actions).getArray().filter(function (action) {
                var count = _.isArray(action.operations) ? action.operations.length : 0;
                operationCount += count;
                return count > 0;
            }).value();

            // no operations: early exit
            if (operationCount === 0) { return $.when(); }

            // apply in action processing mode
            return enterProcessActionsMode(function () {

                var // whether the actions have been received from the server
                    external = Utils.getBooleanOption(options, 'external', false),
                    // whether to execute asynchronously
                    async = Utils.getBooleanOption(options, 'async', false),
                    // invocation filter callback
                    filter = getFilterCallback(options),
                    // resulting Deferred object returned by this method
                    resultDef = $.Deferred();

                // apply actions synchronously if specified
                if (!async) {

                    var // synchronous mode: apply all actions at once, but bail out on error
                        success = _.all(actions, function (action) {
                            // applyOperationsSync() creates undo groups by itself
                            return applyOperationsSync(action.operations, external, filter);
                        });

                    // resolve or reject the resulting Deferred object immediately
                    return success ? resultDef.resolve() : resultDef.reject({ cause: 'operation' });
                }

                var // the timer processing the array asynchronously
                    timer = null,
                    // start index of first operations in the current action (for progress)
                    startIndex = 0,
                    // the promise of the inner loop processing operations of an action
                    innerPromise = null;

                // apply the actions in a browser timeout loop
                timer = this.iterateArraySliced(actions, function (action) {

                    // apply the operations of the current action
                    innerPromise = applyOperationsAsync(action.operations, external, filter);

                    // calculate and notify the total progress of all actions
                    innerPromise.progress(function (progress) {
                        var currIndex = startIndex + action.operations.length * progress;
                        resultDef.notify(currIndex / operationCount);
                    });

                    // update the start index for the next action
                    innerPromise.done(function () { startIndex += action.operations.length; });

                    // clean up after finishing the loop
                    return innerPromise.always(function () { innerPromise = null; });

                }, { delay: 'immediate' });

                // forward the result of the timer to the own Deferred object
                timer.done(function () { resultDef.notify(1).resolve(); }).fail(_.bind(resultDef.reject, resultDef));

                // create an abortable promise for the Deferred object
                return this.createAbortablePromise(resultDef, function () {
                    // abort the outer loop processing the actions
                    timer.abort();
                    // abort the inner loop processing the operations of an action
                    if (innerPromise) { innerPromise.abort(); }
                });
            }, this);
        });

        /**
         * Creates and returns an initialized operations generator instance (an
         * instance of the class specified in the constructor parameter
         * OperationsGeneratorClass).
         *
         * @returns {OperationsGenerator}
         *  A new operations generator instance.
         */
        this.createOperationsGenerator = function () {
            return new OperationsGeneratorClass(app);
        };

        /**
         * Creates a new operations generator, invokes the callback function,
         * and then applies all operations contained in the generator, and
         * sends them to the server.
         *
         * @param {Function} callback
         *  The callback function. Receives a new operations generator as first
         *  parameter. May return the promise of a Deferred object to defer
         *  applying and sending the operations until the promise has been
         *  resolved. Will be called in the context of this instance.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Function} [options.filter]
         *      A predicate function that will be invoked for each operation
         *      contained in the passed actions. Receives the operation as
         *      first parameter, and must return a Boolean specifying whether
         *      to invoke the operation handler. Can be set to the Boolean
         *      value false to prevent applying any operation locally (only
         *      send them to the server).
         *  @param {OperationsGeneratoe} [options.generator]
         *      A custom operations generator instance that will be passed to
         *      the callback function. If omitted, a new default operations
         *      generator will be created (see method
         *      EditModel.createOperationsGenerator() for details).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved after the
         *  operations have been applied and sent successfully (with the result
         *  of the callback function); or rejected, if the callback has
         *  returned a rejected promise (with the result of that promise), or
         *  if applying the operations has failed (with an object containing a
         *  property 'cause' set to the value 'operation').
         */
        this.createAndApplyOperations = logger.profileAsyncMethod('EditModel.createAndApplyOperations()', function (callback, options) {

            // simulate actions processing mode for deferred callbacks
            return enterProcessActionsMode(function () {

                var // the generator passed to the callback function
                    generator = Utils.getObjectOption(options, 'generator'),
                    // the promise returned by this method
                    promise = null;

                // create default operations generator if none has been passed to this method
                if (!_.isObject(generator)) { generator = self.createOperationsGenerator(); }

                // invoke the callback function
                promise = $.when(callback.call(self, generator));

                // apply and send the operations after the callback has finished
                promise = promise.then(function (result) {
                    var success = applyOperationsSync(generator.getOperations(), false, getFilterCallback(options));
                    return success ? result : $.Deferred().reject({ cause: 'operation' });
                });

                return promise;
            });
        });

        /**
         * Returns whether the document can be modified by the current user.
         *
         * @returns {Boolean}
         *  Whether the document can be modified.
         */
        this.getEditMode = function () {
            return editMode;
        };

        /**
         * Enables or disables the edit mode of this document model.
         *
         * @param {Boolean} state
         *  The new edit mode state. If set to true, the user can modify the
         *  document, otherwise the user can view the document contents, but
         *  cannot edit them.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.setEditMode = function (state) {
            if (editMode !== state) {
                editMode = state;
                // notify state change to listeners
                this.trigger('change:editmode', editMode);
            }
            return this;
        };

        /**
         * Returns the undo manager instance of this document containing the
         * undo action stack and the redo action stack.
         *
         * @returns {UndoManager}
         *  The undo manager instance.
         */
        this.getUndoManager = function () {
            return undoManager;
        };

        /**
         * Returns the global collection with the style sheet containers and
         * custom formatting containers of a document.
         *
         * @returns {DocumentStyles}
         *  The global document styles collection of the document.
         */
        this.getDocumentStyles = function () {
            return documentStyles;
        };

        /**
         * Returns the style collection for the passed style family.
         *
         * @param {String} family
         *  The name of the attribute family.
         *
         * @returns {StyleSheets}
         *  The style collection registered for the passed style family.
         */
        this.getStyleCollection = function (styleFamily) {
            return documentStyles.getStyleCollection(styleFamily);
        };

        /**
         * Returns the collection with all fonts registered for this document.
         *
         * @returns {FontCollection}
         *  The collection with all fonts registered for this document.
         */
        this.getFontCollection = function () {
            return documentStyles.getFontCollection();
        };

        /**
         * Returns the collection with all themes registered for this document.
         *
         * @returns {ThemeCollection}
         *  The collection with all themes registered for this document.
         */
        this.getThemeCollection = function () {
            return documentStyles.getThemeCollection();
        };

        /**
         * Returns the total number of operations applied successfully.
         */
        this.getOperationsCount = function () {
            return totalOperations;
        };

        /**
         * Returns the operation state number of the document.
         */
        this.getOperationStateNumber = function () {
            return operationStateNumber;
        };

        /**
         * Sets the operation state number of the document. Used when empty doc is fast displayed
         * @param {Number} osn - new osn number
         */
        this.setOperationStateNumber = function (osn) {
            operationStateNumber = osn;
            this.trigger('change:osn', osn); // update osn value in operations pane
        };

        /**
         * Returns the operation cache with the last two operations.
         *
         * @returns {Object}
         *  An object containing the last two operations in the properties
         *  last and current.
         */
        this.getOperationsCache = function () {
            return operationsCache;
        };

        /**
         * Returns the previous operation saved in the operation cache.
         * This is required for performance reasons. So no copy is returned
         * but the original operation object.
         *
         * @returns {Object}
         *  An object containing the previous operation.
         */
        this.getPreviousOperation = function () {
            return operationsCache.previous;
        };

        // operation handlers -------------------------------------------------

        // setting default attributes is not undoable
        this.registerOperationHandler(Operations.SET_DOCUMENT_ATTRIBUTES, function (operation) {

            var // the attribute set from the passed operation
                attributes = _.clone(Utils.getObjectOption(operation, 'attrs', {}));

            // set the global document attributes
            if (_.isObject(attributes.document)) {
                documentStyles.setDocumentAttributes(attributes.document);
                delete attributes.document;
            }

            // set default values for other attributes
            documentStyles.setDefaultAttributes(attributes);
        });

        this.registerOperationHandler(Operations.INSERT_THEME, function (operation) {
            this.getThemeCollection().insertTheme(operation.themeName, operation.attrs);
        });

        this.registerOperationHandler(Operations.INSERT_FONT_DESCRIPTION, function (operation) {
            this.getFontCollection().insertFont(operation.fontName, operation.attrs);
        });

        this.registerOperationHandler(Operations.INSERT_STYLESHEET, function (operation) {

            var // target style sheet container
                styleSheets = documentStyles.getStyleCollection(operation.type),
                // passed attributes from operation
                attributes = Utils.getObjectOption(operation, 'attrs', {});

            if (!_.isObject(styleSheets)) {
                Utils.warn('EditModel.insertStyleSheet(): invalid style family: "' + operation.type + '"');
                return false;
            }

            // TODO: undo for insertion, needs implementation of DELETE_STYLESHEET
//            undoManager.addUndo({ name: Operations.DELETE_STYLESHEET, type: operation.type, styleId: operation.styleId }, operation);

            return styleSheets.insertStyleSheet(operation.styleId, operation.styleName, operation.parent, attributes, {
                hidden: operation.hidden,
                priority: operation.uiPriority,
                defStyle: operation['default']
            });
        });

        this.registerOperationHandler(Operations.DELETE_STYLESHEET, function (operation) {

            var // target style sheet container
                styleSheets = documentStyles.getStyleCollection(operation.type);

            if (!_.isObject(styleSheets)) {
                Utils.warn('EditModel.deleteStyleSheet(): invalid style family: "' + operation.type + '"');
                return false;
            }

            // TODO: undo for deletion
//            undoManager.addUndo({ name: Operations.INSERT_STYLESHEET, type: operation.type, styleId: operation.styleId, attrs: ..., ... }, operation);

            return styleSheets.deleteStyleSheet(operation.styleId);
        });

        this.registerOperationHandler(Operations.NOOP, _.constant(true));

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            undoManager.destroy();
            documentStyles.destroy();
            undoManager = documentStyles = null;
        });

    } // class EditModel

    // exports ================================================================

    // derive this class from class BaseModel
    return BaseModel.extend({ constructor: EditModel });

});
