/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/control/charttypepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/drawinglayer/lib/canvasjs.min',
    'gettext!io.ox/office/drawing'
], function (Utils, Forms, RadioList, Labels, CanvasJS, gt) {

    'use strict';

    // class ChartTypePicker ==================================================

    /**
     * ChartTypePicker generates preview Images for all assigned Chart-Types
     * With CanvasJS it uses the same Drawer like the Chart in the Sheets
     * Colors are Accents 1 till 4 and the preview are refresh if the theme is
     * changed.
     *
     * @constructor
     *
     * @extends RadioList
     */
    function ChartTypePicker(app, initOptions) {

        var // self reference
            self = this,

            // data-model for CanvasJS to drop all changed value before drawing
            data = {},

            scatterPoints = [2.5, 4, 1.5, 3.5, 2, 2.5];

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            width: 150,
            icon: 'fa-bar-chart-o',
            tooltip: gt('Chart type'),
            updateCaptionMode: 'label',
            itemDesign: 'grid',
            gridColumns: 6,
            smallerVersion: { css: { width: 50 }, hideLabel: true }
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * hook on create items, only called once per type,
         * sets the button text and fill the chain-data for deferred rendering
         */
        function createItemHandler(event, buttonNode) {
            buttonNode.addClass('mini-caption').css({ height: 100, padding: 0 }).append('<img>');
        }

        /**
         * called initial and on theme-change
         * creates CanvasJS structure
         * and draws deferred all types in the chain to images
         */
        var updatePreviewCharts = this.createDebouncedMethod($.noop, function () {

            var // the style collection of the document
                documentStyles = app.getModel().getDocumentStyles(),
                // unique identifier used by CanvasJS
                id = _.uniqueId('io-ox-office-chart-dummy-'),
                // container node for the CanvasJS chart
                containerNode = $('<div id="' + id + '" style="position:fixed;visibility:hidden;left:-100px;top:-100px;width:100px;height:100px;">').appendTo('body'),
                // the CanvasJS renderer
                chart = new CanvasJS.Chart(id, data),
                // the <canvas> node from the renderer
                canvasNode = containerNode.find('canvas');

            if (canvasNode.length === 0) {
                Utils.warn('ChartTypePicker.updatePreviewCharts(): canvas element not found');
                return;
            }

            self.iterateArraySliced(self.getOptionButtons(), function (buttonNode) {

                var id = Forms.getButtonValue(buttonNode),
                    options = $(buttonNode).data('options'),
                    cjs = options.data.cjs,
                    single = /^(pie|doughnut)$/.test(cjs);

                _.each(data.data, function (dataSeries, index) {
                    dataSeries.type = cjs;
                    var cl = { type: 'scheme', value: 'accent' + (index + 1) };
                    dataSeries.color = documentStyles.getCssColor(cl, 'fill');
                });

                _.each(data.data[0].dataPoints, function (dataPoint, index) {
                    if (single) {
                        var cl = { type: 'scheme', value: 'accent' + (index + 1) };
                        dataPoint.color = documentStyles.getCssColor(cl, 'fill');
                    } else {
                        dataPoint.color = null;
                    }
                });

                _.each(data.data[1].dataPoints, function (dataPoint, index) {
                    if (id.indexOf('scatter') === 0) {
                        dataPoint.x = scatterPoints[index];
                    } else {
                        dataPoint.x = index;
                    }
                });

                chart.render();

                // update the image element
                $(buttonNode).find('>img').attr({
                    src: canvasNode[0].toDataURL(),
                    alt: Utils.getStringOption(options, 'label')
                });

            }).done(function () {
                containerNode.remove();
            });

        }, { delay: 50 });

        // initialization -----------------------------------------------------

        // initializes the data-model with disabled axes and random-values
        (function initData() {
            data.axisX = {};
            data.axisY = {};
            data.creditHref = '';
            data.creditText = '';
            data.title = { text: '' };
            data.data = [];

            var pointList = [[6, 4, 5, 3, 4, 1], [1, 2, 3, 4, 6, 7]];

            _.each(pointList, function (points, series) {
                var dataPoints = [];
                data.data[series] = { markerSize: 2, indexLabelFontColor: 'transparent', indexLabelLineColor: 'transparent', showInLegend: false, dataPoints: dataPoints, sourcePoints: {}};
                _.each(points, function (point, index) {
                    dataPoints.push({ label: 'bla ' + (index + 1), y: point, z: (2 + Math.random() * 100)|0});
                });
            });

            data.axisX.labelFontColor = data.axisX.lineColor = data.axisX.tickColor = data.axisX.gridColor = 'transparent';
            data.axisX.labelFontSize = 1;
            data.axisY.labelFontColor = data.axisY.lineColor = data.axisY.tickColor = data.axisY.gridColor = 'transparent';
            data.axisY.labelFontSize = 1;
        })();

        this.getMenuNode().addClass('chart-type-picker');

        // create the (empty) option buttons
        self.getMenu().on('create:item', createItemHandler);
        _.each(Labels.CHART_TYPE_STYLES, function (chartType, id) {
            self.createOptionButton(id, { section: chartType.cat, label: chartType.title, tooltip: chartType.title, data: chartType });
        });

        // lazy initialization of the charts before first usage of the menu
        this.getMenu().one('popup:beforeshow', updatePreviewCharts);

        // refresh preview charts in the drop-down menu after changed theme (color scheme)
        this.listenTo(app.getImportPromise(), 'done', function () {
            self.listenTo(app.getModel().getThemeCollection(), 'triggered', updatePreviewCharts);
        });

    } // class ChartTypePicker

    // exports ================================================================

    return RadioList.extend({ constructor: ChartTypePicker });

});
