/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/control/chartcolorsetpicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'gettext!io.ox/office/drawing'
], function (Utils, Forms, RadioList, ChartStyleUtil, gt) {

    'use strict';

    var // number of color boxes per list entry in the ColorSetPicker control
        COLOR_BOX_COUNT = 3,

        // all predefined color sets
        CHART_COLOR_SETS = ChartStyleUtil.getColorSet();

    // class ChartColorSetPicker ==============================================

    /**
     * A drop-down menu for picking different predefined chart color sets.
     *
     * @constructor
     *
     * @extends RadioList
     */
    function ChartColorSetPicker(app, initOptions) {

        var // self reference
            self = this,

            // the document style collection
            documentStyles = null;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            label: /*#. Menu title: predefined color sets for chart objects */ gt.pgettext('chart-format', 'Color set'),
            tooltip: /*#. Menu tooltip: predefined color sets for chart objects */ gt.pgettext('chart-format', 'Predefined color set'),
            smallerVersion: {
                label: gt.pgettext('chart-format', 'Color')
            }
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Inserts the special mark-up for a color set into the specified list
         * entry of the drop-down menu.
         */
        function createItemHandler(event, buttonNode, value, options) {

            var // the color set data this list item is based on
                colorSet = options.colorSet,
                // the content node for the color boxes
                markup = null;

            // create all color boxes
            markup = '<div class="color-node">';
            _.times(COLOR_BOX_COUNT, function (index) {

                var // extract the base color from the color set
                    color = _.copy(colorSet.colors[index % colorSet.colors.length], true),
                    // extract the variation from the color set
                    variation = (colorSet.variations.length > 0) ? colorSet.variations[index % colorSet.variations.length] : null;

                if (variation) { color.transformations = variation.transformations; }
                markup += '<span class="color-node" style="background-color:' + documentStyles.getCssColor(color, 'fill') + ';"></span>';
            });
            markup += '</div>';

            // insert the color boxes into the caption of the button node
            Forms.getCaptionNode(buttonNode).prepend(markup);
        }

        /**
         * Fills the drop-down menu with all available color sets.
         */
        var initializeColorTable = this.createDebouncedMethod($.noop, function () {
            self.clearMenu();
            _.each(CHART_COLOR_SETS, function (colorSet, index) {
                self.createOptionButton('cs' + index, { tooltip: colorSet.name, colorSet: colorSet });
            });
        }, { delay: 500 });

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('color-set-picker');
        this.getMenu().on('create:item', createItemHandler);

        // initialize class members
        app.onInit(function () {
            documentStyles = app.getModel().getDocumentStyles();
        });

        // insert color buttons in the drop-down menu after import and changed themes
        this.listenTo(app.getImportPromise(), 'done', function () {
            initializeColorTable();
            self.listenTo(app.getModel().getThemeCollection(), 'triggered', initializeColorTable);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = documentStyles = null;
        });

    } // class ChartColorSetPicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: ChartColorSetPicker });

});
