/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.api;

import static com.openexchange.obs.api.Tools.transform;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * {@link SourceFileParser}
 *
 * @author <a href="mailto:felix.marx@open-xchange.com">Felix Marx</a>
 */
public final class SourceFileParser {

    private static final DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

    private SourceFileParser() {
        super();
    }

    public static SourceFile[] parse(InputStream is) throws BuildServiceException {
        final Document document;
        try {
            final DocumentBuilder db = dbf.newDocumentBuilder();
            document = db.parse(is);
            final NodeList childs = document.getChildNodes();
            for (int i = 0; i < childs.getLength(); i++) {
                final Node node = childs.item(i);
                if ("directory".equals(node.getNodeName())) {
                     return parseDirectory(node);
                } else if (Node.TEXT_NODE == node.getNodeType()) {
                    continue;
                } else {
                    throw new BuildServiceException("Unknown node in build result \"" + node.getNodeName() + "\".");
                }
            }
        } catch (ParserConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (SAXException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
        throw new BuildServiceException("Unable to parse build result. " + transform(document));
    }

    private static SourceFile[] parseDirectory(Node directoryNode) throws BuildServiceException {
        final NodeList childs = directoryNode.getChildNodes();
        final NamedNodeMap attributes = directoryNode.getAttributes();
        final String pkgName = attributes.getNamedItem("name").getNodeValue();
        final List<SourceFile> sourceFiles = new ArrayList<SourceFile>();
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("entry".equals(node.getNodeName())) {
                sourceFiles.add(parseStatus(pkgName, node));
            } else if (Node.TEXT_NODE == node.getNodeType()) {
                continue;
            } else {
                throw new BuildServiceException("Unknown node in result \"" + node.getNodeName() + "\" \"" + node.getNodeValue() + "\".");
            }
        }
        return sourceFiles.toArray(new SourceFile[sourceFiles.size()]);
    }

    private static SourceFile parseStatus(String pkgName, Node entry) throws BuildServiceException {
        final NamedNodeMap attributes = entry.getAttributes();
        final Node sizeNode = attributes.getNamedItem("size");
        final String size = null == sizeNode ? null : sizeNode.getNodeValue();
        final Node mtimeNode = attributes.getNamedItem("mtime");
        final String mtime = null == mtimeNode ? null : mtimeNode.getNodeValue();
        final Node errorNode = attributes.getNamedItem("error");
        final SourceFile pkg = new SourceFile(
            pkgName,
            attributes.getNamedItem("name").getNodeValue(),
            attributes.getNamedItem("md5").getNodeValue(),
            size,
            mtime);
        if (null != errorNode) {
            pkg.setError(errorNode.getNodeValue());
        }
        return pkg;
    }
}
