/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.obs.api;

import static com.openexchange.obs.api.Tools.transform;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * {@link ProjectParser}
 *
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
public final class ProjectParser {

    private static final DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

    private ProjectParser() {
        super();
    }

    public static Project parse(final InputStream is) throws BuildServiceException {
        final DocumentBuilder db;
        try {
            db = dbf.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
        Document document = null;
        try {
            document = db.parse(is);
            final NodeList childs = document.getChildNodes();
            for (int i = 0; i < childs.getLength(); i++) {
                final Node node = childs.item(i);
                if ("project".equals(node.getNodeName())) {
                     return parse(node);
                } else if (Node.TEXT_NODE == node.getNodeType()) {
                    continue;
                } else {
                    throw new BuildServiceException("Unknown node in project meta \"" + node.getNodeName() + "\".");
                }
            }
        } catch (NullPointerException e) {
            throw new BuildServiceException("Unable to parse project: " + transform(document), e);
        } catch (BuildServiceException e) {
            throw new BuildServiceException("Unable to parse project: " + transform(document), e);
        } catch (SAXException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
        throw new BuildServiceException("Unable to parse project meta. " + transform(document));
    }

    public static boolean isDisabled(final InputStream is) throws BuildServiceException {
        final Document document;
        try {
            final DocumentBuilder db = dbf.newDocumentBuilder();
            document = db.parse(is);
            final NodeList childs = document.getChildNodes();
            for (int i = 0; i < childs.getLength(); i++) {
                final Node node = childs.item(i);
                if ("project".equals(node.getNodeName())) {
                    return isDisabled(node);
                }
            }
            return false;
        } catch (ParserConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (SAXException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
    }

    private static boolean isDisabled(final Node project) throws DOMException {
        final NodeList childs = project.getChildNodes();
        boolean builddisabled = false;
        boolean publishdisabled = false;
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("build".equals(node.getNodeName())) {
                builddisabled = true;
            }
            if ("publish".equals(node.getNodeName())) {
                publishdisabled = true;
            }
        }
        return builddisabled && publishdisabled;
    }

    public static ProjectReference[] parseAll(final InputStream is) throws BuildServiceException {
        final Document document;
        try {
            final DocumentBuilder db = dbf.newDocumentBuilder();
            document = db.parse(is);
            final NodeList childs = document.getChildNodes();
            for (int i = 0; i < childs.getLength(); i++) {
                final Node node = childs.item(i);
                if ("directory".equals(node.getNodeName())) {
                    return parseProjectlist(node);
                } else if (Node.TEXT_NODE == node.getNodeType()) {
                    continue;
                } else {
                    throw new BuildServiceException("Unknown node in project meta \"" + node.getNodeName() + "\".");
                }
            }
        } catch (ParserConfigurationException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (SAXException e) {
            throw new BuildServiceException(e.getMessage(), e);
        } catch (IOException e) {
            throw new BuildServiceException(e.getMessage(), e);
        }
        throw new BuildServiceException("Unable to parse project meta. " + transform(document));
    }

    private static ProjectReference[] parseProjectlist(final Node projectList) throws DOMException, BuildServiceException {
        final NodeList childs = projectList.getChildNodes();
        final List<ProjectReference> results = new ArrayList<ProjectReference>();
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("entry".equals(node.getNodeName())) {
                results.add(parseReference(node));
            } else if (Node.TEXT_NODE == node.getNodeType()) {
                continue;
            } else {
                throw new BuildServiceException("Unknown node in project list \"" + node.getNodeName() + "\" \"" + node.getNodeValue() + "\".");
            }
        }
        return results.toArray(new ProjectReference[results.size()]);
    }

    private static ProjectReference parseReference(final Node node) {
        final NamedNodeMap attributes = node.getAttributes();
        final String name = attributes.getNamedItem("name").getNodeValue();
        final ProjectReference project = new ProjectReference(name);
        return project;
    }

    private static Project parse(final Node node) throws BuildServiceException {
        final NamedNodeMap attributes = node.getAttributes();
        final String name = attributes.getNamedItem("name").getNodeValue();
        NodeList childs = node.getChildNodes();
        final Project project = new Project(name, parseTitle(childs), parseMaintainer(childs));
        for (Repository repository : parseProject(project, node)) {
            project.addRepository(repository);
        }
        return project;
    }

    private static String parseTitle(NodeList childs) throws BuildServiceException {
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("title".equals(node.getNodeName())) {
                return node.getNodeValue();
            }
        }
        throw new BuildServiceException("Can not find title of project.");
    }

    private static String parseMaintainer(NodeList childs) {
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("person".equals(node.getNodeName())) {
                NamedNodeMap attributes = node.getAttributes();
                if (null == attributes) {
                    continue;
                }
                Node userIdAttribute = attributes.getNamedItem("userid");
                if (null == userIdAttribute) {
                    continue;
                }
                String value = userIdAttribute.getNodeValue();
                if (!"Admin".equals(value)) {
                    return value;
                }
            }
        }
        return null;
    }

    private static Repository[] parseProject(Project project, Node projectNode) throws BuildServiceException {
        final NodeList childs = projectNode.getChildNodes();
        final List<Repository> packages = new ArrayList<Repository>();
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("repository".equals(node.getNodeName())) {
                packages.add(parseRepository(node));
            } else if (Node.TEXT_NODE == node.getNodeType()) {
                continue;
            } else if ("title".equals(node.getNodeName())) {
                continue;
            } else if ("description".equals(node.getNodeName())) {
                continue;
            } else if ("person".equals(node.getNodeName())) {
                continue;
            } else if ("download".equals(node.getNodeName())) {
                // ignore
                continue;
            } else if ("debuginfo".equals(node.getNodeName())) {
                // ignore
                continue;
            } else if ("useforbuild".equals(node.getNodeName())) {
                // ignore
                continue;
            } else if ("remoteurl".equals(node.getNodeName())) {
                continue;
            } else if ("build".equals(node.getNodeName())) {
                parseBuild(project, node);
            } else if ("publish".equals(node.getNodeName())) {
                parsePublish(project, node);
            } else {
                throw new BuildServiceException("Unknown node in result \"" + node.getNodeName() + "\" \"" + node.getNodeValue() + "\".");
            }
        }
        return packages.toArray(new Repository[packages.size()]);
    }

    private static Repository parseRepository(final Node repositoryNode) throws BuildServiceException {
        final NamedNodeMap attributes = repositoryNode.getAttributes();
        String repositoryName = attributes.getNamedItem("name").getNodeValue();
        BuildTrigger rebuild = attributes.getNamedItem("rebuild") == null ? BuildTrigger.TRANSITIVE : BuildTrigger.parse(attributes.getNamedItem("rebuild").getNodeValue());
        BlockMode block = attributes.getNamedItem("block") == null ? BlockMode.ALL : BlockMode.parse(attributes.getNamedItem("block").getNodeValue());
        List<String> arch = new ArrayList<String>();
        final NodeList childs = repositoryNode.getChildNodes();
        for (int i = 0; i < childs.getLength(); i++) {
            Node node = childs.item(i);
            if ("arch".equals(node.getNodeName())) {
                arch.add(node.getTextContent());
            }
        }
        Collection<Architecture> architectures = Architecture.parseArchitecture(arch);
        Repository repository = new Repository(repositoryName, parsePath(repositoryNode), architectures, rebuild, block);
        return repository;
    }

    private static void parseBuild(Project project, Node buildNode) {
        NodeList childs = buildNode.getChildNodes();
        for (int i = 0; i < childs.getLength(); i++) {
            Node node = childs.item(i);
            if ("disable".equals(node.getNodeName())) {
                NamedNodeMap attributes = node.getAttributes();
                if (0 == attributes.getLength()) {
                    project.setBuildDisabled(true);
                } else {
                    Node repositoryAttribute = attributes.getNamedItem("repository");
                    if (null != repositoryAttribute) {
                        project.addDisabledRepository(repositoryAttribute.getNodeValue());
                    }
                    // currently ignoring <disable arch="i586"/>
                }
            }
        }
    }

    private static void parsePublish(Project project, Node buildNode) {
        NodeList childs = buildNode.getChildNodes();
        for (int i = 0; i < childs.getLength(); i++) {
            Node node = childs.item(i);
            if ("disable".equals(node.getNodeName())) {
                NamedNodeMap attributes = node.getAttributes();
                if (0 == attributes.getLength()) {
                    project.setPublishDisabled(true);
                }
            }
        }
    }

    private static List<Path> parsePath(Node repositoryNode) {
        final NodeList childs = repositoryNode.getChildNodes();
        final List<Path> path = new ArrayList<Path>();
        for (int i = 0; i < childs.getLength(); i++) {
            final Node node = childs.item(i);
            if ("path".equals(node.getNodeName())) {
                path.add(new Path(
                    new ProjectReference(node.getAttributes().getNamedItem("project").getNodeValue()),
                    new RepositoryReference(node.getAttributes().getNamedItem("repository").getNodeValue())));
            }
        }
        return path;
    }
}
