/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.drive.storage.execute;

import java.util.List;
import com.openexchange.drive.Action;
import com.openexchange.drive.DriveVersion;
import com.openexchange.drive.actions.AbstractAction;
import com.openexchange.drive.internal.SyncSession;
import com.openexchange.drive.storage.StorageOperation;
import com.openexchange.exception.OXException;


/**
 * {@link BatchActionExecutor}
 *
 * @author <a href="mailto:tobias.friedrich@open-xchange.com">Tobias Friedrich</a>
 */
public abstract class BatchActionExecutor<T extends DriveVersion> extends AbstractActionExecutor<T> {

    protected final boolean allowBatches;

    /**
     * Initializes a new {@link BatchActionExecutor}.
     *
     * @param session The session
     */
    protected BatchActionExecutor(SyncSession session, boolean transactional, boolean allowBatches) {
        super(session, transactional);
        this.allowBatches = allowBatches;
    }

    @Override
    public void execute(List<AbstractAction<T>> actions) throws OXException {
        if (null == actions || 0 == actions.size()) {
            return; // nothing to do
        }
        if (false == allowBatches) {
            super.execute(actions); // execute each action separately
            return;
        }
        int currentBatchStart = 0;
        Action currentAction = actions.get(0).getAction();
        for (int i = 1; i < actions.size(); i++) {
            if (false == currentAction.equals(actions.get(i).getAction())) {
                /*
                 * execute batch
                 */
                execute(currentAction, actions.subList(currentBatchStart, i));
                /*
                 * prepare next batch
                 */
                currentBatchStart = i;
                currentAction = actions.get(i).getAction();
            }
        }
        /*
         * execute remaining batch
         */
        execute(currentAction, actions.subList(currentBatchStart, actions.size()));
    }

    private void execute(final Action action, final List<AbstractAction<T>> actions) throws OXException {
        if (transactional) {
            session.getStorage().wrapInTransaction(new StorageOperation<Void>() {

                @Override
                public Void call() throws OXException {
                    if (1 == actions.size()) {
                        execute(actions.get(0));
                    } else {
                        batchExecute(action, actions);
                    }
                    return null;
                }
            });
        } else {
            if (1 == actions.size()) {
                execute(actions.get(0));
            } else {
                batchExecute(action, actions);
            }
        }
    }

    protected abstract void batchExecute(Action action, List<AbstractAction<T>> actions) throws OXException;

}
