/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.util;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import com.openexchange.usm.api.session.Session;

/**
 * This class stores all information necessary to determine if a client behaves badly, i.e. it performs too many unnecessary synchronizations.
 * It is used and managed by AccessLimiter.
 * 
 * @author afe
 *
 */
public class AccessCounter {

	private static final String DATE_FORMAT_STRING = "yyyy-MM-dd HH:mm:ss.SSS Z";

	private final AccessLimiter _limiter;

	private final long[] _unnecessaryRequests;

	private int _currentIndex;
	private long _blockedUntil;

	public AccessCounter(AccessLimiter limiter) {
		_limiter = limiter;
		_unnecessaryRequests = new long[limiter.getMaxAccesses()];
	}

	/**
	 * Checks whether access is currently denied for this USM session. It will also remove this AccessCounter if the last bad request
	 * is outside the check interval (i.e. stops the counting since there was no bad access within the test interval).
	 * 
	 * @param session
	 * @return true if access should be denied, false if access may be granted
	 */
	public boolean isAccessDenied(Session session) {
		long now = System.currentTimeMillis();
		if (_blockedUntil >= now)
			return true;
		if (_unnecessaryRequests[_currentIndex] < now - _limiter.getTestInterval())
			_limiter.removeCounter(session);
		return false;
	}

	/**
	 * This should be called when an unnecessary access is detected. It will check if there were too many bad accesses within the
	 * test interval, and if true, it will log a warning and enable blocking of all future requests (i.e. isAccessDenied() will return true)
	 * for the configured refusal interval.
	 * 
	 * @param session
	 */
	public void onUnnecessaryAccess(Session session) {
		long now = System.currentTimeMillis();
		if (_blockedUntil >= now)
			return;
		if (_currentIndex >= _unnecessaryRequests.length - 1)
			_currentIndex = 0;
		else
			_currentIndex++;
		long difference = now - _unnecessaryRequests[_currentIndex];
		if (difference < _limiter.getTestInterval()) {
			_blockedUntil = now + _limiter.getRefusalInterval();
			_limiter.getJournal().warn(
					session + " Too many unnecessary synchronizations (" + _limiter.getMaxAccesses() + " in "
							+ difference + " ms.), access will be blocked until "
							+ new SimpleDateFormat(DATE_FORMAT_STRING, Locale.ENGLISH).format(new Date(_blockedUntil)));
		}
		_unnecessaryRequests[_currentIndex] = now;
	}
}
