/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.api.ox.json;

import java.util.Map;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.json.JSONArray;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.resource.ResourceInputStream;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.io.InputStreamProvider;
import com.openexchange.usm.api.session.OXConnectionInformation;
import com.openexchange.usm.api.session.Session;

/**
 * The OXAjaxAccess service wraps the http communication with the OX Ajax-Server interface
 * for easy access from other bundles, especially ContentTypes and Authentication.
 * 
 * <br>TODO Check if POST operation is also required
 * 
 * @author afe
 *
 */
public interface OXJSONAccess {
	/**
	 * Login user into OX server
	 * 
	 * @param session
	 * @throws AuthenticationFailedException if login failed, details are stored in the Exception
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	void login(Session session) throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * End session that was previously started using login()
	 * 
	 * @param session id of session to end
	 */
	void logout(Session session);

	/**
	 * Can be called after a session has been removed to cleanup any OX resources that might still be allocated,
	 * e.g. closing the OX session for the USM session.
	 * 
	 * @param data
	 */
	void logout(OXConnectionInformation data);

	/**
	 * Retrieve the OX Ajax configuration for the user of the Session
	 * 
	 * @param session
	 * @param path optional sub-path of configuration to retrieve
	 * @return
	 * @throws AuthenticationFailedException if the authentication failed on the OX server
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONObject getConfiguration(Session session, String... path) throws AuthenticationFailedException,
			OXCommunicationException;

	/**
	 * Perform a GET operation on the Ajax-Interface using the specified sub-path
	 * and request parameters.
	 * <br>TODO Check if using a Map is the best method for providing request parameters
	 * 	(impossible to specify parameters with multiple values, no types for parameters)
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doGet(String path, String action, Session session, Map<String, String> parameters)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Perform a PUT operation on the Ajax-Interface using the specified sub-path
	 * and request parameters. The provided JSONArray is sent in the request body
	 * <br>TODO Check if using a Map is the best method for providing request parameters
	 * 	(impossible to specify parameters with multiple values, no types for parameters)
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @param array JSONArray to send in request body
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, JSONArray array)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Perform a PUT operation on the Ajax-Interface using the specified sub-path
	 * and request parameters. The provided JSONObject is sent in the request body
	 * <br>TODO Check if using a Map is the best method for providing request parameters
	 * 	(impossible to specify parameters with multiple values, no types for parameters)
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @param object JSONObject to send in request body
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, JSONObject object)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Perform a PUT operation on the Ajax-Interface using the specified sub-path
	 * and request parameters. The provided String is sent in the request body (encoded using UTF-8)
	 * <br>TODO Check if using a Map is the best method for providing request parameters
	 * 	(impossible to specify parameters with multiple values, no types for parameters)
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @param data String to send in request body
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, String data)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Perform a PUT operation on the Ajax-Interface using the specified sub-path
	 * and request parameters. The provided byte array is sent in the request body
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @param data data to send in request body
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, byte[] data)
			throws AuthenticationFailedException, OXCommunicationException;

    /**
     * Perform a PUT operation on the Ajax-Interface using the specified sub-path
     * and request parameters. The data provided by the InputStream is sent in the request body
     * 
     * @param path sub-path to use for request url, e.g. "login", "folders", ...
     * @param action value for always required parameter action (if set to null, it will be omitted in the request)
     * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
     * @param parameters Map containing all request parameters to use
     * @param streamProvider InputStreamProvider for the data to send in request body
     * @return JSONData containing the parsed response
     * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
     * @throws OXCommunicationException if some error occurred during the communication with the OX server
     */
    JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, InputStreamProvider streamProvider) throws AuthenticationFailedException, OXCommunicationException;

	OXResource getResource(Session session, String path, Map<String, String> parameters)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Get a resource as InputStream. The returned InputStream must be closed before any other accesses to OX can be performed.
	 * 
	 * @param session
	 * @param oxAjaxPath
	 * @param parameters
	 * @return
	 * @throws AuthenticationFailedException
	 * @throws OXCommunicationException
	 */
    ResourceInputStream getResourceAsStream(Session session, String oxAjaxPath, Map<String, String> parameters)
        throws AuthenticationFailedException, OXCommunicationException;
	
	JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters,
			JSONObject jsonData, byte[] data, String fileContentType) throws AuthenticationFailedException,
			OXCommunicationException;

	JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters,
			JSONObject jsonData, byte[] data, String fileContentType, String jsonPartName, String filePartName)
			throws AuthenticationFailedException, OXCommunicationException;

    JSONResult storeResourceFromStream(String path, String action, Session session, Map<String, String> parameters,
            JSONObject jsonData, String fileContentType, long size, InputStreamProvider streamProvider)
            throws AuthenticationFailedException, OXCommunicationException;

    JSONResult storeResourceFromStream(String path, String action, Session session, Map<String, String> parameters,
            JSONObject jsonData, String fileContentType, String jsonPartName, String filePartName, long size, InputStreamProvider streamProvider)
            throws AuthenticationFailedException, OXCommunicationException;
	
	JSONResult doPost(String path, String action, Session session, Map<String, String> parameters)
			throws AuthenticationFailedException, OXCommunicationException;
	
	
	/**
	 * @return the used HttpConnectionManager
	 */
	MultiThreadedHttpConnectionManager getHttpConnectionManager();
}
