/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools;

/**
 * Enumeration for OX Documents application types.
 * 
 * @author Carsten Driesner
 */
public enum ApplicationType {

    APP_NONE,
	APP_TEXT,
    APP_SPREADSHEET,
    APP_PRESENTATION;
    
	public final static String APP_TEXT_STRING = "text";
	public final static String APP_SPREADSHEET_STRING = "spreadsheet";
	public final static String APP_PRESENTATION_STRING = "presentation";
	
    /**
     * Provides the pre-defined text for a provided application type.
     *
     * @param type
     *  The application type which should be mapped to a string.
     *
     * @return
     *  The application type as a string or null if the type is not specific
     *  (e.g. APP_NONE).
     */
	static public String enumToString(ApplicationType type) {
    	if (type == APP_TEXT) {
    		return "text";
    	} else if (type == APP_SPREADSHEET) {
    		return APP_SPREADSHEET_STRING;
    	} else if (type == APP_PRESENTATION) {
    		return APP_PRESENTATION_STRING;
    	}
    	return null;
    }
    
    /**
     * Converts a string to a possible application type.
     *
     * @param typeString
     *  A string containing a possible application type.
     *
     * @return
     *  The application type if the string contains the pre-defined text
     *  for the possible application types. Returns APP_NONE if the
     *  string maps to no known application type.
     */
    static public ApplicationType stringToEnum(String typeString) {
    	ApplicationType type = APP_NONE;

    	if ((null != typeString) && (typeString.length() > 0)) {
    		if (APP_TEXT_STRING.equalsIgnoreCase(typeString)) {
    			return APP_TEXT;
    		} else if (APP_SPREADSHEET_STRING.equalsIgnoreCase(typeString)) {
    			return APP_SPREADSHEET;
    		} else if (APP_PRESENTATION_STRING.equalsIgnoreCase(typeString)) {
    			return APP_PRESENTATION;
    		}
    	}

    	return type;
    }

    /**
     * Maps document type to application type.
     *
     * @param docType
     *  The document type which should be mapped to the application type.
     *
     * @return
     *  The application type mapped from the document type or APP_NONE if
     *  no mapping is available.
     */
    static public ApplicationType documentTypeToApplicationType(DocumentType docType) {
    	if (docType == DocumentType.TEXT)
    		return ApplicationType.APP_TEXT;
    	else if (docType == DocumentType.SPREADSHEET)
    		return ApplicationType.APP_SPREADSHEET;
    	else if (docType == DocumentType.PRESENTATION)
    		return ApplicationType.APP_PRESENTATION;
    	else
    		return ApplicationType.APP_NONE;
    }
}
