/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPoint2D;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.STShapeType;
import org.docx4j.dml.chart.CTChartSpace;
import org.docx4j.dml.chart.CTStyle;
import org.docx4j.dml.chart.chartStyle.CTChartColor;
import org.docx4j.dml.spreadsheetdrawing.CTAbsoluteAnchor;
import org.docx4j.dml.spreadsheetdrawing.CTGraphicalObjectFrame;
import org.docx4j.dml.spreadsheetdrawing.CTGraphicalObjectFrameNonVisual;
import org.docx4j.dml.spreadsheetdrawing.CTMarker;
import org.docx4j.dml.spreadsheetdrawing.CTOneCellAnchor;
import org.docx4j.dml.spreadsheetdrawing.CTPicture;
import org.docx4j.dml.spreadsheetdrawing.CTPictureNonVisual;
import org.docx4j.dml.spreadsheetdrawing.CTShape;
import org.docx4j.dml.spreadsheetdrawing.CTShapeNonVisual;
import org.docx4j.dml.spreadsheetdrawing.CTTwoCellAnchor;
import org.docx4j.dml.spreadsheetdrawing.ICellAnchor;
import org.docx4j.dml.spreadsheetdrawing.STEditAs;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.DrawingML.Chart;
import org.docx4j.openpackaging.parts.DrawingML.ChartColor;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONObject;

import com.openexchange.log.LogFactory;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.ooxml.drawingml.DMLChartSpace;
import com.openexchange.office.ooxml.drawingml.DMLGraphic;
import com.openexchange.office.ooxml.drawingml.DMLHelper;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.OperationDocument;

public class Drawings {

    protected static Log log = LogFactory.getLog(Drawings.class);

    public static void createDrawingOperations(JSONArray operationsArray, org.docx4j.openpackaging.parts.DrawingML.Drawing drawingsPart, int sheetIndex)
        throws Exception {

        final org.docx4j.dml.spreadsheetdrawing.CTDrawing drawings = drawingsPart.getJaxbElement();
        final List<ICellAnchor> anchors = drawings.getEGAnchor();
        for(int i=0; i<anchors.size();i++) {
            JSONObject attrs = new JSONObject();
            JSONObject drawingProperties = new JSONObject();
            ICellAnchor anchor = anchors.get(i);

            if(anchor instanceof CTAbsoluteAnchor) {
                drawingProperties.put("anchorType", "absolute");
                CTPoint2D position = ((CTAbsoluteAnchor) anchor).getPos(false);
                CTPositiveSize2D size = ((CTAbsoluteAnchor) anchor).getExt(false);
                if(position!=null) {
                    drawingProperties.put("left", Commons.coordinateTo100TH_MM(position.getX()));
                    drawingProperties.put("top", Commons.coordinateTo100TH_MM(position.getY()));
                }
                if(size!=null) {
                    drawingProperties.put("width", Commons.coordinateTo100TH_MM(size.getCx()));
                    drawingProperties.put("height", Commons.coordinateTo100TH_MM(size.getCy()));
                }
            }
            else if(anchor instanceof CTTwoCellAnchor) {
                CTMarker from = ((CTTwoCellAnchor) anchor).getFrom(false);
                CTMarker to = ((CTTwoCellAnchor) anchor).getTo(false);
                if(from!=null) {
                    drawingProperties.put("startCol", from.getCol());
                    drawingProperties.put("startColOffset", Commons.coordinateTo100TH_MM(from.getColOff()));
                    drawingProperties.put("startRow", from.getRow());
                    drawingProperties.put("startRowOffset", Commons.coordinateTo100TH_MM(from.getRowOff()));
                }
                if(to!=null) {
                    drawingProperties.put("endCol", to.getCol());
                    drawingProperties.put("endColOffset", Commons.coordinateTo100TH_MM(to.getColOff()));
                    drawingProperties.put("endRow", to.getRow());
                    drawingProperties.put("endRowOffset", Commons.coordinateTo100TH_MM(to.getRowOff()));
                }
                if(((CTTwoCellAnchor) anchor).getEditAs()==STEditAs.ABSOLUTE) {
                    drawingProperties.put("anchorType", "twoCellAsAbsolute");
                }
                else if (((CTTwoCellAnchor) anchor).getEditAs()==STEditAs.ONE_CELL) {
                    drawingProperties.put("anchorType", "twoCellAsOneCell");
                }
                else {
                    drawingProperties.put("anchorType", "twoCell");
                }
            }
            else if(anchor instanceof CTOneCellAnchor) {
                drawingProperties.put("anchorType", "oneCell");
                CTMarker from = ((CTOneCellAnchor) anchor).getFrom(false);
                CTPositiveSize2D size = ((CTOneCellAnchor) anchor).getExt(false);
                if(from!=null) {
                    drawingProperties.put("startCol", from.getCol());
                    drawingProperties.put("startColOffset", Commons.coordinateTo100TH_MM(from.getColOff()));
                    drawingProperties.put("startRow", from.getRow());
                    drawingProperties.put("startRowOffset", Commons.coordinateTo100TH_MM(from.getRowOff()));
                }
                if(size!=null) {
                    drawingProperties.put("width", Commons.coordinateTo100TH_MM(size.getCx()));
                    drawingProperties.put("height", Commons.coordinateTo100TH_MM(size.getCy()));
                }
            }
            else {
                throw new FilterException("XLSX import: unknown anchor type for drawing.", ErrorCode.CRITICAL_ERROR);
            }

            final HashMap<String, Object> drawingPropertyMap = DMLGraphic.getDrawingType(drawingsPart, anchor);
            final String drawingType = drawingPropertyMap.containsKey("type") ? (String)drawingPropertyMap.get("type") : "undefined";
            if(drawingType.equals("image")) {
                DrawingPic.createProperties(drawingProperties, drawingsPart, anchor.getPic());
            }
            final JSONObject insertDrawingOperation = new JSONObject();
            insertDrawingOperation.put("name", "insertDrawing");
            final JSONArray jsonStart = new JSONArray();
            jsonStart.put(sheetIndex);
            jsonStart.put(i);
            insertDrawingOperation.put("start", jsonStart);
            insertDrawingOperation.put("type", drawingType);
            if(drawingPropertyMap.containsKey("chartObject")) {
                if(drawingPropertyMap.containsKey("chartType")) {


                    final JSONObject chartProperties = new JSONObject();
                    chartProperties.put("type", drawingPropertyMap.get("chartType"));

                    final CTChartSpace cs = (CTChartSpace)drawingPropertyMap.get("chartObject");
                    final CTStyle chartStyle = cs.getStyle();
                    if(chartStyle!=null){
                        chartProperties.put("chartStyleId", chartStyle.getVal());
                    } else{
                        //2 is the default style of excel
                        chartProperties.put("chartStyleId", 2);
                    }

                    final String chartGroup = (String) drawingPropertyMap.get("chartGroup");
                    if(chartGroup != null){
                        chartProperties.putOpt("stacking", chartGroup);
                    }

                    final Chart chartPart = (Chart)drawingPropertyMap.get("chartPart");
                    final RelationshipsPart chartRelPart = chartPart.getRelationshipsPart(false);
                    if (chartRelPart != null) {
                        // final Relationship chartStyleRel =
                        // chartRelPart.getRelationshipByType("http://schemas.microsoft.com/office/2011/relationships/chartStyle");
                        // final Part chartStylePart = chartRelPart.getPart(chartStyleRel);
                        final Relationship chartColorRel = chartRelPart.getRelationshipByType("http://schemas.microsoft.com/office/2011/relationships/chartColorStyle");
                        if(chartColorRel!=null) {
                            try {
                                final ChartColor chartColorPart = (ChartColor) chartRelPart.getPart(chartColorRel);

                                final CTChartColor el = chartColorPart.getJaxbElement();
                                final JSONObject jsonChartColors = DMLHelper.createJsonFromChartColor(el);
                                if(jsonChartColors!=null) {
                                    chartProperties.put("chartColors", jsonChartColors);
                                }
                            } catch (Exception e) {
                                log.error("error while handling chart colors", e);
                            }
                        }
                    }
                    attrs.put("chart", chartProperties);
                    DMLHelper.createJsonFromChartShapeProperties(attrs, cs.getSpPr());
                }
            }
            if(drawingProperties.length()>0) {
                attrs.put("drawing", drawingProperties);
                insertDrawingOperation.put("attrs", attrs);
            } else if(attrs.length()>0){
                insertDrawingOperation.put("attrs", attrs);
            }
            operationsArray.put(insertDrawingOperation);

            if(drawingPropertyMap.containsKey("chartObject")) {
                DMLChartSpace.createChartSpaceOperations(operationsArray, jsonStart, (CTChartSpace)drawingPropertyMap.get("chartObject"));
                DMLChartSpace.createShapeOperations(attrs, jsonStart, (CTChartSpace)drawingPropertyMap.get("chartObject"));
            }
            
            
        }
        return;
    }

    public static ICellAnchor applyAnchorAttributes(int drawingIndex, org.docx4j.openpackaging.parts.DrawingML.Drawing drawingsPart, JSONObject drawingProperties)
        throws Exception {

        final org.docx4j.dml.spreadsheetdrawing.CTDrawing drawings = drawingsPart.getJaxbElement();
        final List<ICellAnchor> anchors = drawings.getEGAnchor();
        if (drawingIndex>=anchors.size()) {
            throw new RuntimeException("xlsx::applyDrawingOperation: drawing not available");
        }
        if(drawingProperties==null) {
            return anchors.get(drawingIndex);
        }
        final ICellAnchor cellAnchor = setDrawingAnchor(drawingProperties, anchors.get(drawingIndex));
        anchors.set(drawingIndex, cellAnchor);
        return cellAnchor;
    }

    public static void applyDrawingAttributes(ICellAnchor anchor, JSONObject drawingProperties) {

        CTNonVisualDrawingProps nonVisualDrawingProps = null;
        if(anchor.getPic()!=null) {
            final CTPicture pic = anchor.getPic();
            CTPictureNonVisual nonVisualPicProps = pic.getNvPicPr();
            if(nonVisualPicProps==null) {
                nonVisualPicProps = Context.getDmlSpreadsheetDrawingObjectFactory().createCTPictureNonVisual();
                pic.setNvPicPr(nonVisualPicProps);
            }
            nonVisualDrawingProps = nonVisualPicProps.getCNvPr();
            if(nonVisualDrawingProps==null) {
                nonVisualDrawingProps = Context.getDmlObjectFactory().createCTNonVisualDrawingProps();
                nonVisualPicProps.setCNvPr(nonVisualDrawingProps);
            }
        }
        else if(anchor.getGraphicFrame()!=null) {
            final CTGraphicalObjectFrame graphicFrame = anchor.getGraphicFrame();
            CTGraphicalObjectFrameNonVisual nonVisualGraphicProps = graphicFrame.getNvGraphicFramePr();
            if(nonVisualGraphicProps==null) {
                nonVisualGraphicProps = Context.getDmlSpreadsheetDrawingObjectFactory().createCTGraphicalObjectFrameNonVisual();
                graphicFrame.setNvGraphicFramePr(nonVisualGraphicProps);
            }
            nonVisualDrawingProps = nonVisualGraphicProps.getCNvPr();
            if(nonVisualDrawingProps==null) {
                nonVisualDrawingProps = Context.getDmlObjectFactory().createCTNonVisualDrawingProps();
                nonVisualGraphicProps.setCNvPr(nonVisualDrawingProps);
            }
        }
        
        DMLHelper.applyNonVisualDrawingProperties(drawingProperties, nonVisualDrawingProps);
    }

    public static void insertDrawing(OperationDocument operationDocument, int drawingIndex, org.docx4j.openpackaging.parts.DrawingML.Drawing drawingPart, String drawingType, JSONObject attrs)
        throws Exception {

        ICellAnchor anchor = Context.getDmlSpreadsheetDrawingObjectFactory().createCTTwoCellAnchor();
        anchor.setClientData(Context.getDmlSpreadsheetDrawingObjectFactory().createCTAnchorClientData());
        drawingPart.getJaxbElement().getEGAnchor().add(drawingIndex, anchor);
        final JSONObject drawingProperties = attrs!=null ? attrs.optJSONObject("drawing") : null;

        anchor = applyAnchorAttributes(drawingIndex, drawingPart, drawingProperties);

        // we have to create a valid drawing otherwise excel wont open... at least we are creating an default empty rect shape
        if(drawingType.equals("image")) {
            String imageUrl = null;
            if(drawingProperties!=null) {
                imageUrl = Commons.getImageUrlFromImageData(operationDocument.getResourceManager(), drawingProperties.optString("imageData"), "xl/media/");
                if(imageUrl==null) {
                    imageUrl = drawingProperties.optString("imageUrl", null);
                }
            }
            if(imageUrl!=null) {
                anchor.setPic(createImage(operationDocument, drawingPart, imageUrl));
            }
            else {
                anchor.setSp(createDummyShape());
            }
        }
        else if(drawingType.equals("chart")) {
            anchor.setGraphicFrame(createChart(operationDocument, drawingPart, attrs!=null ? attrs.optJSONObject("chart"): null));
        }
        else {
            anchor.setSp(createDummyShape());
        }
        applyDrawingAttributes(anchor, drawingProperties);
    }

    private static CTGraphicalObjectFrame createChart(OperationDocument operationDocument, Part drawingPart, JSONObject chartAttrs)
        throws Exception {

        final CTGraphicalObjectFrame graphicalObjectFrame = Context.getDmlSpreadsheetDrawingObjectFactory().createCTGraphicalObjectFrame();
        graphicalObjectFrame.setNvGraphicFramePr(createGraphicalObjectFrameNonVisual("chart"));
        graphicalObjectFrame.setXfrm(DMLHelper.createTransform2D(0, 0, 500, 500));
        graphicalObjectFrame.setGraphic(DMLGraphic.createDrawingMLChart(operationDocument, drawingPart, "/xl/charts/", chartAttrs));
        return graphicalObjectFrame;
    }

    private static CTPicture createImage(OperationDocument operationDocument, Part drawingPart, String imageUrl)
        throws Exception {

        final CTPicture picture = Context.getDmlSpreadsheetDrawingObjectFactory().createCTPicture();

        // general shape properties
        final CTShapeProperties shapeProperties = Context.getDmlObjectFactory().createCTShapeProperties();
        shapeProperties.setXfrm(DMLHelper.createTransform2D(0, 0, 500, 500));
        shapeProperties.setPrstGeom(DMLHelper.createPresetGeometry2D(STShapeType.RECT));
        picture.setSpPr(shapeProperties);
        picture.setNvPicPr(createPictureNonVisual("image"));

        // picture
        picture.setBlipFill(DMLHelper.createBlipFillProperties(operationDocument, drawingPart, imageUrl));
        return picture;
    }

    private static CTShape createDummyShape() {

        final CTShape shape = Context.getDmlSpreadsheetDrawingObjectFactory().createCTShape();

        // general shape Properties
        final CTShapeProperties shapeProperties = Context.getDmlObjectFactory().createCTShapeProperties();
        shapeProperties.setXfrm(DMLHelper.createTransform2D(0, 0, 500, 500));
        shape.setSpPr(shapeProperties);
        shape.setNvSpPr(createShapeNonVisual("shape"));

        // shape properties (preset)
        shapeProperties.setPrstGeom(DMLHelper.createPresetGeometry2D(STShapeType.RECT));
        return shape;
    }

    private static CTGraphicalObjectFrameNonVisual createGraphicalObjectFrameNonVisual(String name) {
        final CTGraphicalObjectFrameNonVisual graphicalObjectFrameNonVisual = Context.getDmlSpreadsheetDrawingObjectFactory().createCTGraphicalObjectFrameNonVisual();
        graphicalObjectFrameNonVisual.setCNvGraphicFramePr(Context.getDmlObjectFactory().createCTNonVisualGraphicFrameProperties());
        graphicalObjectFrameNonVisual.setCNvPr(DMLHelper.createNonVisualDrawingProps(name));
        return graphicalObjectFrameNonVisual;
    }

    private static CTPictureNonVisual createPictureNonVisual(String name) {
        final CTPictureNonVisual pictureNonVisual = Context.getDmlSpreadsheetDrawingObjectFactory().createCTPictureNonVisual();
        pictureNonVisual.setCNvPicPr(Context.getDmlObjectFactory().createCTNonVisualPictureProperties());
        pictureNonVisual.setCNvPr(DMLHelper.createNonVisualDrawingProps(name));
        return pictureNonVisual;
    }

    private static CTShapeNonVisual createShapeNonVisual(String name) {
        final CTShapeNonVisual shapeNonVisual = Context.getDmlSpreadsheetDrawingObjectFactory().createCTShapeNonVisual();
        shapeNonVisual.setCNvSpPr(Context.getDmlObjectFactory().createCTNonVisualDrawingShapeProps());
        shapeNonVisual.setCNvPr(DMLHelper.createNonVisualDrawingProps(name));
        return shapeNonVisual;
    }

    private static CTMarker getFrom(ICellAnchor cellAnchor) {
        if(cellAnchor instanceof CTTwoCellAnchor) {
            return ((CTTwoCellAnchor)cellAnchor).getFrom(true);
        }
        else if(cellAnchor instanceof CTOneCellAnchor) {
            return ((CTOneCellAnchor)cellAnchor).getFrom(true);
        }
        log.warn("xlsx export:getFrom... wrong anchor type used");
        return new CTMarker();
    }

    private static CTMarker getTo(ICellAnchor cellAnchor) {
        if(cellAnchor instanceof CTTwoCellAnchor) {
            return ((CTTwoCellAnchor)cellAnchor).getTo(true);
        }
        log.warn("xlsx export:getTo... wrong anchor type used");
        return new CTMarker();
    }

    private static CTPositiveSize2D getExt(ICellAnchor cellAnchor) {
        if(cellAnchor instanceof CTOneCellAnchor) {
            return ((CTOneCellAnchor)cellAnchor).getExt(true);
        }
        else if(cellAnchor instanceof CTAbsoluteAnchor) {
            return ((CTAbsoluteAnchor)cellAnchor).getExt(true);
        }
        log.warn("xlsx export:getExt... wrong anchor type used");
        return new CTPositiveSize2D();
    }

    private static CTPoint2D getPos(ICellAnchor cellAnchor) {
        if(cellAnchor instanceof CTAbsoluteAnchor) {
            return ((CTAbsoluteAnchor)cellAnchor).getPos(true);
        }
        log.warn("xlsx export:getPos... wrong anchor type used");
        return new CTPoint2D();
    }

    private static ICellAnchor setDrawingAnchor(JSONObject drawingProperties, ICellAnchor sourceAnchor)
        throws Exception {

        ICellAnchor cellAnchor = sourceAnchor;
        String anchorType = drawingProperties.optString("anchorType", null);

        // check if we have to switch to another anchorType
        if(anchorType!=null) {
            boolean isTwoCell = anchorType.equals("twoCell")||anchorType.equals("twoCellAsOneCell")||anchorType.equals("twoCellAsAbsolute");
            if(isTwoCell) {
                if(!(sourceAnchor instanceof CTTwoCellAnchor)) {
                    cellAnchor = Context.getDmlSpreadsheetDrawingObjectFactory().createCTTwoCellAnchor();
                    if(sourceAnchor instanceof CTOneCellAnchor) {
                        ((CTTwoCellAnchor)cellAnchor).setFrom(((CTOneCellAnchor)sourceAnchor).getFrom(false));
                    }
                }
                if(anchorType.equals("twoCell")) {
                    ((CTTwoCellAnchor)cellAnchor).setEditAs(org.docx4j.dml.spreadsheetdrawing.STEditAs.TWO_CELL);
                }
                else if(anchorType.equals("twoCellAsOneCell")) {
                    ((CTTwoCellAnchor)cellAnchor).setEditAs(org.docx4j.dml.spreadsheetdrawing.STEditAs.ONE_CELL);
                }
                else if(anchorType.equals("twoCellAsAbsolute")) {
                    ((CTTwoCellAnchor)cellAnchor).setEditAs(org.docx4j.dml.spreadsheetdrawing.STEditAs.ABSOLUTE);
                }
            }
            else if(anchorType.equals("oneCell")&&!(sourceAnchor instanceof CTOneCellAnchor)) {
                cellAnchor = Context.getDmlSpreadsheetDrawingObjectFactory().createCTOneCellAnchor();
                if(sourceAnchor instanceof CTTwoCellAnchor) {
                    ((CTOneCellAnchor)cellAnchor).setFrom(((CTTwoCellAnchor)sourceAnchor).getFrom(false));
                }
                else {
                    ((CTOneCellAnchor)cellAnchor).setExt(((CTAbsoluteAnchor)sourceAnchor).getExt(false));
                }
            }
            else if(anchorType.equals("absolute")&&!(sourceAnchor instanceof CTAbsoluteAnchor)) {
                cellAnchor = Context.getDmlSpreadsheetDrawingObjectFactory().createCTAbsoluteAnchor();
                if(sourceAnchor instanceof CTOneCellAnchor) {
                    ((CTAbsoluteAnchor)cellAnchor).setExt(((CTOneCellAnchor)sourceAnchor).getExt(false));
                }
            }
            if(cellAnchor!=sourceAnchor) {
                cellAnchor.setClientData(sourceAnchor.getClientData());
                cellAnchor.setCxnSp(sourceAnchor.getCxnSp());
                cellAnchor.setGraphicFrame(sourceAnchor.getGraphicFrame());
                cellAnchor.setGrpSp(sourceAnchor.getGrpSp());
                cellAnchor.setPic(sourceAnchor.getPic());
                cellAnchor.setSp(sourceAnchor.getSp());
            }
        }

        final Iterator<String> drawingKeys = drawingProperties.keys();
        while(drawingKeys.hasNext()) {
            final String key = drawingKeys.next();
            final Object value = drawingProperties.get(key);
            final Integer intValue = value instanceof Integer ? (Integer)value : null;

            if(key.equals("startCol")&&intValue!=null){
                getFrom(cellAnchor).setCol(intValue);
            }
            else if(key.equals("startColOffset")&&intValue!=null) {
                getFrom(cellAnchor).setColOff(Commons.coordinateFrom100TH_MM(intValue));
            }
            else if(key.equals("startRow")&&intValue!=null) {
                getFrom(cellAnchor).setRow(intValue);
            }
            else if(key.equals("startRowOffset")&&intValue!=null) {
                getFrom(cellAnchor).setRowOff(Commons.coordinateFrom100TH_MM(intValue));
            }
            else if(key.equals("endCol")&&intValue!=null) {
                getTo(cellAnchor).setCol(intValue);
            }
            else if(key.equals("endColOffset")&&intValue!=null) {
                getTo(cellAnchor).setColOff(Commons.coordinateFrom100TH_MM(intValue));
            }
            else if(key.equals("endRow")&&intValue!=null) {
                getTo(cellAnchor).setRow(intValue);
            }
            else if(key.equals("endRowOffset")&&intValue!=null)  {
                getTo(cellAnchor).setRowOff(Commons.coordinateFrom100TH_MM(intValue));
            }
            else if(key.equals("left")&&intValue!=null) {
                getPos(cellAnchor).setX(Commons.coordinateFrom100TH_MM(intValue));
            }
            else if(key.equals("top")&&intValue!=null) {
                getPos(cellAnchor).setY(Commons.coordinateFrom100TH_MM(intValue));
            }
            else if(key.equals("width")&&intValue!=null) {
                getExt(cellAnchor).setCx(Commons.coordinateFrom100TH_MM(intValue));
            }
            else if(key.equals("height")&&intValue!=null) {
                getExt(cellAnchor).setCy(Commons.coordinateFrom100TH_MM(intValue));
            }
        }
        return cellAnchor;
    }
}
