/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.docx4j.dml.Theme;
import org.docx4j.openpackaging.exceptions.Docx4JException;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.ThemePart;
import org.docx4j.openpackaging.parts.SpreadsheetML.SharedStrings;
import org.docx4j.openpackaging.parts.SpreadsheetML.Styles;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.CTCalcPr;
import org.xlsx4j.sml.CTDataValidations;
import org.xlsx4j.sml.CTDataValidations_legacy;
import org.xlsx4j.sml.CTExtension;
import org.xlsx4j.sml.CTExtensionList;
import org.xlsx4j.sml.CTFontSize;
import org.xlsx4j.sml.CTFonts;
import org.xlsx4j.sml.CTRElt;
import org.xlsx4j.sml.CTRst;
import org.xlsx4j.sml.CTSheetFormatPr;
import org.xlsx4j.sml.CTSst;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTXstringWhitespace;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.IDataValidations;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;
import com.openexchange.log.LogFactory;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.PreviewData;
import com.openexchange.office.ooxml.xlsx.operations.ApplyOperationHelper;
import com.openexchange.office.ooxml.xlsx.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;
import com.openexchange.office.ooxml.xlsx.tools.ValidationUtils;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

public class OperationDocument extends com.openexchange.office.ooxml.OperationDocument {

    final private SpreadsheetMLPackage opcPackage;
    protected CTStylesheet stylesheet = null;
    protected CTSst sharedStrings = null;
    private ApplyOperationHelper applyOperationHelper;
    private Double defaultFontSize = null;

    protected static Log log = LogFactory.getLog(OperationDocument.class);

    public OperationDocument(Session session, ServiceLookup _services, InputStream _inputDocumentStream, ResourceManager _resourceManager, DocumentProperties _documentProperties)
        throws FilterException {

        super(session, _services, null, _resourceManager, _documentProperties);
        try {

        	inputDocumentStream = checkDocumentComplexity(_inputDocumentStream, _documentProperties.optLong(DocumentProperties.PROP_MAX_MEMORY_USAGE, -1));

        	org.docx4j.jaxb.Context.setFreeHeapSizeLimit(_documentProperties.optLong(DocumentProperties.PROP_MIN_FREE_HEAPSPACE, 75000000));

            opcPackage = (SpreadsheetMLPackage)SpreadsheetMLPackage.load(inputDocumentStream);

            // styles
            final Styles styles = getPackage().getWorkbookPart().getStyles();
            if(styles!=null)
                stylesheet = styles.getJaxbElement();

            // shared strings
            final SharedStrings _sharedStrings = getPackage().getWorkbookPart().getSharedStrings();
            if(_sharedStrings!=null)
                sharedStrings = _sharedStrings.getJaxbElement();

            final String initialSheetName = getDocumentProperties().optString(DocumentProperties.PROP_INITIAL_SHEETNAME, "Sheet1");
            final Workbook workbook = getPackage().getWorkbookPart().getJaxbElement();
            final Sheet sheet = workbook.getSheets().getSheet().get(0);
            if(sheet.getName().equals("empty_sheet")) {
                sheet.setName(initialSheetName);
            }
        }
        catch(final Docx4JException e) {
            ErrorCode errorCode = ErrorCode.CRITICAL_ERROR;
            if(e.getCause() instanceof java.security.InvalidKeyException) {
                errorCode = ErrorCode.WRONG_PASSWORD;
            }
            else if(e.getCause() instanceof org.apache.poi.EncryptedDocumentException) {
                errorCode = ErrorCode.UNSUPPORTED_ENCRYPTION_USED;
            }
            throw new FilterException(e, errorCode);
        }
    }

    // the constructor without inputStream creates an empty document
    public OperationDocument(ServiceLookup _services, DocumentProperties _documentProperties)
        throws FilterException {

        super(null, _services, null, null, _documentProperties);
        try {
            opcPackage = SpreadsheetMLPackage.createPackage();

            final ThemePart themePart = getDefaultThemePart(new PartName("/xl/theme/theme1.xml"));
            getPackage().getWorkbookPart().addTargetPart(themePart);
            getPackage().createWorksheetPart(new PartName("/xl/worksheets/sheet1.xml"), getDocumentProperties().optString(DocumentProperties.PROP_INITIAL_SHEETNAME, "Sheet1"), 1);

            // applying the default column width...
            final Sheet sheet = SheetUtils.getSheet(opcPackage, 0);
            final RelationshipsPart relationshipPart = opcPackage.getWorkbookPart().getRelationshipsPart();
            final WorksheetPart worksheetPart = (WorksheetPart)relationshipPart.getPart(sheet.getId());
            final Worksheet worksheet = worksheetPart.getJaxbElement();
            CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
            if(sheetFormatPr==null) {
                sheetFormatPr = Context.getsmlObjectFactory().createCTSheetFormatPr();
                worksheet.setSheetFormatPr(sheetFormatPr);
                sheetFormatPr.setParent(worksheet);
                sheetFormatPr.setBaseColWidth(new Long(10L));
                sheetFormatPr.setDefaultRowHeight(14.5);
            }

            getStylesheet(true);
        }
        catch (final Exception e) {
            throw new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    public CTStylesheet getStylesheet(boolean createIfMissing) {

        if(stylesheet==null&&createIfMissing) {
            try {
                final Styles styles = new Styles(new PartName("/xl/styles.xml"));
                stylesheet = createStylesheet();
                styles.setJaxbElement(stylesheet);
                getPackage().getWorkbookPart().addTargetPart(styles);
            }
            catch(final Exception e) {
                stylesheet = null;
            }
        }
        return stylesheet;
    }

    public CTSst getSharedStrings(boolean createIfMissing)
        throws Exception {

        if(sharedStrings==null&&createIfMissing) {
            final SharedStrings _sharedStrings = new SharedStrings(new PartName("/xl/sharedStrings.xml"));
            sharedStrings = Context.getsmlObjectFactory().createCTSst();
            _sharedStrings.setJaxbElement(sharedStrings);
            getPackage().getWorkbookPart().addTargetPart(_sharedStrings);
        }
        return sharedStrings;
    }

    // returns null if something goes wrong ...
    public String getSharedString(String index) {

        if(sharedStrings!=null) {
            final List<CTRst> si = sharedStrings.getSi();
            try  {
                final int i = Integer.parseInt(index);
                if(i>=0&&i<si.size()) {
                    final CTRst ctRst = si.get(i);
                    final CTXstringWhitespace ctxString = ctRst.getT();
                    if(ctxString!=null) {
                        return ctxString.getValue();
                    }
                    final List<CTRElt> ctREltList = ctRst.getR();
                    if(ctREltList!=null) {
                        final StringBuffer stringBuffer = new StringBuffer();
                        for(final CTRElt ctRElt:ctREltList) {
                            final String t = ctRElt.getT().getValue();
                            if(t!=null) {
                                stringBuffer.append(t);
                            }
                        }
                        return stringBuffer.toString();
                    }
                }
            }
            catch(final NumberFormatException e) {
                // TODO: what to do now ?
            }
        }
        return null;
    }

    public String applySharedString(String v)
    	throws InvalidFormatException {

    	if(sharedStrings==null) {
    		final SharedStrings sharedStringsPart = new SharedStrings();
    		sharedStrings = Context.getsmlObjectFactory().createCTSst();
    		sharedStringsPart.setJaxbElement(sharedStrings);
            getPackage().getWorkbookPart().addTargetPart(sharedStringsPart);
    	}
    	int i;
        final List<CTRst> si = sharedStrings.getSi();
        for(i=0;i<si.size();i++) {
        	final CTRst crst = si.get(i);
        	if(crst.getT()!=null&&crst.getT().getValue()!=null&&crst.getT().getValue().equals(v)) {
        		return String.valueOf(i);
        	}
        }
        final CTRst rst = Context.getsmlObjectFactory().createCTRst();
        final CTXstringWhitespace n = Context.getsmlObjectFactory().createCTXstringWhitespace();
        n.setValue(v);
        if(v.charAt(0)==' '||v.charAt(v.length()-1)==' ') {
            n.setSpace("preserve");
        }
        rst.setT(n);
        si.add(rst);
        return String.valueOf(si.size()-1);
    }

    public List<IDataValidations> getDataValidations(Worksheet worksheet) {
        final List<IDataValidations> dataValidations = new ArrayList<IDataValidations>();

        final CTDataValidations validations = worksheet.getDataValidations();
        if(validations!=null) {
            dataValidations.add(validations);
        }
        final CTExtensionList extLst = worksheet.getExtLst();
        if(extLst!=null) {
            for(final CTExtension extension:extLst.getExt()) {
                final Object any = extension.getAny();
                if(any instanceof JAXBElement<?>) {
                    if(extension.getUri().equals("{CCE6A557-97BC-4b89-ADB6-D9C93CAAB3DF}")) {
                        final Object value = ((JAXBElement<?>)any).getValue();
                        if(value instanceof CTDataValidations_legacy) {
                            dataValidations.add((CTDataValidations_legacy)value);
                        }
                    }
                }
            }
        }
        return dataValidations;
    }

    //
    public String escapeString(String s) {
        if(s==null||s.length()==0)
            return s;
        final char char0 = s.charAt(0);
        return ((char0=='=')||(char0=='#')) ? "'" + s : s;
    }

    public ApplyOperationHelper getApplyOperationHelper() {
        return applyOperationHelper;
    }

    @Override
    public void applyOperations(String applyOperations) throws FilterException {
    	if (applyOperations != null){
            int i = 0;
            JSONObject op = null;
            try {
                applyOperationHelper = new ApplyOperationHelper(this);
                final JSONArray aOperations = new JSONArray(new JSONTokener(applyOperations));
                for (i = 0; i < aOperations.length(); i++) {
                    op = (JSONObject) aOperations.get(i);
                    final String opName = op.getString("name");
                    if (opName.equals("setCellContents"))
                        applyOperationHelper.setCellContents(op.getInt("sheet"), op.getJSONArray("start"), op.getJSONArray("contents"));
                    else if(opName.equals("fillCellRange"))
                        applyOperationHelper.fillCellRange(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.opt("value"), op.opt("result"), op.optJSONObject("attrs"), op.optInt("shared", -1), op.optJSONArray("ref"));
                    else if(opName.equals("clearCellRange"))
                        applyOperationHelper.clearCellRange(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), true, true);
                    else if (opName.equals("setRowAttributes"))
                        applyOperationHelper.setRowAttributes(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.getJSONObject("attrs"));
                    else if (opName.equals("setColumnAttributes"))
                        applyOperationHelper.setColumnAttributes(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1), op.getJSONObject("attrs"));
                    else if (opName.equals("insertRows"))
                        applyOperationHelper.insertRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    else if (opName.equals("deleteRows"))
                        applyOperationHelper.deleteRows(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    else if (opName.equals("insertColumns"))
                        applyOperationHelper.insertColumns(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    else if (opName.equals("deleteColumns"))
                        applyOperationHelper.deleteColumns(op.getInt("sheet"), op.getInt("start"), op.optInt("end", -1));
                    else if (opName.equals("insertSheet"))
                        applyOperationHelper.insertSheet(op.getInt("sheet"), op.getString("sheetName"), op.optJSONObject("attrs"));
                    else if (opName.equals("moveSheet"))
                        applyOperationHelper.moveSheet(op.getInt("sheet"), op.getInt("to"));
                    else if (opName.equals("deleteSheet"))
                        applyOperationHelper.deleteSheet(op.getInt("sheet"));
                    else if (opName.equals("copySheet"))
                    	applyOperationHelper.copySheet(op.getInt("sheet"), op.getInt("to"), op.getString("sheetName"));
                    else if (opName.equals("setSheetName"))
                        applyOperationHelper.setSheetName(op.getInt("sheet"), op.getString("sheetName"));
                    else if (opName.equals("setSheetAttributes"))
                        applyOperationHelper.setSheetAttributes(op.getInt("sheet"), op.getJSONObject("attrs"));
                    else if (opName.equals("mergeCells"))
                        applyOperationHelper.mergeCells(op.getInt("sheet"), op.getJSONArray("start"), op.optJSONArray("end"), op.optBoolean("keepContent", false), op.optString("type", "merge"));
                    else if (opName.equals("insertDrawing"))
                        applyOperationHelper.insertDrawing(op.getJSONArray("start"), op.getString("type"), op.optJSONObject("attrs"));
                    else if (opName.equals("deleteDrawing"))
                        applyOperationHelper.deleteDrawing(op.getJSONArray("start"));
                    else if (opName.equals("setDrawingAttributes"))
                        applyOperationHelper.applyDrawingAttributes(op.getJSONArray("start"), op.getJSONObject("attrs"));
                    else if (opName.equals("insertStyleSheet"))
                        applyOperationHelper.insertStyleSheet(op.getString("type"), op.getString("styleId"), op.getString("styleName"), op.getJSONObject("attrs"), op.optString("parent", null), op.optBoolean("hidden", false), op.optInt("uiPriority", 0), op.optBoolean("default", false), op.optBoolean("auto", false));
                    else if (opName.equals("insertChartDataSeries"))
                        applyOperationHelper.insertChartDataSeries(op.getJSONArray("start"), op.getInt("series"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertChartDataSeries"))
                        applyOperationHelper.insertChartDataSeries(op.getJSONArray("start"), op.getInt("series"), op.optJSONObject("attrs"));
                    else if (opName.equals("setChartDataSeriesAttributes"))
                        applyOperationHelper.setChartDataSeriesAttributes(op.getJSONArray("start"), op.getInt("series"), op.optJSONObject("attrs"));
                    else if (opName.equals("deleteChartDataSeries"))
                        applyOperationHelper.deleteChartDataSeries(op.getJSONArray("start"), op.getInt("series"));
                    else if (opName.equals("insertName") || opName.equals("changeName"))
                        applyOperationHelper.insertName(op.optLong("sheet", -1), op.getString("exprName"), op.getString("formula"));
                    else if (opName.equals("deleteName"))
                        applyOperationHelper.deleteName(op.optLong("sheet"), op.getString("exprName"));
                    else if (opName.equals("setChartAxisAttributes"))
                        applyOperationHelper.setChartAxisAttributes(op.getJSONArray("start"), op.getString("axis"), op.optJSONObject("attrs"));
                    else if (opName.equals("setChartGridlineAttributes"))
                        applyOperationHelper.setChartGridlineAttributes(op.getJSONArray("start"), op.getString("axis"), op.optJSONObject("attrs"));
                    else if (opName.equals("setChartTitleAttributes"))
                        applyOperationHelper.setChartTitleAttributes(op.getJSONArray("start"), op.getString("axis"), op.optJSONObject("attrs"));
                    else if (opName.equals("setChartLegendAttributes"))
                        applyOperationHelper.setChartLegendAttributes(op.getJSONArray("start"), op.optJSONObject("attrs"));
                    else if (opName.equals("insertValidation"))
                    	ValidationUtils.insertValidation(this, op);
                    else if (opName.equals("changeValidation"))
                    	ValidationUtils.changeValidation(this, op);
                    else if (opName.equals("deleteValidation"))
                    	ValidationUtils.deleteValidation(this, op.getInt("sheet"), op.getInt("index"));
                    else if (opName.equals("setDocumentAttributes"))
                    	applyOperationHelper.setDocumentAttributes(op.getJSONObject("attrs"));
                    else if (opName.equals("createError"))
                        throw new FilterException("createError operation detected: " + opName, ErrorCode.UNSUPPORTED_OPERATION_USED);
                    else
                        logMessage("warn", "Ignoring unsupported operation: " + opName);
                }
                final WorkbookPart workbookPart = getPackage().getWorkbookPart();
                final Workbook workbook = workbookPart.getJaxbElement();
                CTCalcPr calcPr = workbook.getCalcPr();
                if(calcPr==null) {
                    calcPr = Context.getsmlObjectFactory().createCTCalcPr();
                    workbook.setCalcPr(calcPr);
                }
                // our formula updates should be always up to date now, so it is no longer necessary to
                if(log.isInfoEnabled()) {
                	checkFormulaResults();
                }
                calcPr.setFullCalcOnLoad(true);

                // removing calcChain which might be invalid now
                final RelationshipsPart workbookRelationshipsPart = workbookPart.getRelationshipsPart();
                if(workbookRelationshipsPart!=null) {
                    final Relationship relationship = workbookRelationshipsPart.getRelationshipByType("http://schemas.openxmlformats.org/officeDocument/2006/relationships/calcChain");
                    if(relationship!=null) {
                        workbookRelationshipsPart.removeRelationship(relationship);
                    }
                }
            }
            catch(final Exception e) {
                String message = e.getMessage();
                if(op!=null) {
                    message += ", operation:" + Integer.toString(i) + " " + op.toString();
                }
                throw new FilterException(message, e, ErrorCode.CRITICAL_ERROR);
            }
        }
    }

    /**
     * checkFormulaResults is logging the formula cells that do not have a result
     */
    private void checkFormulaResults() {
        final WorkbookPart workbookPart = getPackage().getWorkbookPart();
        final Workbook workbook = workbookPart.getJaxbElement();
        final RelationshipsPart workbookRelationshipsPart = workbookPart.getRelationshipsPart();
        for (final Sheet sheet:workbook.getSheets().getSheet()) {
            final Part part = workbookRelationshipsPart.getPart(sheet.getId());
            if(part instanceof WorksheetPart) {
                final WorksheetPart sheetPart = (WorksheetPart)part;
                final RelationshipsPart sheetRelationshipsPart = sheetPart.getRelationshipsPart();
                if(sheetRelationshipsPart!=null) {
                    final Worksheet worksheet = sheetPart.getJaxbElement();
                    final SheetData sheetData = worksheet.getSheetData();
                    if(sheetData!=null) {
                        final Iterator<Row> rowIterator = sheetData.createRowIterator();
                        while(rowIterator.hasNext()) {
                            final Row row = rowIterator.next();
                            final Iterator<Cell> cellIterator = row.createCellIterator();
                            while(cellIterator.hasNext()) {
                                final Cell cell = cellIterator.next();
                                if(cell.getF()!=null&&cell.getV()==null) {
                                	log.info("XLSX: Cell:" + cell.getColumn() + " formula without result!");
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    @Override
    public JSONObject getOperations()
        throws Exception {

        final CreateOperationHelper createOperationHelper = new CreateOperationHelper(this, new JSONArray());
        final Object previewObj = getDocumentProperties().get(DocumentProperties.PROP_PREVIEWDATA);
        if((previewObj!=null) && (previewObj instanceof PreviewData)){
            final PreviewData previewData = (PreviewData)previewObj;

        	int activeSheetIndex = createOperationHelper.createPreviewOperations();
        	final JSONObject previewOperations = new JSONObject();
        	previewOperations.put("operations", createOperationHelper.getOperationsArray());
        	previewData.setPreviewOperations(previewOperations);
            previewData.setActiveSheetIndex(activeSheetIndex);
        	createOperationHelper.setOperationsArray(new JSONArray());
            createOperationHelper.createOutstandingOperations(activeSheetIndex);
        }
        else {
        	createOperationHelper.createOperations();
        }

/*
        Writer writer = new FileWriter(new File("/tmp/jsonout.txt"));
        for(int i=0;i<createOperationHelper.getOperationsArray().length();i++) {
            createOperationHelper.getOperationsArray().getJSONObject(i).write(writer);
            writer.append(System.getProperty("line.separator"));
        }
        writer.close();
*/
        final JSONObject aOperations = new JSONObject();
        aOperations.put("operations", createOperationHelper.getOperationsArray());
        return aOperations;
    }

    @Override
    public SpreadsheetMLPackage getPackage() {
        return opcPackage;
    }

    @Override
    public Theme getTheme() {
        try {
            return ((ThemePart)getPackage().getParts().get(new PartName("/xl/theme/theme1.xml"))).getJaxbElement();
        }
        catch (final Exception e) {
            return null;
        }
    }

    public double getDefaultFontSize() {

        if(defaultFontSize==null) {

            try {
                final CTStylesheet stylesheet = getStylesheet(false);
                if(stylesheet!=null) {
                    final CTFonts fonts = stylesheet.getFonts();
                    if(fonts!=null&&!fonts.getFont().isEmpty()) {
                        final Object fontSize = fonts.getFont().get(0).getProperty("sz");
                        if(fontSize!=null&&fontSize instanceof CTFontSize) {
                            defaultFontSize = ((CTFontSize)fontSize).getVal();
                        }
                    }
                }
            }
            catch(final Exception e) {
                // nothing ...
            }
            if(defaultFontSize==null) {
                defaultFontSize = new Double(11.0);
            }
        }
        return defaultFontSize.doubleValue();
    }

    public double getDefaultColumnWidth(Worksheet worksheet) {

        final CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
        if(sheetFormatPr==null) {
            return 8.7d;
        }
        final Double defaultColWidth = sheetFormatPr.getDefaultColWidth();
        if(defaultColWidth!=null&&defaultColWidth.doubleValue()!=0) {
            return defaultColWidth.doubleValue() + 0.7d;
        }
        return sheetFormatPr.getBaseColWidth() + 0.7d;
    }

    public double getDefaultRowHeight(Worksheet worksheet) {

        final CTSheetFormatPr sheetFormatPr = worksheet.getSheetFormatPr();
        return sheetFormatPr!=null ? sheetFormatPr.getDefaultRowHeight() : 14.5d;
    }

    private InputStream checkDocumentComplexity(InputStream _inputStream, long maxMemoryUsage)
    	throws FilterException {

    	InputStream returnValue = null;

    	int maxCellCount  = ConfigurationHelper.getIntegerOfficeConfigurationValue(getServiceLookup(), getSession(), "//spreadsheet/maxCells", 500000);
        if(maxCellCount<=0) {
        	return _inputStream;
        }

        int xmlFileSize = 0;

        byte[] bytes;

        try {
			bytes = IOUtils.toByteArray(_inputStream);
		} catch (IOException e1) {
            throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
		} finally {
	        IOUtils.closeQuietly(_inputStream);
		}

        final InputStream _zipInputStream = new ByteArrayInputStream(bytes);
        final ZipInputStream zipInputStream = new ZipInputStream(_zipInputStream);

        ZipEntry zipEntry = null;
        final byte[] buffer = new byte[0x10000];  // 64kb buffer

        try {
			while((zipEntry = zipInputStream.getNextEntry())!=null) {

				int read;

				final String zipEntryName = zipEntry.getName();
				if(zipEntryName.endsWith(".xml")) {
		        	while((read = zipInputStream.read(buffer))>=0) {
		        		xmlFileSize += read;
		        	}
		        }
			    zipInputStream.closeEntry();
			    if(xmlFileSize>maxCellCount*100) {
		            throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
			    }
			}
            if(maxMemoryUsage>0) {
                if(xmlFileSize*70>maxMemoryUsage) {
                    throw new FilterException("", ErrorCode.MEMORY_USAGE_TOO_HIGH, (long)(xmlFileSize*70));
                }
            }
			returnValue = new ByteArrayInputStream(bytes);
        } catch (IOException e) {
            throw new FilterException("", ErrorCode.COMPLEXITY_TOO_HIGH);
		} finally {
			try {
				zipInputStream.close();
			} catch (IOException e) {

			}
			IOUtils.closeQuietly(_zipInputStream);
		}
        return returnValue;
    }

    private CTStylesheet createStylesheet() throws JAXBException {

        final String defaultStylesheet = "<styleSheet xmlns=\"http://schemas.openxmlformats.org/spreadsheetml/2006/main\"" +
        		"    xmlns:mc=\"http://schemas.openxmlformats.org/markup-compatibility/2006\"" +
        		"    mc:Ignorable=\"x14ac\"" +
        		"    xmlns:x14ac=\"http://schemas.microsoft.com/office/spreadsheetml/2009/9/ac\">" +
        		"    <fonts count=\"1\" x14ac:knownFonts=\"1\">" +
        		"        <font>" +
        		"            <sz val=\"11\" />" +
        		"            <color theme=\"1\" />" +
        		"            <name val=\"Arial\" />" +
        		"            <family val=\"2\" />" +
        		"            <scheme val=\"minor\" />" +
        		"        </font>" +
        		"    </fonts>" +
        		"    <fills count=\"2\">" +
        		"        <fill>" +
        		"            <patternFill patternType=\"none\" />" +
        		"        </fill>" +
        		"        <fill>" +
        		"            <patternFill patternType=\"gray125\" />" +
        		"        </fill>" +
        		"    </fills>" +
        		"    <borders count=\"1\">" +
        		"        <border>" +
        		"            <left />" +
        		"            <right />" +
        		"            <top />" +
        		"            <bottom />" +
        		"            <diagonal />" +
        		"        </border>" +
        		"    </borders>" +
        		"    <cellStyleXfs count=\"1\">" +
        		"        <xf numFmtId=\"0\" fontId=\"0\" fillId=\"0\" borderId=\"0\" />" +
        		"    </cellStyleXfs>" +
        		"    <cellXfs count=\"2\">" +
        		"        <xf numFmtId=\"0\" fontId=\"0\" fillId=\"0\" borderId=\"0\" xfId=\"0\" />" +
        		"    </cellXfs>" +
        		"    <cellStyles count=\"1\">" +
        		"        <cellStyle name=\"Standard\" xfId=\"0\" builtinId=\"0\" />" +
        		"    </cellStyles>" +
        		"    <dxfs count=\"0\" />" +
        		"    <tableStyles count=\"0\" defaultTableStyle=\"TableStyleMedium2\"" +
        		"        defaultPivotStyle=\"PivotStyleLight16\" />" +
        		"</styleSheet>";

        final java.util.HashMap<String, String> mappings = new java.util.HashMap<String, String>();
        final Object o = org.docx4j.XmlUtils.unmarshallFromTemplate(defaultStylesheet, mappings, getPackage().getWorkbookPart().getJAXBContext());
        return o instanceof JAXBElement ? (CTStylesheet)((JAXBElement<?>)o).getValue() : (CTStylesheet)o;
    }
}
