/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.lang.annotation.Annotation;
import java.util.HashSet;
import java.util.List;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.datatype.XMLGregorianCalendar;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.apache.commons.logging.Log;
import org.docx4j.dml.CTBlip;
import org.docx4j.dml.CTBlipFillProperties;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.Graphic;
import org.docx4j.dml.GraphicData;
import org.docx4j.dml.picture.Pic;
import org.docx4j.dml.wordprocessingDrawing.Anchor;
import org.docx4j.dml.wordprocessingDrawing.Inline;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.vml.CTImageData;
import org.docx4j.vml.CTRect;
import org.docx4j.vml.CTShape;
import org.docx4j.vml.officedrawing.STTrueFalse;
import org.docx4j.wml.Br;
import org.docx4j.wml.CTObject;
import org.docx4j.wml.CTParaRPrOriginal;
import org.docx4j.wml.CTSimpleField;
import org.docx4j.wml.CTTblGridChange;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblPrChange;
import org.docx4j.wml.CTTcPrChange;
import org.docx4j.wml.CTTrPrBase;
import org.docx4j.wml.CTTrPrChange;
import org.docx4j.wml.CTTrackChange;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Drawing;
import org.docx4j.wml.IText;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase;
import org.docx4j.wml.ParaRPr;
import org.docx4j.wml.ParaRPrChange;
import org.docx4j.wml.Pict;
import org.docx4j.wml.R;
import org.docx4j.wml.RPr;
import org.docx4j.wml.TblGrid;
import org.docx4j.wml.TblGridBase;
import org.docx4j.wml.TblPr;
import org.docx4j.wml.TrPr;
import org.docx4j.wml.R.Tab;
import org.docx4j.wml.RunDel;
import org.docx4j.wml.RunIns;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.TcPrInner;
import org.docx4j.wml.Tr;
import org.json.JSONArray;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;

import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.tools.Commons;

public abstract class Component {

    protected static Log log = LogFactory.getLog(Component.class);

    private static HashSet<Class<?>> rootElements = new HashSet<Class<?>>();

    // change track flags, the  in the highword specifies the value that should be used
    final public static int CT_NONE  			=  0;
    final public static int CT_INSERTED  		=  1;	// corresponding HIWORD bit == VALUE
    final public static int CT_DELETED   		=  2;	// corresponding HIWORD bit == VALUE
    final public static int CT_MODIFIED 		=  4;	// corresponding HIWORD bit == VALUE
    
    /*
     * simply returns o, but in case o is a JAXBElement and a RootElement annotation is available
     * then the parent content list is updated and the contentModel is returned.
     * */
    public static Object getContentModel(IndexedNode<Object> node, Object parent) {

        if (node.getData() instanceof javax.xml.bind.JAXBElement) {
            boolean hasRootElement = rootElements.contains(((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType());
            if(!hasRootElement) {
                final Annotation[] annotations = ((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType().getAnnotations();
                for(Annotation annotation : annotations) {
                    if(annotation instanceof XmlRootElement) {
                        rootElements.add(((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType());
                        hasRootElement = true;
                        break;
                    }
                }
            }
            if(hasRootElement) {
                node.setData(((JAXBElement<?>)node.getData()).getValue());
            }
        }
        if(node.getData() instanceof Child) {
        	((Child)node.getData()).setParent(parent);
        }
        return node.getData();
    }

    protected IndexedNode<Object> node;
    protected IndexedNode<Object> contextNode;
    protected int componentNumber;

    public Component(IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber) {
        node = _node;
        contextNode = _contextNode;
        componentNumber = _componentNumber;
    }
    public Object getObject() {
        return node.getData();
    }
    public IndexedNode<Object> getNode() {
    	return node;
    }
    public void setNode(IndexedNode<Object> _node) {
    	node = _node;
    }
    public IndexedNode<Object> getContextNode() {
    	return contextNode;
    }
    public void setContextNode(IndexedNode<Object> _contextNode) {
    	contextNode = _contextNode;
    }
    public abstract Component getNextComponent();

    public abstract Component getChildComponent();

    public Object getContextObject() {
        return contextNode.getData();
    }
    public IndexedNode<Object> getContextChildNode() {
    	return node;
    }
    public int getComponentNumber() {
        return componentNumber;
    }
    public int getNextComponentNumber() {
        return componentNumber + 1;
    }
    public void setComponentNumber(int n) {
        componentNumber = n;
    }
    public void splitStart(int n) {
    }
    public void splitEnd(int n) {
    }
    public abstract void setAttributes(OperationDocument operationDocument, JSONObject attrs);

    public static class MainDocumentComponent extends Component {

    	public MainDocumentComponent(MainDocumentPart documentPart) {
    		super(new IndexedNode<Object>(documentPart), null, 0);
    	}
		@Override
		public Component getNextComponent() {
			return null;
		}
		@Override
		public Component getChildComponent() {
			return MainContext.getNextComponent(0, getNode(), -1);
		}
		@Override
		public void setAttributes(OperationDocument operationDocument, JSONObject attrs) {

		}
    }
    /**
     * {@link ParagraphComponent}
     *
     * @author <a href="mailto:Sven.Jacobi@open-xchange.com">Sven Jacobi</a>
     */
    public static class ParagraphComponent extends Component {

    	public ParagraphComponent(P p) {
    		super(new IndexedNode<Object>(p), null, 0);
    	}
        public ParagraphComponent(IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber) {
            super(_node, _contextNode, _componentNumber);
        }
        @Override
		public Component getNextComponent() {
        	final Object contextObject = contextNode.getData();
        	if(contextObject instanceof Tc) {
        		return TableCellContext.getNextComponent(getNextComponentNumber(), getContextNode(), getNode().getIndex());
        	}
        	else {
        		return MainContext.getNextComponent(getNextComponentNumber(), getContextNode(), getNode().getIndex());
        	}
        }
        @Override
		public Component getChildComponent() {
            return ParagraphContext.getNextComponent(0, getNode(), -1);
        }
		@Override
		public void setAttributes(OperationDocument operationDocument, JSONObject attrs) {
			if(attrs==null) {
				return;
			}
			try {
		        final P paragraph = (P)getObject();
		        PPr pPr = paragraph.getPPr();
		        if (pPr==null) {
		            pPr = Context.getWmlObjectFactory().createPPr();
		            pPr.setParent(paragraph);
		            paragraph.setPPr(pPr);
		        }
		        if(attrs.has("styleId")) {
		            final Object styleId = attrs.get("styleId");
		            if(styleId instanceof String) {
		                PPrBase.PStyle pStyle = pPr.getPStyle();
		                if(pStyle==null) {
		                    pStyle = Context.getWmlObjectFactory().createPPrBasePStyle();
		                    pStyle.setParent(pPr);
		                }
		                pStyle.setVal((String)styleId);
		                pPr.setPStyle(pStyle);
		            }
		            else {
		                pPr.setPStyle(null);
		            }
		        }
		        Paragraph.applyParagraphProperties(operationDocument, attrs.optJSONObject("paragraph"), pPr);
		        final Object characterAttrs = attrs.opt("character");
		        if(characterAttrs instanceof JSONObject) {
		            ParaRPr paraRPr = pPr.getRPr();
		            if(paraRPr==null) {
		                paraRPr = Context.getWmlObjectFactory().createParaRPr();
		                paraRPr.setParent(pPr);
		                pPr.setRPr(paraRPr);
		            }
		            com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties(operationDocument, (JSONObject)characterAttrs, paraRPr);
		        }
	            final Object changes = attrs.opt("changes");
	            if(changes!=null) {
		            if(changes instanceof JSONObject) {
	            		ParaRPr paraRPr = pPr.getRPr();
	            		if(paraRPr==null) {
	            			paraRPr = Context.getWmlObjectFactory().createParaRPr();
	            			paraRPr.setParent(pPr);
	            			pPr.setRPr(paraRPr);
	            		}
		            	final Object attrsInserted = ((JSONObject)changes).opt("inserted");
		            	if(attrsInserted!=null) {
		            		if(attrsInserted instanceof JSONObject) {
		            			CTTrackChange ins = paraRPr.getIns();
		            			if(ins==null) {
		            				ins = Context.getWmlObjectFactory().createCTTrackChange();
		            				ins.setParent(paraRPr);
		            				paraRPr.setIns(ins);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsInserted, ins);
		            		}
		            		else {
		            			paraRPr.setIns(null);
		            		}
		            	}
		            	final Object attrsRemoved = ((JSONObject)changes).opt("removed");
		            	if(attrsRemoved!=null) {
		            		if(attrsRemoved instanceof JSONObject) {
		            			CTTrackChange del = paraRPr.getDel();
		            			if(del==null) {
		            				del = Context.getWmlObjectFactory().createCTTrackChange();
		            				del.setParent(paraRPr);
		            				paraRPr.setDel(del);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsRemoved, del);
		            		}
		            		else {
		            			paraRPr.setDel(null);
		            		}
		            	}
		            	final Object attrsModified = ((JSONObject)changes).opt("modified");
		            	if(attrsModified!=null) {
		            		if(attrsModified instanceof JSONObject) {
			            		ParaRPrChange rPrChange = paraRPr.getRPrChange();
			            		if(rPrChange==null) {
			            			rPrChange = Context.getWmlObjectFactory().createParaRPrChange();
			            			rPrChange.setParent(paraRPr);
			            			paraRPr.setRPrChange(rPrChange);
			            		}
		            			final Object attrsModifiedAttrs = ((JSONObject)attrsModified).opt("attrs");
		            			if(attrsModifiedAttrs!=null) {
		            				if(attrsModifiedAttrs instanceof JSONObject) {
		            					CTParaRPrOriginal paraRPrOriginal = rPrChange.getRPr();
		            					if(paraRPrOriginal==null) {
		            						paraRPrOriginal = Context.getWmlObjectFactory().createCTParaRPrOriginal();
		            						paraRPrOriginal.setParent(rPrChange);
		            						rPrChange.setRPr(paraRPrOriginal);
		            					}
		            					// TODO: requires TextRun_base ... TextUtils.setTextRunAttributes(operationDocument, (JSONObject)attrsModifiedAttrs, null, false);
		            				}
		            				else {
		            					rPrChange.setRPr(null);
		            				}
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsModified, rPrChange);
		            		}
		            		else {
		            			paraRPr.setRPrChange(null);
		            		}
		            	}
		            }
		            else {
		            	final ParaRPr paraRPr = pPr.getRPr();
		            	if(paraRPr!=null) {
		            		paraRPr.setIns(null);
		            		paraRPr.setDel(null);
		            		paraRPr.setRPrChange(null);
		            	}
		            }
	            }
			}
	        catch (Exception e) {
				log.error("docx, ParagraphComponent::setAttributes", e);
			}
		}
    }

    public static class TableComponent extends Component {

    	public TableComponent(Tbl tbl) {
    		super(new IndexedNode<Object>(tbl), null, 0);
    	}
        public TableComponent(IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber) {
            super(_node, _contextNode, _componentNumber);
        }
        @Override
		public Component getNextComponent() {
        	final Object contextObject = contextNode.getData();
        	if(contextObject instanceof Tc) {
        		return TableCellContext.getNextComponent(getNextComponentNumber(), getContextNode(), getNode().getIndex());
        	}
        	else {
        		return MainContext.getNextComponent(getNextComponentNumber(), getContextNode(), getNode().getIndex());
        	}
        }
        @Override
		public Component getChildComponent() {
        	return TableContext.getNextComponent(0, getNode(), -1);
        }
		@Override
		public void setAttributes(OperationDocument operationDocument, JSONObject attrs) {
			if(attrs==null) {
				return;
			}
			try {
		        final Tbl tbl = (Tbl)getObject();
        		TblPr tblPr = tbl.getTblPr();
	        	if(tblPr==null) {
	        		tblPr = Context.getWmlObjectFactory().createTblPr();
	        		tblPr.setParent(tbl);
	        		tbl.setTblPr(tblPr);
	        	}
        		if(attrs.has("styleId")) {
		            Table.applyTableStyle(attrs.getString("styleId"), tblPr);
		        }
        		TblGrid tblGrid = tbl.getTblGrid();
        		if(tblGrid==null) {
        			tblGrid = Context.getWmlObjectFactory().createTblGrid();
        			tblGrid.setParent(tbl);
        			tbl.setTblGrid(tblGrid);
        		}
		        Table.applyTableProperties(operationDocument, attrs.optJSONObject("table"), tbl, tblGrid, tblPr);
	        	final Object changes = attrs.opt("changes");
	        	if(changes!=null) {
	        		boolean removeTableChangeTrack = false;
	        		if(changes instanceof JSONObject) {
	        			final Object modified = ((JSONObject)changes).opt("modified");
	        			if(modified!=null) {
	        				if(modified instanceof JSONObject) {
	        					CTTblPrChange tblPrChange = tblPr.getTblPrChange();
	        					if(tblPrChange==null) {
	        						tblPrChange = Context.getWmlObjectFactory().createCTTblPrChange();
	        						tblPrChange.setParent(tblPr);
	        						tblPr.setTblPrChange(tblPrChange);
	        					}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)modified, tblPrChange);	        					
	        					final Object attrsModified = ((JSONObject)modified).opt("attrs");
	        					if(attrsModified!=null) {
	        						if(attrsModified instanceof JSONObject) {
	        							CTTblPrBase tblPrChangePr = tblPrChange.getTblPr();
	        							if(tblPrChangePr==null) {
	        								tblPrChangePr = Context.getWmlObjectFactory().createCTTblPrBase();
	        								tblPrChangePr.setParent(tblPrChange);
	        								tblPrChange.setTblPr(tblPrChangePr);
	        							}
	        							if(((JSONObject)attrsModified).has("styleId")) {
	        								Table.applyTableStyle(((JSONObject) attrsModified).getString("styleId"), tblPrChangePr);
	        							}
	        							final JSONObject tableAttrChanges = ((JSONObject)attrsModified).optJSONObject("table");
	        							if(tableAttrChanges!=null) {
	        								TblGridBase tblGridBase = null;
	        								Object tblGridObj = tableAttrChanges.opt("tableGrid");
	        								if(tblGridObj!=null) {
	        									if(tblGridObj instanceof JSONArray) {
			        								CTTblGridChange tblGridChange = tblGrid.getTblGridChange();
			        								if(tblGridChange==null&&tableAttrChanges.has("tableGrid")) {
			        									tblGridChange=Context.getWmlObjectFactory().createCTTblGridChange();
			        									tblGridChange.setParent(tblGrid);
			        									tblGrid.setTblGridChange(tblGridChange);
			        								}
			        								tblGridBase = tblGridChange.getTblGrid();
			        								if(tblGridBase==null) {
			        									tblGridBase = Context.getWmlObjectFactory().createTblGridBase();
			        									tblGridBase.setParent(tblGridChange);
			        									tblGridChange.setTblGrid(tblGridBase);
			        								}
	        									}
	        									else {
	        										tblGrid.setTblGridChange(null);
	        									}
	        								}
	        								Table.applyTableProperties(operationDocument, tableAttrChanges, null, tblGridBase, tblPrChangePr);
	        							}
	        						}
	        						else {
	        							tblPrChange.setTblPr(null);
	        						}
	        					}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsModified, tblPrChange);
	        				}
	        				else {
	        					removeTableChangeTrack = true;
	        				}
	        			}
	        			// removed / inserted changetracking has to be applied to each row
	        			if(((JSONObject)changes).opt("inserted")!=null||((JSONObject)changes).opt("removed")!=null) {
	        				Component trComponent = getChildComponent();
	        				while(trComponent!=null) {
	        					trComponent.setAttributes(operationDocument, attrs);
	        					trComponent = trComponent.getNextComponent();
	        				}
	        			}
	        		}
	        		else {
	        			removeTableChangeTrack = true;
	        		}
	        		if(removeTableChangeTrack) {
	        			if(tblPr!=null) {
	        				tblPr.setTblPrChange(null);
	        			}
	        			if(tblGrid!=null) {
	        				tblGrid.setTblGridChange(null);
	        			}
	        		}
	        	}
			}
			catch(Exception e) {
				log.error("docx, TableComponent::setAttributes:", e);
			}
		}
    }

    public static class TrComponent extends Component {

    	public TrComponent(Tr tr) {
    		super(new IndexedNode<Object>(tr), null, 0);
    	}
        public TrComponent(IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber) {
            super(_node, _contextNode, _componentNumber);
        }
        @Override
		public Component getNextComponent() {
    		return TableContext.getNextComponent(getNextComponentNumber(), getContextNode(), getNode().getIndex());
        }
        @Override
		public Component getChildComponent() {
        	return TableRowContext.getNextComponent(0, 0, getNode(), -1);
        }
		@Override
		public void setAttributes(OperationDocument operationDocument, JSONObject attrs) {
			if(attrs==null) {
				return;
			}
			try {
		        final Tr tr = (Tr)getObject();
		        TrPr trPr = tr.getTrPr();
		        if(trPr==null) {
		            trPr = Context.getWmlObjectFactory().createTrPr();
		            trPr.setParent(tr);
		            tr.setTrPr(trPr);
		        }
		        Table.applyRowProperties(attrs.optJSONObject("row"), trPr);
		        final Object changes = attrs.opt("changes");
		        if(changes!=null) {
			        if(changes instanceof JSONObject) {
			        	final Object attrsInserted = ((JSONObject)changes).opt("inserted");
			        	if(attrsInserted!=null) {
			        		if(attrsInserted instanceof JSONObject) {
		            			CTTrackChange ins = trPr.getIns();
		            			if(ins==null) {
		            				ins = Context.getWmlObjectFactory().createCTTrackChange();
		            				ins.setParent(trPr);
		            				trPr.setIns(ins);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsInserted, ins);
			        		}
			        		else {
			        			trPr.setIns(null);
			        		}
			        	}
			        	final Object attrsRemoved = ((JSONObject)changes).opt("removed");
			        	if(attrsRemoved!=null) {
				        	if(attrsRemoved instanceof JSONObject) {
		            			CTTrackChange del = trPr.getDel();
		            			if(del==null) {
		            				del = Context.getWmlObjectFactory().createCTTrackChange();
		            				del.setParent(trPr);
		            				trPr.setDel(del);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsRemoved, del);
				        	}
				        	else {
				        		trPr.setDel(null);
				        	}
			        	}
			        	final Object attrsModified = ((JSONObject)changes).opt("modified");
			        	if(attrsModified!=null) {
			        		if(attrsModified instanceof JSONObject) {
			        			CTTrPrChange trPrChange = trPr.getTrPrChange();
			        			if(trPrChange==null) {
			        				trPrChange = Context.getWmlObjectFactory().createCTTrPrChange();
			        				trPrChange.setParent(trPr);
			        				trPr.setTrPrChange(trPrChange);
			        			}
			        			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsModified, trPrChange);
	        					final Object attrsModifiedAttrs = ((JSONObject)attrsModified).opt("attrs");
	        					if(attrsModifiedAttrs!=null) {
	        						if(attrsModifiedAttrs instanceof JSONObject) {
	        							CTTrPrBase trPrBase = trPrChange.getTrPr();
	        							if(trPrBase==null) {
	        								trPrBase = Context.getWmlObjectFactory().createCTTrPrBase();
	        								trPrBase.setParent(trPrChange);
	        								trPrChange.setTrPr(trPrBase);
	        							}
	        							Table.applyRowProperties(((JSONObject)attrsModifiedAttrs).optJSONObject("row"), trPrBase);
	        						}
	        						else {
	        							trPrChange.setTrPr(null);
	        						}
	        					}
			        		}
			        		else {
			        			trPr.setTrPrChange(null);
			        		}
			        	}
			        }
			        else {
			        	trPr.setIns(null);
			        	trPr.setDel(null);
			        	trPr.setTrPrChange(null);
			        }
		        }
			}
			catch(Exception e) {
				log.error("docx, TrComponent::setAttributes:", e);
			}
		}
    }

    public static class TcComponent extends Component {

        private final int gridPosition;

        public TcComponent(IndexedNode<Object> _node, IndexedNode<Object> _contextNode, int _componentNumber, int _gridPosition) {
            super(_node, _contextNode, _componentNumber);
            gridPosition = _gridPosition;
        }

        public int getGridPosition() {
            return gridPosition;
        }

        public int getNextGridPosition() {
            int gridSpan = 1;
            TcPr tcPr = ((Tc)node.getData()).getTcPr();
            if(tcPr!=null) {
                TcPrInner.GridSpan span = tcPr.getGridSpan();
                if (span!=null) {
                    gridSpan = span.getVal().intValue();
                    if (gridSpan==0) {
                        System.out.println("Error: empty gridSpan is zero... this results in problems with the component iterator");
                        gridSpan++; // increasing.. preventing an endless loop
                    }
                }
            }
            return gridPosition + gridSpan;
        }
		@Override
		public Component getNextComponent() {
    		return TableRowContext.getNextComponent(getNextComponentNumber(), getNextGridPosition(), getContextNode(), getNode().getIndex());
        }
		@Override
		public Component getChildComponent() {
			return TableCellContext.getNextComponent(0, getNode(), -1);
		}
		@Override
		public void setAttributes(OperationDocument operationDocument, JSONObject attrs) {
			if(attrs==null) {
				return;
			}
			try {
		        final Tc tc = (Tc)getObject();
		        TcPr tcPr = tc.getTcPr();
		        if(tcPr==null) {
		            tcPr = Context.getWmlObjectFactory().createTcPr();
		            tcPr.setParent(tc);
		            tc.setTcPr(tcPr);
		        }
		        Table.applyCellProperties(operationDocument, attrs.optJSONObject("cell"), tcPr);
		        final Object changes = attrs.opt("changes");
		        if(changes!=null) {
			        if(changes instanceof JSONObject) {
			        	final Object attrsInserted = ((JSONObject)changes).opt("inserted");
			        	if(attrsInserted!=null) {
			        		if(attrsInserted instanceof JSONObject) {
		            			CTTrackChange ins = tcPr.getCellIns();
		            			if(ins==null) {
		            				ins = Context.getWmlObjectFactory().createCTTrackChange();
		            				ins.setParent(tcPr);
		            				tcPr.setCellIns(ins);
		            			}
	                			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsInserted, ins);
			        		}
				        	else {
				        		tcPr.setCellIns(null);
				        	}
			        	}
			        	final Object attrsRemoved = ((JSONObject)changes).opt("removed");
			        	if(attrsRemoved!=null) {
		            		if(attrsRemoved instanceof JSONObject) {
		            			CTTrackChange del = tcPr.getCellDel();
		            			if(del==null) {
		            				del = Context.getWmlObjectFactory().createCTTrackChange();
		            				del.setParent(tcPr);
		            				tcPr.setCellDel(del);
		            			}
		            			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsRemoved, del);
		            		}
		            		else {
		            			tcPr.setCellDel(null);
		            		}
			        	}
			        	final Object attrsModified = ((JSONObject)changes).opt("modified");
			        	if(attrsModified!=null) {
			        		if(attrsModified instanceof JSONObject) {
			        			CTTcPrChange tcPrChange = tcPr.getTcPrChange();
			        			if(tcPrChange==null) {
			        				tcPrChange = Context.getWmlObjectFactory().createCTTcPrChange();
			        				tcPrChange.setParent(tcPr);
			        				tcPr.setTcPrChange(tcPrChange);
			        			}
		            			Utils.applyTrackInfoFromJSON(operationDocument, (JSONObject)attrsModified, tcPrChange);
	        					final Object attrsModifiedAttrs = ((JSONObject)attrsModified).opt("attrs");
	        					if(attrsModifiedAttrs!=null) {
	        						if(attrsModifiedAttrs instanceof JSONObject) {
	        							TcPrInner tcPrInner = tcPrChange.getTcPr();
	        							if(tcPrInner==null) {
	        								tcPrInner = Context.getWmlObjectFactory().createTcPrInner();
	        								tcPrInner.setParent(tcPrChange);
	        								tcPrChange.setTcPr(tcPrInner);
	        							}
	        					        Table.applyCellProperties(operationDocument, ((JSONObject)attrsModifiedAttrs).optJSONObject("cell"), tcPrInner);
	        						}
	        						else {
	        							tcPrChange.setTcPr(null);
	        						}
	        					}
			        		}
			        		else {
			        			tcPr.setTcPrChange(null);
			        		}
			        	}
			        }
			        else {
			        	tcPr.setCellDel(null);
			        	tcPr.setCellIns(null);
			        	tcPr.setCellMerge(null);
			        }
		        }
			}
			catch(Exception e) {
				log.error("docx, TcComponent::setAttributes:", e);
			}
		}
    }

    public abstract static class TextRun_Base extends Component {

        // change tracking mode of this component
        private int changeTrackFlags = 0;

        private IndexedNode<Object> textRunNode;
        private IndexedNode<Object> hyperlinkNode;
        private IndexedNode<Object> runInsNode;
        private IndexedNode<Object> runDelNode;

        protected TextRun_Base(IndexedNode<Object> _node, IndexedNode<Object> _textRunNode, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(_node, _paragraphNode, _componentNumber);
            textRunNode = _textRunNode;
            hyperlinkNode = null;
        }

        @Override
        public Object getContextObject() {
        	Object o = ((Child)node.getData()).getParent();
        	while(!(o instanceof P)&&o instanceof Child) {
        		o = ((Child)o).getParent();
        	}
        	return o;
        }
        @Override
		public IndexedNode<Object> getContextChildNode() {
        	if(hyperlinkNode!=null) {
        		return hyperlinkNode;
        	}
        	else if(runInsNode!=null) {
        		return runInsNode;
        	}
        	else if(runDelNode!=null) {
        		return runDelNode;
        	}
        	else {
        		return textRunNode;
        	}
        }
		@Override
		public Component getNextComponent() {
			Component next = null;

			int index = getNode().getIndex();
			// first checking if there is a further component within the current textrun;
			next = TextRunContext.getNextComponent(getNextComponentNumber(), getContextNode(), hyperlinkNode, runInsNode, runDelNode, textRunNode, index);
			index = textRunNode.getIndex();

			if(next==null&&runDelNode!=null) {
				next = RunDelContext.getNextComponent(getNextComponentNumber(), getContextNode(), hyperlinkNode, runInsNode, runDelNode, index);
				index = runDelNode.getIndex();
			}
			if(next==null&&runInsNode!=null) {
				next = RunInsContext.getNextComponent(getNextComponentNumber(), getContextNode(), hyperlinkNode, runInsNode, index);
				index = runInsNode.getIndex();
			}
			if(next==null&&hyperlinkNode!=null) {
				next = HyperlinkContext.getNextComponent(getNextComponentNumber(), getContextNode(), hyperlinkNode, index);
				index = hyperlinkNode.getIndex();
			}
			if(next==null) {
				next = ParagraphContext.getNextComponent(getNextComponentNumber(), getContextNode(), index);
			}
			return next;
        }
		@Override
		public Component getChildComponent() {
			return null;
		}
		@Override
		public void splitStart(int n) {
			splitAccessor(this, true);
		}
		@Override
		public void splitEnd(int n) {
			splitAccessor(this, false);
		}
		@Override
		public void setAttributes(OperationDocument operationDocument, JSONObject attrs) {
	        try {
				TextUtils.setTextRunAttributes(operationDocument, attrs, this, false);
			}
	        catch (Exception e) {
				log.error("docx, TextRun_Base::setAttributes", e);
			}
		}
        public IndexedNode<Object> getTextRunNode() {
            return textRunNode;
        }
        public R getTextRun() {
            return (R)textRunNode.getData();
        }
        public int getCT() {
        	return changeTrackFlags;
        }
        public void setCT(int _changeTrackFlags) {
        	changeTrackFlags = _changeTrackFlags;
        }
        public void setHyperlinkNode(IndexedNode<Object> _hyperlinkNode) {
        	hyperlinkNode = _hyperlinkNode;
        }
        public IndexedNode<Object> getHyperlinkNode() {
        	return hyperlinkNode;
        }
        public void setRunInsNode(IndexedNode<Object> _runInsNode) {
        	runInsNode = _runInsNode;
        }
        public IndexedNode<Object> getRunInsNode() {
        	return runInsNode;
        }
        public void setRunDelNode(IndexedNode<Object> _runDelNode) {
        	runDelNode = _runDelNode;
        }
        public IndexedNode<Object> getRunDelNode() {
        	return runDelNode;
        }

        protected static void splitAccessor(TextRun_Base component, boolean splitStart) {

        	IndexedNode<Object> node = component.getNode();
        	Object parent = ((Child)node.getData()).getParent();
        	while(!(parent instanceof P)&&parent!=null) {
        		IndexedNode<Object> parentNode = new IndexedNode<Object>(parent, ((ContentAccessor)((Child)parent).getParent()).getContent().indexOf(parent));
        		final IndexedNodeList<Object> parentNodeContent = (IndexedNodeList<Object>)((ContentAccessor)parentNode.getData()).getContent();
        		if((splitStart&&node.getIndex()!=0)||(!splitStart&&node.getIndex()!=parentNodeContent.size()-1)) {

        			IndexedNode<Object> newAccessorNode = null;
        			if(parent instanceof R) {
	        			final R newRun = Context.getWmlObjectFactory().createR();
	    	    		final RPr rPr = ((R)parent).getRPr();
	    	    		if(rPr!=null) {
	    	                final RPr rPrNew = XmlUtils.deepCopy(rPr);
	    	                rPrNew.setParent(newRun);
	    	                newRun.setRPr(rPrNew);
	    	    		}
	    	    		newAccessorNode = new IndexedNode<Object>(newRun);
	    	    		if(splitStart) {
	    	    			component.textRunNode = newAccessorNode;
	    	    		}
	        		}
	        		else if (parent instanceof  P.Hyperlink) {
	    	    		final P.Hyperlink newAccessor = Context.getWmlObjectFactory().createPHyperlink();
	    	            newAccessor.setAnchor(((P.Hyperlink)parent).getAnchor());
	    	            newAccessor.setDocLocation(((P.Hyperlink)parent).getDocLocation());
	    	            newAccessor.setHistory(((P.Hyperlink)parent).isHistory());
	    	            newAccessor.setId(((P.Hyperlink)parent).getId());
	    	            newAccessor.setTgtFrame(((P.Hyperlink)parent).getTgtFrame());
	    	            newAccessor.setTooltip(((P.Hyperlink)parent).getTooltip());
	    	    		newAccessorNode = new IndexedNode<Object>(newAccessor);
	    	    		if(splitStart) {
	    	    			component.hyperlinkNode = newAccessorNode;
	    	    		}
	        		}
	        		else if (parent instanceof RunIns) {
	    	    		final RunIns newAccessor = Context.getWmlObjectFactory().createRunIns();
	    	    		newAccessor.setAuthor(((RunIns)parent).getAuthor());
	    	    		final XMLGregorianCalendar date = ((RunIns)parent).getDate();
	    	    		if(date!=null) {
	    	        		newAccessor.setDate((XMLGregorianCalendar) date.clone());
	    	    		}
	    	    		newAccessor.setId(((RunIns)parent).getId());
	    	    		newAccessorNode = new IndexedNode<Object>(newAccessor);
	    	    		if(splitStart) {
	    	    			component.runInsNode = newAccessorNode;
	    	    		}
	        		}
	        		else if (parent instanceof RunDel) {
	    	    		final RunDel newAccessor = Context.getWmlObjectFactory().createRunDel();
	    	    		newAccessor.setAuthor(((RunDel)parent).getAuthor());
	    	    		final XMLGregorianCalendar date = ((RunDel)parent).getDate();
	    	    		if(date!=null) {
	    	        		newAccessor.setDate((XMLGregorianCalendar) date.clone());
	    	    		}
	    	    		newAccessor.setId(((RunDel)parent).getId());
	    	    		newAccessorNode = new IndexedNode<Object>(newAccessor);
	    	    		if(splitStart) {
	    	    			component.runDelNode = newAccessorNode;
	    	    		}
	        		}
        			if(newAccessorNode!=null) {
        				final Object pParent = ((Child)parentNode.getData()).getParent();
        				final Object newAccessor = newAccessorNode.getData();
        				((Child)newAccessor).setParent(pParent);
        				final IndexedNodeList<Object> pParentNode = (IndexedNodeList<Object>)((ContentAccessor)pParent).getContent();
    		        	pParentNode.addNode(parentNode.getIndex() + 1, newAccessorNode);
    		        	final IndexedNodeList<Object> rContent = (IndexedNodeList<Object>)((ContentAccessor)parentNode.getData()).getContent();
    		            final IndexedNodeList<Object> rContentNew = (IndexedNodeList<Object>)((ContentAccessor)newAccessorNode.getData()).getContent();
    		            int index = node.getIndex();
    		            if(!splitStart) {
    		            	index++;
    		            }
    		            final int count = rContent.size() - index;
    		            if(count>0) {
    		                IndexedNodeList.moveNodes(rContent, rContentNew, index, rContentNew.size(), count, newAccessor);
    		            }
    		            if(splitStart) {
    		            	parentNode = newAccessorNode;
    		            }
        			}
        		}
        		node = parentNode;
        		parent = ((Child)node.getData()).getParent();
        	}
        }
    }

    public static class TextComponent extends TextRun_Base {

        public TextComponent(IndexedNode<Object> _textNode, IndexedNode<Object> _textRunNode, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(_textNode, _textRunNode, _paragraphNode, _componentNumber);
        }

        @Override
        public int getNextComponentNumber() {
            int textLength = ((IText)node.getData()).getValue().length();
            if (textLength==0) {
                System.out.println("Error: empty text string is no component... this results in problems with the component iterator");
                textLength++;       // increasing.. preventing an endless loop
            }
            return componentNumber + textLength;
        }
        @Override
        public void splitStart(int componentPosition) {
        	final int splitPosition = componentPosition-componentNumber;
        	if(splitPosition>0) {
        		splitText(splitPosition, true);
        		componentNumber+=splitPosition;
        	}
    		splitAccessor(this, true);
        }
        @Override
        public void splitEnd(int componentPosition) {
        	if(getNextComponentNumber()>++componentPosition) {
        		splitText(componentPosition-componentNumber, false);
        	}
        	splitAccessor(this, false);
        }
        public static void preserveSpace(IText t) {
            if(t!=null) {
                String s = t.getValue();
                if(s!=null&&s.length()>0) {
                    if(s.charAt(0)==' '||s.charAt(s.length()-1)==' ') {
                        t.setSpace("preserve");
                        return;
                    }
                }
                t.setSpace(null);
            }
        }
        private void splitText(int textSplitPosition, boolean splitStart) {
        	final IText text = (IText)node.getData();
            if(textSplitPosition>0&&textSplitPosition<text.getValue().length()) {
            	final IText newText = text.createText();
                newText.setParent(text.getParent());
                final IndexedNodeList<Object> runContent = getTextRun().getContent();
                final StringBuffer s = new StringBuffer(text.getValue());
                if(splitStart) {
                    runContent.add(node.getIndex(), newText);
                    newText.setValue(s.substring(0, textSplitPosition));
                    text.setValue(s.substring(textSplitPosition));
                }
                else {
                	runContent.add(node.getIndex()+1, newText);
                    text.setValue(s.substring(0, textSplitPosition));
                    newText.setValue(s.substring(textSplitPosition));
                }
                preserveSpace(newText);
                preserveSpace(text);
            }
        }
    }

    public static class TabComponent extends TextRun_Base {

        public TabComponent(IndexedNode<Object> _node, IndexedNode<Object> _textRunNode, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(_node, _textRunNode, _paragraphNode, _componentNumber);
        }
    }

    public static class HardBreakComponent extends TextRun_Base {

        public HardBreakComponent(IndexedNode<Object> _node, IndexedNode<Object> _textRunNode, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(_node, _textRunNode, _paragraphNode, _componentNumber);
        }
    }

    public static class FldSimpleComponent extends Component {

        public FldSimpleComponent(IndexedNode<Object> _node, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(_node, _paragraphNode, _componentNumber);
        }

		@Override
		public Component getNextComponent() {
			return ParagraphContext.getNextComponent(getNextComponentNumber(), getContextNode(), getNode().getIndex());
		}
		@Override
		public Component getChildComponent() {
			return null;
		}
		@Override
		public void setAttributes(OperationDocument operationDocument, JSONObject attrs) {
			try {
		    	final JSONObject characterAttributes = attrs.optJSONObject("character");
		    	com.openexchange.office.ooxml.docx.tools.Character.applyCharacterProperties
		    		(operationDocument, characterAttributes, getTextRun().getRPr());
			}
			catch(Exception e) {
				log.error("docx, FldSimpleComponent::setAttributes:", e);
			}
		}

		public R getTextRun() {
        	final CTSimpleField simpleField = (CTSimpleField)node.getData();
        	final IndexedNodeList<Object> content = simpleField.getContent();
        	for(int i=0; i<content.size();i++) {
        		Object o = getContentModel(content.getNode(i), node.getData());
        		if(o instanceof R)
        			return (R)o;
        	}
        	final R r = Context.getWmlObjectFactory().createR();
        	r.setParent(simpleField);
        	content.add(r);
        	return r;
        }

        public String getInstr() {
            return ((CTSimpleField)node.getData()).getInstr();
        }

        public String getRepresentation() {
            String representation = "";
            IndexedNodeList<Object> content = ((CTSimpleField)node.getData()).getContent();
            if(content!=null) {
                for(int i=0; i<content.size();i++) {
                    Object o = getContentModel(content.getNode(i), node.getData());
                    if(o instanceof R) {
                        IndexedNodeList<Object> run = ((R)o).getContent();
                        if(run!=null) {
                            for(int j=0; j<run.size();j++) {
                                Object ro = getContentModel(run.getNode(j), o);
                                if(ro instanceof IText) {
                                    String t = ((IText)ro).getValue();
                                    if(t!=null)
                                        representation += t;
                                }
                            }
                        }
                    }
                }
            }
            return representation;
        }
    }

    public abstract static class Placeholder_Base extends TextRun_Base {

        public enum GraphicType {
            UNDEFINED,
            IMAGE,                  // undoable
            SHAPE,
            OLE,
            DIAGRAM,
            CHART,
            HORIZONTAL_LINE
        }

        protected Placeholder_Base(IndexedNode<Object> _placeholderNode, IndexedNode<Object> _textRunNode, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(_placeholderNode, _textRunNode, _paragraphNode, _componentNumber);
        }

        abstract public GraphicType getType();
    }

    // parent can either be a Pict or a CTObject

    public static class PictVMLComponent extends Placeholder_Base {
        private GraphicType graphicType = GraphicType.UNDEFINED;

        public PictVMLComponent(IndexedNode<Object> _node, IndexedNode<Object> _textRunNode, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(_node, _textRunNode, _paragraphNode, _componentNumber);

            List<Object> content = getContent();
            if(content != null) {
                for(Object o : content) {
                    if(o instanceof JAXBElement){
                        String type = ((JAXBElement<?>)o).getDeclaredType().getName();
                        if(type.equals("org.docx4j.vml.CTRect")) {
                            o = ((JAXBElement<?>)o).getValue();
                            STTrueFalse stTrueFalse = ((CTRect)o).getHr();
                            if(stTrueFalse == STTrueFalse.T||stTrueFalse == STTrueFalse.TRUE)
                                graphicType = GraphicType.HORIZONTAL_LINE;
                            break;
                        }
                        else if(type.equals("org.docx4j.vml.officedrawing.CTOLEObject")) {
                            graphicType = GraphicType.OLE;
                            break;
                        }
                    }
                }
            }
        }

        @Override
        public GraphicType getType() {
            return graphicType;
        }

        public List<Object> getContent() {
            Object o = getObject();
            if(o instanceof Pict) {
                return ((Pict)o).getAnyAndAny();
            }
            return ((CTObject)o).getAnyAndAny();
        }

        /**
         * @param part (the partName uri is something like: /word/document.xml)
         * @return the imageUrl in following form: word/media/xxxx.png
         */
        public String getImageUrl(Part part) {

            String rId = null;
            List<Object> content = getContent();
            if(content!=null) {
                for(Object o:content) {
                    if (o instanceof JAXBElement && ((JAXBElement<?>)o).getDeclaredType().getName().equals("org.docx4j.vml.CTShape") )
                    {
                        CTShape ctShape = (CTShape)((JAXBElement<?>)o).getValue();
                        if(ctShape!=null) {
                            List<JAXBElement<?>> shapeElements = ctShape.getEGShapeElements();
                            for(JAXBElement<?> shapeElement : shapeElements) {
                                if(shapeElement.getDeclaredType().getName().equals("org.docx4j.vml.CTImageData")) {
                                    CTImageData ctImageData = (CTImageData)((JAXBElement<?>)shapeElement).getValue();
                                    if(ctImageData!=null) {
                                        rId = ctImageData.getId();
                                    }
                                }
                            }
                        }
                        break;
                    }
                }
            }
            return Commons.getUrl(part, rId);
        }
    }

    public static class DrawingComponent extends Placeholder_Base {

        public DrawingComponent(IndexedNode<Object> _node, IndexedNode<Object> _textRunNode, IndexedNode<Object> _paragraphNode, int _componentNumber) {
            super(_node, _textRunNode, _paragraphNode, _componentNumber);
        }

        @Override
        public GraphicType getType() {

            GraphicType type = GraphicType.UNDEFINED;
            GraphicData graphicData = getGraphicData();
            if(graphicData!=null) {
                String uri = graphicData.getUri();
                if(uri.equalsIgnoreCase("http://schemas.microsoft.com/office/word/2010/wordprocessingShape"))
                    type = GraphicType.SHAPE;
                else if(uri.equalsIgnoreCase("http://schemas.openxmlformats.org/drawingml/2006/picture"))
                    type = GraphicType.IMAGE;
                else if(uri.equalsIgnoreCase("http://schemas.openxmlformats.org/drawingml/2006/diagram"))
                    type = GraphicType.DIAGRAM;
                else if(uri.equalsIgnoreCase("http://schemas.openxmlformats.org/drawingml/2006/chart"))
                    type = GraphicType.CHART;
            }
            return type;
        }

        public Object getAnchorOrInline() {
            List<Object> anchorOrInlineContent = ((Drawing)node.getData()).getAnchorOrInline();
            return anchorOrInlineContent.size() > 0 ? anchorOrInlineContent.get(0) : null;
        }

        public void setAnchorOrInline(Object o) {
            List<Object> anchorOrInlineContent = ((Drawing)node.getData()).getAnchorOrInline();
            anchorOrInlineContent.clear();
            anchorOrInlineContent.add(o);
        }

        public CTNonVisualDrawingProps getDocPr() {
            Object anchorOrInline = getAnchorOrInline();
            if(anchorOrInline instanceof Anchor)
                return ((Anchor)anchorOrInline).getDocPr();
            else if(anchorOrInline instanceof Inline)
                return ((Inline)anchorOrInline).getDocPr();
            else
                return null;
        }

        public Graphic getGraphic() {
            Graphic ret = null;
            Object anchorOrInline = getAnchorOrInline();
            if(anchorOrInline instanceof Anchor)
                ret = ((Anchor)anchorOrInline).getGraphic();
            else if (anchorOrInline instanceof Inline)
                ret = ((Inline)anchorOrInline).getGraphic();
            return ret;
        }

        public void setGraphic(Graphic graphic) {
            Object anchorOrInline = getAnchorOrInline();
            if(anchorOrInline instanceof Anchor)
                ((Anchor)anchorOrInline).setGraphic(graphic);
            else if (anchorOrInline instanceof Inline)
                ((Inline)anchorOrInline).setGraphic(graphic);
        }

        public GraphicData getGraphicData() {
            Graphic graphic = getGraphic();
            return graphic!=null ? graphic.getGraphicData() : null;
        }

        public void setGraphicData(GraphicData graphicData) {
            Graphic graphic = getGraphic();
            if(graphic!=null)
                graphic.setGraphicData(graphicData);
        }

        public Pic getPic() {
            Pic pic = null;
            GraphicData graphicData = getGraphicData();
            if(graphicData!=null) {
                List<Object> any = graphicData.getAny();
                for (int i=0;i<any.size();i++) {
                    Object o = any.get(i);
                    if (o instanceof JAXBElement && ((JAXBElement<?>)o).getDeclaredType().getName().equals("org.docx4j.dml.picture.Pic") ) {
                        pic = (Pic)((JAXBElement<?>)o).getValue();
                        any.set(i, pic);
                    }
                    else if (o instanceof Pic) {
                        pic = (Pic)o;
                    }
                    if(pic!=null)
                        break;
                }
            }
            return pic;
        }

        public void setPic(Pic pic) {
            GraphicData graphicData = getGraphicData();
            if(graphicData!=null) {
                int i = 0;
                List<Object> any = graphicData.getAny();
                for(;i<any.size();i++) {
                    Object o = any.get(i);
                    if (o instanceof JAXBElement && ((JAXBElement<?>)o).getDeclaredType().getName().equals("org.docx4j.dml.picture.Pic") )
                        break;
                    else if (o instanceof Pic)
                        break;
                }
                if (any.size()==i)
                    any.add(i, pic);
                else
                    any.set(i, pic);
            }
        }

        public CTBlipFillProperties getBlipFill() {
            Pic pic = getPic();
            return pic!=null ? pic.getBlipFill() : null;
        }

        public void setBlipFill(CTBlipFillProperties blipFillProperties) {
            Pic pic = getPic();
            if(pic!=null)
                pic.setBlipFill(blipFillProperties);
        }

        public CTShapeProperties getSpPr() {
            Pic pic = getPic();
            return pic!=null ? pic.getSpPr() : null;
        }

        public CTBlip getBlip() {
            CTBlipFillProperties blipFillProperties = getBlipFill();
            return blipFillProperties != null ? blipFillProperties.getBlip() : null;
        }

        public void setBlip(CTBlip blip) {
            CTBlipFillProperties blipFillProperties = getBlipFill();
            if(blipFillProperties!=null) {
                blipFillProperties.setBlip(blip);
                blip.setParent(blipFillProperties);
            }
        }

        public CTPositiveSize2D getExtend() {
            CTPositiveSize2D extend = null;
            Object anchorOrInline = getAnchorOrInline();
            if(anchorOrInline instanceof Anchor)
                extend = ((Anchor)anchorOrInline).getExtent();
            else if (anchorOrInline instanceof Inline)
                extend = ((Inline)anchorOrInline).getExtent();
            return extend;
        }
    }

  //--------------------------------------------------------------------------------------------------------------------------------------------------------    

    //                                                                                          Paragraph                             Paragraph 
    //                                   Paragraph                                             ?(hyperlink)                          ?(hyperlink)
    //          TextRun*                SimpleField*       Hyperlink*                             Ins*							         Del*
    //  Text*|Drawing*|Pict*|Tab*|Br*    TextRun(1)         TextRun*                             TextRun*					           TextRun*
    //                                    Text(1)    Text*|Drawing*|Pict*|Tab*|Br*      Text*|Drawing*|Pict*|Tab*|Br*		DelText*|Drawing*|Pict*|Tab*|Br*

  //--------------------------------------------------------------------------------------------------------------------------------------------------------    

    public static class MainContext {

        public static Component getNextComponent(int nextComponentNumber, IndexedNode<Object> bodyNode, int index) {
            Component next = null;
            final IndexedNodeList<Object> content = (IndexedNodeList<Object>)((ContentAccessor)bodyNode.getData()).getContent();
            while(next==null&&++index<content.size()) {
            	final IndexedNode<Object> node = content.getNode(index);
                final Object o = getContentModel(node, bodyNode.getData());
        		if(o instanceof P) {
        			next = new ParagraphComponent(node, bodyNode, nextComponentNumber);
        		}
        		else if(o instanceof Tbl) {
        			next =  new TableComponent(node, bodyNode, nextComponentNumber);
        		}
        	}
        	return next;
        }
    }

    public static class ParagraphContext {

        public static Component getNextComponent(int nextComponentNumber, IndexedNode<Object> paragraphNode, int index) {
            Component next = null;
            final IndexedNodeList<Object> content = ((P)paragraphNode.getData()).getContent();
            while(next==null&&++index<content.size()) {
            	final IndexedNode<Object> node = content.getNode(index);
                final Object o = getContentModel(node, paragraphNode.getData());
                if(o instanceof R)
                	next = TextRunContext.getNextComponent(nextComponentNumber, paragraphNode, null, null, null, node, -1);
                else if(o instanceof P.Hyperlink)
                    next = HyperlinkContext.getNextComponent(nextComponentNumber, paragraphNode, node, -1);
                else if(o instanceof RunIns)
                    next = RunInsContext.getNextComponent(nextComponentNumber, paragraphNode, null, node, -1);
                else if(o instanceof RunDel)
                	next = RunDelContext.getNextComponent(nextComponentNumber, paragraphNode, null, null, node, -1);
                else if(o instanceof CTSimpleField) {
                    next = new FldSimpleComponent(node, paragraphNode, nextComponentNumber);
                }
            }
            return next;
        }
    }

    public static class TableContext {

        public static Component getNextComponent(int nextComponentNumber, IndexedNode<Object> tblNode, int index) {
            Component next = null;
            final IndexedNodeList<Object> content = ((Tbl)tblNode.getData()).getContent();
            while(next==null&&++index<content.size()) {
            	final IndexedNode<Object> node = content.getNode(index);
                final Object o = getContentModel(node, tblNode.getData());
        		if(o instanceof Tr) {
        			next = new TrComponent(node, tblNode, nextComponentNumber);
        		}
        	}
        	return next;
        }
    }

    public static class TableRowContext {

        public static Component getNextComponent(int nextComponentNumber, int nextGridPosition, IndexedNode<Object> tableRowNode, int index) {
            Component next = null;
            final IndexedNodeList<Object> content = ((Tr)tableRowNode.getData()).getContent();
            while(next==null&&++index<content.size()) {
            	final IndexedNode<Object> node = content.getNode(index);
                final Object o = getContentModel(node, tableRowNode.getData());
        		if(o instanceof Tc) {
        			next = new TcComponent(node, tableRowNode, nextComponentNumber, nextGridPosition);
        		}
        	}
        	return next;
        }
    }

    public static class TableCellContext {

        public static Component getNextComponent(int nextComponentNumber, IndexedNode<Object> cellNode, int index) {
            Component next = null;
            final IndexedNodeList<Object> content = (IndexedNodeList<Object>)((ContentAccessor)cellNode.getData()).getContent();
            while(next==null&&++index<content.size()) {
            	final IndexedNode<Object> node = content.getNode(index);
                final Object o = getContentModel(node, cellNode.getData());
        		if(o instanceof P) {
        			next = new ParagraphComponent(node, cellNode, nextComponentNumber);
        		}
        		else if(o instanceof Tbl) {
        			next =  new TableComponent(node, cellNode, nextComponentNumber);
        		}
        	}
        	return next;
        }
    }

//--------------------------------------------------------------------------------------------------------------------------------------------------------    

    public static class TextRunContext {

        public static Component getNextComponent(int nextComponentNumber, IndexedNode<Object> paragraphNode, IndexedNode<Object> hyperlinkNode, 
        											IndexedNode<Object> insNode, IndexedNode<Object> delNode, IndexedNode<Object> textRunNode, int index) {
            TextRun_Base next = null;
            final IndexedNodeList<Object> content = ((R)textRunNode.getData()).getContent();
            while(next==null&&++index<content.size()) {
                final IndexedNode<Object> node = content.getNode(index);
            	final Object o = getContentModel(node, textRunNode.getData());
                if(o instanceof IText) {
                    if(((IText)o).getValue().length()>0) {
                        next = new TextComponent(node, textRunNode, paragraphNode, nextComponentNumber);
                    }
                }
                else if(o instanceof Drawing)
                    next = new DrawingComponent(node, textRunNode, paragraphNode, nextComponentNumber);
                else if(o instanceof Pict||o instanceof CTObject)
                    next = new PictVMLComponent(node, textRunNode, paragraphNode, nextComponentNumber);
                else if(o instanceof Tab)
                    next = new TabComponent(node, textRunNode, paragraphNode, nextComponentNumber);
                else if(o instanceof Br)
                    next = new HardBreakComponent(node, textRunNode, paragraphNode, nextComponentNumber);
            }
            if(next!=null) {
            	next.setHyperlinkNode(hyperlinkNode);
            	int ct = 0;
            	if(delNode!=null) {
            		ct |= (CT_DELETED<<16)|CT_DELETED;
            		next.setRunDelNode(delNode);
            	}
            	if(insNode!=null) {
            		ct |= (CT_INSERTED<<16)|CT_INSERTED;
            		next.setRunInsNode(insNode);
            	}
            	next.setCT(ct);
            }
            return next;
        }
    }
    
    
    public static class HyperlinkContext {

        public static Component getNextComponent(int nextComponentNumber, IndexedNode<Object> paragraphNode, IndexedNode<Object> hyperlinkNode, int index) {
            Component next = null;
            final IndexedNodeList<Object> content = ((P.Hyperlink)hyperlinkNode.getData()).getContent();
            while(next==null&&++index<content.size()) {
                final IndexedNode<Object> node = content.getNode(index);
            	final Object o = getContentModel(node, hyperlinkNode.getData());
                if(o instanceof R) {
                	next = TextRunContext.getNextComponent(nextComponentNumber, paragraphNode, hyperlinkNode, null, null, node, -1);
                }
                else if(o instanceof RunIns) {
                    next = RunInsContext.getNextComponent(nextComponentNumber, paragraphNode, hyperlinkNode, node, -1);
                }
                else if(o instanceof RunDel) {
                	next = RunDelContext.getNextComponent(nextComponentNumber, paragraphNode, hyperlinkNode, null, node, -1);
                }
            }
            return next;
        }
    }

    public static class RunInsContext {

        public static Component getNextComponent(int nextComponentNumber, IndexedNode<Object> paragraphNode, IndexedNode<Object> hyperlinkNode,
        												IndexedNode<Object> runInsNode, int index) {
            Component next = null;
            final IndexedNodeList<Object> content = ((RunIns)runInsNode.getData()).getContent();
            while(next==null&&++index<content.size()) {
                final IndexedNode<Object> node = content.getNode(index);
            	final Object o = getContentModel(node, runInsNode.getData());
                if(o instanceof R) {
                    next = TextRunContext.getNextComponent(nextComponentNumber, paragraphNode, hyperlinkNode, runInsNode, null, node, -1);
                }
                else if(o instanceof RunDel) {
                   	next = RunDelContext.getNextComponent(nextComponentNumber, paragraphNode, hyperlinkNode, runInsNode, node, -1);
                }
            }
            return next;
        }
    }

    public static class RunDelContext {

        public static Component getNextComponent(int nextComponentNumber, IndexedNode<Object> paragraphNode, IndexedNode<Object> hyperlinkNode,
        											IndexedNode<Object> runInsNode, IndexedNode<Object> runDelNode, int index) {
            Component next = null;
            final IndexedNodeList<Object> content = ((RunDel)runDelNode.getData()).getContent();
            while(next==null&&++index<content.size()) {
                final IndexedNode<Object> node = content.getNode(index);
            	final Object o = getContentModel(node, runDelNode.getData());
                if(o instanceof R) {
                    next = TextRunContext.getNextComponent(nextComponentNumber, paragraphNode, hyperlinkNode, runInsNode, runDelNode, node, -1);
                }
            }
            return next;
        }
    }
}