/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.odf;

import java.io.InputStream;

import org.apache.commons.logging.Log;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.dom.OdfMetaDom;
import org.odftoolkit.odfdom.incubator.meta.OdfOfficeMeta;
import org.odftoolkit.odfdom.pkg.OdfPackage;

import com.openexchange.log.LogFactory;
import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.IExporter;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

/**
 * {@link Exporter}
 * 
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class Exporter implements IExporter {

	private final static Log LOG = LogFactory.getLog(Exporter.class);
	final ServiceLookup services;
	String mLastModifiedBy;
	String mCreator;

	/**
	 * Initializes a new {@link Exporter}.
	 */
	public Exporter(ServiceLookup _services) {
		super();
		services = _services;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * com.openexchange.office.IExporter#createDocument(java.io.InputStream,
	 * org.json.JSONObject)
	 */
	@Override
	public InputStream createDocument(Session session, InputStream documentStm,
			String mergeOperations, ResourceManager resourceManager,
			DocumentProperties documentProperties, boolean createFinalDocument)
			throws FilterException {
		
		if (null == mergeOperations) {
			// no operations to add, return the original document stream
			return documentStm;
		}

		InputStream resultStm = null;

		if (null != documentStm) {
			OdfOperationDoc odtOperationDocument = null;

			try {
				odtOperationDocument = new OdfOperationDoc(session, services,
						documentStm, resourceManager, documentProperties);
				final OdfDocument odfDocument = odtOperationDocument
						.getDocument();

				// Always overwrite uniqueDocumentID when writing changes..
				if (documentProperties != null) {
					String statusId = documentProperties.optString(
							DocumentProperties.PROP_UNIQUE_DOCUMENT_IDENTIFIER,
							null);
					if (statusId != null) {
						odtOperationDocument.writeStatusID(statusId);
					}


					String lastModifiedBy = documentProperties.optString(
							DocumentProperties.PROP_LAST_MODIFIED_BY, null);
					String creator = documentProperties.optString(
							DocumentProperties.PROP_CREATOR, null);
					// Manipulating the meta.xml in case the initial author or
					// last author is given..					
					if (lastModifiedBy != null || creator != null) {
						OdfOfficeMeta metaDom = odfDocument.getOfficeMetadata();
						if (metaDom != null) {
							// only apply a change if the previous author was not the same
							if (lastModifiedBy != null
									&& (mLastModifiedBy == null || !mLastModifiedBy
											.equals(lastModifiedBy))) {
								metaDom.setCreator(lastModifiedBy);
								mLastModifiedBy = lastModifiedBy;
							}
							if (creator != null 
									&& (mCreator == null || !mCreator
									.equals(creator))) {
								metaDom.setInitialCreator(creator);
								mCreator = creator;
							}
						}
					}
				}
				odtOperationDocument.applyOperations(mergeOperations);

				if (null != odfDocument) {
					final OdfPackage odfPackage = odfDocument.getPackage();

					if (null != odfPackage) {
						resultStm = odfPackage.getInputStream();
					}
				}
			} catch (Throwable e) {
				LOG.error("ODF Filter could not save the given operation:", e);
				throw (e instanceof FilterException) ? (FilterException) e
						: new FilterException(e, ErrorCode.CRITICAL_ERROR);
			} finally {
				if (null != odtOperationDocument) {
					odtOperationDocument.close();
				}
			}
		}

		return resultStm;
	}
}
