/**
 * **********************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 *
 * Copyright 2008, 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0. You can also
 * obtain a copy of the License at http://odftoolkit.org/docs/license.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************************
 */
package org.odftoolkit.odfdom.component;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import static org.odftoolkit.odfdom.component.JsonOperationProducer.OX_DEFAULT_LIST;
import static org.odftoolkit.odfdom.component.JsonOperationProducer.normalizeLength;
import static org.odftoolkit.odfdom.component.OdfOperationDocument.MAX_TABLE_CELLS;
import static org.odftoolkit.odfdom.component.OdfOperationDocument.MAX_TABLE_COLUMNS;
import static org.odftoolkit.odfdom.component.OdfOperationDocument.MAX_TABLE_ROWS;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.odftoolkit.odfdom.dom.OdfContentDom;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.OdfSchemaDocument;
import org.odftoolkit.odfdom.dom.OdfStylesDom;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.OdfStyleableShapeElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawFrameElement;
import org.odftoolkit.odfdom.dom.element.draw.DrawImageElement;
import org.odftoolkit.odfdom.dom.element.style.StyleFontFaceElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableColumnPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTablePropertiesElement;
import org.odftoolkit.odfdom.dom.element.svg.SvgDescElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableCellElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnGroupElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderColumnsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableHeaderRowsElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowGroupElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableRowsElement;
import org.odftoolkit.odfdom.dom.element.text.TextAElement;
import org.odftoolkit.odfdom.dom.element.text.TextHElement;
import org.odftoolkit.odfdom.dom.element.text.TextLineBreakElement;
import org.odftoolkit.odfdom.dom.element.text.TextListElement;
import org.odftoolkit.odfdom.dom.element.text.TextListHeaderElement;
import org.odftoolkit.odfdom.dom.element.text.TextListItemElement;
import org.odftoolkit.odfdom.dom.element.text.TextListStyleElement;
import org.odftoolkit.odfdom.dom.element.text.TextNoteCitationElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.dom.element.text.TextParagraphElementBase;
import org.odftoolkit.odfdom.dom.element.text.TextSElement;
import org.odftoolkit.odfdom.dom.element.text.TextSpanElement;
import org.odftoolkit.odfdom.dom.element.text.TextTabElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.dom.style.props.OdfStyleProperty;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeAutomaticStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeStyles;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.incubator.doc.text.OdfTextListStyle;
import org.odftoolkit.odfdom.pkg.OdfAttribute;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.odftoolkit.odfdom.pkg.OdfNamespace;
import org.odftoolkit.odfdom.type.Length;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * @author svante.schubertATgmail.com
 */
public class OdfFileSaxHandler extends DefaultHandler {

    private static final Logger LOG = Logger.getLogger(OdfFileSaxHandler.class.getName());
    private static final String ROW_SPAN = "rowSpan";
    // ToDo: Fix API with its 'ugly' property name
    private static final String COLUMN_SPAN = "gridSpan";
    // ODF value types used for cell content
    private static final String VALUE_TYPE_BOOLEAN = "boolean";
    private static final String VALUE_TYPE_TIME = "time";
    private static final String VALUE_TYPE_DATE = "date";
    private static final String VALUE_TYPE_CURRENCY = "currency";
    private static final String VALUE_TYPE_FLOAT = "float";
    private static final String VALUE_TYPE_PERCENTAGE = "percentage";
    private static final String VALUE_TYPE_STRING = "string";
    private static final String VALUE_TYPE_VOID = "void";
    static final String VISIBLE = "visible";
    static final String COLLAPSE = "collapse";
    static final String APOSTROPHE = "'";
    static final String APOSTROPHE_AND_EQUATION = "'=";
    static final String EMPTY_STRING = "";
    static final String SPACE_CHAR = " ";
    static final String NONE = "none";
    static final String DOT_CHAR = ".";
    static final String DOT = "dot";
    static final String EQUATION = "=";
    static final String HYPHEN_CHAR = "-";
    static final String HYPHEN = "hyphen";
    static final String UNDERSCORE_CHAR = "_";
    static final String UNDERSCORE = "underscore";
    private static final String LIBRE_OFFICE_MS_INTEROP_NAMESPACE = "urn:openoffice:names:experimental:ooo-ms-interop:xmlns:field:1.0";
    private static final String LIBRE_OFFICE_MS_INTEROP_TYPE_CHECKBOX = "vnd.oasis.opendocument.field.FORMCHECKBOX";
    private static final String LIBRE_OFFICE_MS_INTEROP_CHECKBOX_UNICODE = "\u25A1";
    private static final Integer ONE = 1;
    // the empty XML file to which nodes will be added
    private OdfFileDom mFileDom;
    private JsonOperationProducer mJsonOperationProducer;
    private Map<String, TextListStyleElement> mAutoListStyles = null;
    /**
     * Represents a stack of TextSpanElement. The text span will be added during
     * startElement(..) with the start address of text span And during
     * endElement(..) the correct span will be returned and the end address can
     * be provided as well.
     *
     *
     */
    private ArrayDeque<TextSelection> mTextSelectionStack;
    // the context node
    private Node mCurrentNode;
    private StringBuilder mCharsForElement = new StringBuilder();
    // Text for operations will be collected separately to allow to push not at every new delimiter (e.g. span).
    // In addition will be <text:s/> in the output string exchange to spaces
    private StringBuilder mCharsForOperation = new StringBuilder();
    // Gatheres the start text position for operations
    private boolean mIsCharsBeginning = true;
    List<Integer> mCharsStartPosition = null;
    private int mComponentDepth = -1;       // as component depth starts with zero
    // the actual component. Linking each other building the tree view of the document
    private Component mCurrentComponent;
    // the document itself, accessed via "/", not accessible via integer position (!!) TODO?!?
    //private Component mRootComponent;
    // the postion of the component, being updated for the operations being generated
    private LinkedList<Integer> mLastComponentPositions = new LinkedList<Integer>();
    /**
     * DOM is created by default, but is in general not needed
     */
    private boolean domCreationEnabled = true;
    private ArrayDeque<ShapeProperties> mShapePropertiesStack;
    // *** TABLE PROPERTIES ***
    // ToDo: Move this table member variables to a special Table related parser/evaluator (Strategy Pattern?)
    // name of the table or spreadsheet
    private String mTableName;
    private TableTableElement mTableElement;
    private List<TableTableColumnElement> mColumns;
    // The relative widths of the columns of a table
    private List<Integer> mColumnRelWidths;

    private int mColumnCount;
    // Required as the component table/draw can only be delayed created,
    // After the first child has been parsed.. And should ONLY ONCE created!
    private boolean isTableNew = false;
    Map<String, Object> mTableHardFormatting = null;
    // *** LIST Properties ***
    // @text:start-value is provided to the first paragraph only
    private int mListStartValue = -1;
    private ArrayDeque<ParagraphListProperties> mListStyleStack;
    // used to track in a text:h/text:p if currently whitespace is being deleted/trimmed
    private ArrayDeque<WhitespaceStatus> mWhitespaceStatusStack;
    /**
     * Quick cache to get the correct linked list. Key is the xml:id of the
     * first list. The sequence of all continued lists and usability functions
     * are provided by ContinuedList
     */
    private Map<String, ContinuedList> mLinkedLists = new HashMap<String, ContinuedList>();
    //*** FOR BLOCKING OPERATIONS
    // the number of elements above the current element during parsing.
    // Required to find out if the correct blocking element for the UI was found
    int mElementDepth = 0;
    // The depth of the element responsible of blocking further operations
    int mBlockingElementDepth = 0;
    boolean mNoOperationsAllowed = false;
    // All following blocking modes have different behavior
    boolean mIsBlockingFrame = false; // itself and children are allowed
    boolean mIsIgnoredElement = false; // not even itself allwed
    boolean mIsBlockingShape = false; // itself allowed
    // RunTimeConfiguration given by the caller of the ODF Adapter
    private int mMaxTableColumnCount;
    private int mMaxTableRowCount;
    private int mMaxTableCellCount;
    private boolean mIsSpreadsheet = true;
    private StringBuilder mSpreadsheetCellContent;
    // ToDo: Possible to bundle all component names?
    private static final String CELLS = "Cells";
    private static final String TABLE = "Table";
    private static final String ROWS = "Rows";
    private static final String COLUMNS = "Columns";
    // a range is currently only used within spreadsheet documents to optimize the data transfer
    private static final String RANGE = "Range";
    private static final String PARAGRAPH = "Paragraph";
    private static final String IMAGE = "Image";
    private static final String SHAPE = "Shape";
    private static final String HARD_BREAK = "HardBreak";
    private static final String TAB = "Tab";
    private static final String TEXT = "Text";
    private static final String FIELD = "Field";
    private static final String ATTRIBUTES = "Attributes";
    private static final int MAX_REPEATING_EMPTY_CELLS = 3;
    private static final int MIN_REPEATING_CONTENT_CELLS = 2;
    /**
     * LO/AOO/Calligra are applying to Hyperlinks the "Internet_20_link" style,
     * without writing out the dependency into XML. Therefore whenever a
     * Hyperlink existists without character style properties, the reference
     * will be set. see as well OX issue #29658#
     */
    private static final String HYERLINK_DEFAULT_STYLE = "Internet_20_link";
    private boolean mHasHyperlinkTemplateStyle = false;

    /**
     * Required as the order of linked-list is important! All xml:ids of a
     * connected/linked lists are put into a single list. This collection is
     * used to get the correct reference to the xml:id of the preceding list and
     * have to be updated, when linked lists are created, deleted or moved. Only
     * the text:continue-list of a new list will be evaluated
     */
    class ContinuedList {

        private String mListId;
        private List<String> mSortedIds = null;

        public ContinuedList(String precedingListId, String currentListId) {
            if (precedingListId != null && !precedingListId.isEmpty()) {
                mListId = precedingListId;
            } else {
                if (currentListId != null && !currentListId.isEmpty()) {
                    mListId = currentListId;
                }
            }
            mSortedIds = new LinkedList<String>();
        }

        public void add(String listId) {
            mSortedIds.add(listId);
        }

        public List<String> getListIds() {
            return mSortedIds;
        }

        public String getListId() {
            return mListId;
        }
    }

    /**
     * Checks if the preceding list is already part of a continued list,
     * otherwise creates a new continued list and adds both ids to it
     */
    ContinuedList newContinuedList(String precedingListId, String currentListId) {
        ContinuedList continuedList;
        if (!mLinkedLists.containsKey(precedingListId)) {
            continuedList = new ContinuedList(precedingListId, currentListId);
            continuedList.add(precedingListId);
            mLinkedLists.put(precedingListId, continuedList);
        } else {
            continuedList = mLinkedLists.get(precedingListId);
        }
        if (currentListId != null && !currentListId.isEmpty()) {
            continuedList.add(currentListId);
            mLinkedLists.put(currentListId, continuedList);
        }
        return continuedList;
    }

    /**
     * Checks if the preceding list is already part of a continued list,
     * otherwise creates a new continued list and adds both id to it
     */
    ContinuedList newContinuedList(String currentListId) {
        ContinuedList continuedList = null;
        if (currentListId != null && !currentListId.isEmpty()) {
            if (!mLinkedLists.containsKey(currentListId)) {
                continuedList = new ContinuedList(null, currentListId);
                mLinkedLists.put(currentListId, continuedList);
            } else {
                continuedList = mLinkedLists.get(currentListId);
            }
        }
        return continuedList;
    }

    /**
     * The status of frames
     */
    class ShapeProperties {

        // *** FRAME PROPERTIES ***
        // Required as the component draw frame can only be delayed created,
        // After the first child has been parsed.. And should ONLY ONCE created!
        DrawFrameElement mDrawFrameElement = null;
        List<Integer> mShapePosition = null;
        Map<String, Object> mShapeHardFormatations = null;
        List<OdfElement> mFrameChildren = null;
        int mFrameChildrenNumber = 0;
        boolean mIsImageFrame = false;

        public DrawFrameElement getDrawFrameElement() {
            return mDrawFrameElement;
        }

        private ShapeProperties() {
        }

        public ShapeProperties(List<Integer> start, Map<String, Object> hardFormatations) {
            // Maps are being reused, for upcoming components, therefore the collections have to be cloned
            mShapePosition = new LinkedList<Integer>(start);
            if (hardFormatations != null) {
                mShapeHardFormatations = new HashMap<String, Object>();
                mShapeHardFormatations.putAll(hardFormatations);
                JSONObject originalDrawingProps = (JSONObject) hardFormatations.get("drawing");
                // Unfortunately the JSON lib being used, does not support deep cloning
                JSONObject newDrawingProps = new JSONObject();
                if (originalDrawingProps != null) {
                    // copying hard
                    for (String key : originalDrawingProps.keySet()) {
                        try {
                            newDrawingProps.put(key, originalDrawingProps.get(key));
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    mShapeHardFormatations.put("drawing", newDrawingProps);
                }
            }
        }

        /**
         * If the frame has one or more image elements this method dispatches an
         * operation for the first image
         *
         * @param desc description of the image
         */
        public void createImageOperation(String desc) {
            createShapeOperation(desc, true);
        }

        public void createShapeOperation(String desc) {
            createShapeOperation(desc, false);
        }

        private void createShapeOperation(String desc, boolean isImage) {
            if (desc != null && !desc.isEmpty()) {
                JSONObject drawingProps = (JSONObject) this.mShapeHardFormatations.get("drawing");
                try {
                    drawingProps.put("description", desc);
                } catch (JSONException ex) {
                    Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
			// No drawing (e.g. frames) allowed on top level. ODF uses those for page aligned images, OX Text will fail as only tables and paragraphs are allwed root components.
            // Instead a dummy paragraph will be added around the drawing via an operation

            // see if top level drawing
            if (this.mShapePosition.size() == 1) {
                // create root level paragraph for the drawing
                if (mWithinTable) {
                    cacheTableOperation(PARAGRAPH, this.mShapePosition, null, null, null);
                } else {
                    mJsonOperationProducer.add(PARAGRAPH, this.mShapePosition, null);
                }

                // nest the drawing within the paragraph
                this.mShapePosition.add(0);
                // Do not create shapes delayed otherwise descendant components would be created first confusing the client
                if (isImage) {
                    if (mWithinTable) {
                        cacheTableOperation(IMAGE, this.mShapePosition, this.mShapeHardFormatations, null, null);
                    } else {
                        mJsonOperationProducer.addImage(this.mShapePosition, this.mShapeHardFormatations);
                    }
                } else {
                    if (mWithinTable) {
                        if (!mIsSpreadsheet) { // No shapes support for spreadsheet, yet!
                            cacheTableOperation(SHAPE, this.mShapePosition, this.mShapeHardFormatations, null, null);
                        }
                    } else {
                        mJsonOperationProducer.addShape(this.mShapePosition, this.mShapeHardFormatations);
                    }
                }
            } else {
                // Do not create shapes delayed otherwise descendant components would be created first confusing the client
                if (isImage) {
                    if (mWithinTable) {
                        cacheTableOperation(IMAGE, this.mShapePosition, this.mShapeHardFormatations, null, null);
                    } else {
                        mJsonOperationProducer.addImage(this.mShapePosition, this.mShapeHardFormatations);
                    }
                } else {
                    if (mWithinTable) {
                        if (!mIsSpreadsheet) { // No shapes support for spreadsheet, yet!
                            cacheTableOperation(SHAPE, this.mShapePosition, this.mShapeHardFormatations, null, null);
                        }
                    } else {
                        mJsonOperationProducer.addShape(this.mShapePosition, this.mShapeHardFormatations);
                    }
                }

            }
        }

        /**
         * If the frame has one or more image elements this method dispatches an
         * operation for the first image
         */
        public void createImageOperation() {
            createShapeOperation(null, true);
        }

        /**
         * If the frame has one or more image elements this method dispatches an
         * operation for the first image
         */
        public void createShapeOperation() {
            createShapeOperation(null, false);
        }

        public void setDrawFrameElement(DrawFrameElement drawFrameElement) {
            mDrawFrameElement = drawFrameElement;
        }

        public Map<String, Object> getShapeHardFormatting() {
            return mShapeHardFormatations;
        }

        public boolean hasImageSibling() {
            return mIsImageFrame;
        }

        public void declareImage() {
            mIsImageFrame = true;
        }

        /**
         * In a frame there might be multiple objects, but only the first
         * applicable is shown. For instance, there are replacement images after
         * an OLE object *
         */
        public int incrementChildNumber() {
            return ++mFrameChildrenNumber;
        }

        public int decrementChildNumber() {
            return --mFrameChildrenNumber;
        }

        public int getChildNumber() {
            return mFrameChildrenNumber;
        }
    }

    /**
     * The status of the list style
     */
    class ParagraphListProperties {

        // on each list
        String mStyleName = null;
        // on each list item
        String mStyleOverride = null;
        boolean mShowListLabel = false;
        boolean mIsListStart = false;
        String mListId;
        String mListXmlId;
        String mListItemXmlId;

        public String getListId() {
            return mListId;
        }

        public void setListId(String listId) {
            mListId = listId;
        }

        public String getListXmlId() {
            return mListXmlId;
        }

        public void setListXmlId(String listXmlId) {
            mListXmlId = listXmlId;
        }

        public String getListItemXmlId() {
            return mListItemXmlId;
        }

        public void setListItemXmlId(String listItemXmlId) {
            mListItemXmlId = listItemXmlId;
        }

        public void setListStart(boolean isListStart) {
            mIsListStart = isListStart;
        }

        public ParagraphListProperties() {
        }

        public void showListLabel(boolean showListLabel) {
            mShowListLabel = showListLabel;
        }

        public boolean hasListLabel() {
            return mShowListLabel;
        }

        public boolean isListStart() {
            return mIsListStart;
        }

        /**
         * Overrides the list style given by a <text:list> element.
         *
         * @param styleName the new list style, or null to unset a previous
         * override
         */
        public void overrideListStyle(String styleName) {
            mStyleOverride = styleName;
        }

        void setListStyleName(String styleName) {
            mStyleName = styleName;
        }

        String getListStyleName() {
            String styleName = null;
            if (mStyleOverride != null && !mStyleOverride.isEmpty()) {
                styleName = mStyleOverride;
            } else if (mStyleName != null && !mStyleName.isEmpty()) {
                styleName = mStyleName;
            }
            return styleName;
        }
    }

    /**
     * The whitespace status of a text container (ie. paragraph or heading).
     * Required for whitespace handling
     */
    class WhitespaceStatus {

        WhitespaceStatus(boolean isParagraphIgnored, int depth) {
            mDepth = depth;
            mIsParagraphIgnored = isParagraphIgnored;
        }
        int mDepth = -1;
        // FIXME: Instead of this context variabel, we might as well use the OdfElement.
        boolean mIsParagraphIgnored = false;

        public int getParagraphDepth() {
            return mDepth;
        }
        boolean mOnlyWhiteSpaceSoFar = true;
        int mFirstSpaceCharPosition = -1;

        public boolean hasOnlyWhiteSpace() {
            return mOnlyWhiteSpaceSoFar;
        }

        public boolean isParagraphIgnored() {
            return mIsParagraphIgnored;
        }

        public void setOnlyWhiteSpace(boolean onlyWhiteSpace) {
            mOnlyWhiteSpaceSoFar = onlyWhiteSpace;
        }

        /**
         * During parsing the first character of space siblings. -1 if there is
         * no space sibling
         */
        public int getFirstSpaceCharPosition() {
            return mFirstSpaceCharPosition;
        }

        /**
         * During parsing the first character of space siblings. -1 if there is
         * no space sibling
         */
        public void setFirstSpaceCharPosition(int currentSpaceCharPosition) {
            mFirstSpaceCharPosition = currentSpaceCharPosition;
        }

        /**
         * @return true if the previous character was a white space character
         */
        public boolean hasSpaceBefore() {
            return mFirstSpaceCharPosition > -1;
        }
    }
    OdfSchemaDocument mSchemaDoc = null;

    // Candidate Component Mode
    // Some components consist of multiple XML elements.
    // Even some ODF components start with the same
    // 2DO - DRAGON BOOK - Parser Look-ahead funzt net bei SAX? ;)
    //private boolean isCandidateComponentMode = true;
    public OdfFileSaxHandler(Node rootNode) {
        // Initialize starting DOM node
        if (rootNode instanceof OdfFileDom) {
            mFileDom = (OdfFileDom) rootNode;
        } else {
            mFileDom = (OdfFileDom) rootNode.getOwnerDocument();
        }
        mCurrentNode = rootNode;

        // *** COMPONENT HANDLING ***
        // Initialize starting Component
        // Make the root of component tree (to be created) accessible via the ODF schema document
        mSchemaDoc = (OdfSchemaDocument) mFileDom.getDocument();
        // The current component is the root component
        mCurrentComponent = null;

        // Getting Configuration
        Map configuration = mSchemaDoc.getPackage().getRunTimeConfiguration();
        if (mSchemaDoc instanceof OdfTextDocument) {
            mMaxTableColumnCount = 15; // Test default for text documents
            mMaxTableRowCount = 1500;
            mMaxTableCellCount = 1500;
            mIsSpreadsheet = false;
        } else {
            mMaxTableColumnCount = Integer.MAX_VALUE; // Test default for spreadsheet documents
            mMaxTableRowCount = Integer.MAX_VALUE;
            mMaxTableCellCount = Integer.MAX_VALUE;
            mIsSpreadsheet = true;
        }
        if (configuration != null && !mIsSpreadsheet) {
            if (configuration.containsKey(MAX_TABLE_COLUMNS)) {
                mMaxTableColumnCount = (Integer) configuration.get(MAX_TABLE_COLUMNS);
            }
            if (configuration.containsKey(MAX_TABLE_ROWS)) {
                mMaxTableRowCount = (Integer) configuration.get(MAX_TABLE_ROWS);
            }
            if (configuration.containsKey(MAX_TABLE_CELLS)) {
                mMaxTableCellCount = (Integer) configuration.get(MAX_TABLE_CELLS);
            }
        }
        LOG.log(Level.FINEST, "mMaxTableColumnCount{0}", mMaxTableColumnCount);
        LOG.log(Level.FINEST, "mMaxTableRowCount{0}", mMaxTableRowCount);
        LOG.log(Level.FINEST, "mMaxTableCellCount{0}", mMaxTableCellCount);

        // Make the Operation Queue to be created accessible via the Schema Document
        mJsonOperationProducer = mSchemaDoc.getJsonOperationQueue();
        if (mJsonOperationProducer == null) {
            // temporary initated here as all the tests are not using the OperationTextDocument
            mJsonOperationProducer = new JsonOperationProducer();
            mSchemaDoc.setJsonOperationQueue(mJsonOperationProducer);
        }
        try {
            mAutoListStyles = new HashMap<String, TextListStyleElement>();
            // initalize all template styles
            OdfStylesDom stylesDom = mSchemaDoc.getStylesDom();
            OdfOfficeStyles templateStyles = stylesDom.getOfficeStyles();
            if (templateStyles != null) {
                // check if the default hyperlinkstyle do exist (more info in test for OX issue #29658#)
                mHasHyperlinkTemplateStyle = templateStyles.getStyle(HYERLINK_DEFAULT_STYLE, OdfStyleFamily.Text) != null;

                List<OdfStyle> paragraphStyles = mapIterableToList(templateStyles.getStylesForFamily(OdfStyleFamily.Paragraph));
                // The sort is for testing purpose to receive across different JDK an equal result
                Collections.sort(paragraphStyles, new AlphanumComparator());
                for (OdfStyle style : paragraphStyles) {
                    mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.Paragraph, style);
                }
                List<OdfStyle> textStyles = mapIterableToList(templateStyles.getStylesForFamily(OdfStyleFamily.Text));
                // The sort is for testing purpose to receive across different JDK an equal result
                Collections.sort(textStyles, new AlphanumComparator());
                for (OdfStyle style : textStyles) {
                    mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.Text, style);
                }
                List<OdfStyle> graphicStyles = mapIterableToList(templateStyles.getStylesForFamily(OdfStyleFamily.Graphic));
                Collections.sort(graphicStyles, new AlphanumComparator());

//				Collections.sort(graphicStyles,
//                 new Comparator<OdfStyle>()
//                 {
//                     public int compare(OdfStyle s1, OdfStyle s2)
//                     {
//                         return s1.getStyleNameAttribute().compareTo(s2.getStyleNameAttribute());
//                     }
//                 });
                for (OdfStyle style : templateStyles.getStylesForFamily(OdfStyleFamily.Graphic)) {
                    mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.Graphic, style);
                }

//			for(OdfStyle style : templateStyles.getStylesForFamily(OdfStyleFamily.Table)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.Table, style);
//			}
//			for(OdfStyle style : templateStyles.getStylesForFamily(OdfStyleFamily.TableRow)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.TableRow, style);
//			}
//			for(OdfStyle style : templateStyles.getStylesForFamily(OdfStyleFamily.TableColumn)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.TableColumn, style);
//			}
                for (OdfStyle style : templateStyles.getStylesForFamily(OdfStyleFamily.TableCell)) {
                    mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.TableCell, style);
//				mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.TableCell, (OdfStyleBase) templateStyles.getDefaultStyle(OdfStyleFamily.TableCell));
                }
//			for(OdfStyle style : templateStyles.getStylesForFamily(OdfStyleFamily.Section)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.Section, style);
//			}
//			for(OdfStyle style : templateStyles.getStylesForFamily(OdfStyleFamily.List)){
//				mJsonOperationProducer.triggerStyleHierarchyOps(OdfStyleFamily.List, style);
//			}
                List<OdfTextListStyle> listStyles = mapIterableToList(templateStyles.getListStyles());
                Collections.sort(listStyles, new AlphanumComparator());
                for (OdfTextListStyle listStyle : listStyles) {
                    mJsonOperationProducer.addListStyle(listStyle);
                }
            }

            // Handle default master page attributes
            mJsonOperationProducer.insertDocumentProperties(stylesDom, "Standard");

        } catch (Exception ex) {
            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
        }
        // Stack to remember/track the nested delimiters not being componenets (spans) open-up by SAX events
        mTextSelectionStack = new ArrayDeque<TextSelection>();
        mListStyleStack = new ArrayDeque<ParagraphListProperties>();
        mShapePropertiesStack = new ArrayDeque<ShapeProperties>();
        mWhitespaceStatusStack = new ArrayDeque<WhitespaceStatus>();
    }

    /**
     * Maps an iterable object to a list.
     */
    private static <T> List<T> mapIterableToList(Iterable<T> iterable) {
        List<T> copy = null;
        if (!(iterable instanceof List)) {
            Iterator iter = iterable.iterator();
            copy = new ArrayList<T>();
            while (iter.hasNext()) {
                copy.add((T) iter.next());
            }
        } else {
            copy = (List) iterable;
        }
        return copy;
    }

    @Override
    public void startDocument() throws SAXException {
    }

    @Override
    public void endDocument() throws SAXException {
    }

    /**
     * There are areas that are not allowed to addChild further components
     * beyond. All further operations have to be blocked, but the creation of
     * the DOM tree must not be disturbed.
     */
    private boolean isBlockedSubTree() {
        return mNoOperationsAllowed;
    }

    /**
     * There are areas that are not allowed to addChild further components
     * beyond. All further operations have to be blocked, but the creation of
     * the DOM tree must not be disturbed.
     */
    private boolean checkEndOfBlockedSubTree(String uri, String localName) {

        boolean isBlocked = mNoOperationsAllowed;
        if (mNoOperationsAllowed) {
            isBlocked = isBlockedSubTree(uri, localName, false);
        }
        mElementDepth--;
        return isBlocked;
    }

    private boolean checkStartOfBlockedSubTree(String uri, String localName) {
        mElementDepth++;
        boolean isBlocked = mNoOperationsAllowed;
        if (!mNoOperationsAllowed) {
            isBlocked = isBlockedSubTree(uri, localName, true);
        } else if (mIsBlockingFrame) {
            if (mBlockingElementDepth == mElementDepth - 1 && !localName.equals("table")) {
                isBlocked = false;
            } else {
                isBlocked = true;
            }
        }
        return isBlocked;
    }

    //ToDo: Differentiate  if there is a shapeBlock, ImageBlock or ParagraphBlock
    private boolean isBlockedSubTree(String uri, String localName, boolean isStart) {
        // within a paragraph within a paragraph
        boolean isBlocked = mNoOperationsAllowed;
        if (isStart) {
            // if it is a second text component (ie. text:p or text:h element)
            if (!mWhitespaceStatusStack.isEmpty() && Component.isTextComponentRoot(uri, localName) || OdfElement.isIgnoredElement(uri, localName)) {
                isBlocked = true;
                mNoOperationsAllowed = true;
                mIsIgnoredElement = true;
                mBlockingElementDepth = mElementDepth;
                // if it is a <draw:frame>
            } else if (uri != null && uri.equals(DrawFrameElement.ELEMENT_NAME.getUri())) {
                if (localName.equals(DrawFrameElement.ELEMENT_NAME.getLocalName())) {
                    isBlocked = false;
                    mNoOperationsAllowed = true;
                    mIsBlockingFrame = true;
                    mBlockingElementDepth = mElementDepth;
                } else if (Component.isShapeElement(uri, localName)) {
                    isBlocked = false;
                    mNoOperationsAllowed = true;
                    mIsBlockingShape = true;
                    mBlockingElementDepth = mElementDepth;
                }
            }
        } else { // if this is the closing event of an element
            if (mNoOperationsAllowed) {
                if (mBlockingElementDepth == mElementDepth) {
                    if (mIsIgnoredElement && (!mWhitespaceStatusStack.isEmpty() && Component.isTextComponentRoot(uri, localName) || OdfElement.isIgnoredElement(uri, localName))) {
                        mIsIgnoredElement = false;
                        mBlockingElementDepth = -1;
                        mNoOperationsAllowed = false;
                        isBlocked = true;
                        // if it is a <draw:frame>
                    } else if (uri != null && uri.equals(DrawFrameElement.ELEMENT_NAME.getUri())) {
                        if (mIsBlockingFrame && localName.equals(DrawFrameElement.ELEMENT_NAME.getLocalName())) {
                            mNoOperationsAllowed = false;
                            mBlockingElementDepth = -1;
                            mIsBlockingFrame = false;
                            isBlocked = false;
                            // still the children of a frame are allowed
                        } else if (mIsBlockingShape && Component.isShapeElement(uri, localName)) {
                            mNoOperationsAllowed = false;
                            mBlockingElementDepth = -1;
                            mIsBlockingShape = false;
                            isBlocked = false;
                        }
                    }
                } else if (mIsBlockingFrame && mBlockingElementDepth == mElementDepth - 1 && !localName.equals("table")) {
                    isBlocked = false;
                }
            } else { // closing will never enabled a blocking
                if (mIsIgnoredElement || mIsBlockingShape) {
                    // close this element, but afterwards
                    mNoOperationsAllowed = true;
                    isBlocked = false;
                } else if (mIsBlockingFrame) {
                    if (mBlockingElementDepth == mElementDepth - 1 && !localName.equals("table")) {
                        isBlocked = false;
                    } else {
                        isBlocked = true;
                    }
                }

            }
        }
        return isBlocked;
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
        flushTextAtStart(uri, localName, qName);
        // if there is a specilized handler on the stack, dispatch the event
        OdfElement element = null;
        // ToDo: Should be able to create operations without creating the DOM Tree
        // ToDo: Are there use-cases that text:s still resides in the DOM? if(isWhiteSpaceElement) ??
        // If Paragraph is not being edited, it will be saved as it is..

        if (domCreationEnabled) {
            if (uri.equals(EMPTY_STRING) || qName.equals(EMPTY_STRING)) {
                element = mFileDom.createElement(localName);
            } else {
                element = mFileDom.createElementNS(uri, qName);
            }
            addAttributes(element, attributes);
        }

        // Font declarations are before the component
        if (element instanceof StyleFontFaceElement) {
            String fontName = element.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "name");
            if (fontName != null && !fontName.isEmpty()) {
                Set fontNames = ((OdfDocument) mSchemaDoc).getFontNames();
                if (!fontNames.contains(fontName)) {
                    mJsonOperationProducer.insertFontDescription(fontName, null, element.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "font-family"), element.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-family-generic"), element.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-pitch"), element.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "panose-1"));
                    fontNames.add(fontName);
                }
            }
        }
        if (element instanceof TextListStyleElement) {
            // We need the reference for later gettin the list styles
            TextListStyleElement listStyle = (TextListStyleElement) element;
            String styleName = listStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "name");
            if (styleName != null && !styleName.isEmpty()) {
                mAutoListStyles.put(styleName, listStyle);
            }
        }

        if (!checkStartOfBlockedSubTree(uri, localName)) {
            if (Component.isComponentRoot(uri, localName)) { // || Component.isCoveredComponentRoot(uri, localName)) {

                // It is not allowed to addChild further components..,
                // within a paragraph within a paragraph
                // ToDo ? -- HashMap with KEY - URL+localname, VALUE - ComponentName
                if (element instanceof TextPElement || element instanceof TextHElement) {
                    // Paragraphs that are not child of a known component should be ignored, otherwise the client gets into trouble with nested paragraphs
                    boolean isNestedParagraph = true;
                    if (mWhitespaceStatusStack.size() > 0) {
                        if (mWhitespaceStatusStack.getLast().getParagraphDepth() != mComponentDepth) {
                            isNestedParagraph = true;
                        }
                    } else {
                        isNestedParagraph = false;
                    }

                    if (!isNestedParagraph) {
                        mComponentDepth++;
                        TextParagraphElementBase p = (TextParagraphElementBase) element;
                        Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(p);
                        if (hardFormatting == null) {
                            hardFormatting = new HashMap<String, Object>();
                        }
                        if (element instanceof TextHElement || !mListStyleStack.isEmpty()) {
                            if (!hardFormatting.containsKey("paragraph")) {
                                // if there are absolute styles, but not the main property set, where the styleId should be placed in
                                hardFormatting.put("paragraph", new JSONObject());
                            }
                            JSONObject paraProps = (JSONObject) hardFormatting.get("paragraph");

                            try {
                                if (!mListStyleStack.isEmpty()) {
                                    paraProps.put("listLevel", mListStyleStack.size() - 1);
                                    // Only the first paragraph within a list item should show a label!
                                    ParagraphListProperties listProps = mListStyleStack.getLast();
                                    if (listProps.hasListLabel()) {
                                        listProps.showListLabel(Boolean.FALSE);
                                    } else {
                                        paraProps.put("listLabelHidden", Boolean.TRUE);
                                    }
                                    String listId = listProps.getListId();
                                    if (listId != null && !listId.isEmpty()) {
                                        paraProps.put("listId", listId);
                                    }
                                    boolean foundListXmlId = false;
                                    boolean foundListItemXmlId = false;
                                    Iterator<ParagraphListProperties> listPropsIter = mListStyleStack.descendingIterator();
                                    while ((!foundListXmlId || !foundListItemXmlId) && listPropsIter.hasNext()) {
                                        ParagraphListProperties currentListProp = listPropsIter.next();

                                        String listXmlId = currentListProp.getListXmlId();
                                        if (!foundListXmlId && listXmlId != null && !listXmlId.isEmpty()) {
                                            foundListXmlId = true;
                                            paraProps.put("listXmlId", listXmlId);
                                        }
                                        String listItemXmlId = currentListProp.getListItemXmlId();
                                        if (!foundListItemXmlId && listItemXmlId != null && !listItemXmlId.isEmpty()) {
                                            foundListItemXmlId = true;
                                            paraProps.put("listItemXmlId", listItemXmlId);
                                        }
                                    }

                                    if (listProps.isListStart()) {
                                        paraProps.put("listStart", Boolean.TRUE);
                                    }
                                    String listStyleId = mJsonOperationProducer.getListStyle(mListStyleStack, p);
                                    if (listStyleId != null && !listStyleId.isEmpty()) {
                                        mJsonOperationProducer.addListStyle(mSchemaDoc, mAutoListStyles, listStyleId);
                                        paraProps.put("listStyleId", listStyleId);
                                    } else {
                                        paraProps.put("listStyleId", OX_DEFAULT_LIST);
                                    }
                                    if (mListStartValue != -1) {
                                        paraProps.put("listStartValue", mListStartValue);
                                        mListStartValue = -1;
                                    }
                                }
                                // Add heading outline numbering
                                if (element instanceof TextHElement) {
                                    Integer outlineLevel = ((TextHElement) p).getTextOutlineLevelAttribute();
                                    if (outlineLevel != null) {
                                        outlineLevel--;
                                        paraProps.put("outlineLevel", outlineLevel);
                                    }
                                }
                            } catch (JSONException ex) {
                                Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                            }
                        }

                        List<Integer> position = updateComponentPosition();
                        OdfStyle templateStyle = p.getDocumentStyle();
                        String styleId = null;
                        if (templateStyle != null) {
                            styleId = templateStyle.getStyleNameAttribute();
                            if (styleId != null && !styleId.isEmpty()) {
                                hardFormatting.put("styleId", styleId);
                            }
                        }
                        if (mWithinTable) {
                            if (!mIsSpreadsheet) { // within a spreadsheet the content is given by ranges not via paragraphs..
                                cacheTableOperation(PARAGRAPH, position, hardFormatting, null, null);
                            }
                        } else {
                            mJsonOperationProducer.add(PARAGRAPH, position, hardFormatting);
                        }
                        mCurrentComponent = mCurrentComponent.createChildComponent(p);

                        // For each new paragraph/heading addChild a new context information for their whitespace, required for normalization
                        mWhitespaceStatusStack.add(new WhitespaceStatus(false, mComponentDepth));
                        element.markAsComponentRoot(true);
// ToDo: NEW COMPONENTS - SECTION
//			} else if (element instanceof TextSectionElement) {
//                mJsonOperationProducer.addChild("Section", position);
//                mCurrentComponent = mCurrentComponent.addChild((TextSectionElement) element);
                    } else {
                        // a nested text component without known component in-between
                        // ignore nested paragraph content
                        mWhitespaceStatusStack.add(new WhitespaceStatus(true, mComponentDepth));
                        ((OdfElement) element).ignoredComponent(true);
                    }
                } else if (element instanceof DrawFrameElement || Component.isShapeElement(uri, localName)) {
                    OdfElement shape = element;
                    Map<String, Object> hardFormatting = null;
                    if (element instanceof OdfStyleableShapeElement) {
                        hardFormatting = mJsonOperationProducer.getHardStyles((OdfStyleableShapeElement) shape);
                    }
                    if (hardFormatting == null || !hardFormatting.containsKey("drawing")) {
                        // if there are absolute styles, but not the main property set, where the styleId should be placed in
                        if (hardFormatting == null) {
                            hardFormatting = new HashMap<String, Object>();
                        }
                        hardFormatting.put("drawing", new JSONObject());
                    }
                    JSONObject drawingProps = (JSONObject) hardFormatting.get("drawing");
                    if (shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "width")) {
                        try {
                            drawingProps.put("width", JsonOperationProducer.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "width")));
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    if (shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "height")) {
                        try {
                            drawingProps.put("height", JsonOperationProducer.normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "height")));
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    if (shape.hasAttributeNS(OdfDocumentNamespace.DRAW.getUri(), "name")) {
                        try {
                            drawingProps.put("name", shape.getAttributeNS(OdfDocumentNamespace.DRAW.getUri(), "name"));
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
// Might be confusing for the user..
//					} else {
//						if (element instanceof DrawFrameElement) {
//							// if there is no name, the type will be added
//							drawingProps.put("name", "image");
//						}else{
//							// if there is no name, the type will be added
//							drawingProps.put("name", localName);
//						}
                    }
                    if (shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "x")) {
                        try {
                            int x = normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "x"));
                            if (x != 0) {
                                drawingProps.put("anchorHorOffset", x);
                            }
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    if (shape.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y")) {
                        try {
                            int y = normalizeLength(shape.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y"));
                            if (y != 0) {
                                drawingProps.put("anchorVertOffset", y);
                            }
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    //<attribute name="text:anchor-type">
                    //	<choice>
                    //		<value>page</value>
                    //		<value>frame</value>
                    //		<value>paragraph</value>
                    //		<value>char</value>
                    //		<value>as-char</value>
                    //	</choice>
                    //</attribute>
				/*	API:
                     anchorHorBase:  Horizontal anchor mode:		One of 'margin', 'page', 'column', 'character', 'leftMargin', 'rightMargin', 'insideMargin', or 'outsideMargin'.
                     /*
                     @text:anchor-type: h=anchorHorBase & v=anchorVerBase
                     page		=> h=page	v=page
                     frame		=> h=column v=margin
                     paragraph	=> h=column v=paragraph
                     char		=> h=character v=paragraph
                     as-char		=> inline & h & v weglassen*/
                    if (shape.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "anchor-type")) {
                        try {
                            String anchorType = shape.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "anchor-type");
                            if (anchorType.equals("page")) {
                                // OX API: true: image as character, false: floating mode
                                drawingProps.put("inline", Boolean.FALSE);
                                drawingProps.put("anchorHorBase", "page");
                                drawingProps.put("anchorVertBase", "page");
                            } else if (anchorType.equals("frame")) {
                                // OX API: true: image as character, false: floating mode
                                drawingProps.put("inline", Boolean.FALSE);
                                drawingProps.put("anchorHorBase", "column");
                                drawingProps.put("anchorVertBase", "margin");
                            } else if (anchorType.equals("paragraph")) {
                                // OX API: true: image as character, false: floating mode
                                drawingProps.put("inline", Boolean.FALSE);
                                drawingProps.put("anchorHorBase", "column");
                                drawingProps.put("anchorVertBase", "paragraph");
                            } else if (anchorType.equals("char")) {
                                // OX API: true: image as character, true: floating mode
                                drawingProps.put("inline", Boolean.FALSE);
                                drawingProps.put("anchorHorBase", "character");
                                drawingProps.put("anchorVertBase", "paragraph");
                            } else if (anchorType.equals("as-char")) {
                                // OX API: true: image as character, false: floating mode
                                drawingProps.put("inline", Boolean.TRUE);
                            }
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    hardFormatting.put("drawing", drawingProps);

                    mComponentDepth++;
                    List<Integer> pos = updateComponentPosition();
                    // the delay of the operation was not the solution, as the children would be added fist
                    // instead a setAttribute would be more appropriate
                    // even if there is a automatic style, only the template style is required
                    if (element instanceof OdfStyleableShapeElement) {
                        String styleId = ((OdfStyleableShapeElement) shape).getDocumentStyleName();
                        if (styleId != null && !styleId.isEmpty()) {
                            hardFormatting.put("styleId", styleId);
                        }
                    }
                    ShapeProperties shapeProps = new ShapeProperties(pos, hardFormatting);
                    // special handling for frames as together with the image child they are a single component
                    if (element instanceof DrawFrameElement) {
                        shapeProps.setDrawFrameElement((DrawFrameElement) shape);
                    }
                    element.markAsComponentRoot(true);
                    mCurrentComponent = mCurrentComponent.createChildComponent(element);
                    mShapePropertiesStack.push(shapeProps);

                    // table component (table within a text document or a spreadsheet)
                } else if (element instanceof TableTableElement) {
                    mComponentDepth++;
                    // The table will be created with column width, after columns are parsed (just before first row!)
                    updateComponentPosition();
                    // tables are not written out directly, but its operation collected and only flushed
                    // if they are not exceeding a maximum size

                    isTableNew = true;
                    mTableElement = (TableTableElement) element;
                    mCurrentComponent = mCurrentComponent.createChildComponent(mTableElement);
                    // initialize a new list for the relative column widths
                    //ToDo: Receive the styles from the root component
                    //ToDo: If I do not want a DOM, do I have to parse the styles and addChild them to component?
                    //		Do I have to parse the styles.xml first to get the props as maps (hashmaps)?
                    if (mTableElement.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name")) {
                        mTableHardFormatting = mJsonOperationProducer.getHardStyles(mTableElement);
                        String styleId = mTableElement.getDocumentStyleName();
                        if (styleId != null && !styleId.isEmpty()) {
                            if (mTableHardFormatting == null) {
                                mTableHardFormatting = new HashMap<String, Object>();
                            }
                            mTableHardFormatting.put("styleId", styleId);
                            //	All ODF styles are hard formatted
                            //	JSONObject tableProps = mTableHardFormatting.get("table");
                            //	mTableHardFormatting.put("styleId", table.getDocumentStyle().getStyleNameAttribute());
                            //	OdfStyle tableStyle = table.getDocumentStyle();
                            //	if(tableStyle != null){
                            //	mTableDisplayName = tableStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "display-name");
                        }
                    }
                    mTableName = mTableElement.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "name");
                    if (mIsSpreadsheet) {
                        // The grid is known after columns had been parsed, updating later to row positino
                        List<Integer> tablePosition = new LinkedList<Integer>(mLastComponentPositions);
                        cacheTableOperation(TABLE, tablePosition, mTableHardFormatting, null, mTableName);
                    } else {
                        mColumnRelWidths = new LinkedList<Integer>();
                    }
                    element.markAsComponentRoot(true);
                } else if (element instanceof TableTableRowElement) {
                    mComponentDepth++;
                    if (isTableNew) {
                        if (!mIsSpreadsheet) {
                            // In case neiter relative nor absolute table column width were given, all column are equal sized (given rel size of '1')
                            mColumnRelWidths = collectColumnWidths(mTableElement, mColumns);
                            mColumns.clear();
                            if (mColumnRelWidths != null && mColumnRelWidths.isEmpty()) {
                                for (int i = 0; i < mColumnCount; i++) {
                                    mColumnRelWidths.add(ONE);
                                }
                            }

                            // The grid is known after columns had been parsed, updating later to row positino
                            List<Integer> tablePosition = new LinkedList<Integer>(mLastComponentPositions);
                            cacheTableOperation(TABLE, tablePosition, mTableHardFormatting, mColumnRelWidths, mTableName);
                        }
                        mTableHardFormatting = null;
                        isTableNew = false;
                        mTableName = null;
                        mColumnCount = 0;
                        mColumnRelWidths = null;
                    }
                    List<Integer> position = updateComponentPosition();
                    TableTableRowElement row = (TableTableRowElement) element;
                    mCurrentComponent = mCurrentComponent.createChildComponent(row);
                    // repeatition can cause a different positioning
                    int repeatedRows = 1;
                    if (row.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-repeated")) {
                        repeatedRows = Integer.parseInt(row.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-repeated"));
                        mCurrentComponent.hasRepeated(true);
                    }
                    boolean isVisible = Boolean.TRUE;
                    if (row.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "visibility")) {
                        isVisible = VISIBLE.equals(row.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "visibility"));
                    }
                    Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(row);
                    OdfStyle templateStyle = row.getDocumentStyle();
                    String styleId = null;
                    if (templateStyle != null) {
                        styleId = templateStyle.getStyleNameAttribute();
                        if (styleId != null && !styleId.isEmpty()) {
                            hardFormatting.put("styleId", styleId);
                        }
                    }
                    if (!isVisible) {
                        JSONObject rowProps;
                        if (hardFormatting == null) {
                            // if there are absolute styles, but not the main property set, where the styleId should be placed in
                            if (hardFormatting == null) {
                                hardFormatting = new HashMap<String, Object>();
                            }
                        }
                        if (!hardFormatting.containsKey("row")) {
                            rowProps = new JSONObject();
                            hardFormatting.put("row", rowProps);
                        } else {
                            rowProps = (JSONObject) hardFormatting.get("row");
                            if (rowProps == null) {
                                rowProps = new JSONObject();
                            }
                        }
                        try {
                            rowProps.put("visible", Boolean.FALSE);
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    if (mIsSpreadsheet) {
                        int firstRow = position.get(position.size() - 1);
                        mCachedTableOps.mFirstRow = firstRow;
                        mCachedTableOps.mLastRow = firstRow + repeatedRows - 1;
                        mCachedTableOps.mRepeatedRows = repeatedRows - 1;
                        cacheTableOperation(ROWS, position, hardFormatting, firstRow, firstRow + repeatedRows - 1, mCachedTableOps.mPreviousRepeatedRows);
                    } else {
                        cacheTableOperation(ROWS, position, hardFormatting);
                    }
                    element.markAsComponentRoot(true);
                } else if (element instanceof TableTableCellElement) {
                    mComponentDepth++;
                    TableTableCellElement cell = (TableTableCellElement) element;
                    mCurrentComponent = mCurrentComponent.createChildComponent(cell);
                    mCachedTableOps.mCellRepetition = 1;
                    Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(cell);
                    if (cell.hasAttributes()) {
                        try {
                            // if there are absolute styles, but not the main property set, where the styleId should be placed in
                            if (hardFormatting == null || !hardFormatting.containsKey("cell")) {
                                if (hardFormatting == null) {
                                    hardFormatting = new HashMap<String, Object>();
                                }
                            }
                            JSONObject cellProps = (JSONObject) hardFormatting.get("cell");
                            if (cellProps == null) {
                                cellProps = new JSONObject();
                            }
                            // repeatition and covering can cause a different positioning
                            // ToDo: To make DOM optional, work on the component instead of the element. Check directly SAX attributes parameter!
                            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated")) {
                                // cellProps.put("repeatedColumns", cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeatedColumns"));
                                mCachedTableOps.mCellRepetition = Integer.parseInt(cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated"));
                                mCurrentComponent.hasRepeated(true);
                            }
                            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-spanned")) {
                                cellProps.put(COLUMN_SPAN, Integer.parseInt(cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-spanned")));
                            }
                            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-spanned")) {
                                cellProps.put(ROW_SPAN, Integer.parseInt(cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-spanned")));
                            }
                            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula") || cell.hasAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value-type")) {
                                String valueType = cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula");
                                Object cellValue = getCellValue(cell, valueType);
                                if (cellValue != null) {
                                    cellProps.put("value", cellValue);
                                }
                                // API MISSING: cellProps.put("numberFormat", mJsonOperationProducer.getCellNumberFormat(valueType));
                            }
                            if (cellProps.length() != 0) {
                                hardFormatting.put("cell", cellProps);
                            }
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    List<Integer> position = updateComponentPosition();
                    OdfStyle templateStyle = cell.getDocumentStyle();
                    if (templateStyle != null) {
                        String styleId = templateStyle.getStyleNameAttribute();
                        if (styleId != null && !styleId.isEmpty()) {
                            hardFormatting.put("styleId", styleId);
                        }
                    }
                    if (mIsSpreadsheet) { // caching the formats, when traversing through possible sub tables
                        cacheTableOperation(CELLS, mLastComponentPositions, hardFormatting, mCurrentComponent);
                        // all content of all paragraphs had been added to this builder, as our spreadsheet cell has only ONE format at this time.
                        mSpreadsheetCellContent = new StringBuilder();
                    } else {
                        cacheTableOperation(CELLS, position, hardFormatting);
                    }
                    element.markAsComponentRoot(true);
                } else if (element instanceof TextLineBreakElement) {
                    mComponentDepth++;
                    TextLineBreakElement lineBreak = (TextLineBreakElement) element;
                    List<Integer> position = updateComponentPosition();
                    mCurrentComponent = mCurrentComponent.createChildComponent(lineBreak);
                    if (mWithinTable) { // no HardBreak within a spreadsheet cell atm
                        if (!mIsSpreadsheet) { // paragraphs, text content, format and text breaks are given via cell range
                            cacheTableOperation(HARD_BREAK, position, null, null, null);
                        }
                    } else {
                        mJsonOperationProducer.add(HARD_BREAK, position, null);
                    }
                    element.markAsComponentRoot(true);
                } else if (element instanceof TextTabElement) {
                    mComponentDepth++;
                    TextTabElement tab = (TextTabElement) element;
                    List<Integer> position = updateComponentPosition();
                    mCurrentComponent = mCurrentComponent.createChildComponent(tab);
                    if (mWithinTable) { // no tabs within a spreadsheet cell atm
                        if (!mIsSpreadsheet) { // paragraphs, text content, format and text breaks are given via cell range
                            cacheTableOperation(TAB, position, null, null, null);
                        }
                    } else {
                        mJsonOperationProducer.add(TAB, position, null);
                    }
                    element.markAsComponentRoot(true);
                } else if (Component.isField(uri, localName)) {
                    mComponentDepth++;
                    List<Integer> position = updateComponentPosition();
                    mCurrentComponent = mCurrentComponent.createChildComponent(element);
                    TextFieldSelection selection = null;
                    if (element.hasAttributeNS(LIBRE_OFFICE_MS_INTEROP_NAMESPACE, "type") && element.getAttributeNS(LIBRE_OFFICE_MS_INTEROP_NAMESPACE, "type").equals(LIBRE_OFFICE_MS_INTEROP_TYPE_CHECKBOX)) {
                        selection = new TextFieldSelection(element, position, LIBRE_OFFICE_MS_INTEROP_CHECKBOX_UNICODE);
                    } else {
                        selection = new TextFieldSelection(element, position);
                    }
                    mTextSelectionStack.add(selection);
                } else {
                    mComponentDepth++;
                    element.markAsComponentRoot(true);
                }

            } else if (element instanceof TextSpanElement) {
                // Span <text:span> will be triggering an operation after the text content is parsed
                TextSpanSelection selection = new TextSpanSelection((TextSpanElement) element, getTextPosition());
                mTextSelectionStack.add(selection);
            } else if (element instanceof TextAElement) {
                TextHyperlinkSelection selection = new TextHyperlinkSelection((TextAElement) element, getTextPosition());
                mTextSelectionStack.add(selection);
            } else if (element instanceof TextSElement) { // IMPROVABLE: Currently no component, as it will be removed anyway and would burden removal from automatic path counting
                mComponentDepth++;
                List<Integer> position = updateComponentPosition();
                if (mIsCharsBeginning) {
                    mCharsStartPosition = position;
                    mIsCharsBeginning = false;
                }
                // No operation triggering as client knows only space characters. We keep the parsing/mapping to the more performant server
                TextSElement spaces = (TextSElement) element;
                mCurrentComponent = mCurrentComponent.createChildComponent(spaces);
                Integer quantity = spaces.getTextCAttribute();
                if (quantity == null) {
                    mCharsForOperation.append('\u0020');
                } else {
                    for (int i = 0; i < quantity; i++) {
                        mCharsForOperation.append('\u0020');
                    }
                }

            } else if (element instanceof TableTableColumnElement) {
                // Columns can be grouped by <table:table-columns> and <table:table-column-group>, these would addChild metadata to the following columns
                // Column command should be triggered when one of the grouping starts or closes or if the first row arrives
                TableTableColumnElement column = (TableTableColumnElement) element;
                // Adjust Column Count
                mColumnCount++;
                int repeatedColumns = 1;
                int firstColumn = mColumnCount - 1;
                if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated")) {
                    repeatedColumns = Integer.parseInt(column.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated"));
                    if (repeatedColumns > 1) {
                        mColumnCount += (repeatedColumns - 1);
                    }
                }
                Map<String, Object> mappedDefaultCellStyleProperties = null;
                if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "default-cell-style-name")) {
                    String defaultCellStyleId = column.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "default-cell-style-name");
                    if (defaultCellStyleId != null && !defaultCellStyleId.isEmpty()) {
                        // first check for automatic style of default cell style
                        // the automatic styles are in a spreadsheet always in the content.xml, only when a table is in a header/footer it would be in the styles.xml (latter we do not support)
                        OdfOfficeAutomaticStyles autoStyles = ((OdfContentDom) this.mFileDom).getAutomaticStyles();
                        OdfStyle defaultCellStyle = null;
                        defaultCellStyle = autoStyles.getStyle(defaultCellStyleId, OdfStyleFamily.TableCell);
                        if (defaultCellStyle == null) {
                            // if there is no automatic style, check if the style exist in the templates
                            OdfOfficeStyles documentStyles = mSchemaDoc.getDocumentStyles();
                            defaultCellStyle = documentStyles.getStyle(defaultCellStyleId, OdfStyleFamily.TableCell);
                        }
                        if (defaultCellStyle != null) {
                            // map all the style properties directly to JSON to be added to the column creation later
                            mappedDefaultCellStyleProperties = mJsonOperationProducer.getMappedStyleProperties(defaultCellStyle);
                        }
                    }
                }
                if (mIsSpreadsheet) { // Text are handling columns only as an array of widths
                    Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(column);
                    String styleId = null;
                    OdfStyle templateStyle = column.getDocumentStyle();
                    if (templateStyle != null) {
                        styleId = templateStyle.getStyleNameAttribute();
                    }
                    boolean isVisible = Boolean.TRUE;
                    if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "visibility")) {
                        isVisible = VISIBLE.equals(column.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "visibility"));
                    }
                    if (styleId != null || mappedDefaultCellStyleProperties != null || !isVisible) {
                        JSONObject columnProps;
                        if (hardFormatting == null) {
                            // if there are absolute styles, but not the main property set, where the styleId should be placed in
                            if (hardFormatting == null) {
                                hardFormatting = new HashMap<String, Object>();
                            }
                        }
                        if (!hardFormatting.containsKey("column")) {
                            columnProps = new JSONObject();
                            hardFormatting.put("column", columnProps);
                        } else {
                            columnProps = (JSONObject) hardFormatting.get("column");
                        }
                        if (!isVisible) {
                            try {
                                columnProps.put("visible", Boolean.FALSE);
                            } catch (JSONException ex) {
                                Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                            }
                        }
                        if (!hardFormatting.containsKey("cell")) {
                            if (mappedDefaultCellStyleProperties != null) {
                                if (mappedDefaultCellStyleProperties.containsKey("cell")) {
                                    hardFormatting.put("cell", mappedDefaultCellStyleProperties.get("cell"));
                                }
                                if (mappedDefaultCellStyleProperties.containsKey("character")) {
                                    hardFormatting.put("character", mappedDefaultCellStyleProperties.get("character"));
                                }
                            }
                        } else {
                            LOG.warning("******* BEWARE: There should exist no cell attributes!! ***********");
                        }
                        if (styleId != null && !styleId.isEmpty()) {
                            hardFormatting.put("styleId", styleId);
                        }
                    }
                    cacheTableOperation(COLUMNS, mLastComponentPositions, hardFormatting, firstColumn, mColumnCount - 1);
                } else { // if text
                    if (mColumns == null) {
                        mColumns = new ArrayList();
                    }
                    mColumns.add(column);
                }
            } else if (element instanceof TextListElement) {
                TextListElement list = (TextListElement) element;
                // in case it is a new list
                if (mListStyleStack.isEmpty()) {
                    // Add always a style, so it can be popped of the stack EVERY time a list element ends
                    ParagraphListProperties paragraphListProps = new ParagraphListProperties();
                    paragraphListProps.setListStart(true);

                    // There are two continuation mechanisms for lists in ODF.
                    // ODF 1.0/1.1 uses @text:continue-numbering using true/false
                    // ODF 1.2 added @text:continue-list using an IDRef to an xml:id of another list.
                    String continuedListId = list.getTextContinueListAttribute();
                    String listXmlId = list.getXmlIdAttribute();
                    if (continuedListId != null && !continuedListId.isEmpty()) {
                        paragraphListProps.setListId(newContinuedList(continuedListId, listXmlId).getListId());
                    } else if (listXmlId != null && !listXmlId.isEmpty()) {
                        paragraphListProps.setListId(newContinuedList(listXmlId).getListId());
                    }
                    if (listXmlId != null && !listXmlId.isEmpty()) {
                        paragraphListProps.setListXmlId(listXmlId);
                    } else {
                        paragraphListProps.setListXmlId(null);
                    }
                    mListStyleStack.add(paragraphListProps);
                } else {
                    // Add always a style, so it can be popped of the stack EVERY time a list element ends
                    mListStyleStack.add(new ParagraphListProperties());
                }

                // @text:continue-numbering LATER
                // @text:continue-list LATER
                // @xml-id - LATER
                // @text:style-name is the given list style unless overwritten by a decendent list
                //	Check if the list style was used already in the document,
                //	if not, map the list properties. Check first in auto than in template.
                //	(Due to MSO issue the style might be even in auto in styles.xml - shall I move them back to content?)
                if (list.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-name")) {
                    String listStyle = list.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-name");
                    mListStyleStack.getLast().setListStyleName(listStyle);
                }
            } else if (element instanceof TextListItemElement || element instanceof TextListHeaderElement) {
                ParagraphListProperties paragraphListStyle = mListStyleStack.getLast();
                OdfElement listItem = element;
                if (listItem instanceof TextListHeaderElement) {
                    // list header never show a label
                    paragraphListStyle.showListLabel(false);
                } else {
                    // As a new list item starts, the next paragraph needs to provide the list label
                    paragraphListStyle.showListLabel(true);
                }
                //	@text:start-value is provided to the first paragraph only
                if (listItem.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "start-value")) {
                    mListStartValue = Integer.parseInt(listItem.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "start-value"));
                }
                //	@text:style-override overrides within this list item the list style
                if (listItem.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-override")) {
                    String styleOverride = listItem.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-override");
                    if (styleOverride != null && !styleOverride.isEmpty()) {
                        paragraphListStyle.overrideListStyle(styleOverride);
                    } else {
                        paragraphListStyle.overrideListStyle(null);
                    }
                } else {
                    paragraphListStyle.overrideListStyle(null);
                }
                //  @xml-id
                String listXmlId = null;
                if (listItem instanceof TextListItemElement) {
                    listXmlId = ((TextListItemElement) listItem).getXmlIdAttribute();
                } else if (listItem instanceof TextListHeaderElement) {
                    listXmlId = ((TextListHeaderElement) listItem).getXmlIdAttribute();
                }
                if (listXmlId != null && !listXmlId.isEmpty()) {
                    mListStyleStack.getLast().setListItemXmlId(listXmlId);
                } else {
                    mListStyleStack.getLast().setListItemXmlId(null);
                }
            } else if (element instanceof DrawImageElement) {
                DrawImageElement image = (DrawImageElement) element;
                ShapeProperties frameProps = mShapePropertiesStack.peekFirst();
                int childNo = frameProps.incrementChildNumber();
                if (childNo == 1) {

                    Map<String, Object> hardFormatting = new HashMap<String, Object>();
                    hardFormatting.putAll(frameProps.getShapeHardFormatting());
                    JSONObject drawingProps = (JSONObject) hardFormatting.get("drawing");
                    if (image.hasAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "href")) {
                        try {
                            String href = image.getAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "href");
                            drawingProps.put("imageUrl", href);
                            // if there is cropping from the frame, we need to do further calculation based on real graphic size
                            if (drawingProps.has("cropRight") && (drawingProps.has("height") || drawingProps.has("width"))) {
                                JsonOperationProducer.calculateCrops(image, href, drawingProps);
                            }
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    if (image.hasAttributeNS(OdfDocumentNamespace.XML.getUri(), "id")) {
                        try {
                            drawingProps.put("imageXmlId", image.getAttributeNS(OdfDocumentNamespace.XML.getUri(), "id"));
                        } catch (JSONException ex) {
                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                    //ToDo: Need test document with child element having office:binary-data with base64 content

                    DrawFrameElement frameElement = frameProps.getDrawFrameElement();
                    frameElement.markAsComponentRoot(true);
                    mShapePropertiesStack.pollFirst();
                    mShapePropertiesStack.addFirst(frameProps);
                    frameProps.declareImage();
                    hardFormatting.put("drawing", drawingProps);
//					}else {
//						drawingProps.put("viewAlternative", childNo - 1);
                }

//				if (!frameProps.hasImageSibling()) {
//					mComponentDepth++;
//					frameProps.setFramePosition(updateComponentPosition());
//					mCurrentComponent = mCurrentComponent.createChildComponent(frameElement);
//					//			position.set(position.size() - 1, position.get(position.size() - 1) +1);
//				}
//				if (childNo == 1) { // DISABLING REPLACEMENT IMAGE FEATURE AS LONG CLIENT DOES NOT SUPPORT IT
//					 ToDo: Dependencies for frame replacement feature has to be updated in OdfElement.raiseComponentSize() for every Frame child/feature enabled
//					frameProps.saveShapeProps(frameProps.getShapePosition(), hardFormatting);
//				}
            }
            // but within a shape being a frame, the child elements are still of interest (currently only <draw:image> supported)
//		} else if (!mShapePropertiesStack.isEmpty() && mShapePropertiesStack.peekLast().mDrawFrameElement != null) {

            if (Component.isDocumentRoot(uri, localName)) {
                // temporary initated here as all the tests are not using the OperationTextDocument
                mCurrentComponent = new Component(element);
                mSchemaDoc.setRootComponent(mCurrentComponent);
            }
        } else {
            if (element instanceof OdfElement) {
                ((OdfElement) element).ignoredComponent(true);
            }
        }

        // push the new element as the context node...
        //mCurrentNode = element;
        // addChild the new element as a child of the current context node
        mCurrentNode = mCurrentNode.appendChild(element);
    }

    private static String getFormulaValue(TableTableCellElement cell) {
        String value = null;
        if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula")) {
            String formula = cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula");
            if (formula.startsWith("of:")) {
                value = formula.substring(3);
            } else {
                value = formula;
            }
        }
        return value;
    }

    private static Object getCellValue(TableTableCellElement cell, String valueType) {
        Object value = null;
        if (valueType != null && !valueType.isEmpty()) {
            if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula")) {
                //table:formula
                value = getFormulaValue(cell);
            } else if (valueType.equals("boolean")) {
                //office:boolean-value
                value = Boolean.parseBoolean(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "boolean-value"));
            } else if (valueType.equals("currency")) {
                //office:value and office:currency
                // 2Do: Missing Mapping for currency!!
                value = Long.parseLong(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value"));
            } else if (valueType.equals("date")) {
                // office:date-value
                value = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "date-value");
            } else if (valueType.equals("float")) {
                //office:value
                value = Double.parseDouble(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value"));
            } else if (valueType.equals("percentage")) {
                //office:value
                value = Long.parseLong(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value"));
            } else if (valueType.equals("string")) {
                //office:string-value
                value = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "string-value");
            } else if (valueType.equals("time")) {
                //office:time-value
                value = cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "time-value");
            }
            // do not allow empty strings.
            if (value != null && value instanceof String && ((String) value).isEmpty()) {
                value = null;
            }
        } else if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "formula")) {
            value = getFormulaValue(cell);
        } else if (cell.hasAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value")) {
            value = Double.parseDouble(cell.getAttributeNS(OdfDocumentNamespace.OFFICE.getUri(), "value"));
        }

        return value;
    }

    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException {
        flushTextAtEnd(uri, localName, qName);

        // at the end of a table check if it can be flushed
        if (uri != null && localName != null && localName.equals(TableTableElement.ELEMENT_NAME.getLocalName()) && uri.equals(OdfDocumentNamespace.TABLE.getUri())) {
            endTableSizeEvaluation();
        }
        // if we remove the current element, the current node shall not be changed in the end
        boolean selectionNormalization = false;
        // SPECIAL HANDLING FOR DESCRIPTION OF SHAPES: draw:frame as the shape is one of the children, e.g a draw:image child
        if (!checkEndOfBlockedSubTree(uri, localName)) {
            boolean isImageComponent = false;
            if ((uri != null && uri.equals(DrawFrameElement.ELEMENT_NAME.getUri()) && localName.equals(DrawFrameElement.ELEMENT_NAME.getLocalName()) || Component.isShapeElement(uri, localName))) {
                mComponentDepth--;
                // Check for description of shape/frame about to be closed
                ShapeProperties shapeProps = mShapePropertiesStack.removeLast();
                isImageComponent = shapeProps.hasImageSibling();
                NodeList descList = mCurrentComponent.mRootElement.getElementsByTagNameNS(OdfDocumentNamespace.SVG.getUri(), SvgDescElement.ELEMENT_NAME.getLocalName());
                String description = null;
                if (descList.getLength() > 0) {
                    SvgDescElement desc = (SvgDescElement) descList.item(0);
                    Node descText = desc.getFirstChild();
                    if (descText != null && descText instanceof Text) {
                        description = ((Text) descText).getTextContent();
                    }
                }

                if (description != null) {
                    if (isImageComponent) {
                        // create the image that has to be created, what was delayed as the description was a sibling
                        shapeProps.createImageOperation(description);
                    } else {
                        shapeProps.createShapeOperation(description);
//						// Shape operation for frames have to be generated at this point
//						if (uri.equals(DrawFrameElement.ELEMENT_NAME.getUri()) && localName.equals(DrawFrameElement.ELEMENT_NAME.getLocalName())) {
//							shapeProps.createShapeOperation(description);
//						} else {
//							// All other shapes receive a description via setAttributes
//							Map attr = new HashMap<String, Object>();
//							JSONObject drawingProps = new JSONObject();
//							try {
//								drawingProps.put("description", (description));
//								attr.put("drawing", drawingProps);
//								// Shape was alreay created, just trigger a setAttribute operation with the description afterwards
//								mJsonOperationProducer.setAttributes(shapeProps.mShapePosition, null, attr);
//							} catch (JSONException ex) {
//								Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
//							}
//						}
                    }
                } else {
                    if (isImageComponent) {
                        // only the image has to be created, what was delayed as the description was a sibling
                        shapeProps.createImageOperation();
                    } else {
                        shapeProps.createShapeOperation();
                    }
                }
                mCurrentComponent = mCurrentComponent.getParent();
//			} else if (Component.isCoveredComponentRoot(uri, localName)) { // adjust counting for table cells without numbering
//				//ToDO: Instead to count the covered someone should count the spanning (BUT this is against OOXML cell numbering!)
//				mComponentDepth--;
            } else if (isSpaceElement(uri, localName)) {
                mComponentDepth--;
                mCurrentComponent = mCurrentComponent.getParent();
//		} else if (uri != null && uri.equals(DrawImageElement.ELEMENT_NAME.getUri()) && localName.equals(DrawImageElement.ELEMENT_NAME.getLocalName())) {
//			mFramePropertiesStack.getFirst().decrementChildNumber();
            } else if (uri != null && Component.isComponentRoot(uri, localName)) {
                if (Component.isTextComponentRoot(uri, localName) && mWhitespaceStatusStack.size() > 0 && mWhitespaceStatusStack.getLast().mIsParagraphIgnored) {
                    // do nothing for a ignored paragraph
                    mWhitespaceStatusStack.removeLast();
                    // SPECIAL HANDLING FOR IMAGE: draw:image (not a component root T- replacement for draw:frame)
                } else if (localName.equals(DrawFrameElement.ELEMENT_NAME.getLocalName()) && uri.equals(DrawFrameElement.ELEMENT_NAME.getUri()) && isImageComponent
                    || !(localName.equals(DrawFrameElement.ELEMENT_NAME.getLocalName()) && uri.equals(DrawFrameElement.ELEMENT_NAME.getUri()))) {
                    // if the current component is a text container flush spans
                    if (Component.isTextComponentRoot(mCurrentNode)) {
                        Collection<TextSelection> selections = ((TextParagraphElementBase) mCurrentNode).getTextSelections();
                        if (selections != null) {
                            for (TextSelection s : selections) {
                                OdfStylableElement selectionElement = (OdfStylableElement) s.getSelectionElement();
                                Map<String, Object> hardFormatting = mJsonOperationProducer.getHardStyles(selectionElement);
                                String styleId = null;
                                OdfStyle templateStyle = selectionElement.getDocumentStyle();
                                if (templateStyle != null) {
                                    styleId = templateStyle.getStyleNameAttribute();
                                }
                                if (s.hasUrl() || styleId != null) {
                                    try {
                                        JSONObject charProps;
                                        if (hardFormatting == null) {
                                            // if there are absolute styles, but not the main property set, where the styleId should be placed in
                                            if (hardFormatting == null) {
                                                hardFormatting = new HashMap<String, Object>();
                                            }
                                        }
                                        if (s.hasUrl()) {
                                            if (!hardFormatting.containsKey("character")) {
                                                charProps = new JSONObject();
                                                hardFormatting.put("character", charProps);
                                            } else {
                                                charProps = (JSONObject) hardFormatting.get("character");
                                            }
                                            charProps.put("url", s.getURL());
                                        }
                                        if (styleId != null && !styleId.isEmpty()) {
                                            hardFormatting.put("styleId", styleId);
                                        } else {
                                            // add the implicit by LO/AOO used hyperlink style
                                            if (mHasHyperlinkTemplateStyle) {
                                                hardFormatting.put("styleId", HYERLINK_DEFAULT_STYLE);
                                            }
                                        }
                                    } catch (JSONException ex) {
                                        Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                                    }
                                }
                                if (hardFormatting != null) {
                                    if (mWithinTable) {
                                        if (mIsSpreadsheet) {
                                            // Currently all hardformatting text styles are being mapped to the cell
                                            CachedInnerTableOperation cellOperation = null;
                                            int lastOperation = mCachedTableOps.size() - 1;
                                            cellOperation = (CachedInnerTableOperation) mCachedTableOps.get(lastOperation);
                                            if (cellOperation.mComponentType.equals(CELLS)) {
                                                if (cellOperation.mHardFormattingProperties == null) {
                                                    cellOperation.mHardFormattingProperties = new HashMap<String, Object>();

                                                }
                                                if (hardFormatting.containsKey("character")) {
                                                    if (!cellOperation.mHardFormattingProperties.containsKey("character")) {
                                                        // addChild map directly
                                                        cellOperation.mHardFormattingProperties.put("character", hardFormatting.get("character"));
                                                    } else {
                                                        // merge map items
                                                        JSONObject existingFormatting = (JSONObject) cellOperation.mHardFormattingProperties.get("character");
                                                        // all text styles are mapped to the cell
                                                        JSONObject newFormatting = (JSONObject) hardFormatting.get("character");
                                                        if (existingFormatting != null && newFormatting != null) {
                                                            Set<String> keys = existingFormatting.keySet();
                                                            for (String key : keys) {
                                                                try {
                                                                    newFormatting.put(key, existingFormatting.get(key));
                                                                } catch (JSONException ex) {
                                                                    Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                                                                }
                                                            }
                                                        }
                                                        cellOperation.mHardFormattingProperties.put("character", newFormatting);
                                                    }
                                                    // cache the style, in case the content is being removed, the style will be kept
                                                    // Cell cellComponent = (Cell) cellOperation.mComponentProperties[0];
                                                    //cellComponent.setInternalCellStyle(cellOperation.mHardFormattingProperties);
                                                }
                                                if (hardFormatting.containsKey("cell")) {
                                                    if (!cellOperation.mHardFormattingProperties.containsKey("cell")) {
                                                        // addChild map directly
                                                        cellOperation.mHardFormattingProperties.put("cell", hardFormatting.get("cell"));
                                                    } else {
                                                        // merge map items
                                                        JSONObject existingStyles = (JSONObject) cellOperation.mHardFormattingProperties.get("cell");
                                                        JSONObject newStyles = (JSONObject) hardFormatting.get("cell");
                                                        if (existingStyles != null && newStyles != null) {
                                                            Set<String> keys = existingStyles.keySet();
                                                            for (String key : keys) {
                                                                try {
                                                                    newStyles.put("cell", existingStyles.get(key));
                                                                } catch (JSONException ex) {
                                                                    Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                                                                }
                                                            }
                                                        }
                                                    }
                                                }
                                            } else {
                                                LOG.warning("This should have been a cell operation..");
                                            }
                                        } else {
                                            cacheTableOperation(ATTRIBUTES, s.getStartPosition(), hardFormatting, s.getEndPosition(), null);
                                        }
                                    } else {
                                        mJsonOperationProducer.setAttributes(s.getStartPosition(), s.getEndPosition(), hardFormatting);
                                    }
                                }
                            }
                        }
//					// in this case check if the closing descendent (this element)
//					// had any none whitespace text and apply if necessary the change
//					// remove the current whitespace properties from the stack
//					int depth = mWhitespaceStatusStack.size();
//					boolean childHasWhiteSpace = true;
//					boolean parentHasOnlyWhiteSpace;
                        // if there is a parent text container
                        if (mWhitespaceStatusStack.size() > 0) {
                            mWhitespaceStatusStack.removeLast();
// BEFORE WE DID NOT ALLOWED TO FOLLOWING PARAGRAPHS
//						// see if the child only had whitespaces
//						childHasWhiteSpace = mWhitespaceStatusStack.getLast().hasOnlyWhiteSpace();
//						// switch to parent
//						mWhitespaceStatusStack.pop();
//						// see if the parent had only whitespaces
//						WhitespaceStatus parentWhiteSpaceStatus = mWhitespaceStatusStack.getLast();
//						parentHasOnlyWhiteSpace = parentWhiteSpaceStatus.hasOnlyWhiteSpace();
//						// if the parent had only whitespaces, but not the child
//						if (parentHasOnlyWhiteSpace && !childHasWhiteSpace) {
//							// remove the only whitespace modus from the parent
//							parentWhiteSpaceStatus.setOnlyWhiteSpace(childHasWhiteSpace);
//						}
//					} else {
//						// otherwise just end the state collection of this paragraph/heading
//						mWhitespaceStatusStack.pop();
                        }
                    }
                    // removing the last in the list of positions, when a component is closed
                    if (mCurrentComponent != null) {
                        if (mIsSpreadsheet && mCachedTableOps != null) {
                            Element root = mCurrentComponent.getRootElement();
                            if (root instanceof TableTableCellElement) {
                                JSONObject cellRepresentation = null;
                                CachedInnerTableOperation cellOperation = null;
                                int lastOperation = mCachedTableOps.size() - 1;
                                cellOperation = mCachedTableOps.get(lastOperation);
                                if (cellOperation != null && cellOperation.mComponentType.equals(CELLS)) {
                                    cellOperation = (CachedInnerTableOperation) mCachedTableOps.remove(lastOperation);
                                    // CELL STYLE: apply styles to the cell
                                    if ((cellOperation.mHardFormattingProperties != null && !cellOperation.mHardFormattingProperties.isEmpty()) || (mSpreadsheetCellContent != null && mSpreadsheetCellContent.length() > 0)) {
                                        cellRepresentation = new JSONObject();
                                        try {
                                            if (cellOperation.mHardFormattingProperties != null && !cellOperation.mHardFormattingProperties.isEmpty()) {
                                                if (cellOperation.mHardFormattingProperties.containsKey("cell")) {
                                                    JSONObject cellProps = (JSONObject) cellOperation.mHardFormattingProperties.get("cell");
                                                    if (cellProps.hasAndNotNull("value")) {
                                                        cellRepresentation.put("value", cellProps.opt("value"));
                                                        cellProps.remove("value");
                                                        if (cellProps.length() == 0) {
                                                            cellOperation.mHardFormattingProperties.remove("cell");
                                                        }
                                                    }
                                                }
                                                if (!cellOperation.mHardFormattingProperties.isEmpty()) {
                                                    JSONObject attrs = new JSONObject();
                                                    for (String arg : cellOperation.mHardFormattingProperties.keySet()) {
                                                        attrs.put(arg, cellOperation.mHardFormattingProperties.get(arg));
                                                    }
                                                    cellRepresentation.put("attrs", attrs);
                                                }
                                            }
                                            // if there is content within the cell element, use this as value, unless there is already a none-null value (given by one of the value attributes)
                                            if (mSpreadsheetCellContent != null && mSpreadsheetCellContent.length() > 0 && !cellRepresentation.hasAndNotNull("value")) {
                                                // SPREADSHEET SVANTE 2DO DATA FORMATS
                                                //cellRepresentation.put("value", castSpreadsheetCellContent(mSpreadsheetCellContent.toString(), ((TableTableCellElement) root).getOfficeValueTypeAttribute()));
                                                String cellValue = mSpreadsheetCellContent.toString();
                                                // if there is a content starting with = we have to mask it to distinguish it from formula
                                                if (cellValue.startsWith(EQUATION)) {
                                                    cellRepresentation.put("value", APOSTROPHE + cellValue);
                                                } else {
                                                    cellRepresentation.put("value", cellValue);
                                                }

                                            }
                                        } catch (JSONException ex) {
                                            Logger.getLogger(OdfFileSaxHandler.class.getName()).log(Level.SEVERE, null, ex);
                                        }
                                    }
                                    mSpreadsheetCellContent = null;
                                    evaluateSimilarCells(cellOperation.mStart, cellRepresentation, false);
                                    Cell cellComponent = (Cell) cellOperation.mComponentProperties[0];
                                    if (cellComponent.hasRepeated()) {
                                        mLastComponentPositions.set(mComponentDepth, mLastComponentPositions.get(mComponentDepth) + cellComponent.repetition() - 1);
//										String pos = cellComponent.getPosition(mCurrentComponent);
//										int i = mComponentDepth;
//										int lastCP = mLastComponentPositions.size() - 1;
                                    }
                                } else {
                                    LOG.warning("This should have been a cell operation..");
                                }
                            } else if (root instanceof TableTableRowElement) {
                                // flush cells not being written out yet
                                evaluateSimilarCells(null, null, true);
                                // Have to be added when rows is closed, as it cumulates the repeatedColumns rows of PREVIOUS rows
                                mCachedTableOps.mPreviousRepeatedRows += mCachedTableOps.mRepeatedRows;
                            }
                        }
                        /*
                         if (mCurrentComponent.hasRepeated()) {
                         // Position adjustments due to repetition
                         if (root instanceof TableTableRowElement) {
                         TableTableRowElement row = (TableTableRowElement) root;
                         // row repeatedColumns should not occur in text documents for AOO/LO and MSO15
                         if (row.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-repeatedColumns")) {
                         int nextRowNo = mLastComponentPositions.get(mComponentDepth) + Integer.parseInt(row.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-rows-repeatedColumns"));
                         // Adjustment to the component tree due do "repeatedColumns" attributes
                         mLastComponentPositions.set(mComponentDepth, nextRowNo - 1);
                         }
                         } else if (root instanceof TableTableCellElement) {
                         TableTableCellElement cell = (TableTableCellElement) root;
                         if (cell.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeatedColumns")) {
                         // Adjustment to the component tree due do "repeatedColumns" attributes
                         int nextCellNo = mLastComponentPositions.get(mComponentDepth) + Integer.parseInt(cell.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeatedColumns"));
                         mLastComponentPositions.set(mComponentDepth, nextCellNo - 1);
                         }
                         }
                         }*/
                        mComponentDepth--;
                    }
                    mCurrentComponent = mCurrentComponent.getParent();
                }
            } // if text delimiter - addChild text
            else if (Component.isTextSelection(mCurrentNode)) {
                // dropping the last (most inner) selection from the stack
                TextSelection textSelection = mTextSelectionStack.pollLast();
                if (textSelection != null) {
                    textSelection.setEndPosition(getTextPosition());
                    OdfElement root = ((OdfElement) mCurrentNode).getComponentRoot();
                    if (Component.isTextComponentRoot(root)) {
                        // sometimes when spans share the same text area, they are condensed to one. The remaining one is being returned, or in case of an empty element the parent
                        mCurrentNode = ((TextContainingElement) root).appendTextSelection(textSelection);
                        // selectionNormalization might delete an element in this case the change of the current node would be an error!
                        selectionNormalization = true;
                    }
                }
            } else if (uri.equals(OdfDocumentNamespace.TEXT.getUri()) && localName.equals("list-item")) {
                mListStyleStack.getLast().overrideListStyle(null);
            } else if (uri.equals(OdfDocumentNamespace.TEXT.getUri()) && localName.equals("list")) {
                // POP UP NEW LIST STYLE
                mListStyleStack.removeLast();
            }
        }
        // selectionNormalization might delete an element in this case the change of the current node would be an error!
        if (mCurrentNode != null && !selectionNormalization) {
//			// Remove Space elements
//			if (isSpaceElement(mCurrentNode)) {
//				TextSElement spaceElement = (TextSElement) mCurrentNode;
//				// remove the space element as it was already added as space character
//				mCurrentNode = mCurrentNode.getParentNode();
//				mCurrentNode.removeChild(spaceElement);
//			} else {

            // pop to the parent node
            mCurrentNode = mCurrentNode.getParentNode();
//			}
        }

    }

    /**
     * Supported types by API are: Empty Number String Boolean Error Formula
     */
    // Used later for data formats 
    private Object castSpreadsheetCellContent(String cellContent, String cellType) {
        Object castedCellContent = cellContent;
        if (cellType != null && !cellType.equals(VALUE_TYPE_STRING) || cellType.equals(VALUE_TYPE_CURRENCY)) {
            if (cellType.equals(VALUE_TYPE_BOOLEAN)) {
                castedCellContent = Boolean.parseBoolean(cellContent);
            } else if (cellType.equals(VALUE_TYPE_FLOAT)
                || cellType.equals(VALUE_TYPE_PERCENTAGE)) {
                castedCellContent = Double.parseDouble(cellContent);
            } else if (cellType.equals(VALUE_TYPE_TIME)
                || cellType.equals(VALUE_TYPE_DATE)) {
                // we need to convert the values to an Integer
                //ToDo
            } else if (cellType.equals(VALUE_TYPE_VOID)) {
                castedCellContent = null;
            }
        }
        return castedCellContent;
    }

    private void addAttributes(Element element, Attributes attributes) {
        String attrQname;
        String attrURL;
        OdfAttribute attr;
        for (int i = 0; i < attributes.getLength(); i++) {
            attrURL = attributes.getURI(i);
            attrQname = attributes.getQName(i);
            // if no namespace exists
            if (attrURL.equals(EMPTY_STRING) || attrQname.equals(EMPTY_STRING)) {
                // create attribute without prefix
                attr = mFileDom.createAttribute(attributes.getLocalName(i));
            } else {
                if (attrQname.startsWith("xmlns:")) {
                    // in case of xmlns prefix we have to create a new OdfNamespace
                    OdfNamespace namespace = mFileDom.setNamespace(attributes.getLocalName(i), attributes.getValue(i));
                    // if the file Dom is already associated to parsed XML addChild the new namespace to the root element
                    Element root = mFileDom.getRootElement();
                    if (root == null) {
                        root = element;
                    }
                    root.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns:" + namespace.getPrefix(), namespace.getUri());
                }
                // create all attributes, even namespace attributes
                attr = mFileDom.createAttributeNS(attrURL, attrQname);
            }

            // namespace attributes will not be created and return null
            if (attr != null) {
                element.setAttributeNodeNS(attr);
                // don't exit because of invalid attribute values
                try {
                    // set Value in the attribute to allow validation in the attribute
                    attr.setValue(attributes.getValue(i));
                } // if we detect an attribute with invalid value: remove attribute node
                catch (IllegalArgumentException e) {
                    element.removeAttributeNode(attr);
                }
            }
        }
    }

    private void flushTextAtStart(String uri, String localName, String qName) {
        flushText(uri, localName, qName, false);
    }

    private void flushTextAtEnd(String uri, String localName, String qName) {
        flushText(uri, localName, qName, true);
    }

    /**
     * Consumers shall collapse white space characters that occur in <ul> <li>a
     * <text:p> or <text:h> element (so called paragraph elements), and </li>
     * <li>in their descendant elements, if the OpenDocument schema permits the
     * inclusion of character data for the element itself and all its ancestor
     * elements up to the paragraph element. </li></ul> Collapsing white space
     * characters is defined by the following algorithm: 1)The following
     * [UNICODE] characters are replaced by a " " (U+0020, SPACE) character:
     * \ue570HORIZONTAL TABULATION (U+0009) \ue570CARRIAGE RETURN (U+000D)
     * \ue570LINE FEED (U+000A) 2)The character data of the paragraph element
     * and of all descendant elements for which the OpenDocument schema permits
     * the inclusion of character data for the element itself and all its
     * ancestor elements up to the paragraph element, is concatenated in
     * document order. 3)Leading " " (U+0020, SPACE) characters at the start of
     * the resulting text and trailing SPACE characters at the end of the
     * resulting text are removed. 4)Sequences of " " (U+0020, SPACE) characters
     * are replaced by a single " " (U+0020, SPACE) character.
     */
    private void flushText(String uri, String localName, String qName, boolean isEndOfElement) {
        // check if there is was text found to be added to the element
        if (mCharsForElement.length() > 0) {
            // every text will be kept from the XML file (e.g. indent)
            String newString = mCharsForElement.toString();
            mCharsForElement.setLength(0);
            Text text = mFileDom.createTextNode(newString);
            if (isEndOfElement && Component.isField(uri, localName)) {
                TextSelection textSelection = mTextSelectionStack.pollLast();
                if (!isBlockedSubTree()) {
                    // Currently only check-box have an UTF-8 square as replacementText
                    String replacementText = ((TextFieldSelection) textSelection).getReplacementText();
                    if (mWithinTable && !mIsSpreadsheet) {
                        if (replacementText != null) {
                            cacheTableOperation(FIELD, textSelection.getStartPosition(), null, localName, replacementText);
                        } else {
                            cacheTableOperation(FIELD, textSelection.getStartPosition(), null, localName, newString);
                        }
                    } else {
                        if (replacementText != null) {
                            mJsonOperationProducer.insertField(textSelection.getStartPosition(), localName, replacementText);
                        } else {
                            mJsonOperationProducer.insertField(textSelection.getStartPosition(), localName, newString);
                        }
                    }
                }
                mCurrentNode.appendChild(text);
            } else {
                // if the text is within a text aware component
                if (mCurrentNode instanceof OdfElement) {
                    // ToDo: Uncertain what with text should happen that is not within a text component? Neglectable by
                    if ((Component.isTextComponentRoot(mCurrentNode) || Component.isTextComponentRoot(((OdfElement) mCurrentNode).getComponentRoot())) && !isBlockedSubTree() && mWhitespaceStatusStack.size() > 0 && !mWhitespaceStatusStack.getLast().isParagraphIgnored() && !(mCurrentNode instanceof TextNoteCitationElement)) {
                        mComponentDepth++;
                        if (mIsCharsBeginning) {
                            mCharsStartPosition = updateTextPosition();
                            mIsCharsBeginning = false;
                        }
                        mComponentDepth--;
                        // The new charPosition adds the text lenght, but inserted will be without
                        // 1) insertion
                        mCharsForOperation.append(newString);
                        // muss ich rekursiv die gr\u00f6sse nach oben reichen? f\u00fcr jeden none component descendant? K\u00f6nnte ich in OdfElement implemenentieren!
                        // \u00fcberschreibe alle addChild/delete Funktionalit\u00e4t!
                        // Merge/split/delete Text Funktionalit\u00e4t f\u00fcr alle ELEMENTE? Komponenten m\u00fcssen mit einbezogen werden! Reuse of Recursion -- ACTION KLASSE.operate() aufruf!?!?
                        //					OdfElement element = (OdfElement) mCurrentNode;
                        //					element.appendChild(text);
                    }
                    if (isSpaceElement(mCurrentNode)) {
                        mCurrentNode.getParentNode().appendChild(text);
                    } else {
                        mCurrentNode.appendChild(text);
                    }
                }
            }
        } else {
            if (isEndOfElement && Component.isField(uri, localName)) {
                TextSelection textSelection = mTextSelectionStack.pollLast();
                if (!isBlockedSubTree()) {
                    // Currently only check-box have an UTF-8 square as replacementText
                    String replacementText = ((TextFieldSelection) textSelection).getReplacementText();
                    if (mWithinTable && !mIsSpreadsheet) {
                        if (replacementText != null) {
                            cacheTableOperation(FIELD, textSelection.getStartPosition(), null, localName, replacementText);
                        } else {
                            cacheTableOperation(FIELD, textSelection.getStartPosition(), null, localName, null);
                        }
                    } else {
                        if (replacementText != null) {
                            mJsonOperationProducer.insertField(textSelection.getStartPosition(), localName, replacementText);
                        } else {
                            mJsonOperationProducer.insertField(textSelection.getStartPosition(), localName, null);
                        }
                    }
                }
            }
        }
        if (mCharsForOperation.length() > 0 && Component.isComponentRoot(uri, localName) && !isSpaceElement(uri, localName)) {
            if (mWithinTable) {
                if (mIsSpreadsheet) {
                    mSpreadsheetCellContent.append(mCharsForOperation.toString());
                } else {
                    cacheTableOperation(TEXT, mCharsStartPosition, null, mCharsForOperation.toString(), null);
                }
            } else {
                mJsonOperationProducer.addText(mCharsStartPosition, mCharsForOperation.toString());
            }
            mCharsForOperation.setLength(0);
            mIsCharsBeginning = true;
        }
    }

    static boolean isSpaceElement(Node node) {
        return node instanceof TextSElement;
    }

    static boolean isSpaceElement(String uri, String localName) {
        return uri != null && uri.equals(TextSElement.ELEMENT_NAME.getUri()) && localName.equals(TextSElement.ELEMENT_NAME.getLocalName());
    }

    @Override
    /**
     * http://xerces.apache.org/xerces2-j/faq-sax.html#faq-2 : SAX may deliver
     * contiguous text as multiple calls to characters, for reasons having to do
     * with parser efficiency and input buffering. It is the programmer's
     * responsibility to deal with that appropriately, e.g. by accumulating text
     * until the next non-characters event. This method will finalize the text
     * of an element, by flushing/appending it to the element node. It is called
     * at the beginning of startElement/endElement. In case of startElement the
     * text will be referred to the previous element node (before the new
     * started). In case of endElement the text will be referred to the current
     * element node.
     */
    public void characters(char[] ch, int startPosition, int length) throws SAXException {
        if (mCurrentComponent instanceof TextContainer) {
            // ODF Whitespacehandling
            WhitespaceStatus currentWhiteSpaceStatus = mWhitespaceStatusStack.getLast();
            // Note: The delta between startPosition and endPosition marks the text to be written out
            // startPosition will only be raised to endposition, when characters have to be skipped!
            int endPosition = startPosition;
            int lastPos = startPosition + length;
            boolean previousContentWritten = false;
            char c;
            // Go through all characters found by the parser..
            for (int i = startPosition; i < lastPos; i++) {
                c = ch[i];
                // first part is trimming in the beginning of the element
                if (currentWhiteSpaceStatus.hasOnlyWhiteSpace()) {
                    // \t (tabulator = 0x09)
                    if (c == '\u0020' // space
                        || c == '\t'
                        // \r (carriage return = 0x0D)
                        || c == '\r'
                        // \n (line feed = 0x0A)
                        || c == '\n') {
                        // skipt this character, keeping the difference between start & end (length) equal
                        startPosition++;
                        endPosition++;
                    } else {
                        // first character being found worth to be written
                        currentWhiteSpaceStatus.setOnlyWhiteSpace(false);
                        endPosition++;
                    }
                    // second part is about collapsing multiple whitespaces
                } else {
                    if (c == '\u0020' // space
                        || c == '\t' // \t (tabulator = 0x09)
                        // \r (carriage return = 0x0D)
                        || c == '\r'
                        // \n (line feed = 0x0A)
                        || c == '\n') {
                        // if we have aleady a preceding whitespace character
                        if (currentWhiteSpaceStatus.hasSpaceBefore()) {
                            if (!previousContentWritten) {
                                // as we have to skip a character in the array, write what we have
                                if (endPosition - startPosition > 0) {
                                    mCharsForElement.append(ch, startPosition, endPosition - startPosition);
                                }
                                previousContentWritten = true;
                            }
                            // NOT including this character
                            endPosition++;
                            startPosition = endPosition;
                        } else {
                            currentWhiteSpaceStatus.setFirstSpaceCharPosition(i);
                            ch[i] = '\u0020'; // overwrite all
                            endPosition++;
                        }
                    } else {
                        if (currentWhiteSpaceStatus.hasSpaceBefore()) {
                            currentWhiteSpaceStatus.setFirstSpaceCharPosition(-1);
                        }
                        endPosition++; // including this character
                    }
                }
            }
            if (endPosition - startPosition > 0) {
                mCharsForElement.append(ch, startPosition, endPosition - startPosition);
            }
        } else {
            /*
             * ToDo: The following will be ignored for now:
             In addition, OpenDocument Consumers shall ignore all element children ([RNG] section 5,
             Data Model) of elements defined in this specification that are strings consisting entirely of whitespace characters and
             which do not satisfy a pattern of the OpenDocument schema definition for the element.
             */
            // See http://docs.oasis-open.org/office/v1.2/os/OpenDocument-v1.2-os-part1.html#a3_18White_Space_Processing_and_EOL_Handling
            mCharsForElement.append(ch, startPosition, length);
        }
    }

    @Override
    public InputSource resolveEntity(String publicId, String systemId) throws IOException, SAXException {
        return super.resolveEntity(publicId, systemId);
    }

    /**
     * This method is being called whenever an element being the root element is
     * found, within startElement function. It considers mComponentDepth and the
     * size of the list mLastComponentPositions. It updates the position (ie.
     * list mLastComponentPositions) by adding a child or a sibling. This ONLY
     * works if mComponentDepth has already been updated before this method is
     * being called. Currently at the beginning of startElement, when realizing
     * it is a component root element.
     *
     * The only reason to update the position (mLastComponentPositions) outside
     * of this function is to handle attributes representing repeatedColumns
     * components	* (i.e. for cells & rows).
     *
     * @return the path of the current found component as Integer array
     */
    private List<Integer> updateComponentPosition() {
        List<Integer> pos = updatePosition(true);
        return pos;
    }

    /**
     * This method is being called whenever an element being the root element is
     * found, within startElement function. It considers mComponentDepth and the
     * size of the list mLastComponentPositions. It updates the position (ie.
     * list mLastComponentPositions) by adding a child or a sibling. This ONLY
     * works if mComponentDepth has already been updated before this method is
     * being called. Currently at the beginning of startElement, when realizing
     * it is a component root element.
     *
     * The only reason to update the position (mLastComponentPositions) outside
     * of this function is to handle attributes representing repeatedColumns
     * components	* (i.e. for cells & rows).
     *
     * @return the path of the current found component as Integer array
     */
    private List<Integer> updateTextPosition() {
        return updatePosition(false);
    }

    /**
     * This method is being called whenever an element being the root element is
     * found, within startElement function. It considers mComponentDepth and the
     * size of the list mLastComponentPositions. It updates the position (ie.
     * list mLastComponentPositions) by adding a child or a sibling. This ONLY
     * works if mComponentDepth has already been updated before this method is
     * being called. Currently at the beginning of startElement, when realizing
     * it is a component root element.
     *
     * The only reason to update the position (mLastComponentPositions) outside
     * of this function is to handle attributes representing repeatedColumns
     * components	* (i.e. for cells & rows).
     *
     * @return the path of the current found component as Integer array
     */
    private List<Integer> updatePosition(boolean isComponent) {

        /**
         * Used components being siblings of text, where the position is
         * determined by the text count, not by the previous component number
         */
        // take care of position handling
        // NEW COMPONENT CHILD: If the component Level is deeper than the last position
        // Actually at the first component: componentDepth start with 0, but the size with 1
        if (mComponentDepth == mLastComponentPositions.size()) {
            // addChild a new level
            mLastComponentPositions.add(mComponentDepth, 0);

            // NEW SIBLING: If the depth is "equal" (ie. lower) than addChild a sibling
        } else if (mComponentDepth == mLastComponentPositions.size() - 1) {
            int positionUpdate;
            if (mCurrentComponent instanceof TextContainer) {
                // increment the last position number as a sibling was added
                positionUpdate = mCurrentComponent.size();
            } else if (mCurrentComponent instanceof Cell) {
                positionUpdate = mLastComponentPositions.get(mComponentDepth) + 1;

//				positionUpdate = mCurrentComponent.repetition();
            } else {
                positionUpdate = mLastComponentPositions.get(mComponentDepth) + 1;
            }
            mLastComponentPositions.set(mComponentDepth, positionUpdate);
// FINISHED COMPONENT - REMOVE DEPTH - If a component was closed and the position needds to be added
        } else if (mComponentDepth < mLastComponentPositions.size() - 1) {
            // remove the last position and addChild a new one
            mLastComponentPositions.removeLast();
            updatePosition(isComponent);
        } else {
            LOG.warning("Houston, we have a problem..");
        }
        //ToDo: Do I need a new LIST for every component? Or may I addChild a position to the component?
        return new LinkedList<Integer>(mLastComponentPositions);
        //ToDo: Below a bad idea?
        //return Collections.unmodifiableList(mLastComponentPositions);
    }

    /**
     * @return the path of the current found component as Integer array
     */
    private List<Integer> getTextPosition() {
        // The text is one level further down
        mComponentDepth++;
        List<Integer> position = updateTextPosition();
        mComponentDepth--;
        return position;
    }

    /**
     *
     * @return a property value.
     */
//	private String getPropertyValue(OdfStyleProperty prop) {
//		String value = null;
//
//		OdfStylePropertiesBase properties = getPropertiesElement(prop.getPropertySet());
//		if (properties != null) {
//			if (properties.hasAttributeNS(prop.getName().getUri(), prop.getName().getLocalName())) {
//				return properties.getOdfAttribute(prop.getName()).getValue();
//			}
//		}
//
//		OdfStyleBase parent = getParentStyle();
//		if (parent != null) {
//			return parent.getProperty(prop);
//		}
//
//		return value;
//	}
    public static String getProperty(OdfStyleProperty prop, OdfStylableElement element) {
        String value = null;
        OdfStyle style = element.getAutomaticStyle();
        if (style == null) {
            element.getOrCreateUnqiueAutomaticStyle();
            style = element.getAutomaticStyle();
            if (style == null) {
                style = element.getDocumentStyle();
            }
        }
        if (style != null) {
            value = style.getProperty(prop);
        }
        return value;
    }

    private static Length getPropertyLength(OdfStyleProperty prop, OdfStylableElement element) {
        Length length = null;
        String propValue = getProperty(prop, element);
        if (propValue != null && !propValue.isEmpty()) {
            length = new Length(propValue);
        }
        return length;
    }

    /**
     * This class compares alphanumeric Strings consistent across multiple OS
     * and JDK. Required to sort Styles within a List to have a consistent order
     * of createStyle operations in regression tests.
     */
    private static class AlphanumComparator implements Comparator {

        public int compare(Object o1, Object o2) {
            if (!(o1 instanceof String)) {
                o1 = o1.toString();
            }

            if (!(o2 instanceof String)) {
                o2 = o2.toString();
            }
            String s1 = (String) o1;
            String s2 = (String) o2;

            int thisMarker = 0;
            int thatMarker = 0;
            int s1Length = s1.length();
            int s2Length = s2.length();

            while (thisMarker < s1Length && thatMarker < s2Length) {
                String thisPart = getPart(s1, s1Length, thisMarker);
                thisMarker += thisPart.length();

                String thatPart = getPart(s2, s2Length, thatMarker);
                thatMarker += thatPart.length();

                // If both parts contain numeric characters, sort them numerically
                int result = 0;
                if (isDigit(thisPart.charAt(0)) && isDigit(thatPart.charAt(0))) {
                    // Simple parts comparison by length.
                    int thisPartLength = thisPart.length();
                    result = thisPartLength - thatPart.length();
                    // If equal, the first different number counts
                    if (result == 0) {
                        for (int i = 0; i < thisPartLength; i++) {
                            result = thisPart.charAt(i) - thatPart.charAt(i);
                            if (result != 0) {
                                return result;
                            }
                        }
                    }
                } else {
                    result = thisPart.compareTo(thatPart);
                }

                if (result != 0) {
                    return result;
                }
            }
            return s1Length - s2Length;
        }

        /**
         * Length of string for improved efficiency (only calculated once) *
         */
        private String getPart(String s, int slength, int marker) {
            StringBuilder part = new StringBuilder();
            char c = s.charAt(marker);
            part.append(c);
            marker++;
            if (isDigit(c)) {
                while (marker < slength) {
                    c = s.charAt(marker);
                    if (!isDigit(c)) {
                        break;
                    }
                    part.append(c);
                    marker++;
                }
            } else {
                while (marker < slength) {
                    c = s.charAt(marker);
                    if (isDigit(c)) {
                        break;
                    }
                    part.append(c);
                    marker++;
                }
            }
            return part.toString();
        }

        private boolean isDigit(char ch) {
            return ch >= 48 && ch <= 57;
        }
    }
    CachedTable mCachedTableOps = null;

    // based on document position the tables and subtables are being flushed after each end
    HashMap<List<Integer>, CachedTable> mAllTables = null;
    private ArrayList<CachedTable> mTableStack;
    boolean mWithinTable = false;
    private int mNumberOfNestedTables = 0;

    class CachedInnerTableOperation {

        String mComponentType;
        List<Integer> mStart;
        Map<String, Object> mHardFormattingProperties;
        Object[] mComponentProperties;
        // required to track repeatedColumns rows for spreadsheets..
        int mCurrentRowNo = 0;
        int mCurrentRowRepetition = 0;

        CachedInnerTableOperation(String componentType, List<Integer> start, Map<String, Object> styleFormatting, Object... componentProperties) {
            mComponentType = componentType;
            mStart = start;
            mHardFormattingProperties = styleFormatting;
            mComponentProperties = componentProperties;
        }
    }

    class CachedTable extends ArrayList<CachedInnerTableOperation> {

        // START *** SPREADSHEET PROPERTIES
        // required to track repeatedColumns rows for spreadsheets..
        int mPreviousRepeatedRows = 0;
        int mRepeatedRows = 0;
        // spreadsheet range, representing a subset of a single row.
        // added cell data (content & format) neglecting starting/ending empty cells and within the row any adjacent empty cells above MAX_REPEATING_EMPTY_CELLS
        JSONArray mCurrentRange;
        // only cells with content should trigger an operatoin
        int mFirstContentCellNo = -1;
        // equal cell contents can be given out compressed
        int mFirstEqualCellNo = -1;
        int mCurrentColumnNo = 0;
        // after there had been content, three following empty cells are being counted, before a new insert operation is being triggered
        int mEmptyCellCount = 0;
        int mFirstEmptyCell = -1;
        // the JSONArray does not allow to delete empty cells, therefore they have to be counted
        JSONObject mPreviousCell = null;
        private int mCellRepetition = 1;
        // get the sheetNo
        Integer mSheetNo;
        Integer mFirstRow;
        Integer mLastRow;
        ArrayList<CachedInnerTableOperation> mCachedTableContentOps = null;
        // END *** SPREADSHEET PROPERTIES
        int mColumnCount = 0;
        int mRowCount = 0;
        boolean mWhiteSpaceOnly = true;
        int mCellCount = 0;
        boolean mIsTooLarge = false;
        List<Integer> mTableGrid;
        List<Integer> mStart;
        String tableName;

        CachedTable() {
            super();
        }
    }

    private void cacheTableOperation(String componentType, List<Integer> start, Map<String, Object> hardFormattingProperties, Object... componentProperties) {
        LinkedList<Integer> position = null;
        if (start != null) {
            position = new LinkedList<Integer>(start);
        }
        if (!mIsSpreadsheet && componentType.equals(CELLS)) { // does not effect a spreadsheet
            mCachedTableOps.mCellCount++;
            if (mCachedTableOps.mCellCount > mMaxTableCellCount) {
                mCachedTableOps.mIsTooLarge = true;
            }
        } else if (componentType.equals(ROWS)) { // does not effect a spreadsheet
            if (mCachedTableOps.mCachedTableContentOps == null) {
                mCachedTableOps.mCachedTableContentOps = new ArrayList<CachedInnerTableOperation>();
            }
            mCachedTableOps.mRowCount++;
            if (mCachedTableOps.mRowCount > mMaxTableRowCount) {
                mCachedTableOps.mIsTooLarge = true;
            }
        } else if (mIsSpreadsheet && componentType.equals(COLUMNS)) { // in a text document the columns are defined by an array of interger (widths)
            mCachedTableOps.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
        } else if (componentType.equals(RANGE)) { // spreadsheet only
            mCachedTableOps.mCachedTableContentOps.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
        } else if (componentType.equals(TABLE)) { // all formats (Text & Spreadsheet atm)
            CachedTable newCachedTable = startTableSizeEvaluation(position, (List<Integer>) componentProperties[0]);
            if (newCachedTable.mTableGrid != null) {
                if (newCachedTable.mTableGrid.size() > mMaxTableColumnCount) {
                    newCachedTable.mIsTooLarge = true;
                    // adding the table now, as it would not be below as already too large..
                } else {
                    newCachedTable.mColumnCount = newCachedTable.mTableGrid.size();
                }
            }
            // if it is the first root table
            if (mCachedTableOps == null) {
                // only for the root table the table itself will be added at the beginning
                mCachedTableOps = newCachedTable;
                mCachedTableOps.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
            } else { // as subtable the table will be added twice:
                // once for the parent as notifier
                mCachedTableOps.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
                mCachedTableOps = newCachedTable;
                // once for the child to create
                mCachedTableOps.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
            }
        }
        if (!mCachedTableOps.mIsTooLarge && !componentType.equals(TABLE) && !componentType.equals(COLUMNS) && !componentType.equals(RANGE)) {
            mCachedTableOps.add(new CachedInnerTableOperation(componentType, position, hardFormattingProperties, componentProperties));
        }
    }

    /**
     * According to user run-time configuration only tables of a certain size
     * are allowed to be created. Tables exceeding the limit are being shown by
     * a replacement object, otherwise the client performance might not be
     * sufficient.
     */
    private CachedTable startTableSizeEvaluation(List<Integer> position, List<Integer> tableGrid) {
        if (mNumberOfNestedTables == 0) {
            mWithinTable = true;
        }
        CachedTable newCachedTable = new CachedTable();

        if (mAllTables == null) {
            mAllTables = new HashMap<List<Integer>, CachedTable>();
        }
        if (mTableStack == null) {
            mTableStack = new ArrayList<CachedTable>();
        }
        newCachedTable.mTableGrid = tableGrid;
        mTableStack.add(mNumberOfNestedTables, newCachedTable);
        mAllTables.put(position, newCachedTable);
        mNumberOfNestedTables++;
        return newCachedTable;
    }

    /**
     * According to user run-time configuration only tables of a certain size
     * are allowed to be created. Tables exceeding the limit are being shown by
     * a replacement object, otherwise the client performance might not be
     * sufficient.
     *
     * As the limit is being checked on sub table level, the complete table have
     * to be parsed before giving green light for any table. On the opposite, if
     * a subtable is already too large, it can be neglected collecting
     * operations for that subtable.
     */
    private void endTableSizeEvaluation() {
        mNumberOfNestedTables--;
        if (mNumberOfNestedTables == 0) {
            flushTableOperations(mCachedTableOps, true);
            mCachedTableOps = null;            
            mWithinTable = false;
            mAllTables.clear();
        } else if (mNumberOfNestedTables > 0) {
            // when leaving a table, continue with the parent table
            mCachedTableOps = mTableStack.get(mNumberOfNestedTables - 1);
        } else { // below zero might appear, when table had started in blocked area
            mNumberOfNestedTables = 0;
            mCachedTableOps = null;
            mWithinTable = false;
            mAllTables = null;
        }
    }

    private void flushTableOperations(CachedTable currentTable, boolean isStartOfTable) {
        ListIterator<CachedInnerTableOperation> tableOperationIterator = currentTable.listIterator();
        while (tableOperationIterator.hasNext()) {
            CachedInnerTableOperation operation = (CachedInnerTableOperation) tableOperationIterator.next();
            if (operation.mComponentType.equals(TABLE)) {
                if (isStartOfTable) {
                    isStartOfTable = false;
                    if (currentTable.mIsTooLarge) {
                        // replacement table
                        mJsonOperationProducer.addExceededTable(operation.mStart, ((List) operation.mComponentProperties[0]).size(), currentTable.mRowCount, (List<Integer>) operation.mComponentProperties[0]);
                        break;
                    } else {
                        if (currentTable.mRowCount > mMaxTableRowCount || currentTable.mColumnCount > mMaxTableColumnCount || currentTable.mCellCount > mMaxTableCellCount) {
                            mJsonOperationProducer.addExceededTable(operation.mStart, ((List) operation.mComponentProperties[0]).size(), currentTable.mRowCount, (List<Integer>) operation.mComponentProperties[0]);
                            break;
                        } else {
                            // the last parameter are: mColumnRelWidths, mTableName, mIsTableVisible);
                            mJsonOperationProducer.addTable(operation.mStart, operation.mHardFormattingProperties, (List<Integer>) operation.mComponentProperties[0], (String) operation.mComponentProperties[1], mIsSpreadsheet);
                        }
                    }
                } else {
                    flushTableOperations(mAllTables.get(operation.mStart), true);
                }
            } else if (operation.mComponentType.equals(TEXT)) {
                mJsonOperationProducer.addText(operation.mStart, (String) operation.mComponentProperties[0]);
            } else if (operation.mComponentType.equals(ATTRIBUTES)) {
                mJsonOperationProducer.setAttributes(operation.mStart, (List<Integer>) operation.mComponentProperties[0], operation.mHardFormattingProperties);
            } else if (mIsSpreadsheet && operation.mComponentType.equals(ROWS)) { // spreadsheet function
                // instead of counting the correct rows, the repeatedColumns offset will be added to the cell position
                int firstRow = (Integer) operation.mComponentProperties[0];
                int lastRow = (Integer) operation.mComponentProperties[1];
                int repeatedRowOffset = (Integer) operation.mComponentProperties[2];
                mJsonOperationProducer.formatRows(operation.mStart, operation.mHardFormattingProperties, firstRow, lastRow, repeatedRowOffset);

            } else if (mIsSpreadsheet && operation.mComponentType.equals(COLUMNS)) { // spreadsheet function
                mJsonOperationProducer.formatColumns(operation.mStart, operation.mHardFormattingProperties, (Integer) operation.mComponentProperties[0], (Integer) operation.mComponentProperties[1]);
            } else if (operation.mComponentType.equals(SHAPE)) {
                mJsonOperationProducer.addShape(operation.mStart, operation.mHardFormattingProperties);
            } else if (operation.mComponentType.equals(IMAGE)) {
                mJsonOperationProducer.addImage(operation.mStart, operation.mHardFormattingProperties);
            } else if (operation.mComponentType.equals(FIELD)) {
                mJsonOperationProducer.insertField(operation.mStart, (String) operation.mComponentProperties[0], (String) operation.mComponentProperties[1]);
            } else {
                mJsonOperationProducer.add(operation.mComponentType, operation.mStart, operation.mHardFormattingProperties);
            }
        }
        if (mIsSpreadsheet) {
            tableOperationIterator = currentTable.mCachedTableContentOps.listIterator();
            while (tableOperationIterator.hasNext()) {
                CachedInnerTableOperation operation = (CachedInnerTableOperation) tableOperationIterator.next();
                if (operation.mComponentType.equals(RANGE)) {
                    // Documentation: addRange(int mSheetNo, Integer mFirstRow, Integer mLastRow, int repeatedRowOffset, int firstContentCell, int mCurrentCellNo, JSONObject repeatedCell, List singleRow)
                    mJsonOperationProducer.addRange((Integer) operation.mComponentProperties[0], (Integer) operation.mComponentProperties[1], (Integer) operation.mComponentProperties[2], (Integer) operation.mComponentProperties[3], (Integer) operation.mComponentProperties[4], (Integer) operation.mComponentProperties[5], (JSONObject) operation.mComponentProperties[6], (JSONArray) operation.mComponentProperties[7], (Boolean) operation.mComponentProperties[8]);
                }
            }
        }
    }

    /**
     * Optimizes the operations of spreadsheet cells neglecting
     * starting/trailing empty cells and for cells with content or style
     * bundling similar cells to single operations.
     *
     * Repeated rows are automatically a range.
     *
     * There are three pointer (variables), that are updated during parsing the
     * spreadsheet: mCurrentCellNo is the actual column number
     * mFirstContentCellNo is mFirstEqualCellNo is set to the first cell to be
     * written, after a cell was written out or an empty precessor
     *
     * ToDo: Refactoring - As soon every component got its own parser, the
     * mCachedTableOps. have to be replaced by the Context of the component
     */
    private void evaluateSimilarCells(final List<Integer> start, JSONObject currentCell, boolean closingRow) {
        // An Operation will always be triggered in the end of the function
        boolean triggerOperation = false;

        // every repeatedColumns row will result into a fillRange operation
        int previousContentRepetition = 1;

        // if the previous cells are equal
        if (mCachedTableOps.mFirstEqualCellNo > -1) {
            previousContentRepetition = mCachedTableOps.mCurrentColumnNo - mCachedTableOps.mFirstEqualCellNo;
        }

        boolean isRepeatedRow = mCachedTableOps.mLastRow != null && !mCachedTableOps.mFirstRow.equals(mCachedTableOps.mLastRow);

        // do not trigger the operation if the spreadsheetRow is null and its only member is null
        if (mCachedTableOps.mSheetNo == null && start != null) {
            // we have a cell position and require the two above (first parent row, afterwards cell) => already -2
            // and an additional - 1 as size of 1 would result in zero position ==> finally -3
            mCachedTableOps.mSheetNo = start.get(start.size() - 3);
        }
        // ** There are four variations for previous/current cell we have to check:
        // 1) Current Content Cell, Previous Content empty
        if (currentCell != null && mCachedTableOps.mPreviousCell != null) {
            // if the two cells are NOT the same
            if (!currentCell.equals(mCachedTableOps.mPreviousCell)) {
                if (previousContentRepetition > MIN_REPEATING_CONTENT_CELLS) {
                    triggerOperation = true;
                } else {
                    if (mCachedTableOps.mCurrentRange == null) {
                        mCachedTableOps.mCurrentRange = new JSONArray();
                    }
                    // Resolving repetition, explicitly adding cells to the range
                    for (int i = 0; mCachedTableOps.mCurrentColumnNo - mCachedTableOps.mFirstEqualCellNo > i; i++) {
                        mCachedTableOps.mCurrentRange.put(mCachedTableOps.mPreviousCell);
                    }
                    // if the row is being repeatedColumns, there are always vertical spans (fill same content multiple times
                    if (mCachedTableOps.mLastRow - mCachedTableOps.mFirstRow > 0) {
                        triggerOperation = true;
                    }
                    // there is an upcoming fill operation, the previous content has to be flushed
                    if (mCachedTableOps.mCellRepetition > MIN_REPEATING_CONTENT_CELLS) {
                        triggerOperation = true;
                    }
                    mCachedTableOps.mFirstEqualCellNo = mCachedTableOps.mCurrentColumnNo;
                }
            }
            // 2) Current Content Cell, Previous Empty Cell (never have saved anything)
        } else if (currentCell != null && mCachedTableOps.mPreviousCell == null) { // && mCachedTableOps.mCurrentRange == null
            mCachedTableOps.mFirstEqualCellNo = mCachedTableOps.mCurrentColumnNo;
            if (mCachedTableOps.mFirstContentCellNo == -1) {
                // reset the empty cell counter - if previous was empty
                mCachedTableOps.mFirstContentCellNo = mCachedTableOps.mCurrentColumnNo;
                mCachedTableOps.mEmptyCellCount = 0;
            } else {
                if (mCachedTableOps.mCellRepetition > MIN_REPEATING_CONTENT_CELLS) {
                    triggerOperation = true;
                } else {
                    if (mCachedTableOps.mCurrentRange == null) {
                        mCachedTableOps.mCurrentRange = new JSONArray();
                    }
                    for (int i = 0; mCachedTableOps.mEmptyCellCount > i; i++) {
                        mCachedTableOps.mCurrentRange.put(JSONObject.NULL);
                    }
                    mCachedTableOps.mEmptyCellCount = 0;
                }
            }
            // 3) Content Cell empty, Previo Cell full
        } else if (currentCell == null && mCachedTableOps.mPreviousCell != null) {
            mCachedTableOps.mEmptyCellCount += mCachedTableOps.mCellRepetition;
            // as there had been previously content
            // check if it was repeating content
            if (previousContentRepetition > MIN_REPEATING_CONTENT_CELLS) {
                triggerOperation = true;
            } else {
                if (mCachedTableOps.mCurrentRange == null) {
                    mCachedTableOps.mCurrentRange = new JSONArray();
                }
                // save the previous cell for later compressed output
                for (int i = 0; mCachedTableOps.mCurrentColumnNo - mCachedTableOps.mFirstEqualCellNo > i; i++) {
                    mCachedTableOps.mCurrentRange.put(mCachedTableOps.mPreviousCell);
                }
                // if the row is being repeatedColumns, there are always vertical spans (fill same content multiple times
                if (mCachedTableOps.mLastRow - mCachedTableOps.mFirstRow > 0) {
                    triggerOperation = true;
                }
            }
            mCachedTableOps.mFirstEqualCellNo = -1; // if there was previously repeating content cells
            // 4) Both are null
        } else if (currentCell == null && mCachedTableOps.mPreviousCell == null) {
            // note that an empty cell was passed
            mCachedTableOps.mEmptyCellCount += mCachedTableOps.mCellRepetition;
            // if this is the first empty cell
            if (mCachedTableOps.mFirstEmptyCell == -1) {
                // remember when it started
                mCachedTableOps.mFirstEmptyCell = mCachedTableOps.mCurrentColumnNo;

                // else check if the maximum repeatedColumns empty cells was reached and existing content has to be dispatched as an operation
            } else if (mCachedTableOps.mFirstContentCellNo != -1 && MAX_REPEATING_EMPTY_CELLS > mCachedTableOps.mEmptyCellCount) {
                triggerOperation = true;
            }
        }

        // if there are content cells
        if (closingRow && mCachedTableOps.mFirstContentCellNo > -1 || triggerOperation) {
            // WRITING WHITESPACE TO ROW
            // if the last cell used content, but there was previous whitespace, the whitespace has to be explicitly set
            if (mCachedTableOps.mEmptyCellCount > 0 && currentCell != null) {
                for (int i = 0; mCachedTableOps.mEmptyCellCount > i; i++) {
                    if (mCachedTableOps.mCurrentRange == null) {
                        mCachedTableOps.mCurrentRange = new JSONArray();
                    }
                    mCachedTableOps.mCurrentRange.put(JSONObject.NULL);
                }
                mCachedTableOps.mEmptyCellCount = 0;
            }
            // WRITING CELL TO ROW
            // if content to flush exist and the operation was triggered
            //  OR there is horizontal repeatedColumns content
            //  OR there is vertical repeatedColumns content
            if (mCachedTableOps.mCurrentRange != null || previousContentRepetition > MIN_REPEATING_CONTENT_CELLS || isRepeatedRow) {
                cacheTableOperation(RANGE, null, null, mCachedTableOps.mSheetNo, mCachedTableOps.mFirstRow, mCachedTableOps.mLastRow, mCachedTableOps.mPreviousRepeatedRows, mCachedTableOps.mFirstContentCellNo, previousContentRepetition, mCachedTableOps.mPreviousCell, mCachedTableOps.mCurrentRange, previousContentRepetition > MIN_REPEATING_CONTENT_CELLS);
                // REMOVE ME: mJsonOperationProducer.addRange(mCachedTableOps.mSheetNo, mCachedTableOps.mFirstRow, mCachedTableOps.mLastRow, mCachedTableOps.mPreviousRepeatedRows, mCachedTableOps.mFirstContentCellNo, previousContentRepetition, mCachedTableOps.mPreviousCell, mCachedTableOps.mCurrentRange, previousContentRepetition > MIN_REPEATING_CONTENT_CELLS);
            }
            // if a fill sufficent repeating is now after a content, the previous content was flushed
            if (mCachedTableOps.mFirstEqualCellNo == mCachedTableOps.mCurrentColumnNo) {
                // but still a content and repeating content exits
                mCachedTableOps.mFirstContentCellNo = mCachedTableOps.mFirstEqualCellNo;
            } else {
                if (currentCell != null) {
                    mCachedTableOps.mFirstContentCellNo = mCachedTableOps.mCurrentColumnNo;
                    mCachedTableOps.mFirstEqualCellNo = mCachedTableOps.mCurrentColumnNo;
                } else {
                    mCachedTableOps.mFirstContentCellNo = -1;
                    mCachedTableOps.mFirstEqualCellNo = -1;
                }
            }
            mCachedTableOps.mCurrentRange = null;
        }
        if (!closingRow) {
            // Making the current cell the previous for next round
            mCachedTableOps.mPreviousCell = currentCell;
            mCachedTableOps.mCurrentColumnNo += mCachedTableOps.mCellRepetition;
            mCachedTableOps.mCellRepetition = 1;
        } else {
            // after the end of a row reset all values
            mCachedTableOps.mSheetNo = null;
            mCachedTableOps.mCurrentRange = null;
            mCachedTableOps.mPreviousCell = null;
            mCachedTableOps.mFirstContentCellNo = -1;
            mCachedTableOps.mFirstEqualCellNo = -1;
            mCachedTableOps.mCurrentColumnNo = 0;
            mCachedTableOps.mEmptyCellCount = 0;
            mCachedTableOps.mFirstEmptyCell = -1;
            mCachedTableOps.mCellRepetition = 1;
        }
    }

    static void stashColumnWidths(TableTableElement tableElement) {
        List<TableTableColumnElement> existingColumnList = getTableColumnElements(tableElement, new LinkedList<TableTableColumnElement>());
        List<Integer> tableColumWidths = OdfFileSaxHandler.collectColumnWidths(tableElement, existingColumnList);
        tableElement.pushTableGrid(tableColumWidths);
    }

    static List<Integer> collectColumnWidths(TableTableElement tableElement, List<TableTableColumnElement> columns) {
        boolean hasRelColumnWidth = false;
        boolean hasAbsColumnWidth = false;
        boolean hasColumnWithoutWidth = false;
        List<Integer> columnRelWidths = new ArrayList();
        for (TableTableColumnElement column : columns) {
            if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "style-name")) {
                Length tableWidth = getPropertyLength(StyleTablePropertiesElement.Width, tableElement);

                int repeatedColumns = 1;
                if (column.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated")) {
                    repeatedColumns = Integer.parseInt(column.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "number-columns-repeated"));
                }

                String columnRelWidth = getProperty(StyleTableColumnPropertiesElement.RelColumnWidth, column);

                // it is being assumed, when the columnRelWidth is once set, it is always set
                if (columnRelWidth != null && !columnRelWidth.isEmpty()) {
                    hasRelColumnWidth = true;
                    if (hasAbsColumnWidth) {
                        LOG.warning("******* BEWARE: Absolute and relative width are not supposed to be mixed!! ***********");
                    }
                    columnRelWidth = columnRelWidth.substring(0, columnRelWidth.indexOf('*'));
                    Integer relWidth = Integer.parseInt(columnRelWidth);
                    for (int i = 0; i < repeatedColumns; i++) {
                        columnRelWidths.add(relWidth);
                    }
                } else { // if there is no relative column width
                    if (hasRelColumnWidth) {
                        LOG.warning("******* BEWARE: Absolute and relative width are not supposed to be mixed!! ***********");
                    }

                    Length columnWidth = getPropertyLength(StyleTableColumnPropertiesElement.ColumnWidth, column);
                    // there can be only table width and ..
                    if (tableWidth != null) {
                        // columnwidth, with a single one missing
                        if (columnWidth != null) {
                            hasAbsColumnWidth = true;
                            int widthFactor = (int) Math.round((columnWidth.getMillimeters() * 100) / tableWidth.getMillimeters());
                            for (int i = 0; i < repeatedColumns; i++) {
                                columnRelWidths.add(widthFactor);
                            }
                        } else {
                            if (hasColumnWithoutWidth) {
                                LOG.warning("******* BEWARE: Two columns without width and no column width are not expected!! ***********");
                            }
                            hasColumnWithoutWidth = true;
                        }
                        // if the table is not set, it will always be unset..
                    } else {
                        if (columnWidth != null) {
                            hasAbsColumnWidth = true;
                            int widthFactor = (int) Math.round((columnWidth.getMicrometer() * 10));
                            for (int i = 0; i < repeatedColumns; i++) {
                                columnRelWidths.add(widthFactor);
                            }
                        } else {
                            LOG.warning("******* BEWARE: Two columns without width and no column width are not expected!! ***********");
                        }
                    }
                }
            }
        }
        return columnRelWidths;
    }

    /**
     * Returns all TableTableColumn descendants that exist within the
     * tableElement, even within groups, columns and header elements
     */
    static List<TableTableColumnElement> getTableColumnElements(Element parent, List columns) {
        NodeList children = parent.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);
            if (child instanceof Element) {
                if (child instanceof TableTableColumnElement) {
                    columns.add((TableTableColumnElement) child);
                } else if (child instanceof TableTableColumnGroupElement
                    || child instanceof TableTableHeaderColumnsElement
                    || child instanceof TableTableColumnsElement) {
                    columns = getTableColumnElements((Element) child, columns);
                } else if (child instanceof TableTableRowGroupElement
                    || child instanceof TableTableHeaderRowsElement
                    || child instanceof TableTableRowElement
                    || child instanceof TableTableRowsElement) {
                    break;
                }
            }
        }
        return columns;
    }
}
