/*
 *  Copyright 2010-2013, Plutext Pty Ltd.
 *
 *  This file is part of xlsx4j, a component of docx4j.

    docx4j is licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.

    You may obtain a copy of the License at

        http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.

 */
package org.xlsx4j.sml;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;

import org.apache.commons.logging.Log;
import org.docx4j.XmlUtils;
import org.jvnet.jaxb2_commons.ppp.Child;
import org.xlsx4j.jaxb.Context;

import com.openexchange.log.LogFactory;


/**
 * <p>Java class for CT_SheetData complex type.
 *
 * <p>The following schema fragment specifies the expected content contained within this class.
 *
 * <pre>
 * &lt;complexType name="CT_SheetData">
 *   &lt;complexContent>
 *     &lt;restriction base="{http://www.w3.org/2001/XMLSchema}anyType">
 *       &lt;sequence>
 *         &lt;element name="row" type="{http://schemas.openxmlformats.org/spreadsheetml/2006/main}CT_Row" maxOccurs="unbounded" minOccurs="0"/>
 *       &lt;/sequence>
 *     &lt;/restriction>
 *   &lt;/complexContent>
 * &lt;/complexType>
 * </pre>
 *
 *
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "CT_SheetData", propOrder = {
    "row"
})
@XmlRootElement(name = "sheetData")
public class SheetData implements Child
{
    @XmlTransient
    protected static Log log = LogFactory.getLog(SheetData.class);
    protected List<Row> row;
    @XmlTransient
    private Object parent;
    @XmlTransient
    private TreeMap<Integer, Row> rows_impl;
    @XmlTransient
    private Long maxSi;

    /**
     * Gets the value of the row property.
     *
     * <p>
     * This accessor method returns a reference to the live list,
     * not a snapshot. Therefore any modification you make to the
     * returned list will be present inside the JAXB object.
     * This is why there is not a <CODE>set</CODE> method for the row property.
     *
     * <p>
     * For example, to add a new item, do as follows:
     * <pre>
     *    getRow().add(newItem);
     * </pre>
     *
     *
     * <p>
     * Objects of the following type(s) are allowed in the list
     * {@link Row }
     *
     *
     */
    // this method is now private to ensure that Map<Long, Row> stays up to date..
    private List<Row> getListArray() {
    	
        if (row==null) {
        	if(rows_impl!=null) {
        		row = new ArrayList<Row>(rows_impl.values());
        	}
        	else {
        		row = new ArrayList<Row>();
        	}
        	rows_impl = null;
        }
        return row;
    }

    // the List<Row> may include multiple Rows addressing the same row index, so our Map<Long, Row>
    // takes care only of the first (TODO merging of rows with the same index)
    // if a row is not having an explicit index its index is always previous rowIndex + 1..
    // if a rowIndex is greater than the index of the previous row we have an invalid document
    private Map<Integer, Row> getTreeMap() {
        if(rows_impl==null) {
            rows_impl = new TreeMap<Integer, Row>();
            if(row!=null) {
            	for(Row r:row) {
            		rows_impl.put(r.getRow(), r);
            	}
            	row = null;
            }
        }
        return rows_impl;
    }

    public Row getRow(int index, boolean forceCreate) {

    	final Map<Integer, Row> treeMap = getTreeMap();
        Row r = treeMap.get(index);
        if(forceCreate&&r==null) {

            // creating a new row...
        	org.docx4j.jaxb.Context.checkMemory();

            r = Context.getsmlObjectFactory().createRow();
            r.setParent(this);
            r.setRow(index);
            treeMap.put(index, r);
        }
        return r;
    }

    private static boolean equals(CTBorderPr border1, CTBorderPr border2) {
        if(border1==border2)
            return true;
        if(border1==null||border2==null)
            return false;
        if(border1.getStyle()!=border2.getStyle())
            return false;
        CTColor col1 = border1.getColor();
        CTColor col2 = border2.getColor();
        if(col1==col2)
            return true;
        if(col1==null||col2==null)
            return false;
        if(col1.isAuto()!=null&&col2.isAuto()!=null) {
            if(col1.isAuto().booleanValue()!=col2.isAuto().booleanValue())
                return false;
        }
        else if (col1.isAuto()!=col2.isAuto())
            return false;
        if(col1.getIndexed()!=null&&col2.getIndexed()!=null) {
            if(col1.getIndexed().longValue()!=col2.getIndexed().longValue())
                return false;
        }
        else if(col1.getIndexed()!=col2.getIndexed())
            return false;
        if(!Arrays.equals(col1.getRgb(), col2.getRgb()))
            return false;
        if(col1.getTheme()!=null&&col2.getTheme()!=null) {
            if(col1.getTheme().longValue()!=col2.getTheme().longValue())
                return false;
        }
        else if(col1.getTheme()!=col2.getTheme())
            return false;

        if(col1.tint!=null&&col2.tint!=null) {
            if(col1.tint.doubleValue()!=col2.tint.doubleValue())
                return false;
        }
        else if(col1.getTint()!=col2.getTint())
            return false;
        return true;
    }

    private static CTBorderPr createClone(CTBorderPr border) {
        if(border==null)
            return null;
        CTBorderPr newBorder = Context.getsmlObjectFactory().createCTBorderPr();
        newBorder.setStyle(border.getStyle());

        CTColor color = border.getColor();
        if(color!=null) {
            CTColor newColor = Context.getsmlObjectFactory().createCTColor();
            newColor.setAuto(color.isAuto());
            newColor.setIndexed(color.getIndexed());
            if(color.getRgb()!=null) {
                newColor.setRgb(Arrays.copyOf(color.getRgb(), color.getRgb().length));
            }
            newColor.setTheme(color.getTheme());
            newColor.setTint(color.getTint());
        }
        return newBorder;
    }

    public static void ApplyBorderStyle(CTStylesheet stylesheet, Cell cellClone, Cell nextRowCell) {
        final CTXf ctXfPrev = stylesheet.getCellXfByIndex(cellClone.getS());
        // is prevRow using a hard border ?
        if(ctXfPrev!=null&&ctXfPrev.isApplyBorder()!=null&&ctXfPrev.isApplyBorder()&&ctXfPrev.getBorderId()!=null) {
            long borderId = ctXfPrev.getBorderId().longValue();
            long newBorderId = 0;   // this is our new borderId if the cell of the nextRow is not having hard border attributes
            if(nextRowCell!=null) {
                final CTXf ctXfNext = stylesheet.getCellXfByIndex(nextRowCell.getS());
                if(ctXfNext!=null&&ctXfNext.isApplyBorder()!=null&&ctXfNext.isApplyBorder()&&ctXfNext.getBorderId()!=null) {
                    long nextBorderId = ctXfNext.getBorderId().longValue();
                    if(nextBorderId==borderId) {
                        newBorderId = borderId;     // no special border handling...
                    }
                    else {
                        CTBorder prevBorder = stylesheet.getBorderByIndex(borderId);
                        CTBorder nextBorder = stylesheet.getBorderByIndex(nextBorderId);
                        if(prevBorder!=null&&nextBorder!=null) {
                            // we have to create a new border...
                            CTBorder newBorder = Context.getsmlObjectFactory().createCTBorder();
                            if(equals(prevBorder.getBottom(), nextBorder.getBottom()))
                                newBorder.setBottom(createClone(prevBorder.getBottom()));
                            if(equals(prevBorder.getLeft(), nextBorder.getLeft()))
                                newBorder.setLeft(createClone(prevBorder.getLeft()));
                            if(equals(prevBorder.getTop(), nextBorder.getTop()))
                                newBorder.setTop(createClone(prevBorder.getTop()));
                            if(equals(prevBorder.getRight(), nextBorder.getRight()))
                                newBorder.setRight(createClone(prevBorder.getRight()));
                            if(equals(prevBorder.getDiagonal(), nextBorder.getDiagonal()))
                                newBorder.setDiagonal(createClone(prevBorder.getDiagonal()));
                            if(equals(prevBorder.getHorizontal(), nextBorder.getHorizontal()))
                                newBorder.setHorizontal(createClone(prevBorder.getHorizontal()));
                            if(equals(prevBorder.getVertical(), nextBorder.getVertical()))
                                newBorder.setVertical(createClone(prevBorder.getVertical()));
                            newBorder.setDiagonalDown(prevBorder.isDiagonalDown());
                            newBorder.setDiagonalUp(prevBorder.isDiagonalUp());

                            // this is our new border id...
                            newBorderId = stylesheet.getOrApplyBorder(newBorder);
                        }
                    }
                }
            }
            if(borderId!=newBorderId) {
                CTXf detachedXf = XmlUtils.deepCopy(ctXfPrev, Context.getJcSML());
                detachedXf.setApplyBorder(newBorderId!=0);
                detachedXf.setBorderId(newBorderId);
                long index = stylesheet.getOrApplyCellXf(detachedXf);
                if(index>0) {
                    cellClone.setS(index);
                }
            }
        }
    }

    public void insertRows(CTStylesheet stylesheet, int start, int insertCount) {

    	final List<Row> rows = getListArray();

        int lastRowNumber = 0;
        int i, insertIndex;

        Row _r = null;
        for(insertIndex=0; insertIndex<rows.size(); insertIndex++) {
            _r = rows.get(insertIndex);
            if(_r.getRow()>=start) {
                break;
            }
            lastRowNumber = _r.getRow();
        }

        Row prevRow = null;
        // check if we can inherit row properties from the preceding row
        if(insertIndex>0&&lastRowNumber+1==start) {
            prevRow = rows.get(insertIndex-1);
        }
        Row nextRow = null;
        if(insertIndex<rows.size()&&_r.getRow()==start) {
            nextRow = rows.get(insertIndex);
        }

        // now each following row and cell ref behind our insert position
        // is to be increased by the insertCount
        for(i=insertIndex;i<rows.size();i++) {
            Row r = rows.get(i);
            Long rowRef = r.getR();
            if(rowRef!=null) {
                rowRef += insertCount;
                r.setR(rowRef);
            }
        }

        // now we create and insert our new rows
        for(i=0; i<insertCount; i++) {
            Row newRow = Context.getsmlObjectFactory().createRow();
            newRow.setR(new Long(start+1+i));
            rows.add(insertIndex+i, newRow);
            if(prevRow!=null) {
                newRow.setCollapsed(prevRow.isCollapsed());
                newRow.setCustomFormat(prevRow.isCustomFormat());
                newRow.setCustomHeight(prevRow.isCustomHeight());
                newRow.setHidden(false);
                newRow.setHt(prevRow.getHt());
                newRow.setOutlineLevel(prevRow.getOutlineLevel());
                newRow.setParent(prevRow.getParent());
                newRow.setPh(prevRow.isPh());
                newRow.setS(prevRow.getS());
                newRow.setThickBot(prevRow.isThickBot());
                newRow.setThickTop(prevRow.isThickTop());

                // now cloning cells
                final Iterator<Cell> cellIterator  = prevRow.createCellIterator();
                while(cellIterator.hasNext()) {
                    Cell cell = cellIterator.next();
                    Cell cellClone = newRow.getCell(cell.getColumn(), true);
                    cellClone.setS(cell.getS());

                    // we need to take care of border attributes only if
                    // inserting the first row, second insertion is always
                    // a clone of the previous row
                    if(i==0) {
                        Cell nextRowCell = null;
                        if(nextRow!=null) {
                            nextRowCell = nextRow.getCell(cell.getColumn(), false);
                        }
                        ApplyBorderStyle(stylesheet, cellClone, nextRowCell);
                    }
                }
                prevRow = newRow;
            }
        }
    }

    public void deleteRows(int start, int deleteCount) {

    	final List<Row> rows = getListArray();
    	for(int i=0; i<rows.size(); i++) {
    		final Row r = rows.get(i);
    		final int row = r.getRow();
    		if(row>=start) {
	    		if(row<=start+deleteCount-1) {
	    			rows.remove(i--);
	    		}
	    		else {
	    			r.setRow(row - deleteCount);
	    		}
    		}
    	}
    }

    public void insertColumns(CTStylesheet stylesheet, int start, int insertCount) {
        final Iterator<Row> rowIterator = createRowIterator();
        while(rowIterator.hasNext()) {
            final Row r = rowIterator.next();
            r.insertCells(stylesheet, start, insertCount);
        }
    }

    public void deleteColumns(int start, int deleteCount) {
    	final Iterator<Row> rowIterator = createRowIterator();
    	while(rowIterator.hasNext()) {
    		final Row r = rowIterator.next();
    		r.deleteCells(start, deleteCount);
    	}
    }

    public Iterator<Row> createRowIterator() {
    	if(row!=null) {
    		return row.iterator();
    	}
    	else if (rows_impl!=null) {
    		return rows_impl.values().iterator();
    	}
    	return getListArray().iterator();
    }

    /**
     * Gets the parent object in the object tree representing the unmarshalled xml document.
     *
     * @return
     *     The parent object.
     */
    @Override
    public Object getParent() {
        return this.parent;
    }

    @Override
    public void setParent(Object parent) {
        this.parent = parent;
    }

    public void setSi(long si) {
        if(maxSi==null) {
            maxSi = si;
        }
        else if(si>maxSi) {
            maxSi = si;
        }
    }

    public long getNextSi() {
        if(maxSi==null) {
            maxSi = 0L;
        }
        else {
            maxSi++;
        }
        return maxSi;
    }

    public int getLastRowNumber() {
    	if(row!=null&&!row.isEmpty()) {
    		return row.get(row.size()-1).getRow();
    	}
    	else if (rows_impl!=null&&!rows_impl.isEmpty()) {
    		return rows_impl.lastKey();
    	}
    	return -1;
    }
    
    @SuppressWarnings("unused")
    public void beforeMarshal(Marshaller marshaller) {
        getListArray();
    }
    
    /**
     * This method is invoked by the JAXB implementation on each instance when unmarshalling completes.
     *
     * @param parent
     *     The parent object in the object tree.
     * @param unmarshaller
     *     The unmarshaller that generated the instance.
     */
    public void afterUnmarshal(Unmarshaller unmarshaller, Object _parent) {
        setParent(_parent);
    }
}
