/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.jms.core.plain;

import java.util.Enumeration;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.jms.Destination;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.Queue;
import javax.jms.Session;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openxchange.office_communication.jms.core.plain.beans.JMSBeanMapper;
import com.openxchange.office_communication.jms.core.plain.beans.JMSMessageBean;
import com.openxchange.office_communication.jms.core.plain.beans.JMSMessageUtils;

//=============================================================================
/** Base class providing all you need to work with JMS as client.
 *
 *  It's a 'pure' client sending requests to one queue
 *  and retrieve response messages within another queue.
 *  
 *  All those base on request/response pairs and corresponding queues.
 *  Loops, worker threads, request-response correlation is done by these
 *  helper.
 *  
 *  There exists two modes:
 *  
 *  Mode A )
 *  
 *  YOU define a queue explicit where all outgoing messages has to be sent to.
 *  WE define a temporary queue where responses comes in.
 *  
 *  To enable that mode You have to call:
 *  <code>
 *  setFullQualifiedOutQueue ();
 *  start();
 *  ...
 *  </code>
 *  
 *  Mode B)
 *
 *  YOU define a queue namespace and a queue type for outgoing messages.
 *  WE search for a suitable queue within the current JMS context and use it.
 *  Further WE define a temporary queue where responses comes in.
 *  
 *  To enable that mode You have to call:
 *  <code>
 *  setQueueNamespace ();
 *  setTargetQueueType ();
 *  start();
 *  ...
 *  </code>
 *  
 *  Independent from the mode you use ...
 *  we use setJMSReplyTo() to define our temporary queue where
 *  we expect to get our answer back.
 */
public class JMSClientBase
{
    //-------------------------------------------------------------------------
	public static final Logger LOG = LoggerFactory.getLogger(JMSClientBase.class);
	
    //-------------------------------------------------------------------------
	public static final String QUEUE_TYPE = "simple-client";

    //-------------------------------------------------------------------------
	public static final String QUEUE_NAME_4_RESPONSES = "response";

	//-------------------------------------------------------------------------
	public static final int DEFAULT_TIMEOUT_IN_MS = 60000;
	
    //-------------------------------------------------------------------------
	public JMSClientBase ()
		throws Exception
	{}

	//-------------------------------------------------------------------------
	/** set an unique ID for this client.
	 *  It's up to you what 'unique' means.
	 *  
	 *  If this method is not called ...
	 *  WE generate such unique ID by our own automatically.
	 *
	 *	@param	sID [IN]
	 *			the unique ID.
	 */
	protected void setClientID (final String sID)
	    throws Exception
	{
		m_sClientID = sID;
	}

	//-------------------------------------------------------------------------
	/**
	 */
	protected void setQueueNamespace (final String sNamespace)
	    throws Exception
	{
		m_sQueueNamespace = sNamespace;
	}

	//-------------------------------------------------------------------------
	protected void setTargetQueueType (final String sType)
	    throws Exception
	{
		m_sTargetQueueType = sType;
	}

	//-------------------------------------------------------------------------
	protected void setTimeout (final int nTimeout)
	    throws Exception
	{
		m_nTimeout = nTimeout;
	}

	//-------------------------------------------------------------------------
	public void setFullQualifiedRequestQueue (final String sQueue)
		throws Exception
	{
		m_sFullQualifiedRequestQueue = sQueue;
	}
	
	//-------------------------------------------------------------------------
	public synchronized void start ()
	    throws Exception
	{
		LOG.trace ("start ...");

		if (m_aResponseCollectorThread != null)
		{
			LOG.trace("... already started");
			return;
		}
		impl_registerInQueueContext ();

		m_aResponseCollectorThread = new Thread (new Runnable ()
		{
			@Override
			public void run()
			{
				impl_collectResponses ();
			}
		});
		m_aResponseCollectorThread.start ();

		LOG.trace ("started.");
	}

	//-------------------------------------------------------------------------
	public synchronized void stop ()
	    throws Exception
	{
		LOG.trace ("stop ...");

		// tricky :-)
		// We reset our thread member first ...
		// but then those instance might die ...
		// Holding one last reference within this method scope ...
		// will ensure those instance will live till the end of this method .-)

		@SuppressWarnings("unused")
		Thread aCollectorThreadHold = null;

		synchronized (this)
		{
			if (m_aResponseCollectorThread == null)
				return;
			
			aCollectorThreadHold       = m_aResponseCollectorThread;
			m_aResponseCollectorThread = null;
		}
		
		synchronized (m_aShutdownSync)
		{
			m_aShutdownSync.setValue(true);
			m_aShutdownSync.wait (); // TODO timeout ?
		}
		
		if (m_aClientConsumer != null)
			m_aClientConsumer.close();
		
		if (m_aClientProducer != null)
			m_aClientProducer.close();
		
		if (m_aSession != null)
			m_aSession.close();

		LOG.trace ("stopped.");
	}

	//-------------------------------------------------------------------------
	@SuppressWarnings("unchecked")
	public < T extends JMSMessageBean > T requestSynchronous (final T aRequest)
	    throws Exception
	{
		final Destination               aResponseQueue = mem_ResponseQueue ();
		final MessageProducer           aWrite         = mem_ClientProducer ();
		final JMSRequestResponseManager aRRManager     = mem_RequestResponseManager ();
		final Message                   aJMSRequest    = impl_newMessage ();
		final JMSBeanMapper             aBeanMapper    = mem_BeanMapper ();

		aBeanMapper.mapBeanToMessage(aRequest, aJMSRequest);
		aJMSRequest.setJMSReplyTo(aResponseQueue);

		final JMSRequestResponsePair aResponseSync = aRRManager.registerRequest (aJMSRequest);
		
		LOG.debug ("client : send request ...");
		aWrite.send(aJMSRequest);

		LOG.debug ("client : wait for response ...");
		final boolean bFinished = aResponseSync.await(m_nTimeout, TimeUnit.MILLISECONDS);
		if ( ! bFinished)
		{
			LOG.debug ("client : timeout ...");
			throw new TimeoutException ("No response in tim. timeout=["+m_nTimeout+"]");
		}
		
		LOG.debug ("client : got response ...");
		final Message        aJMSResponse = aResponseSync.getResponse ();
		System.err.println ("##### response : "+aJMSResponse);
		final JMSMessageBean aResponse    = JMSMessageUtils.emptyBeanFitsToMessage(aJMSResponse);
		aBeanMapper.mapMessageToBean(aJMSResponse, aResponse);

		return (T) aResponse;
	}

	//-------------------------------------------------------------------------
	private javax.jms.Message impl_newMessage ()
	    throws Exception
	{
		final Session           aSession    = mem_Session ();
		final javax.jms.Message aJmsMessage = aSession.createTextMessage();
		return aJmsMessage;
	}
	
	//-------------------------------------------------------------------------
	private void impl_collectResponses ()
	{
		try
		{
			LOG.trace ("start ...");
			
			final MessageConsumer           aRead      = mem_ClientConsumer ();
			final JMSRequestResponseManager aRRManager = mem_RequestResponseManager ();
			
			while (true)
			{
				synchronized (m_aShutdownSync)
				{
					if (m_aShutdownSync.booleanValue() == true)
					{
						LOG.trace ("accept shutdown request !");
						m_aShutdownSync.notifyAll();
						break;
					}
				}

				LOG.trace ("poll for response ...");
				final Message aJMSResponse = aRead.receive(200);
				if (aJMSResponse == null)
					continue;

				impl_dumpMessage ("RESPONSE : ", aJMSResponse);

				LOG.trace ("notify response ...");
				aRRManager.registerResponseAndNotify(aJMSResponse);
				
				LOG.trace ("ack response ...");
				aJMSResponse.acknowledge();
			}
		}
		catch (final Throwable ex)
		{
			LOG.error (ex.getMessage (), ex);
			// TODO error handling ?!
		}
	}

	//-------------------------------------------------------------------------
	private void impl_registerInQueueContext ()
	    throws Exception
	{
	    if (
	    	( ! StringUtils.isEmpty(m_sQueueNamespace)) &&
	    	( ! StringUtils.isEmpty(m_sClientID      ))
	       )
	    {
	    	impl_registerInQueueContext4NamespaceMode ();
	    }
	    else
	    if (
		    (StringUtils.isEmpty(m_sQueueNamespace)) ||
		    (StringUtils.isEmpty(m_sClientID      ))
		   )
	    {
	    	impl_registerInQueueContext4TempMode ();
	    }
	    else
	    	throw new UnsupportedOperationException ("No namspace nor any worker ID ... which queue name mode do you want to use ?");

	    LOG.debug ("my consumer ID   = '"+m_sMyConsumerID   +"'");
	    LOG.debug ("my request queue = '"+m_sMyResponseQueue+"'");
	}
	
	//-------------------------------------------------------------------------
	private void impl_registerInQueueContext4NamespaceMode ()
	    throws Exception
	{
		final JMSQueueManager aManager       = JMSQueueManager.create(m_sQueueNamespace);
		final String          sMe            = m_sClientID;
		final String          sResponseQueue = JMSQueueManager.buildQueueName(sMe, QUEUE_NAME_4_RESPONSES);

		aManager.bindConsumer2Queue(sMe           , sResponseQueue);
		aManager.bindQueue2Type    (sResponseQueue, QUEUE_TYPE   );

		m_sMyConsumerID    = sMe           ;
		m_sMyResponseQueue = sResponseQueue;
		m_aQueueManager    = aManager      ;
	}

	//-------------------------------------------------------------------------
	private void impl_registerInQueueContext4TempMode ()
	    throws Exception
	{
		 final JMSQueueManager aManager   = JMSQueueManager.create(null);
		 final Session         aSession   = mem_Session ();
		 final Queue           aTempQueue = (Queue) aSession.createTemporaryQueue();
		 final String          sTempQueue = aTempQueue.getQueueName();
		 final String          sMe        = UUID.randomUUID().toString();
		 
		 aManager.bindConsumer2Queue(sMe       , sTempQueue);
		 aManager.bindQueue2Type    (sTempQueue, QUEUE_TYPE);

		 m_sClientID        = sMe;
		 m_sMyConsumerID    = sMe;
		 m_sMyResponseQueue = sTempQueue;
		 m_aQueueManager    = aManager;
	}

	//-------------------------------------------------------------------------
	private void impl_dumpMessage (final String  sDesc   ,
								   final Message aMessage)
     	throws Exception
	{
		final StringBuffer sDump = new StringBuffer ();
		
		sDump.append (sDesc);
		sDump.append ("\n" );
		
		final Enumeration< ? > lProps = aMessage.getPropertyNames();
		while (lProps.hasMoreElements())
		{
			final String sProp  = (String) lProps.nextElement();
			final String sValue = aMessage.getStringProperty(sProp);
			
			sDump.append ("'"+sProp+"' = '"+sValue+"'\n");
		}
		
		System.out.println (sDump.toString ());
	}
	
	//-------------------------------------------------------------------------
    private synchronized MessageProducer mem_ClientProducer ()
    	throws Exception
    {
    	if (m_aClientProducer == null)
    	{
    		final Session     aSession          = mem_Session ();
    		final Destination aRequestQueue     = mem_RequestQueue ();
    		                  m_aClientProducer = aSession.createProducer(aRequestQueue);
    	}
    	return m_aClientProducer;
    }
    
    //-------------------------------------------------------------------------
    private synchronized MessageConsumer mem_ClientConsumer ()
    	throws Exception
    {
    	if (m_aClientConsumer == null)
    	{
    		final Session     aSession          = mem_Session ();
    		final Destination aResponseQueue    = mem_ResponseQueue ();
    		                  m_aClientConsumer = aSession.createConsumer(aResponseQueue);
    	}
    	return m_aClientConsumer;
    }

    //-------------------------------------------------------------------------
	private synchronized Destination mem_RequestQueue ()
	    throws Exception
	{
		if (m_aRequestQueue == null)
		{
			String sRequestQueueAbsolute = null;
			
			if ( ! StringUtils.isEmpty(m_sFullQualifiedRequestQueue))
			{
				sRequestQueueAbsolute = m_sFullQualifiedRequestQueue;
			}
			else
			{
				final String sRequestQueue         = m_aQueueManager.getBalancedQueue4Type(m_sTargetQueueType);
				             sRequestQueueAbsolute = m_aQueueManager.getQueueNameAbsolute (sRequestQueue     );
			}
			
			final Session aSession        = mem_Session ();
                          m_aRequestQueue = aSession.createQueue(sRequestQueueAbsolute);
            LOG.debug ("request queue was defined as '"+sRequestQueueAbsolute+"' / "+m_aRequestQueue+" ...");
		}
		return m_aRequestQueue;
	}
	
    //-------------------------------------------------------------------------
	private synchronized Destination mem_ResponseQueue ()
	    throws Exception
	{
		if (m_aResponseQueue == null)
		{
			// note : impl_registerInQueueContext () has to be called before this method ...
			// we need m_sMyResponseQueue well defined here .-)

			final Session     aSession               = mem_Session ();
			final String      sResponseQueueAbsolute = m_aQueueManager.getQueueNameAbsolute(m_sMyResponseQueue);
			final Destination aResponseQueue         = aSession.createQueue(sResponseQueueAbsolute);

		    m_aResponseQueue = aResponseQueue;
			LOG.debug ("response queue was defined as '"+sResponseQueueAbsolute+"' ...");
		}
		return m_aResponseQueue;
	}

    //-------------------------------------------------------------------------
    private synchronized Session mem_Session ()
    	throws Exception
    {
    	if (m_aSession == null)
    	{
    		final IJMSClient iClient  = mem_JMSClient ();
    		final Session    aSession = iClient.newSession();
    		m_aSession = aSession;
    	}
    	return m_aSession;
    }
    
    //-------------------------------------------------------------------------
    private synchronized IJMSClient mem_JMSClient ()
        throws Exception
    {
    	if (m_iJMSClient == null)
    		m_iJMSClient = JMSClient.get();
    	return m_iJMSClient;
    }

    //-------------------------------------------------------------------------
	private JMSRequestResponseManager mem_RequestResponseManager ()
	    throws Exception
	{
		if (m_aRequestResponseManager == null)
			m_aRequestResponseManager = new JMSRequestResponseManager ();
		return m_aRequestResponseManager;
	}
	
	//-------------------------------------------------------------------------
    private JMSBeanMapper mem_BeanMapper ()
        throws Exception
    {
    	if (m_aBeanMapper == null)
    		m_aBeanMapper = JMSBeanMapper.get();
    	return m_aBeanMapper;
    }
   
    //-------------------------------------------------------------------------
	private String m_sClientID = null;
	
    //-------------------------------------------------------------------------
    private IJMSClient m_iJMSClient = null;

    //-------------------------------------------------------------------------
    private Session m_aSession = null;

    //-------------------------------------------------------------------------
    private int m_nTimeout = DEFAULT_TIMEOUT_IN_MS;
    
    //-------------------------------------------------------------------------
    private Destination m_aRequestQueue = null;
    
    //-------------------------------------------------------------------------
    private Destination m_aResponseQueue = null;

    //-------------------------------------------------------------------------
    private MessageProducer m_aClientProducer = null;

    //-------------------------------------------------------------------------
    private MessageConsumer m_aClientConsumer = null;

    //-------------------------------------------------------------------------
    private Thread m_aResponseCollectorThread = null;

	//-------------------------------------------------------------------------
	private JMSRequestResponseManager m_aRequestResponseManager = null;

    //-------------------------------------------------------------------------
    private JMSQueueManager m_aQueueManager = null;

    //-------------------------------------------------------------------------
    private String m_sFullQualifiedRequestQueue = null;
    
    //-------------------------------------------------------------------------
    private String m_sQueueNamespace = null;

    //-------------------------------------------------------------------------
    private String m_sMyConsumerID = null;
    
    //-------------------------------------------------------------------------
    private String m_sTargetQueueType = null;

    //-------------------------------------------------------------------------
    private String m_sMyResponseQueue = null;

    //-------------------------------------------------------------------------
    private MutableBoolean m_aShutdownSync = new MutableBoolean (false);
    
	//-------------------------------------------------------------------------
    private JMSBeanMapper m_aBeanMapper = null;
}
