/**
 * OPEN-XCHANGE legal information
 *
 * All intellectual property rights in the Software are protected by
 * international copyright laws.
 *
 *
 * In some countries OX, OX Open-Xchange, open xchange and OXtender
 * as well as the corresponding Logos OX Open-Xchange and OX are registered
 * trademarks of the Open-Xchange, Inc. group of companies.
 * The use of the Logos is not covered by the GNU General Public License.
 * Instead, you are allowed to use these Logos according to the terms and
 * conditions of the Creative Commons License, Version 2.5, Attribution,
 * Non-commercial, ShareAlike, and the interpretation of the term
 * Non-commercial applicable to the aforementioned license is published
 * on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 * Please make sure that third-party modules and libraries are used
 * according to their respective licenses.
 *
 * Any modifications to this package must retain all copyright notices
 * of the original copyright holder(s) for the original code used.
 *
 * After any such modifications, the original and derivative code shall remain
 * under the copyright of the copyright holder(s) and/or original author(s)per
 * the Attribution and Assignment Agreement that can be located at
 * http://www.open-xchange.com/EN/developer/. The contributing author shall be
 * given Attribution for the derivative code and a license granting use.
 *
 *  Copyright (C) 2004-2014 Open-Xchange, Inc.
 *  Mail: info@open-xchange.com
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License, Version 2 as published
 *  by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc., 59
 *  Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package com.openxchange.office_communication.jms.core.plain.beans;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.jms.ObjectMessage;
import javax.jms.TextMessage;

import org.apache.activemq.command.ActiveMQMessage;
import org.apache.activemq.util.ByteSequence;
import org.apache.commons.lang3.StringUtils;

import com.openxchange.office_communication.tools.common.TypeConverter;

//=============================================================================
public class JMSBeanMapper
{
	//-------------------------------------------------------------------------
	private JMSBeanMapper ()
		throws Exception
	{}

	//-------------------------------------------------------------------------
	public static JMSBeanMapper get ()
	    throws Exception
	{
		if (m_gSingleton == null)
			m_gSingleton = new JMSBeanMapper ();
		return m_gSingleton;
	}

	//-------------------------------------------------------------------------
	public < T extends JMSMessageBean > void mapBeanToMessage (final T                        aBean   ,
															   final org.apache.camel.Message aMessage)
	    throws Exception
	{
		aBean.validate ();
		
		final JMSBeanMapping< T > aMapping = impl_getOrCreateBeanMapping (aBean);
		final Set< String >       lHeader  = aMapping.listHeader();
		final Set< String >       lProps   = aMapping.listProperties();
		
		for (final String sHeader : lHeader)
		{
			final Field  aField = aMapping.accessHeader(sHeader);
			final String sValue = impl_toString(aBean, aField);
			
			if (sValue == null) // TODO check for empty strings to ?
				continue;

			aMessage.setHeader(sHeader, sValue);
		}
		
		for (final String sProp : lProps)
		{
			final Field  aField = aMapping.accessProperty(sProp);
			final String sValue = impl_toString(aBean, aField);
			
			if (sValue == null) // TODO check for empty strings to ?
				continue;

			aMessage.setHeader(sProp, sValue);
		}

		final Object aBody = aBean.getBody();
		if (aBody != null)
			aMessage.setBody(aBody);
	}

	//-------------------------------------------------------------------------
	public < T extends JMSMessageBean > void mapBeanToMessage (final T                 aBean   ,
								                               final javax.jms.Message aMessage)
	    throws Exception
	{
		aBean.validate ();
		
		final JMSBeanMapping< T > aMapping = impl_getOrCreateBeanMapping (aBean);
		final Set< String >       lHeader  = aMapping.listHeader();
		final Set< String >       lProps   = aMapping.listProperties();
		
		for (final String sHeader : lHeader)
		{
			final Field  aField = aMapping.accessHeader(sHeader);
			final String sValue = impl_toString(aBean, aField);
			
			if (sValue == null) // TODO check for empty strings to ?
				continue;
			
			if (StringUtils.startsWith(sHeader, "JMS"))
			{
				if (StringUtils.equals(sHeader, JMSHeader.JMSHEADER_MESSAGEID))
					aMessage.setJMSMessageID(sValue);
				else
				if (StringUtils.equals(sHeader, JMSHeader.JMSHEADER_CORRELATIONID))
					aMessage.setJMSCorrelationID(sValue);
				else
					throw new UnsupportedOperationException ("No support for JMS Header '"+sHeader+"' implemented yet.");
			}
			else
			{
				aMessage.setStringProperty(sHeader, sValue);
			}
		}

		for (final String sProp : lProps)
		{
			final Field  aField = aMapping.accessProperty(sProp);
			final String sValue = impl_toString(aBean, aField);
			
			if (sValue == null) // TODO check for empty strings to ?
				continue;
			
			aMessage.setStringProperty(sProp, sValue);
		}
		
		final Object aBody = aBean.getBody();
		if (aBody != null)
		{
			if (aMessage instanceof TextMessage)
				((TextMessage)aMessage).setText(TypeConverter.toString(aBody, aBody.getClass()));
			else
			if (aMessage instanceof ObjectMessage)
				((ObjectMessage)aMessage).setObject((Serializable)aBody);
			else
				throw new UnsupportedOperationException ("no support for message type '"+aMessage.getClass()+"' implemented yet :-)");
		}
	}

	//-------------------------------------------------------------------------
	public < T extends JMSMessageBean > void mapMessageToBean (final org.apache.camel.Message aMessage,
															   final T                        aBean   )
	    throws Exception
	{
		final JMSBeanMapping< T >                 aMapping = impl_getOrCreateBeanMapping (aBean);
		final Iterator< Entry< String, Object > > lHeader  = aMessage.getHeaders().entrySet().iterator();
		
		while (lHeader.hasNext())
		{
			final Entry< String, Object > aHeader = lHeader.next ();
			final String                  sHeader = aHeader.getKey();
			final Object                  aValue  = aHeader.getValue();
			final Field                   aField  = aMapping.accessHeader(sHeader);

			if (aField == null)
				continue;
			
			final Object aRealValue = impl_fromString (aValue, aField);
			aField.set(aBean, aRealValue);
		}
		
		final Object aBody = aMessage.getBody();
		if (aBody != null)
			aBean.setBody((Serializable)aBody); // let it crash .-)
	}

	//-------------------------------------------------------------------------
	@SuppressWarnings("unchecked")
	public < T extends JMSMessageBean > void mapMessageToBean (final javax.jms.Message aMessage,
															   final T                 aBean   )
	    throws Exception
	{
		final JMSBeanMapping< T >   aMapping = impl_getOrCreateBeanMapping (aBean);
		final Enumeration< String > lHeader  = aMessage.getPropertyNames();
		
		// handle custom header
		
		while (lHeader.hasMoreElements())
		{
			final String sHeader = lHeader.nextElement();
			final String sValue  = aMessage.getStringProperty(sHeader);
			final Field  aField  = aMapping.accessHeader(sHeader);

			if (aField == null)
				continue;
			
			final Object aValue = impl_fromString (sValue, aField);
			aField.set(aBean, aValue);
		}
		
		// handle special JMS header
		
		      Field  aField            = null;
		final String sJMSMessageID     = aMessage.getJMSMessageID();
		final String sJMSCorrelationID = aMessage.getJMSCorrelationID();
		
		aField = aMapping.accessHeader(JMSHeader.JMSHEADER_MESSAGEID);
		if (aField != null && ! StringUtils.isEmpty(sJMSMessageID))
			aField.set(aBean, sJMSMessageID);

		aField = aMapping.accessHeader(JMSHeader.JMSHEADER_CORRELATIONID);
		if (aField != null && ! StringUtils.isEmpty(sJMSCorrelationID))
			aField.set(aBean, sJMSCorrelationID);

		// handle body

		if (aMessage instanceof TextMessage)
		{
			final String sBody = ((TextMessage)aMessage).getText();
			if ( ! StringUtils.isEmpty(sBody))
				aBean.setBody(sBody);
		}
		else
		if (aMessage instanceof ObjectMessage)
		{
			final Object aBody = ((ObjectMessage)aMessage).getObject();
			if (aBody != null)
				aBean.setBody((Serializable)aBody);
		}
		else
		if (aMessage instanceof ActiveMQMessage)
		{
			final ByteSequence aBody = ((ActiveMQMessage)aMessage).getContent();
			if (aBody != null)
			{
				final String sContent = new String(aBody.getData(), "utf-8");
				aBean.setBody(sContent);
			}
		}
		else
			throw new UnsupportedOperationException ("No support for message type '"+aMessage.getClass ()+"' implemented yet.");
	}

	//-------------------------------------------------------------------------
	private String impl_toString (final Object aBean ,
								  final Field  aField)
	    throws Exception
	{
		final Class< ? > aType   = aField.getType();
		final Object     aValue  = aField.get(aBean);
		final String     sString = TypeConverter.toString(aValue, aType);
		return sString;
	}
	
	//-------------------------------------------------------------------------
	private Object impl_fromString (final Object aValue,
								    final Field  aField)
	    throws Exception
	{
		if (aValue == null)
			return null;
		
		if ( ! aValue.getClass().isAssignableFrom(String.class))
			return aValue;
		
		final Class< ? > aType      = aField.getType();
		final Object     aRealValue = TypeConverter.fromString((String)aValue, aType);
		return aRealValue;
	}

	//-------------------------------------------------------------------------
	@SuppressWarnings("unchecked")
	private < T extends JMSMessageBean > JMSBeanMapping< T > impl_getOrCreateBeanMapping (final T aBean)
	    throws Exception
	{
		final Map< String, JMSBeanMapping< ? extends JMSMessageBean > > aRegistry  = mem_MappingRegistry ();
		final Class< T >                                                aBeanClass = (Class< T >)aBean.getClass ();
		final String                                                    sBeanClass = aBeanClass.getName ();
		      JMSBeanMapping< ? extends JMSMessageBean >                aMapping   = aRegistry.get(sBeanClass);
		
		if (aMapping == null)
		{
			aMapping = JMSBeanMapping.create(aBeanClass);
			aRegistry.put(sBeanClass, aMapping);
		}
		
		return (JMSBeanMapping< T >) aMapping;
	}
	
	//-------------------------------------------------------------------------
	private Map< String, JMSBeanMapping< ? extends JMSMessageBean > > mem_MappingRegistry ()
	    throws Exception
	{
		if (m_aMappingRegistry == null)
			m_aMappingRegistry = new HashMap< String, JMSBeanMapping< ? > >();
		return m_aMappingRegistry;
	}
	
	//-------------------------------------------------------------------------
	private static JMSBeanMapper m_gSingleton = null;
	
	//-------------------------------------------------------------------------
	private Map< String, JMSBeanMapping< ? extends JMSMessageBean > > m_aMappingRegistry = null;
}
